/*
 * Copyright © 2015 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Alexandros Frantzis <alexandros.frantzis@canonical.com>
 */

#include "unity_display_service.h"
#include "screen.h"
#include "dbus_message_handle.h"
#include "dbus_event_loop.h"
#include "dbus_connection_handle.h"
#include "scoped_dbus_error.h"

#include "unity_display_service_introspection.h" // autogenerated

namespace
{

char const* const dbus_display_interface = "com.canonical.Unity.Display";
char const* const dbus_display_path = "/com/canonical/Unity/Display";
char const* const dbus_display_service_name = "com.canonical.Unity.Display";

}

usc::UnityDisplayService::UnityDisplayService(
    std::shared_ptr<usc::DBusEventLoop> const& loop,
    std::string const& address,
    std::shared_ptr<usc::Screen> const& screen)
    : screen{screen},
      loop{loop},
      connection{std::make_shared<DBusConnectionHandle>(address.c_str())}
{
    loop->add_connection(connection);
    connection->request_name(dbus_display_service_name);
    connection->add_filter(handle_dbus_message_thunk, this);
}

::DBusHandlerResult usc::UnityDisplayService::handle_dbus_message_thunk(
    ::DBusConnection* connection, DBusMessage* message, void* user_data)
{
    auto const dbus_display_service = static_cast<usc::UnityDisplayService*>(user_data);
    return dbus_display_service->handle_dbus_message(connection, message, user_data);
}

DBusHandlerResult usc::UnityDisplayService::handle_dbus_message(
    ::DBusConnection* connection, DBusMessage* message, void* user_data)
{
    ScopedDBusError args_error;

    if (dbus_message_is_method_call(message, "org.freedesktop.DBus.Introspectable", "Introspect"))
    {
        DBusMessageHandle reply{
            dbus_message_new_method_return(message),
            DBUS_TYPE_STRING, &unity_display_service_introspection,
            DBUS_TYPE_INVALID};

        dbus_connection_send(connection, reply, nullptr);
    }
    else if (dbus_message_is_method_call(message, dbus_display_interface, "TurnOn"))
    {
        dbus_TurnOn();

        DBusMessageHandle reply{dbus_message_new_method_return(message)};
        dbus_connection_send(connection, reply, nullptr);
    }
    else if (dbus_message_is_method_call(message, dbus_display_interface, "TurnOff"))
    {
        dbus_TurnOff();

        DBusMessageHandle reply{dbus_message_new_method_return(message)};
        dbus_connection_send(connection, reply, nullptr);
    }
    else if (dbus_message_get_type(message) == DBUS_MESSAGE_TYPE_METHOD_CALL)
    {
         DBusMessageHandle reply{
             dbus_message_new_error(message, DBUS_ERROR_FAILED, "Not supported")};

        dbus_connection_send(connection, reply, nullptr);
    }

    if (args_error)
    {
         DBusMessageHandle reply{
             dbus_message_new_error(message, DBUS_ERROR_FAILED, "Invalid arguments")};

        dbus_connection_send(connection, reply, nullptr);
    }

    return DBUS_HANDLER_RESULT_HANDLED;
}

void usc::UnityDisplayService::dbus_TurnOn()
{
    screen->turn_on();
}

void usc::UnityDisplayService::dbus_TurnOff()
{
    screen->turn_off();
}
