/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "shellserverconfiguration.h"

#include "initialsurfaceplacementstrategy.h"
#include "sessionlistener.h"
#include "surfaceconfigurator.h"
#include "surfacesource.h"
#include "sessionauthorizer.h"
#include "surfacecontroller.h"
#include "focussetter.h"
#include "logging.h"

// mir
#include "mir/shell/organising_surface_factory.h"

namespace msh = mir::shell;
namespace ms = mir::surfaces;

ShellServerConfiguration::ShellServerConfiguration(int argc, char const* argv[], QObject* parent)
    : QObject(parent)
    , DefaultServerConfiguration(argc, argv)
{
    DLOG("ShellServerConfiguration created");
}

ShellServerConfiguration::~ShellServerConfiguration()
{
}

std::shared_ptr<msh::PlacementStrategy>
ShellServerConfiguration::the_shell_placement_strategy()
{
    return shell_placement_strategy(
        [this]
        {
            return std::make_shared<InitialSurfacePlacementStrategy>(the_shell_display_layout());
        });
}

std::shared_ptr<msh::SessionListener>
ShellServerConfiguration::the_shell_session_listener()
{
    return shell_session_listener(
        [this]
        {
            return std::make_shared<SessionListener>();
        });
}

std::shared_ptr<msh::SurfaceConfigurator>
ShellServerConfiguration::the_shell_surface_configurator()
{
    return shell_surface_configurator(
        [this]()
        {
            return std::make_shared<SurfaceConfigurator>();
        });
}

std::shared_ptr<ms::SurfaceController>
ShellServerConfiguration::the_surface_controller()
{
    return surface_controller(
        [this]()
        {
            return std::make_shared<SurfaceController>(the_surface_stack_model());
        });
}

std::shared_ptr<mir::frontend::SessionAuthorizer>
ShellServerConfiguration::the_session_authorizer()
{
    return session_authorizer(
    [this]()
    {
        return std::make_shared<SessionAuthorizer>(the_session_manager());
    });
}

// FIXME: Needed to detect the shell's surface - there must be a better way
std::shared_ptr<msh::SurfaceFactory>
ShellServerConfiguration::the_shell_surface_factory()
{
    return shell_surface_factory(
        [this]()
        {
            m_surfaceSource = std::make_shared<SurfaceSource>(
                the_surface_builder(), the_shell_surface_configurator());

            return std::make_shared<msh::OrganisingSurfaceFactory>(
                m_surfaceSource,
                the_shell_placement_strategy());
        });
}

std::shared_ptr<msh::FocusSetter>
ShellServerConfiguration::the_shell_focus_setter()
{
    return shell_focus_setter(
       [this]
       {
           return std::make_shared<FocusSetter>(DefaultServerConfiguration::the_shell_focus_setter(), the_input_targeter());
       });
}

/************************************ Shell side ************************************/
SessionAuthorizer *ShellServerConfiguration::sessionAuthorizer()
{
    SessionAuthorizer *ptr = static_cast<SessionAuthorizer*>(the_session_authorizer().get());
    return q_check_ptr(ptr); //CHECKME will this always be ready before shell comes up?
}

SessionListener *ShellServerConfiguration::sessionListener()
{
    SessionListener *ptr = static_cast<SessionListener*>(the_shell_session_listener().get());
    return q_check_ptr(ptr); //CHECKME will this always be ready before shell comes up?
}

SurfaceConfigurator *ShellServerConfiguration::surfaceConfigurator()
{
    SurfaceConfigurator *ptr = static_cast<SurfaceConfigurator*>(the_shell_surface_configurator().get());
    return q_check_ptr(ptr); //CHECKME will this always be ready before shell comes up?
}

SurfaceSource *ShellServerConfiguration::surfaceSource()
{
    return m_surfaceSource.get();
}

FocusSetter *ShellServerConfiguration::focusSetter()
{
    return static_cast<FocusSetter*>(the_shell_focus_setter().get());
}
