/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Ricardo Mendoza <ricardo.mendoza@canonical.com>
 */

// local
#include "taskcontroller.h"

// unity-mir
#include <logging.h>

// Qt
#include <QStringList>

// glib
#include <glib.h>

// std
#include <signal.h>
#include <unistd.h>


TaskController* TaskController::m_theTaskController = nullptr;

TaskController* TaskController::singleton()
{
    if (!m_theTaskController) {
        m_theTaskController = new TaskController();
    }
    return m_theTaskController;
}

TaskController::TaskController(QObject *parent) :
    QObject(parent)
{
    startCallback = [](const gchar * appId, gpointer userData) {
        Q_UNUSED(userData)
        Q_EMIT TaskController::singleton()->processStartReport(QString(appId), false);
    };

    stopCallback = [](const gchar * appId, gpointer userData) {
        Q_UNUSED(userData)
        Q_EMIT TaskController::singleton()->processStopped(QString(appId), false);
    };

    focusCallback = [](const gchar * appId, gpointer userData) {
        Q_UNUSED(userData)
        Q_EMIT TaskController::singleton()->requestFocus(QString(appId));
    };

    resumeCallback = [](const gchar * appId, gpointer userData) {
        Q_UNUSED(userData)
        Q_EMIT TaskController::singleton()->requestResume(QString(appId));
    };

    failureCallback = [](const gchar * appId, upstart_app_launch_app_failed_t failureType, gpointer userData) {
        Q_UNUSED(userData)
        if (failureType == UPSTART_APP_LAUNCH_APP_FAILED_CRASH) {
            Q_EMIT TaskController::singleton()->processStopped(QString(appId), true);
        } else if (failureType == UPSTART_APP_LAUNCH_APP_FAILED_START_FAILURE) {
            Q_EMIT TaskController::singleton()->processStartReport(QString(appId), true);
        } else {
            LOG("TaskController: unknown failure type returned from upstart-app-launch");
        }
        Q_EMIT TaskController::singleton()->requestResume(QString(appId));
    };

    upstart_app_launch_observer_add_app_start(startCallback, nullptr);
    upstart_app_launch_observer_add_app_stop(stopCallback, nullptr);
    upstart_app_launch_observer_add_app_focus(focusCallback, nullptr);
    upstart_app_launch_observer_add_app_resume(resumeCallback, nullptr);
    upstart_app_launch_observer_add_app_failed(failureCallback, nullptr);
}

TaskController::~TaskController()
{
    upstart_app_launch_observer_delete_app_start(startCallback, nullptr);
    upstart_app_launch_observer_delete_app_stop(stopCallback, nullptr);
    upstart_app_launch_observer_delete_app_focus(focusCallback, nullptr);
    upstart_app_launch_observer_delete_app_resume(resumeCallback, nullptr);
    upstart_app_launch_observer_delete_app_failed(failureCallback, nullptr);
}

bool TaskController::start(const QString& appId, const QStringList& arguments)
{
    DLOG("TaskController::start appId='%s'", qPrintable(appId));
    gchar ** upstartArgs = nullptr;
    bool result = false;

    // Convert arguments QStringList into format suitable for upstart-app-launch
    upstartArgs = g_new0(gchar *, arguments.length());

    for (int i=0; i<arguments.length(); i++) {
        upstartArgs[i] = arguments.at(i).toLatin1().data();
    }

    result = upstart_app_launch_start_application(appId.toLatin1().constData(),
                                                  static_cast<const gchar * const *>(upstartArgs));
    g_free(upstartArgs);

    DLOG_IF(!result, "TaskController::startApplication appId='%s' FAILED", qPrintable(appId));
    return result;
}

bool TaskController::stop(const QString& appId)
{
    DLOG("TaskController::stop appId='%s'", qPrintable(appId));
    bool result = false;

    result = upstart_app_launch_stop_application(appId.toLatin1().constData());

    DLOG_IF(!result, "TaskController::stopApplication appId='%s' FAILED", qPrintable(appId));
    return result;
}

bool TaskController::appIdHasProcessId(const QString& appId, const quint64 pid)
{
    DLOG("TaskController::isApplicationPid appId='%s', pid=%lld", qPrintable(appId), pid);
    return upstart_app_launch_pid_in_app_id(pid, appId.toLatin1().constData());
}

bool TaskController::suspend(const QString& appId)
{
    DLOG("TaskController::suspend (this=%p, application=%p)", this, qPrintable(appId));
    pid_t pid = upstart_app_launch_get_primary_pid(appId.toLatin1().constData());

    if (pid) {
        kill(pid, SIGSTOP);
        return true;
    } else {
        return false;
    }
}

bool TaskController::resume(const QString& appId)
{
    DLOG("TaskController::resume (this=%p, application=%p)", this, qPrintable(appId));
    pid_t pid = upstart_app_launch_get_primary_pid(appId.toLatin1().constData());

    if (pid) {
        kill(pid, SIGCONT);
        return true;
    } else {
        return false;
    }
}
