/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ubuntukeyboardinfo.h"

#include <QTimer>

namespace {
    const int gConnectionAttemptIntervalMs = 5000;
    const int gMaxFailedAttempts = 10;
    const char gServerName[] = "ubuntu-keyboard-info";
}

UbuntuKeyboardInfo::UbuntuKeyboardInfo(QObject *parent)
    : QObject(parent),
    m_failedAttempts(0),
    m_lastWidth(0),
    m_lastHeight(0)
{
    connect(&m_socket, &QLocalSocket::connected, this, &UbuntuKeyboardInfo::onConnected);
    connect(&m_socket, &QLocalSocket::disconnected, this, &UbuntuKeyboardInfo::onDisconnected);
    connect(&m_socket, &QIODevice::readyRead,
            this, &UbuntuKeyboardInfo::readAllBytesFromSocket);

    typedef void (QLocalSocket::*MemberFunctionType)(QLocalSocket::LocalSocketError);
    MemberFunctionType funcPointer = &QLocalSocket::error;
    connect(&m_socket, funcPointer,
            this, &UbuntuKeyboardInfo::onSocketError);

    tryConnectingToServer();
}

void UbuntuKeyboardInfo::tryConnectingToServer()
{
    m_socket.connectToServer(gServerName, QIODevice::ReadOnly);
}

void UbuntuKeyboardInfo::onConnected()
{
    m_failedAttempts = 0;
}

void UbuntuKeyboardInfo::onDisconnected()
{
    qWarning("UbuntuKeyboardInfo - disconnected");
}

void UbuntuKeyboardInfo::onSocketError(QLocalSocket::LocalSocketError socketError)
{
    // Polling every gConnectionAttemptIntervalMs. Not the best approach but could be worse.
    Q_UNUSED(socketError);
    ++m_failedAttempts;
    if (m_failedAttempts < gMaxFailedAttempts) {
        QTimer::singleShot(gConnectionAttemptIntervalMs, this, SLOT(tryConnectingToServer()));
    } else {
        qCritical() << "Failed to connect to" << gServerName << "after"
                    << m_failedAttempts << "failed attempts";
    }
}

void UbuntuKeyboardInfo::readAllBytesFromSocket()
{
    while (m_socket.bytesAvailable() > 0) {
        readInfoFromSocket();
    }
}

void UbuntuKeyboardInfo::readInfoFromSocket()
{
    const size_t sharedInfoSize = sizeof(struct SharedInfo);
    QByteArray bytes = m_socket.read(sharedInfoSize);
    if (bytes.size() != sharedInfoSize) {
        qWarning() << "UbuntuKeyboardInfo: expected to receive" << sharedInfoSize
                   << "but got" << bytes.size();
        return;
    }

    {
        struct SharedInfo *sharedInfo = reinterpret_cast<struct SharedInfo*>(bytes.data());
        if (m_lastWidth != sharedInfo->keyboardWidth) {
            m_lastWidth = sharedInfo->keyboardWidth;
            Q_EMIT widthChanged(m_lastWidth);
        }

        if (m_lastHeight != sharedInfo->keyboardHeight) {
            m_lastHeight = sharedInfo->keyboardHeight;
            Q_EMIT heightChanged(m_lastHeight);
        }
    }
}
