/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// local
#include "application_manager.h"
#include "application.h"
#include "desktopfilereader.h"
#include "dbuswindowstack.h"

// unity-mir
#include "qmirserverapplication.h"
#include "shellserverconfiguration.h"
#include "sessionlistener.h"
#include "sessionauthorizer.h"
#include "logging.h"

// mir
#include <mir/shell/session_manager.h>

// Qt
#include <QCoreApplication>
#include <QProcess>

#define USE_UPSTART 0

#if !USE_UPSTART
  #include <pwd.h>
#endif

namespace msh = mir::shell;

using namespace unity::shell::application;

ApplicationManager::ApplicationManager(QObject *parent)
:   ApplicationManagerInterface(parent)
,   m_focusedApplication(NULL)
{
    DLOG("ApplicationManager::ApplicationManager (this=%p)", this);

    QMirServerApplication* mirServerApplication = dynamic_cast<QMirServerApplication*>(QCoreApplication::instance());
    if (mirServerApplication == NULL) {
        LOG("Need to use QMirServerApplication");
        QCoreApplication::quit();
        return;
    }
    m_mirServer = mirServerApplication->server();

    QObject::connect(m_mirServer->sessionListener(), &SessionListener::sessionStarting,
                     this, &ApplicationManager::onSessionStarting);
    QObject::connect(m_mirServer->sessionListener(), &SessionListener::sessionStopping,
                     this, &ApplicationManager::onSessionStopping);
    QObject::connect(m_mirServer->sessionListener(), &SessionListener::sessionFocused,
                     this, &ApplicationManager::onSessionFocused);
    QObject::connect(m_mirServer->sessionListener(), &SessionListener::sessionUnfocused,
                     this, &ApplicationManager::onSessionUnfocused);
    QObject::connect(m_mirServer->sessionListener(), &SessionListener::sessionCreatedSurface,
                     this, &ApplicationManager::onSessionCreatedSurface);
    QObject::connect(m_mirServer->sessionAuthorizer(), &SessionAuthorizer::requestAuthorizationForSession,
                     this, &ApplicationManager::authorizeSession, Qt::BlockingQueuedConnection);

    // we use random numbers, so seed the generator
    qsrand(QTime::currentTime().msec());

    m_dbusWindowStack = new DBusWindowStack(this);
    m_taskController = new TaskController(this);
}

ApplicationManager::~ApplicationManager()
{
    DLOG("ApplicationManager::~ApplicationManager");
}

int ApplicationManager::rowCount(const QModelIndex &parent) const
{
    return !parent.isValid() ? m_applications.size() : 0;
}

QVariant ApplicationManager::data(const QModelIndex &index, int role) const
{
    if (index.row() >= 0 && index.row() < m_applications.size()) {
        Application *application = m_applications.at(index.row());
        switch (role) {
            case RoleAppId:
                return QVariant::fromValue(application->appId());
            case RoleName:
                return QVariant::fromValue(application->name());
            case RoleComment:
                return QVariant::fromValue(application->comment());
            case RoleIcon:
                return QVariant::fromValue(application->icon());
            case RoleStage:
                return QVariant::fromValue((int)application->stage());
            case RoleState:
                return QVariant::fromValue((int)application->state());
            case RoleFocused:
                return QVariant::fromValue(application->focused());
            default:
                return QVariant();
        }
    } else {
        return QVariant();
    }
}

Application* ApplicationManager::get(int index) const
{
    return m_applications.at(index);
}

Application* ApplicationManager::findApplication(const QString &appId) const
{
    for (Application *app : m_applications) {
        if (app->appId() == appId) {
            return app;
        }
    }
    return nullptr;
}

QString ApplicationManager::focusedApplicationId() const
{
    if (m_focusedApplication) {
        return m_focusedApplication->appId();
    } else {
        return QString();
    }
}

bool ApplicationManager::focusApplication(const QString &appId)
{
    DLOG("ApplicationManager::focusApplication (this=%p, appId=%s)", this, appId.toLatin1().data());
    Application *application = findApplication(appId);

    m_mirServer->the_session_manager()->set_focus_to(application->session());

    // FIXME(dandrader): lying here. The operation is async. So we will only know whether
    // the focusing was successful once the server replies. Maybe the API in unity-api should
    // reflect that?
    return true;
}

void ApplicationManager::unfocusCurrentApplication()
{
    DLOG("ApplicationManager::unfocusCurrentApplication (this=%p)", this);

    m_mirServer->the_session_manager()->set_focus_to(NULL); //FIXME(greyback)
}

Application* ApplicationManager::respawnApplication(Application* application)
{
    DLOG("ApplicationManager::respawnApplication(this=%p, application=%p)", this, application);

    // Start process - set correct environment
    setenv("QT_QPA_PLATFORM", "ubuntumirclient", 1);

    bool result;
    qint64 pid = 0;
    struct passwd* passwd = getpwuid(getuid());
    DLOG("current working directory: '%s' - args='%s'", passwd ? passwd->pw_dir : "/", application->m_arguments.join(' ').toLatin1().data());
    QProcess builder;
    builder.setProcessChannelMode(QProcess::ForwardedChannels);
    QString exec(application->m_arguments[0]);
    application->m_arguments.removeFirst();
    result = builder.startDetached(exec, application->m_arguments, QString(passwd ? passwd->pw_dir : "/"), &pid);
    application->m_arguments.prepend(exec);
    DLOG_IF(result == false, "process failed to start");
    if (result) {
        // Set existing application's pid to new instance
        application->setPid(pid);

        // Push to front
        if (m_applications.count() > 1)
            move(m_applications.indexOf(application), 0);

        DLOG("builder '%s' respawned with pid %lld", application->name().toLatin1().data(), pid);
        return application;
    } else {
        DLOG("builder '%s' failed to respawn", application->name().toLatin1().data());
        return NULL;
    }
}

Application* ApplicationManager::startApplication(const QString &appId,
                                                  const QStringList &arguments)
{
    return startApplication(appId, NoFlag, arguments);
}

Application *ApplicationManager::startApplication(const QString &appId, ExecFlags flags,
                                                  const QStringList &constArgs)
{
    DLOG("ApplicationManager::startApplication (this=%p, appId=%s)", this, appId.toLatin1().data());

    QStringList arguments = constArgs;

    // Load desktop file.
    QString desktopFile = QString("/usr/share/applications/%1.desktop").arg(appId);
    DesktopFileReader* desktopData = new DesktopFileReader(desktopFile);
    if (!desktopData->loaded()) {
        delete desktopData;
        return NULL;
    }

#if USE_UPSTART
    /* launch via upstart */
    arguments.prepend("APP_ID=" + desktopData->appId());
    arguments.prepend("application");

    // Start process.
    //FIXME(greyback) use upstart via DBUS
    QProcess process;
    process.start("start", arguments);
    //FIXME(greyback) this is blocking, instead listen for finished() signal
    process.waitForFinished(1000);

    DLOG_IF(process.error(), "process 'start application' failed to start with error: %s", process.errorString().toLatin1().data());
    if (!process.error()) {
        // fetch pid
        QString output = process.readAllStandardOutput();
        int lastSpace = output.lastIndexOf(' ');
        qint64 pid = output.remove(0, lastSpace).toInt();
#else
    /* Launch manually */

    // Format arguments.
    // FIXME(loicm) Special field codes are simply ignored for now.
    //     http://standards.freedesktop.org/desktop-entry-spec/latest/ar01s06.html
    QStringList execArguments = desktopData->exec().split(" ", QString::SkipEmptyParts);
    DASSERT(execArguments.size() > 0);
    QString exec(execArguments[0]);
    const int kSize = execArguments.size();
    for (int i = kSize - 1; i > 0; i--) {
        if ((execArguments[i].size() == 2) && (execArguments[i][0].toLatin1() == '%')) {
            const char kChar = execArguments[i][1].toLatin1();
            if (kChar == 'F' || kChar == 'u' || kChar == 'U' || kChar == 'd' || kChar == 'D'
                    || kChar == 'n' || kChar == 'N' || kChar == 'i' || kChar == 'c' || kChar == 'k'
                    || kChar == 'v' || kChar == 'm') {
                continue;
            }
        }
        arguments.prepend(execArguments[i]);
    }
    arguments.append(QString("--desktop_file_hint=") + desktopData->file());
    if (flags.testFlag(ApplicationManager::ForceMainStage))
        arguments.append(QString("--stage_hint=main_stage"));
    else if (desktopData->stageHint() == "SideStage")
        arguments.append(QString("--stage_hint=side_stage"));

    // Start process - set correct environment
    setenv("QT_QPA_PLATFORM", "ubuntumirclient", 1);

    bool result;
    qint64 pid = 0;
    QString path = "/";
    // respect Path from .desktop file
    if (desktopData->path() != "") {
        path = desktopData->path();
    } else {
        struct passwd* passwd = getpwuid(getuid());
        if (passwd)
            path = passwd->pw_dir;
    }
    DLOG("current working directory: '%s'", path.toLatin1().data());
        QProcess builder;
    builder.setProcessChannelMode(QProcess::ForwardedChannels);
    result = builder.startDetached(exec, arguments, path, &pid);

    DLOG_IF(result == false, "process failed to start");
    if (result) {
#endif
        DLOG("process started with pid %lld, adding '%s' to application lists", pid, desktopData->name().toLatin1().data());
        arguments.prepend(exec);
        Application* application = new Application(
                    desktopData, pid,
                    (flags.testFlag(ApplicationManager::ForceMainStage)
                        || desktopData->stageHint() != "SideStage")
                        ? Application::MainStage : Application::SideStage,
                    Application::Starting, //FIXME(greyback): assuming running immediately
                    arguments,
                    m_taskController
                    );

        add(application);
        return application;
    } else {
        return NULL;
    }
}

bool ApplicationManager::stopApplication(const QString &appId)
{
    DLOG("ApplicationManager::stopApplication (this=%p, appId=%s)", this, qPrintable(appId));

    Application *application = findApplication(appId);

    if (application != NULL) {
        if (application == m_focusedApplication) {
            // TODO(greyback) What to do?? Focus next app, or unfocus everything??
            m_focusedApplication = NULL;
            Q_EMIT focusedApplicationIdChanged();
        }
        remove(application);
        m_dbusWindowStack->WindowDestroyed(0, application->name());

        // Start process.
        bool result;

#if USE_UPSTART
        result = QProcess::startDetached("stop", QStringList() << "application"
                                         << ("APP_ID=" + application->appId()));
        DLOG_IF(result == false, "process 'stop application' failed to execute");
#else
        result = QProcess::startDetached("/bin/kill", QStringList() << "-9" << QString::number(application->m_pid));
        DLOG_IF(result == false, "process '/bin/kill -9 %lld' failed to execute", application->m_pid);
#endif
        LOG_IF(!result, "ApplicationManager: Unable to stop process '%s'",
               application->name().toLatin1().constData());
        delete application;
    }

    // FIXME(dandrader): lying here. The operation is async. So we will only know whether
    // the focusing was successful once the server replies. Maybe the API in unity-api should
    // reflect that?
    return true;
}

/************************************* Mir-side methods *************************************/

void ApplicationManager::authorizeSession(const quint64 pid, bool &authorized)
{
    authorized = false; //to be proven wrong

    DLOG("ApplicationManager::authorizeSession (this=%p, pid=%lld)", this, pid);
    Application* application = findApplicationWithPid(pid);
    if (application) {
        QString name = application->appId() + QString::number(qrand());
        application->setSessionName(name);
        authorized = true;
        return;
    }

    /*
     * Hack: Allow applications to be launched externally, but must be executed with the
     * "desktop_file_hint" parameter attached. This exists until upstart-app-launch can
     * notify shell it is starting an application and so shell should allow it. Also reads
     * the --stage parameter to determine the desired stage
     */
    QFile cmdline(QString("/proc/%1/cmdline").arg(pid));
    if (!cmdline.open(QIODevice::ReadOnly | QIODevice::Text)) {
        DLOG("ApplicationManager REJECTED connection from app with pid %lld as unable to read process command", pid);
        return;
    }

    QByteArray command = cmdline.readLine().replace('\0', ' ');

    // FIXME: special exception for the OSK - maliit-server - not very secure
    if (command.startsWith("maliit-server")) {
        authorized = true;
        return;
    }

    QString pattern = QRegularExpression::escape("--desktop_file_hint=") + "(\\S+)";
    QRegularExpression regExp(pattern);
    QRegularExpressionMatch regExpMatch = regExp.match(command);

    if (!regExpMatch.hasMatch()) {
        LOG("ApplicationManager REJECTED connection from app with pid %lld as no desktop_file_hint specified", pid);
        return;
    }

    QString desktopFileName = regExpMatch.captured(1);
    DLOG("Process supplied desktop_file_hint, loading '%s'", desktopFileName.toLatin1().data());

    DesktopFileReader* desktopData = new DesktopFileReader(desktopFileName);
    if (!desktopData->loaded()) {
        delete desktopData;
        LOG("ApplicationManager REJECTED connection from app with pid %lld as desktop_file_hint file not found", pid);
        return;
    }

    // some naughty applications use a script to launch the actual application. Check for the
    // case where shell actually launched the script.
    application = findApplication(desktopData->appId());
    if (application && application->state() == Application::Starting) {
        DLOG("Process with pid %lld appeared, attached to existing entry '%s' in application lists",
             pid, application->appId().toLatin1().data());
        delete desktopData;
        QString name = application->appId() + QString::number(qrand());
        application->setSessionName(name);
        authorized = true;
        return;
    }

    // if stage supplied in CLI, fetch that
    Application::Stage stage = Application::MainStage;
    pattern = QRegularExpression::escape("--stage=") + "(\\S+)";
    regExp.setPattern(pattern);
    regExpMatch = regExp.match(command);

    if (regExpMatch.hasMatch() && regExpMatch.captured(1) == "side_stage") {
        stage = Application::SideStage;
    }

    DLOG("Existing process with pid %lld appeared, adding '%s' to application lists", pid, desktopData->name().toLatin1().data());

    QString argStr(command.data());
    QStringList arguments(argStr.split(' '));
    application = new Application(desktopData, pid, stage, Application::Starting, arguments, m_taskController);
    add(application);
    authorized = true;
}

void ApplicationManager::onSessionStarting(std::shared_ptr<msh::ApplicationSession> const& session)
{
    DLOG("ApplicationManager::onSessionStarting (this=%p, application=%s)", this, session->name().c_str());

    //FIXME(greyback) Mir not supplying any identifier that we can use to link the PID to the session
    // so am assuming that the *most recently* launched application session is the one that connects
    Application* application = findLastExecutedApplication();
    if (application && application->state() != Application::Running) {
        application->setSession(session);
    } else {
        DLOG("ApplicationManager::onSessionStarting - unauthorized application!!");
    }
}

void ApplicationManager::onSessionStopping(std::shared_ptr<msh::ApplicationSession> const& session)
{
    DLOG("ApplicationManager::onSessionStopping (this=%p, application=%s)", this, session->name().c_str());

    // in case application closed not by hand of shell, check again here:
    Application* application = findApplicationWithSession(session);
    if (application) {
        if (application == m_focusedApplication) {
            // TODO(greyback) What to do?? Focus next app, or unfocus everything??
            m_focusedApplication = NULL;
            remove(application);
            Q_EMIT focusedApplicationIdChanged();
        }
        application->setState(Application::Stopped);
        m_dbusWindowStack->WindowDestroyed(0, application->name());
    }
}

void ApplicationManager::onSessionFocused(std::shared_ptr<msh::ApplicationSession> const& session)
{
    DLOG("ApplicationManager::onSessionFocused (this=%p, application=%s)", this, session->name().c_str());
    Application* application = findApplicationWithSession(session);

    // Don't give application focus until it has created it's surface, when it is set as state "Running"
    if (application && application->state() != Application::Starting
            && application != m_focusedApplication) {
        setFocused(application);
        QModelIndex appIndex = findIndex(application);
        Q_EMIT dataChanged(appIndex, appIndex, QVector<int>() << RoleFocused);
    }
}

void ApplicationManager::onSessionUnfocused()
{
    DLOG("ApplicationManager::onSessionUnfocused (this=%p)", this);
    if (NULL != m_focusedApplication) {
        Q_ASSERT(m_focusedApplication->focused());
        m_focusedApplication->setFocused(false);
        m_focusedApplication->setState(Application::Suspended);
        m_focusedApplication = NULL;
        Q_EMIT focusedApplicationIdChanged();
        m_dbusWindowStack->FocusedWindowChanged(0, QString(), 0);

        QModelIndex appIndex = findIndex(m_focusedApplication);
        Q_EMIT dataChanged(appIndex, appIndex, QVector<int>() << RoleFocused << RoleState);
    }
}

void ApplicationManager::onSessionCreatedSurface(msh::ApplicationSession const* session,
                                               std::shared_ptr<msh::Surface> const& surface)
{
    DLOG("ApplicationManager::onSessionCreatedSurface (this=%p)", this);
    Q_UNUSED(surface);

    Application* application = findApplicationWithSession(session);
    if (application && application->state() == Application::Starting) {
        m_dbusWindowStack->WindowCreated(0, application->name());
        // only when Session creates a Surface will we actually mark it focused
        setFocused(application);
        QModelIndex appIndex = findIndex(application);
        Q_EMIT dataChanged(appIndex, appIndex, QVector<int>() << RoleFocused);
    }
}

void ApplicationManager::setFocused(Application *application)
{
    if (application == m_focusedApplication)
        return;

    m_focusedApplication = application;
    m_focusedApplication->setFocused(true);
    m_focusedApplication->setState(Application::Running);
    Q_EMIT focusedApplicationIdChanged();
    m_dbusWindowStack->FocusedWindowChanged(0, application->name(), application->stage());
}

Application* ApplicationManager::findApplicationWithSession(const std::shared_ptr<msh::Session> &session)
{
    return findApplicationWithSession(session.get());
}

Application* ApplicationManager::findApplicationWithSession(const msh::Session *session)
{
    for (Application *app : m_applications) {
        if (app->session().get() == session) {
            return app;
        }
    }
    return nullptr;
}

Application* ApplicationManager::findApplicationWithPid(const int pid)
{
    for (Application *app : m_applications) {
        if (app->m_pid == pid) {
            return app;
        }
    }
    return nullptr;
}

Application* ApplicationManager::findLastExecutedApplication()
{
    if (m_applications.length() > 0) {
        return m_applications.last();
    } else {
        return NULL;
    }
}

void ApplicationManager::add(Application* application)
{
    DASSERT(application != NULL);
    DLOG("ApplicationManager::add (this=%p, application='%s')", this, qPrintable(application->name()));

    beginInsertRows(QModelIndex(), m_applications.size(), m_applications.size());
    m_applications.append(application);
    endInsertRows();
    emit countChanged();
}

void ApplicationManager::remove(Application *application)
{
    DASSERT(application != NULL);
    DLOG("ApplicationManager::remove (this=%p, application='%s')", this, qPrintable(application->name()));

    int i = m_applications.indexOf(application);
    if (i != -1) {
        beginRemoveRows(QModelIndex(), i, i);
        m_applications.removeAt(i);
        endRemoveRows();
        emit countChanged();
    }
}

void ApplicationManager::move(int from, int to) {
    DLOG("ApplicationManager::move (this=%p, from=%d, to=%d)", this, from, to);
    if (from == to) return;

    if (from >= 0 && from < m_applications.size() && to >= 0 && to < m_applications.size()) {
        QModelIndex parent;
        /* When moving an item down, the destination index needs to be incremented
           by one, as explained in the documentation:
           http://qt-project.org/doc/qt-5.0/qtcore/qabstractitemmodel.html#beginMoveRows */
        beginMoveRows(parent, from, from, parent, to + (to > from ? 1 : 0));
        m_applications.move(from, to);
        endMoveRows();
    }
}

QModelIndex ApplicationManager::findIndex(Application* application)
{
    for (int i = 0; i < m_applications.size(); ++i) {
        if (m_applications.at(i) == application) {
            return index(i);
        }
    }

    return QModelIndex();
}
