/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef APPLICATION_H
#define APPLICATION_H

// std
#include <memory>

// Mir
#include <mir/shell/snapshot.h>

//Qt
#include <QtCore/QtCore>
#include <QImage>

class QImage;
class DesktopFileReader;
class TaskController;
namespace mir { namespace shell { class ApplicationSession; }}

class Application : public QObject {
    Q_OBJECT
    Q_ENUMS(Stage)
    Q_ENUMS(State)
    Q_PROPERTY(QString desktopFile READ desktopFile CONSTANT)
    Q_PROPERTY(QString appId READ appId CONSTANT)
    Q_PROPERTY(QString name READ name CONSTANT)
    Q_PROPERTY(QString comment READ comment CONSTANT)
    Q_PROPERTY(QString icon READ icon CONSTANT)
    Q_PROPERTY(QString exec READ exec CONSTANT)
    Q_PROPERTY(Stage stage READ stage NOTIFY stageChanged)
    Q_PROPERTY(State state READ state NOTIFY stateChanged)
    Q_PROPERTY(bool focus READ focus NOTIFY focusChanged)
    Q_PROPERTY(bool fullscreen READ fullscreen NOTIFY fullscreenChanged)

public:
    enum Stage { MainStage, SideStage };
    enum State { Starting, Running, Suspended, Stopped };

    Application(DesktopFileReader* desktopData, qint64 pid, Stage stage, State state, const QStringList& arguments, TaskController* taskController);
    ~Application();

    QString desktopFile() const;
    QString appId() const;
    QString name() const;
    QString comment() const;
    QString icon() const;
    QString exec() const;
    Stage stage() const;
    State state() const;
    bool focus() const;
    bool fullscreen() const;
    std::shared_ptr<mir::shell::ApplicationSession> session() const;

public slots:
    void suspend();
    void resume();
    void respawn();

Q_SIGNALS:
    void stageChanged();
    void stateChanged();
    void focusChanged();
    void fullscreenChanged();

private:
    pid_t pid() const;
    void setPid(pid_t pid);
    void setStage(Stage stage);
    void setState(State state);
    void setFocus(bool focus);
    void setFullscreen(bool fullscreen);
    void setSession(const std::shared_ptr<mir::shell::ApplicationSession>& session);
    void setSessionName(const QString& name);

    DesktopFileReader* m_desktopData;
    qint64 m_pid;
    Stage m_stage;
    State m_state;
    bool m_focus;
    bool m_fullscreen;
    std::shared_ptr<mir::shell::ApplicationSession> m_session;
    QString m_sessionName;
    QStringList m_arguments;
    TaskController* m_taskController;
    QTimer* m_suspendTimer;

    friend class TaskController;
    friend class ApplicationManager;
    friend class ApplicationListModel;
};

Q_DECLARE_METATYPE(Application*)

#endif  // APPLICATION_H
