/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef APPLICATIONMANAGER_H
#define APPLICATIONMANAGER_H

// std
#include <memory>

// Qt
#include <QObject>
#include <QStringList>

#include "application.h"
#include "application_list_model.h"

class ShellServerConfiguration;
class DBusWindowStack;

namespace mir {
    namespace shell {
        class ApplicationSession;
        class Surface;
    }
}

class ApplicationManager : public QObject
{
    Q_OBJECT
    Q_ENUMS(Role)
    Q_ENUMS(StageHint)
    Q_ENUMS(FormFactorHint)
    Q_ENUMS(FavoriteApplication)
    Q_FLAGS(ExecFlags)

    Q_PROPERTY(ApplicationListModel* mainStageApplications READ mainStageApplications
               NOTIFY mainStageApplicationsChanged)
    Q_PROPERTY(Application* mainStageFocusedApplication READ mainStageFocusedApplication
               NOTIFY mainStageFocusedApplicationChanged)

public:
    explicit ApplicationManager(QObject *parent = 0);
    ~ApplicationManager();

    // Mapping enums to Ubuntu Platform API enums.
    enum Role {
      Dash, Default, Indicators, Notifications, Greeter,
      Launcher, OnScreenKeyboard, ShutdownDialog
    };
    enum StageHint {
      MainStage, IntegrationStage, ShareStage, ContentPickingStage,
      SideStage, ConfigurationStage
    };
    enum FormFactorHint {
      DesktopFormFactor, PhoneFormFactor, TabletFormFactor
    };
    enum FavoriteApplication {
      CameraApplication, GalleryApplication, BrowserApplication, ShareApplication
    };
    enum Flag {
      NoFlag = 0x0,
      ForceMainStage = 0x1,
    };
    Q_DECLARE_FLAGS(ExecFlags, Flag)

    ApplicationListModel* mainStageApplications() const;
    Application* mainStageFocusedApplication() const;

    Q_INVOKABLE void focusApplication(Application *application);
    Q_INVOKABLE void focusFavoriteApplication(FavoriteApplication application);
    Q_INVOKABLE void unfocusCurrentApplication(StageHint stageHint);
    Q_INVOKABLE Application* startProcess(QString desktopFile, ExecFlags flags = NoFlag, QStringList arguments = QStringList());
    Q_INVOKABLE void stopProcess(Application* application);

Q_SIGNALS:
    void mainStageApplicationsChanged();
    void mainStageFocusedApplicationChanged();
    void focusRequested(FavoriteApplication favoriteApplication);

public Q_SLOTS:
    void authorizedSession(const quint64 pid, bool &authorized);

    void sessionStarting(std::shared_ptr<mir::shell::ApplicationSession> const& session);
    void sessionStopping(std::shared_ptr<mir::shell::ApplicationSession> const& session);
    void sessionFocused(std::shared_ptr<mir::shell::ApplicationSession> const& session);
    void sessionUnfocused();

    void sessionCreatedSurface(mir::shell::ApplicationSession const*, std::shared_ptr<mir::shell::Surface> const&);

private:
    void setFocused(Application *application);

    ApplicationListModel* m_mainStageApplications;
    Application* m_mainStageFocusedApplication; // remove as Mir has API for this
    ShellServerConfiguration* m_mirServer;
    DBusWindowStack* m_dbusWindowStack;

    friend class DBusWindowStack;
};

Q_DECLARE_OPERATORS_FOR_FLAGS(ApplicationManager::ExecFlags)

Q_DECLARE_METATYPE(ApplicationManager*)

#endif // APPLICATIONMANAGER_H
