/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// local
#include "application_manager.h"
#include "application_list_model.h"
#include "application.h"
#include "desktopfilereader.h"
#include "dbuswindowstack.h"

// unity-mir
#include "qmirserverapplication.h"
#include "shellserverconfiguration.h"
#include "sessionlistener.h"
#include "sessionauthorizer.h"
#include "logging.h"

// mir
#include <mir/shell/session_manager.h>

// Qt
#include <QCoreApplication>
#include <QProcess>

#define USE_UPSTART 0

#if !USE_UPSTART
  #include <pwd.h>
#endif

namespace msh = mir::shell;

ApplicationManager::ApplicationManager(QObject *parent)
:   QObject(parent)
,   m_mainStageApplications(new ApplicationListModel())
,   m_mainStageFocusedApplication(NULL)
{
    DLOG("ApplicationManager::ApplicationManager (this=%p)", this);

    QMirServerApplication* mirServerApplication = dynamic_cast<QMirServerApplication*>(QCoreApplication::instance());
    if (mirServerApplication == NULL) {
        LOG("Need to use QMirServerApplication");
        QCoreApplication::quit();
        return;
    }
    m_mirServer = mirServerApplication->server();

    QObject::connect(m_mirServer->sessionListener(), &SessionListener::sessionStarting,
                     this, &ApplicationManager::sessionStarting);
    QObject::connect(m_mirServer->sessionListener(), &SessionListener::sessionStopping,
                     this, &ApplicationManager::sessionStopping);
    QObject::connect(m_mirServer->sessionListener(), &SessionListener::sessionFocused,
                     this, &ApplicationManager::sessionFocused);
    QObject::connect(m_mirServer->sessionListener(), &SessionListener::sessionUnfocused,
                     this, &ApplicationManager::sessionUnfocused);
    QObject::connect(m_mirServer->sessionListener(), &SessionListener::sessionCreatedSurface,
                     this, &ApplicationManager::sessionCreatedSurface);

    std::shared_ptr<SessionAuthorizer> sessionAuthorizer
            = std::dynamic_pointer_cast<SessionAuthorizer>(m_mirServer->the_session_authorizer());

    QObject::connect(sessionAuthorizer.get(), &SessionAuthorizer::requestAuthorizationForSession,
                     this, &ApplicationManager::authorizedSession, Qt::BlockingQueuedConnection);

    // we use random numbers, so seed the generator
    qsrand(QTime::currentTime().msec());

    m_dbusWindowStack = new DBusWindowStack(this);
}

ApplicationManager::~ApplicationManager()
{
    DLOG("ApplicationManager::~ApplicationManager");
    delete m_mainStageApplications;
}

ApplicationListModel* ApplicationManager::mainStageApplications() const
{
    DLOG("ApplicationManager::mainStageApplications (this=%p)", this);
    return m_mainStageApplications;
}

Application* ApplicationManager::mainStageFocusedApplication() const
{
    DLOG("ApplicationManager::mainStageFocusedApplication (this=%p)", this);
    return m_mainStageFocusedApplication;
}

void ApplicationManager::focusApplication(Application* application)
{
    DLOG("ApplicationManager::focusApplication (this=%p, application=%p)", this, application);

    m_mirServer->the_session_manager()->set_focus_to(application->session());
}

void ApplicationManager::focusFavoriteApplication(
        ApplicationManager::FavoriteApplication application)
{
    DLOG("ApplicationManager::focusFavoriteApplication (this=%p, application=%d)",
         this, static_cast<int>(application));
    Q_UNUSED(application);
    // TODO(greyback) - upstart should handle favourite applications
}

void ApplicationManager::unfocusCurrentApplication(ApplicationManager::StageHint stageHint)
{
    DLOG("ApplicationManager::unfocusCurrentApplication (this=%p, stageHint=%d)", this,
         static_cast<int>(stageHint));
    Q_UNUSED(stageHint);

    m_mirServer->the_session_manager()->set_focus_to(NULL); //FIXME(greyback)
}

Application* ApplicationManager::startProcess(QString desktopFile, ApplicationManager::ExecFlags flags,
                                           QStringList arguments)
{
    DLOG("ApplicationManager::startProcess (this=%p, flags=%d)", this, (int) flags);
    // Load desktop file.
    DesktopFileReader* desktopData = new DesktopFileReader(desktopFile);
    if (!desktopData->loaded()) {
        delete desktopData;
        return NULL;
    }

#if USE_UPSTART
    /* launch via upstart */
    arguments.prepend("APP_ID=" + desktopData->appId());
    arguments.prepend("application");

    // Start process.
    //FIXME(greyback) use upstart via DBUS
    QProcess process;
    process.start("start", arguments);
    //FIXME(greyback) this is blocking, instead listen for finished() signal
    process.waitForFinished(1000);

    DLOG_IF(process.error(), "process 'start application' failed to start with error: %s", process.errorString().toLatin1().data());
    if (!process.error()) {
        // fetch pid
        QString output = process.readAllStandardOutput();
        int lastSpace = output.lastIndexOf(' ');
        qint64 pid = output.remove(0, lastSpace).toInt();
#else
    /* Launch manually */

    // Format arguments.
    // FIXME(loicm) Special field codes are simply ignored for now.
    //     http://standards.freedesktop.org/desktop-entry-spec/latest/ar01s06.html
    QStringList execArguments = desktopData->exec().split(" ", QString::SkipEmptyParts);
    DASSERT(execArguments.size() > 0);
    QString exec(execArguments[0]);
    const int kSize = execArguments.size();
    for (int i = 1; i < kSize; i++) {
        if ((execArguments[i].size() == 2) && (execArguments[i][0].toLatin1() == '%')) {
            const char kChar = execArguments[i][1].toLatin1();
            if (kChar == 'F' || kChar == 'u' || kChar == 'U' || kChar == 'd' || kChar == 'D'
                    || kChar == 'n' || kChar == 'N' || kChar == 'i' || kChar == 'c' || kChar == 'k'
                    || kChar == 'v' || kChar == 'm') {
                continue;
            }
        }
        arguments.prepend(execArguments[i]);
    }
    arguments.append(QString("--desktop_file_hint=") + desktopData->file());
    if (flags.testFlag(ApplicationManager::ForceMainStage))
        arguments.append(QString("--stage_hint=main_stage"));
    else if (desktopData->stageHint() == "SideStage")
        arguments.append(QString("--stage_hint=side_stage"));

    // Start process - set correct environment
    setenv("QT_QPA_PLATFORM", "ubuntumirclient", 1);

    bool result;
    qint64 pid = 0;
    struct passwd* passwd = getpwuid(getuid());
    DLOG("current working directory: '%s'", passwd ? passwd->pw_dir : "/");
    result = QProcess::startDetached(exec, arguments, QString(passwd ? passwd->pw_dir : "/"), &pid);
    DLOG_IF(result == false, "process failed to start");
    if (result) {
#endif

        DLOG("process started with pid %lld, adding '%s' to application lists", pid, desktopData->name().toLatin1().data());
        Application* application = new Application(
                    desktopData, pid,
                    (flags.testFlag(ApplicationManager::ForceMainStage)
                        || desktopData->stageHint() != "SideStage")
                        ? Application::MainStage : Application::SideStage,
                    Application::Starting //FIXME(greyback): assuming running immediately
                    );

        m_mainStageApplications->add(application);
        return application;
    } else {
        return NULL;
    }
}

void ApplicationManager::stopProcess(Application* application)
{
    DLOG("ApplicationManager::stopProcess (this=%p, application=%p)", this, application);

    if (application != NULL) {
        if (application == m_mainStageFocusedApplication) {
            // TODO(greyback) What to do?? Focus next app, or unfocus everything??
            m_mainStageFocusedApplication = NULL;
            Q_EMIT mainStageFocusedApplicationChanged();
        }
        m_mainStageApplications->remove(application);
        m_dbusWindowStack->WindowDestroyed(0, application->name());

        // Start process.
        bool result;

#if USE_UPSTART
        result = QProcess::startDetached("stop", QStringList() << "application"
                                         << ("APP_ID=" + application->appId()));
        DLOG_IF(result == false, "process 'stop application' failed to execute");
#else
        result = QProcess::startDetached("/bin/kill", QStringList() << "-9" << QString::number(application->m_pid));
        DLOG_IF(result == false, "process '/bin/kill -9 %lld' failed to execute", application->m_pid);
#endif
        LOG_IF(!result, "ApplicationManager: Unable to stop process '%s'", application->name().toLatin1().constData());
        delete application;
    }
}

/************************************* Mir-side methods *************************************/

void ApplicationManager::authorizedSession(const quint64 pid, bool &authorized)
{
    authorized = false; //to be proven wrong

    DLOG("ApplicationManager::authorizedSession (this=%p, pid=%lld)", this, pid);
    Application* application = m_mainStageApplications->getApplicationWithPid(pid);
    if (application) {
        QString name = application->appId() + QString::number(qrand());
        application->setSessionName(name);
        authorized = true;
        return;
    }

    /*
     * Hack: Allow applications to be launched externally, but must be executed with the
     * "desktop_file_hint" parameter attached. This exists until upstart-app-launch can
     * notify shell it is starting an application and so shell should allow it. Also reads
     * the --stage parameter to determine the desired stage
     */
    QFile cmdline(QString("/proc/%1/cmdline").arg(pid));
    if (!cmdline.open(QIODevice::ReadOnly | QIODevice::Text)) {
        DLOG("ApplicationManager REJECTED connection from app with pid %lld as unable to read process command", pid);
        return;
    }

    QByteArray command = cmdline.readLine().replace('\0', ' ');

    // FIXME: special exception for the OSK - maliit-server - not very secure
    if (command.startsWith("maliit-server")) {
        authorized = true;
        return;
    }

    QString pattern = QRegularExpression::escape("--desktop_file_hint=") + "(\\S+)";
    QRegularExpression regExp(pattern);
    QRegularExpressionMatch regExpMatch = regExp.match(command);

    if (!regExpMatch.hasMatch()) {
        LOG("ApplicationManager REJECTED connection from app with pid %lld as no desktop_file_hint specified", pid);
        return;
    }

    QString desktopFileName = regExpMatch.captured(1);
    DLOG("Process supplied desktop_file_hint, loading '%s'", desktopFileName.toLatin1().data());

    DesktopFileReader* desktopData = new DesktopFileReader(desktopFileName);
    if (!desktopData->loaded()) {
        delete desktopData;
        LOG("ApplicationManager REJECTED connection from app with pid %lld as desktop_file_hint file not found", pid);
        return;
    }

    // some naughty applications use a script to launch the actual application. Check for the
    // case where shell actually launched the script.
    application = m_mainStageApplications->getApplicationWithAppId(desktopData->appId());
    if (application && application->state() == Application::Starting) {
        DLOG("Process with pid %lld appeared, attached to existing entry '%s' in application lists",
             pid, application->appId().toLatin1().data());
        delete desktopData;
        QString name = application->appId() + QString::number(qrand());
        application->setSessionName(name);
        authorized = true;
        return;
    }

    // if stage supplied in CLI, fetch that
    Application::Stage stage = Application::MainStage;
    pattern = QRegularExpression::escape("--stage=") + "(\\S+)";
    regExp.setPattern(pattern);
    regExpMatch = regExp.match(command);

    if (regExpMatch.hasMatch() && regExpMatch.captured(1) == "side_stage") {
        stage = Application::SideStage;
    }

    DLOG("Existing process with pid %lld appeared, adding '%s' to application lists", pid, desktopData->name().toLatin1().data());
    application = new Application(desktopData, pid, stage, Application::Starting);
    m_mainStageApplications->add(application);
    authorized = true;
}

void ApplicationManager::sessionStarting(std::shared_ptr<msh::ApplicationSession> const& session)
{
    DLOG("ApplicationManager::sessionStarting (this=%p, application=%s)", this, session->name().c_str());

    //FIXME(greyback) Mir not supplying any identifier that we can use to link the PID to the session
    // so am assuming that the *most recently* launched application session is the one that connects
    Application* application = m_mainStageApplications->getLastExecutedApplication();
    if (application && application->state() != Application::Running) {
        application->setSession(session);
    } else {
        DLOG("ApplicationManager::sessionStarting - unauthorized application!!");
    }
}

void ApplicationManager::sessionStopping(std::shared_ptr<msh::ApplicationSession> const& session)
{
    DLOG("ApplicationManager::sessionStopping (this=%p, application=%s)", this, session->name().c_str());

    // in case application closed not by hand of shell, check again here:
    Application* application = m_mainStageApplications->getApplicationWithSession(session);
    if (application) {
        if (application == m_mainStageFocusedApplication) {
            // TODO(greyback) What to do?? Focus next app, or unfocus everything??
            m_mainStageFocusedApplication = NULL;
            Q_EMIT mainStageFocusedApplicationChanged();
        }
        m_mainStageApplications->remove(application);
        m_dbusWindowStack->WindowDestroyed(0, application->name());
    }
}

void ApplicationManager::sessionFocused(std::shared_ptr<msh::ApplicationSession> const& session)
{
    DLOG("ApplicationManager::sessionFocused (this=%p, application=%s)", this, session->name().c_str());
    Application* application = m_mainStageApplications->getApplicationWithSession(session);

    // Don't give application focus until it has created it's surface, when it is set as state "Running"
    if (application && application->state() != Application::Starting
            && application != m_mainStageFocusedApplication) {
        setFocused(application);
    }
}

void ApplicationManager::sessionUnfocused()
{
    DLOG("ApplicationManager::sessionUnfocused (this=%p)", this);
    if (NULL != m_mainStageFocusedApplication) {
        m_mainStageFocusedApplication->setFocus(false);
        m_mainStageFocusedApplication = NULL;
        Q_EMIT mainStageFocusedApplicationChanged();
        m_dbusWindowStack->FocusedWindowChanged(0, QString(), 0);
    }
}

void ApplicationManager::sessionCreatedSurface(msh::ApplicationSession const* session,
                                               std::shared_ptr<msh::Surface> const& surface)
{
    DLOG("ApplicationManager::sessionCreatedSurface (this=%p)", this);
    Q_UNUSED(surface);

    Application* application = m_mainStageApplications->getApplicationWithSession(session);
    if (application && application->state() == Application::Starting) {
        application->setState(Application::Running);

        m_dbusWindowStack->WindowCreated(0, application->name());

        // only when Session creates a Surface will we actually mark it focused
        setFocused(application);
    }
}

void ApplicationManager::setFocused(Application *application)
{
    m_mainStageFocusedApplication = application;
    m_mainStageFocusedApplication->setFocus(true);
    Q_EMIT mainStageFocusedApplicationChanged();
    m_dbusWindowStack->FocusedWindowChanged(0, application->name(), application->stage());
}
