/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// local
#include "application_list_model.h"
#include "application.h"

// unity-mir
#include "logging.h"

// mir
#include <mir/shell/application_session.h>

namespace msh = mir::shell;

ApplicationListModel::ApplicationListModel(QObject* parent)
:   QAbstractListModel(parent)
,   m_applications()
{
    m_roleNames.insert(0, "application");
    DLOG("ApplicationListModel::ApplicationListModel (this=%p, parent=%p)", this, parent);
}

ApplicationListModel::~ApplicationListModel()
{
    DLOG("ApplicationListModel::~ApplicationListModel");
    const int kSize = m_applications.size();
    for (int i = 0; i < kSize; i++)
        delete m_applications.at(i);
    m_applications.clear();
}

int ApplicationListModel::rowCount(const QModelIndex& parent) const
{
    DLOG("ApplicationListModel::rowCount (this=%p), returns %d", this, !parent.isValid() ? m_applications.size() : 0);
    return !parent.isValid() ? m_applications.size() : 0;
}

QVariant ApplicationListModel::data(const QModelIndex& index, int role) const
{
    DLOG("ApplicationListModel::data (this=%p, role=%d)", this, role);
    if (index.row() >= 0 && index.row() < m_applications.size() && role == 0)
        return QVariant::fromValue(m_applications.at(index.row()));
    else
        return QVariant();
}

QVariant ApplicationListModel::get(int row) const
{
    DLOG("ApplicationListModel::get (this=%p, row=%d)", this, row);
    return data(index(row), 0);
}

void ApplicationListModel::move(int from, int to)
{
    DLOG("ApplicationListModel::move (this=%p, from=%d, to=%d)", this, from, to);
    if (from >= 0 && from < m_applications.size() && to >= 0 && to < m_applications.size()) {
        QModelIndex parent;
        /* When moving an item down, the destination index needs to be incremented
       by one, as explained in the documentation:
       http://qt-project.org/doc/qt-5.0/qtcore/qabstractitemmodel.html#beginMoveRows */
        beginMoveRows(parent, from, from, parent, to + (to > from ? 1 : 0));
        m_applications.move(from, to);
        endMoveRows();
    }
}

void ApplicationListModel::add(Application* application)
{
    DASSERT(application != NULL);
    DLOG("ApplicationListModel::add (this=%p, application='%s')", this,
         application->name().toLatin1().data());
#if !defined(QT_NO_DEBUG)
    for (int i = 0; i < m_applications.size(); i++)
        ASSERT(m_applications.at(i) != application);
#endif
    beginInsertRows(QModelIndex(), m_applications.size(), m_applications.size());
    m_applications.append(application);
    endInsertRows();
    emit countChanged();
}

void ApplicationListModel::remove(Application* application)
{
    DASSERT(application != NULL);
    DLOG("ApplicationListModel::remove (this=%p, application='%s')", this,
         application->name().toLatin1().data());
    int i = m_applications.indexOf(application);
    if (i != -1) {
        beginRemoveRows(QModelIndex(), i, i);
        m_applications.removeAt(i);
        endRemoveRows();
        emit countChanged();
    }
}

Application* ApplicationListModel::getApplicationWithSession(const std::shared_ptr<msh::Session> &session)
{
    return getApplicationWithSession(session.get());
}

Application* ApplicationListModel::getApplicationWithSession(const msh::Session *session)
{
    for (Application *app : m_applications) {
        if (app->session().get() == session) {
            return app;
        }
    }
    return nullptr;
}

Application* ApplicationListModel::getApplicationWithAppId(const QString &appId)
{
    for (Application *app : m_applications) {
        if (app->appId() == appId) {
            return app;
        }
    }
    return nullptr;
}

Application* ApplicationListModel::getApplicationWithPid(const int pid)
{
    for (Application *app : m_applications) {
        if (app->m_pid == pid) {
            return app;
        }
    }
    return nullptr;
}

Application* ApplicationListModel::getLastExecutedApplication()
{
    if (m_applications.length() > 0) {
        return m_applications.last();
    } else {
        return NULL;
    }
}

