/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "shellserverconfiguration.h"

#include "initialsurfaceplacementstrategy.h"
#include "sessionlistener.h"
#include "surfacesource.h"
#include "sessionauthorizer.h"
#include "surfacebuilder.h"

// mir
#include "mir/shell/organising_surface_factory.h"

// Qt
#include <QDebug>

namespace msh = mir::shell;

ShellServerConfiguration::ShellServerConfiguration(int argc, char const* argv[], QObject* parent)
    : QObject(parent)
    , DefaultServerConfiguration(argc, argv)
    , m_surfaceBuilder(new SurfaceBuilder(the_surface_stack_model()))
    , m_sessionListener(new SessionListener)
    , m_surfaceSource(new SurfaceSource(the_surface_builder(), the_shell_surface_configurator()))
    , m_sessionAuthorizer(new SessionAuthorizer(the_session_manager()))
{
    qDebug() << "ShellServerConfiguration created";
}

ShellServerConfiguration::~ShellServerConfiguration()
{
}

std::shared_ptr<msh::PlacementStrategy>
ShellServerConfiguration::the_shell_placement_strategy()
{
    return shell_placement_strategy(
        [this]
        {
            return std::make_shared<InitialSurfacePlacementStrategy>(the_shell_display_layout());
        });
}

std::shared_ptr<msh::SessionListener>
ShellServerConfiguration::the_shell_session_listener()
{
    std::shared_ptr<SessionListener> sessionListener = m_sessionListener;
    return shell_session_listener(
        [this, &sessionListener]
        {
            return sessionListener;
        });
}

std::shared_ptr<msh::SurfaceFactory>
ShellServerConfiguration::the_shell_surface_factory()
{
    return shell_surface_factory(
        [this]
        {
            return std::make_shared<mir::shell::OrganisingSurfaceFactory>(
                m_surfaceSource,
                the_shell_placement_strategy());
        });
}

std::shared_ptr<msh::SurfaceBuilder>
ShellServerConfiguration::the_surface_builder()
{
    return m_surfaceBuilder;
}

std::shared_ptr<mir::frontend::SessionAuthorizer>
ShellServerConfiguration::the_session_authorizer()
{
    return m_sessionAuthorizer;
}

SessionListener* ShellServerConfiguration::sessionListener() const
{
    return m_sessionListener.get();
}

SurfaceSource *ShellServerConfiguration::surfaceSource() const
{
    return m_surfaceSource.get();
}
