#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009-2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" Base tests cases and test utilities """
from __future__ import with_statement

import dbus
from dbus.mainloop.glib import DBusGMainLoop
import contextlib
import os
import shutil

from twisted.internet import defer
from twisted.python import failure
from unittest import TestCase
from oauth import oauth


@contextlib.contextmanager
def environ(env_var, new_value):
    """context manager to replace/add an environ value"""
    old_value = os.environ.get(env_var, None)
    os.environ[env_var] = new_value
    yield
    if old_value is None:
        os.environ.pop(env_var)
    else:
        os.environ[env_var] = old_value


class FakeOAuthClient(object):
    """ Fake OAuthClient"""

    def __init__(self, realm):
        """ create the instance. """
        self.realm = realm
        self.consumer = oauth.OAuthConsumer('ubuntuone', 'hammertime')

    def get_access_token(self):
        """ returns a Token"""
        return 'a token'


class FakeDBusInterface(object):
    """A fake DBusInterface..."""
    def shutdown(self, with_restart=False):
        """...that only knows how to go away"""


class BaseTestCase(TestCase):
    """Base TestCase with helper methods to handle temp dir.

    This class provides:
        mktemp(name): helper to create temporary dirs
        rmtree(path): support read-only shares
        makedirs(path): support read-only shares
    """

    def mktemp(self, name='temp'):
        """ Customized mktemp that accepts an optional name argument. """
        tempdir = os.path.join(self.tmpdir, name)
        if os.path.exists(tempdir):
            self.rmtree(tempdir)
        self.makedirs(tempdir)
        return tempdir

    @property
    def tmpdir(self):
        """Default tmpdir: module/class/test_method."""
        # check if we already generated the root path
        root_dir = getattr(self, '__root', None)
        if root_dir:
            return root_dir
        MAX_FILENAME = 32 # some platforms limit lengths of filenames
        base = os.path.join(self.__class__.__module__[:MAX_FILENAME],
                            self.__class__.__name__[:MAX_FILENAME],
                            self._testMethodName[:MAX_FILENAME])
        # use _trial_temp dir, it should be os.gwtcwd()
        # define the root temp dir of the testcase, pylint: disable=W0201
        self.__root = os.path.join(os.getcwd(), base)
        return self.__root

    def rmtree(self, path):
        """Custom rmtree that handle ro parent(s) and childs."""
        if not os.path.exists(path):
            return
        # change perms to rw, so we can delete the temp dir
        if path != getattr(self, '__root', None):
            os.chmod(os.path.dirname(path), 0755)
        if not os.access(path, os.W_OK):
            os.chmod(path, 0755)
        # pylint: disable=W0612
        for dirpath, dirs, files in os.walk(path):
            for dirname in dirs:
                if not os.access(os.path.join(dirpath, dirname), os.W_OK):
                    os.chmod(os.path.join(dirpath, dirname), 0777)
        shutil.rmtree(path)

    def makedirs(self, path):
        """Custom makedirs that handle ro parent."""
        parent = os.path.dirname(path)
        if os.path.exists(parent):
            os.chmod(parent, 0755)
        os.makedirs(path)

    def setUp(self):
        TestCase.setUp(self)

    def tearDown(self):
        """ cleanup the temp dir. """
        return TestCase.tearDown(self)


class DBusTestCase(BaseTestCase):
    """ Test the DBus event handling """

    def setUp(self):
        """ Setup the infrastructure fo the test (dbus service). """
        BaseTestCase.setUp(self)
        self.loop = DBusGMainLoop(set_as_default=True)
        self.bus = dbus.bus.BusConnection(mainloop=self.loop)
        # monkeypatch busName.__del__ to avoid errors on gc
        # we take care of releasing the name in shutdown
        dbus.service.BusName.__del__ = lambda _: None
        self.bus.set_exit_on_disconnect(False)
        self.signal_receivers = set()

    def tearDown(self):
        """ Cleanup the test. """
        d = self.cleanup_signal_receivers(self.signal_receivers)
        d.addBoth(self._tearDown)
        d.addBoth(lambda _: BaseTestCase.tearDown(self))
        return d

    def _tearDown(self):
        """ shutdown """
        self.bus.flush()
        self.bus.close()

    def error_handler(self, error):
        """ default error handler for DBus calls. """
        if isinstance(error, failure.Failure):
            self.fail(error.getErrorMessage())

    def cleanup_signal_receivers(self, signal_receivers):
        """ cleanup self.signal_receivers and returns a deferred """
        deferreds = []
        for match in signal_receivers:
            d = defer.Deferred()
            def callback(*args):
                """ callback that accepts *args. """
                if not d.called:
                    d.callback(args)
            self.bus.call_async(dbus.bus.BUS_DAEMON_NAME,
                                dbus.bus.BUS_DAEMON_PATH,
                                dbus.bus.BUS_DAEMON_IFACE, 'RemoveMatch', 's',
                                (str(match),), callback, self.error_handler)
            deferreds.append(d)
        if deferreds:
            return defer.DeferredList(deferreds)
        else:
            return defer.succeed(True)

