# --------------------------------------------------------------------
# Copyright © 2014-2015 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# --------------------------------------------------------------------

# --------------------------------------------------------------------
# TODO:
#
# - Stop services, unpack, restart services?
# -   XXX: *DO* unpack device files once above handled.
# - Fix bug triggered by restarting systemd which causes
#   Upgrader.systemd.connection* to hang for 30 seconds even when
#   recreated.
# --------------------------------------------------------------------

'''
Apply an Ubuntu Core system image update.

WARNING : If running an in-place upgrade, should be run within a new
          mount namespace to avoid disturbing the existing (read-only)
          root filesystem.

NOTE    : The signature files associated with each image file is *NOT*
          re-verified, since this must already have been handled by
          system-image-cli(1).
'''

import sys
import os
import logging
import re
import shutil
import subprocess
import tempfile
import tarfile
import stat
import errno
import dbus

from enum import Enum
from time import sleep

script_name = os.path.basename(__file__)

log = logging.getLogger()

# Name of the writable user data partition label as created by
# ubuntu-device-flash(1).
WRITABLE_DATA_LABEL = 'writable'

WRITABLE_MOUNTPOINT = '/writable'

# name of primary root filesystem partition label as created by
# ubuntu-device-flash(1).
SYSTEM_DATA_A_LABEL = 'system-a'

# name of primary root filesystem partition label as created by
# ubuntu-device-flash(1). Note that this partition will
# only be present if this is an A/B upgrade system.
SYSTEM_DATA_B_LABEL = 'system-b'

# The tar file contains a 'system/' directory with the new files
# to apply to the real system. It may also contain a top-level
# file called 'removed' that lists files relative to the system
# mount for files on the system that should be removed before
# unpacking the rest of the archive.
#
# Examples:
#
# - if the file '/etc/foo' should be removed, it
#   would be specified in the removed_file as 'system/etc/foo'.
#
# - if the file '/etc/bar' should be modified, it
#   would be specified in the tar file as member 'system/etc/bar'.
TAR_FILE_SYSTEM_PREFIX = 'system/'

# boot assets (kernel, initrd, .dtb's)
# (XXX: note no trailing slash to ensure we unpack the directory itself).
TAR_FILE_ASSETS_PREFIX = 'assets/'

TAR_FILE_REMOVED_FILE = 'removed'

SYSTEM_IMAGE_CHANNEL_CONFIG = '/etc/system-image/channel.ini'


def remove_prefix(path, prefix=TAR_FILE_SYSTEM_PREFIX):
    '''
    Remove specified prefix from path and return the result.

    Prefix must end with a slash.

    If @prefix is not a prefix of @path, returns None.
    '''

    assert(prefix.endswith('/'))

    i = path.find(prefix)

    if i < 0:
        return None

    # ensure that the returned path has a leading slash
    return path[len(prefix)-1:]


def get_command(pid):
    '''
    Returns full path to binary associated with @pid, or None if the
    lookup failed.
    '''
    try:
        exe = os.readlink('/proc/' + str(pid) + '/exe')
    except:
        # pid probably went away
        return None

    return exe


def get_root_partitions_by_label():
    '''
    Returns a list of tuples of the recognised root filesystem partitions
    available on this system. The tuples contain the following triplet:

        ( <partition-name>, <full-device-path>, <mountpoint> )

    '''
    cmd = 'lsblk'
    recognised = (SYSTEM_DATA_A_LABEL, SYSTEM_DATA_B_LABEL)

    matches = []

    args = [cmd, '--ascii', '--output-all', '--pairs']
    log.debug('running: {}'.format(args))

    proc = subprocess.Popen(args,
                            stdout=subprocess.PIPE,
                            stderr=subprocess.DEVNULL,
                            universal_newlines=True)

    if proc.wait() != 0:
        return matches

    stdout = proc.communicate()[0]
    lines = stdout.split('\n')

    for line in lines:
        line = line.rstrip()
        if not line:
            continue

        fields = {}

        # split the line into 'NAME="quoted value"' fields
        results = re.findall(r'(?:[^\s"]|"(?:[^"])*")+', line)
        for result in results:
            name, value = result.split('=')

            # remove quotes
            value = value.lstrip('"').rstrip('"')

            fields[name] = value

        if fields['NAME'] != fields['KNAME']:
            # For SCSI devices, these fields match
            continue

        if 'LABEL' in fields and fields['LABEL'] in recognised:
            # reconstructing the device name like this is valid
            # for SCSI devices
            device = '/dev/{}'.format(fields['NAME'])

            if not os.path.exists(device):
                continue
            matches.append((fields['LABEL'],
                            device,
                            fields['MOUNTPOINT']))

    return matches


def uses_ab_partitions():
    '''
    Returns: True if the system uses A/B partitions, else False.
    '''
    return len(get_root_partitions_by_label()) == 2


def get_writable_disk():
    '''
    Establish the disk partition for the writable user data partition.
    '''
    cmd = "blkid -L '{}' 2>/dev/null".format(WRITABLE_DATA_LABEL)
    log.debug('running: {}'.format(cmd))

    output = subprocess.getoutput(cmd)
    return output.rstrip()


def fsck(device):
    '''
    Run fsck(8) on specified device.
    '''
    assert (os.path.exists(device))

    failed = False

    cmd = '/sbin/fsck'
    args = []

    args.append(cmd)

    # Paranoia - don't fsck if already mounted
    args.append('-M')

    args.append('-av')
    args.append(device)

    log.debug('running: {}'.format(args))

    proc = subprocess.Popen(args,
                            stdout=subprocess.PIPE,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)

    ret = proc.wait()

    if ret == 0:
        return

    stdout, stderr = proc.communicate()

    # fsck's return code 1 means: "Filesystem errors corrected"
    # (aka a warning - FS is consistent [now]).
    failed = False if ret == 1 else True

    log.error('{} returned {} ({}): {}, {}'
              .format(args,
                      ret,
                      "failed" if failed else "warning",
                      stdout,
                      stderr))

    if failed:
        sys.exit(1)


def remount(mountpoint, options):
    """
    Remount mountpoint using the specified options string (which is
    passed direct to the mount command and which must not contain
    spaces).
    """
    cmd = 'mount'

    args = []

    args.append(cmd)
    args.append('-oremount,{}'.format(options))
    args.append(mountpoint)

    log.debug('running: {}'.format(args))

    proc = subprocess.Popen(args,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)
    if proc.wait() != 0:
        stderr = proc.communicate()[1]
        log.error('failed to remount ({}): {}'
                  .format(args, stderr))
        sys.exit(1)


def mount(source, target, options=None):
    '''
    Mount @source on @target using optional specified
    options string (which must not contain spaces).
    '''

    cmd = 'mount'

    args = []

    args.append(cmd)
    if options:
        args += ['-o', options]
    args.append(source)
    args.append(target)

    log.debug('running: {}'.format(args))

    proc = subprocess.Popen(args,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)
    if proc.wait() != 0:
        stderr = proc.communicate()[1]
        log.error('failed to mount ({}): {}'
                  .format(args, stderr))
        sys.exit(1)


def bind_mount(source, target):
    '''
    Bind mount @source to @target.
    '''
    log.debug('bind mounting existing root')
    mount(source, target, 'bind')


def mount_all():
    '''
    Mount all filesystems if not already mounted.
    '''

    args = ['mount', '-a']
    log.debug('running: {}'.format(args))
    proc = subprocess.Popen(args,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)

    if proc.wait() != 0:
        stderr = proc.communicate()[1]
        log.error('failed to mount all ({}): {}'
                  .format(args, stderr))


def lazy_unmount(target):
    '''
    async unmount the specified mount target.
    '''

    unmount(target, ['--lazy'])


def unmount(target, options=None):
    '''
    Unmount the specified mount target, using the specified list of
    options.
    '''

    log.debug('unmounting {}'.format(target))

    args = []
    args.append('umount')
    if options:
        args.extend(options)
    args.append(target)

    log.debug('running: {}'.format(args))

    proc = subprocess.Popen(args,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)

    if proc.wait() != 0:
        stderr = proc.communicate()[1]
        log.error('failed to unmount ({}): {}'
                  .format(args, stderr))

        # its a lazy umount, do not fail hard just now.
        #
        # FIXME: entry in mount table for /etc/writable/hostname really
        # does not seem to have been mounted by systemd. No problem when
        # booting under upstart as mountall does bind mount that path.
        #
        # sys.exit(1)


def make_mount_private(target):
    '''
    Make the specified filesystem private.
    '''

    args = ['mount', '--make-rprivate', target]
    log.debug('running: {}'.format(args))

    proc = subprocess.Popen(args,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)

    if proc.wait() != 0:
        stderr = proc.communicate()[1]
        log.error('failed to make {} private ({}): {}'
                  .format(target, args, stderr))


def lazy_unmount_specified(mounts):
    '''
    async nnmount all targets specified by @mounts.
    '''

    for mount in mounts:
        lazy_unmount(mount)


def get_writable_mounts():
    '''
    Returns a list of (bind) mounts whose source location is the
    writable partition.

    Note that the list of bind mounts is derived from the current
    mounts. This is safer than simply checking fstab since it is
    guaranteed correct.

    Note also that the list is reverse sorted so that assuming the list
    is processed in order, child mounts will be handled before parent
    mounts.
    '''
    disk = get_writable_disk()
    file = '/proc/mounts'

    mounts = []

    try:
        with open(file, 'r') as fh:
            for mount in fh.readlines():
                mount = mount.strip()
                fields = mount.split()

                # only consider user data mounts
                if fields[0] != disk:
                    continue

                # ignore the primary mount
                if fields[1] == WRITABLE_MOUNTPOINT:
                    continue

                # ignore root
                if fields[1] == '/' or fields[1] == '/root':
                    continue

                mounts.append(fields[1])
    except:
        sys.exit('Failed to read command file: {}'.format(file))

    # Reverse sort to ensure each child mount is handled before its
    # parent.
    return sorted(mounts, reverse=True)


def get_affected_pids(unpack_inodes):
    '''
    @unpack_inodes: list of inodes representing the files the system image
    upgrade will modify.

    Returns a dict keyed by filename whose value is an array of pids
    that are using the file currently.

    '''

    # Command to list all open files as quickly as possible.
    #
    # XXX: Note that although we nominally only care about files open on
    # the rootfs, we don't constrain lsof to only consider such files
    # since the system-images contain files that are mounted in writable
    # partitions (examples being /var/log/dpkg.log and
    # /var/log/apt/history.log).
    #
    # By considering all files, we increase the likelihood of a reboot
    # but in doing avoid unexpected system behaviour (where a process is
    # seeking through an old copy of a config file for example).
    #
    # Discard stderr to avoid the annoying
    # 'lsof: WARNING can't stat() ...' messages.
    cmd = 'lsof -lnPR 2>/dev/null'

    log.debug('running: {}'.format(cmd))

    output = subprocess.getoutput(cmd)

    # Hash showing which processes are using files that the
    # upgrader needs to replace.
    #
    # key: filename.
    # value: array of pids.
    pid_file_map = {}

    # Read the lsof output. Note that we do _not_ ignore deleted files
    # since in fact the files we are looking for have been deleted
    # (unlinked, but not fully).
    for line in output.split('\n'):
        fields = line.split()

        # ignore header
        if line.startswith('COMMAND'):
            continue

        # ignore files with no inode
        if fields[5] in ('netlink', 'unknown'):
            continue

        pid = int(fields[1])

        # Deleted files have one less field (size/offset).
        if len(fields) == 9 and fields[4] == 'DEL':

            potential_inode = fields[7]

            if not potential_inode.isdigit():
                continue

            inode = int(potential_inode)
            file = fields[8]
        else:
            potential_inode = fields[8]

            if not potential_inode.isdigit():
                continue

            inode = int(potential_inode)
            file = fields[9]

        # ignore kernel threads
        if file == '/':
            continue

        # ignore anything that doesn't look like a file
        if file[0] != '/':
            continue

        # ignore files that don't relate to files the upgrade will
        # change.
        if inode not in unpack_inodes:
            continue

        # create a hash of arrays / dict of lists
        if file not in pid_file_map:
            pid_file_map[file] = []

        pid_file_map[file].append(pid)

    return pid_file_map


class Systemd():
    '''
    Interface to systemd init daemon.

    The upgrader talks to systemd via its private socket since this
    protects the upgrader...

    - from issues with a broken dbus-daemon (admittedly unlikely).

    - from getting disconnected from systemd should the dbus-daemon need
      to be restarted by the upgrader if the dbus-daemon is holding inodes
      open (much more likely to happen).

    - against changes in the format of systemd's command-line tooling
      output (as happened when 'systemctl show' output changed between
      systemd 208 and 215).

    '''

    SYSTEMD_BUS_NAME = 'org.freedesktop.systemd1'
    SYSTEMD_MGR_INTERFACE = 'org.freedesktop.systemd1.Manager'
    SYSTEMD_UNIT_INTERFACE = 'org.freedesktop.systemd1.Unit'
    SYSTEMD_OBJECT_PATH = '/org/freedesktop/systemd1'

    SYSTEMD_PRIVATE_SOCKET = 'unix:path=/run/systemd/private'

    FREEDESKTOP_PROPERTIES = 'org.freedesktop.DBus.Properties'

    def __init__(self):
        self.connection = \
            dbus.connection.Connection(self.SYSTEMD_PRIVATE_SOCKET)
        self.proxy = self.connection.get_object(self.SYSTEMD_BUS_NAME,
                                                self.SYSTEMD_OBJECT_PATH)
        self.properties = dbus.Interface(self.proxy,
                                         self.FREEDESKTOP_PROPERTIES)

        self.interface = dbus.Interface(self.proxy,
                                        self.SYSTEMD_MGR_INTERFACE)

        # method for handling service start/stop/restart
        self.mode = 'replace'

    def version(self):
        '''
        Returns the currently running version of systemd.
        '''
        return self.properties.Get(self.SYSTEMD_MGR_INTERFACE, 'Version')

    def find_unit(self, pid):
        '''
        Find the systemd unit associated with @pid.

        Returns D-Bus object path for service associated with @pid,
         or None.
        '''
        try:
            return self.interface.GetUnitByPID(pid)
        except:
            return None

    def get_service(self, name):
        '''
        @name: D-Bus path for service.

        Return the unit associated with the specified @name.
        '''
        unit = self.connection.get_object(self.SYSTEMD_BUS_NAME,
                                          name)
        interface = dbus.Interface(unit, self.SYSTEMD_UNIT_INTERFACE)
        return interface

    def stop_service(self, name):
        '''
        Stop the specified service.

        @name: D-Bus path for service.

        Returns: True on success, else False
        '''

        log.debug('stopping service {}'.format(name))

        interface = self.get_service(name)

        try:
            interface.Stop(self.mode)
            return True
        except:
            return False

    def start_service(self, name):
        '''
        Start the specified service.

        @name: D-Bus path for service.

        Returns: True on success, else False
        '''

        log.debug('starting service {}'.format(name))

        interface = self.get_service(name)

        try:
            interface.Start(self.mode)
            return True
        except:
            return False

    def restart_service(self, name):
        '''
        Restart the specified service.

        @name: D-Bus path for service
         (for example, "/org/freedesktop/systemd1/unit/cron_2eservice").

        Returns: True on success, else False
        '''
        try:
            interface = self.get_service(name)
            interface.Restart(self.mode)
            return True
        except:
            return False

    def stop_service_by_pid(self, pid):
        '''
        Stop service specified by @pid.
        '''
        unit = self.find_unit(pid)
        self.stop_service(unit)

    # FIXME: should return new pid
    def restart_service_by_pid(self, pid):
        '''
        Restart service specified by @pid.
        '''
        unit = self.find_unit(pid)
        self.restart_service(unit)

    # FIXME:
    #
    # We should use D-Bus here, rather than calling the binary.
    #
    # However, even if we use D-Bus, currently the D-Bus logic will fail
    # since after a restart, systemd will sever all D-Bus connections
    # and even after closing the connection, deleting the
    # appropriate objects, and recreating those objects, method calls
    # (such as self.version()) hang for approximately 30 seconds and
    # then fail with:
    #
    # dbus.exceptions.DBusException: org.freedesktop.DBus.Error.NoReply
    #
    def restart_manager(self):
        '''
        Cause systemd to re-exec itself.

        Returns: True on success, else False.
        '''

        # FIXME:
        #
        # Don't attempt to restart for now to avoid disrupting the
        # existing D-Bus connection.
        log.error('FIXME: not restarting systemd')
        return False

        log.debug('restarting systemd service manager')

        args = ['systemctl', 'daemon-reexec']
        log.debug('running: {}'.format(args))
        proc = subprocess.Popen(args,
                                stderr=subprocess.PIPE,
                                universal_newlines=True)
        if proc.wait() != 0:
            stderr = proc.communicate()[1]
            log.error('failed to restart systemd ({}): {}'
                      .format(args, stderr))
            return False

        return True


class Upgrader():

    # FIXME: Should query system-image-cli (see bug LP:#1380574).
    DEFAULT_CACHE_DIR = '/writable/cache'

    # Directory to mount writable root filesystem below the cache
    # diretory.
    MOUNT_TARGET = 'system'

    # Magic file that records time last system image was successfully applied.
    TIMESTAMP_FILE = '.last_update'

    DIR_MODE = 0o750

    def update_timestamp(self):
        '''
        Update the timestamp file to record the time the last upgrade
        completed successfully.
        '''
        file = os.path.join(self.get_cache_dir(), self.TIMESTAMP_FILE)
        open(file, 'w').close()

    def get_cache_dir(self):
        '''
        Returns the full path to the cache directory, which is used as a
        scratch pad, for downloading new images to and bind mounting the
        rootfs.
        '''
        return self.options.tmpdir \
            if self.options.tmpdir \
            else self.DEFAULT_CACHE_DIR

    def get_mount_target(self):
        '''
        Get the full path to the mount target directory.
        '''
        return os.path.join(self.get_cache_dir(), self.MOUNT_TARGET)

    def __init__(self, options, commands, remove_list):
        self.dispatcher = {
            'format': self._cmd_format,
            'load_keyring': self._cmd_load_keyring,
            'mount': self._cmd_mount,
            'unmount': self._cmd_unmount,
            'update': self._cmd_update,
        }

        # Records why a reboot is required (or REBOOT_NONE
        # if no reboot necessary)
        self.reboot_reasons = Enum('REBOOT_REASON',
                                   'REBOOT_NONE ' +
                                   'REBOOT_BOOTME ' +
                                   'REBOOT_SERVICE ' +
                                   'REBOOT_OPEN_FILE ' +
                                   'REBOOT_NEW_IMAGE ')

        # List of recognised methods the upgrader supports
        # to upgrade a system.
        #
        # If dual root filesystem partitions are found, the upgrader
        # will upgrade to the "other" partition. Otherwise, an in-place
        # upgrade will be applied.
        self.upgrade_types = Enum('UPGRADE_TYPE',
                                  'UPGRADE_IN_PLACE ' +
                                  'UPGRADE_AB_PARTITIONS ')

        self.options = options

        # array of imperative commands to run, as generated by
        # system-image-cli(1).
        self.commands = commands

        # files to remove before unpacking new image.
        self.remove_list = remove_list
        self.full_image = False

        # path => inode map set by save_links()
        self.file_map = {}

        self.lost_found = '/lost+found'

        # Set on any of the following:
        #
        # - Failure to restart all services.
        # - Failure to determine service associated with a pid.
        # - System-image requested it ('bootme' flag).
        self.reboot_reason = self.reboot_reasons.REBOOT_NONE

        self.removed_file = TAR_FILE_REMOVED_FILE

        # Identify the directory the files the command file refers to
        # live in.
        if self.options.cmdfile:
            self.file_dir = os.path.dirname(self.options.cmdfile)

        self.systemd = Systemd()

        # list of systemd unit D-Bus paths (such as
        # '/org/freedesktop/systemd1/unit/cron_2eservice') that need to
        # be restarted since they have running processes that are
        # holding open files that the upgrader needs to replace.
        self.services = []

        # If True, the other partition is considered empty. In this
        # scenario, prior to unpacking the latest image, the _current_
        # rootfs image is copied to the other partition.
        #
        # Note: Only used by UPGRADE_IN_PLACE.
        self.other_is_empty = False

        target = self.get_mount_target()

        # Note that we need the stat of the mountpoint,
        # *NOT* the device itself.
        self.mountpoint_stat = os.stat(target)

    def set_reboot_reason(self, reason):
        '''
        Set the reboot reason, if not already set. This ensures the
        primary reason is retained.
        '''
        if self.reboot_reason != self.reboot_reasons.REBOOT_NONE:
            # ignore
            return

        self.reboot_reason == reason

    def determine_upgrade_type(self):

        self.current_rootfs_device = self.get_rootfs()

        # Determine what sort of upgrade will be performed
        if uses_ab_partitions():
            self.upgrade_type = self.upgrade_types.UPGRADE_AB_PARTITIONS
            self.other_rootfs_device = self.get_other_rootfs()
            self.rootfs_to_modify = self.other_rootfs_device
        else:
            self.upgrade_type = self.upgrade_types.UPGRADE_IN_PLACE
            self.rootfs_to_modify = self.current_rootfs_device

    def prepare(self):
        '''
        Required setup.
        '''

        self.determine_upgrade_type()

        log.debug('System is using systemd version {}'
                  .format(self.systemd.version()))

        log.debug('system upgrade type is {}'.format(self.upgrade_type))
        if self.options.debug > 1:
            log.debug('current rootfs device is {}'
                      .format(self.current_rootfs_device))
            log.debug('rootfs to update is {}'
                      .format(self.rootfs_to_modify))

        if self.options.force_inplace_upgrade:
            self.upgrade_type = self.upgrade_types.UPGRADE_IN_PLACE
            self.rootfs_to_modify = self.current_rootfs_device
            log.debug('forcing upgrade type to be {}'
                      .format(self.upgrade_type))

        if self.options.dry_run:
            return

        if self.options.clean_only:
            self.cleanup_inodes()
            return

        if self.options.root_dir != '/':
            # Don't modify root when running in test mode
            return

        if self.upgrade_type != self.upgrade_types.UPGRADE_IN_PLACE:
            return

        # Necessary since systemd makes the rootfs shared, which allows
        # mount operations visible across all mount namespaces.
        log.debug('making {} private'.format(self.options.root_dir))

        make_mount_private(self.options.root_dir)

        # Unmount all the bind mounts to avoid any possibility
        # of writing to the writable partition.
        mounts = get_writable_mounts()

        log.debug('unmounting writable partitions')

        lazy_unmount_specified(mounts)

        self.cleanup_inodes()

    def run(self):
        '''
        Execute the commands in the command file
        '''
        self.prepare()

        if self.options.clean_only:
            return

        for cmdline in self.commands:
            cmdline = cmdline.strip()

            args = cmdline.split()
            cmd = args[0]
            args = args[1:]

            if self.dispatcher[cmd]:
                log.debug('running dispatcher {} ({})'.format(cmd, args))
                self.dispatcher[cmd](args)
            else:
                log.warning('ignoring bogus input line: {}'
                            .format(cmdline))
        self.finish()

    def finish(self):
        '''
        Final tidy-up.
        '''

        if self.options.dry_run:
            self.show_outcome()
            return

        if self.options.leave_files:
            log.debug('not removing files')
        else:
            for file in self.remove_list:
                log.debug('removing file {}'.format(file))
                os.remove(file)

        # Don't remount or reboot in test mode.
        if self.options.root_dir != '/':
            return

        if self.upgrade_type == self.upgrade_types.UPGRADE_IN_PLACE:
            log.debug('remounting all writable partitions')
            mount_all()

            try:
                prefix = self.get_saved_link_dir_prefix()
                os.rmdir(prefix)
            except:
                # there must be remaining links, so leave them to be cleaned
                # up via the initramfs.
                pass

        self.show_outcome()

        self.update_timestamp()

        if self.reboot_reason == self.reboot_reasons.REBOOT_NONE:
            return

        # A system with A/B partitions should not automatically reboot;
        # let snappy advise the admin that a reboot should be performed.
        if self.upgrade_type == \
                self.upgrade_types.UPGRADE_AB_PARTITIONS:
                return

        if self.options.no_reboot:
            log.warning('Not rebooting at user request')
            return

        # give the admin a chance to see the message
        log.debug('Waiting for {} seconds before rebooting'
                  .format(self.options.reboot_delay))
        sleep(self.options.reboot_delay)
        os.system('/sbin/reboot')

    def _cmd_format(self, args):
        try:
            target = args[0]
        except:
            log.warning('expected target')
            return

        if target == 'system':
            self.full_image = True

        # Don't modify the system state
        if self.options.dry_run or self.options.root_dir != '/':
            return

        log.warning('ingoring format target {} (unsupported operation)'
                    .format(target))
        return

    def _cmd_load_keyring(self, args):
        try:
            keyring = args[0]
            signature = args[1]
        except:
            log.warning('expected keyring and signature')
            return

        keyring = os.path.join(self.file_dir, keyring)
        signature = os.path.join(self.file_dir, signature)

        self.remove_list.append(keyring)
        self.remove_list.append(signature)

        # already handled by system-image-cli(1) on image download
        log.info('ignoring keyring {} and signature {}'
                 ' (already handled)'
                 .format(keyring, signature))

    def _cmd_mount(self, args):
        """
        Although the name of this method matches the verb in the command
        file, it actually remounts "other" r/w now since dual partition
        systems have the other partition permanently mounted read-only.
        """

        try:
            target_type = args[0]
        except:
            log.warning('expected target type')
            return

        # the mount command is an imperative one: this script decides
        # what to mount and where :)
        if target_type != 'system':
            log.warning('unknown mount target type: {}'
                        .format(target_type))
            return

        self.remount_rootfs(writable=True)

        if self.upgrade_type != self.upgrade_types.UPGRADE_AB_PARTITIONS:
            return

        if self.other_is_empty:
            # Copy current rootfs data to the other rootfs's blank
            # partition.
            log.debug('syncing root partitions')
            self.sync_partitions()

    def _cmd_unmount(self, args):
        """
        Although the name of this method matches the verb in the command
        file, it actually remounts "other" r/o now since dual partition
        systems have the other partition permanently mounted read-only.
        """

        try:
            target_type = args[0]
        except:
            log.warning('expected target type')
            return

        # the unmount command is an imperative one: this script decides
        # what to mount and where :)
        if target_type != 'system':
            log.warning('unknown mount target type: {}'
                        .format(target_type))
            return

        self.remount_rootfs(writable=False)

    def remount_rootfs(self, writable=False):
        target = self.get_mount_target()

        if writable:
            # ro->rw so need to fsck first.
            unmount(target)

            root = self.get_other_rootfs()

            # needs to be mounted writable, so check it first!
            fsck(root)

            mount(root, target, "rw")

            channel_ini = '{}/{}'.format(target,
                                         SYSTEM_IMAGE_CHANNEL_CONFIG)
            channel_ini = os.path.normpath(channel_ini)

            if not os.path.exists(channel_ini):
                log.debug('root partition on {} is empty'.format(root))
                self.other_is_empty = True
        else:
            # rw->ro so no fsck required.
            remount(target, "ro")

    def get_rootfs(self):
        '''
        Returns the full path to the currently booted root partition.
        '''
        roots = get_root_partitions_by_label()
        return roots[0][1] if roots[0][2] else roots[1][1]

    def get_other_rootfs(self):
        '''
        Returns the full device path to the "other" partition for
        systems that have dual root filesystems (A/B partitions)
        '''
        roots = get_root_partitions_by_label()
        return roots[0][1] if not roots[0][2] else roots[1][1]

    def show_other_partition_details(self):
        '''
        Mount the other partition and dump the contents of the
        channel.ini file to stdout.
        '''
        self.determine_upgrade_type()

        if not uses_ab_partitions():
            log.error('System does not have dual root partitions')
            sys.exit(1)

        target = self.get_mount_target()

        file = os.path.normpath('{}/{}'
                                .format(target,
                                        SYSTEM_IMAGE_CHANNEL_CONFIG))

        log.debug('Reading file from other partition: {}'.format(file))

        try:
            with open(file, 'r') as f:
                sys.stdout.write(f.read())
        except:
            # no output by default, denoting an error
            log.debug('Cannot find file on other partition: {}'
                      .format(file))

    def show_outcome(self):
        '''
        Show the user whether a reboot is required and if so, why.
        '''
        log.info('System update completed for rootfs on {}'
                 .format(self.rootfs_to_modify))

        if self.reboot_reason == self.reboot_reasons.REBOOT_NONE:
            log.info('System update completed - no reboot required')
        else:
            log.warning('System update requires a reboot to finalise')
            log.warning('(Reboot reason: {})'
                        .format(self.reboot_reason))

    def get_saved_link_path(self, file):
        '''
        Returns the full path to the hard-link copy of the inode
        associated with @file.
        '''
        prefix = self.get_saved_link_dir_prefix()
        link_path = os.path.normpath('{}/{}'.format(prefix, file))

        return link_path

    def get_saved_link_dir(self, file):
        '''
        Returns the full path to the directory where the hard-link inode
        copy for file @file is located.
        '''
        prefix = self.get_saved_link_dir_prefix()
        dirname = os.path.dirname(file)
        saved_link_dirname = os.path.normpath('{}{}'.format(prefix, dirname))

        return saved_link_dirname

    def get_saved_link_dir_prefix(self):
        '''
        Returns the full path to the directory under which the
        hard-link inode file copies are stored.
        '''
        self.make_lost_and_found()

        return os.path.join(self.lost_found,
                            'ubuntu-core-upgrader')

    def make_lost_and_found(self):
        '''
        Create a lost+found directory on the root filesystem.
        '''
        if os.path.exists(self.lost_found) and \
            os.path.isdir(self.lost_found) and \
                not os.path.islink(self.lost_found):
                return

        cwd = os.getcwd()
        os.chdir('/')

        if os.path.islink(self.lost_found) or os.path.isfile(self.lost_found):
                os.remove(self.lost_found)

        cmd = 'mklost+found'
        args = [cmd]
        proc = subprocess.Popen(args,
                                stderr=subprocess.PIPE,
                                universal_newlines=True)
        if proc.wait() != 0:
            stderr = proc.communicate()[1]
            log.error('failed to run {}: {}'.format(cmd, stderr))
            sys.exit(1)

        os.chdir(cwd)

        # create a subdirectory to work in
        dir = os.path.join(self.lost_found, 'ubuntu-core-upgrader')
        try:
            os.makedirs(dir, mode=self.DIR_MODE, exist_ok=True)
        except Exception as e:
            log.error('cannot create directory {}: {}'
                      .format(dir, e))
            sys.exit(1)

    def get_file_contents(self, tar, file):
        '''
        @tar: tarfile object.
        @file: full path to file within @tar to extract.

        Returns: contents of @file from within @tar.
        '''
        tmpdir = tempfile.mkdtemp(prefix=script_name)
        tar.extract(path=tmpdir, member=tar.getmember(file))

        path = os.path.join(tmpdir, file)
        lines = [line.rstrip() for line in open(path, 'r')]

        shutil.rmtree(tmpdir)

        return lines

    def cleanup_inodes(self):
        '''
        Remove stale inodes from below '/lost+found'.

        If the upgrader has already been run and forced a reboot
        due to unknown processes holding open inodes, hard links will
        still exist below '/lost+found'.

        These need to be removed before the current upgrade process can
        continue since it the next upgrade may affect the same files as
        last time.

        Cleanup up via the upgrader isn't ideal since inodes are being
        wasted in the time window between calls to the upgrader.
        However, the only other alternatives are to perform the
        cleanup at boot (initramfs) or shutdown and these options are
        not ideal as problems could result in an unbootable system (they
        would also require yet more toggling of the root FS to rw and we
        try to avoid that wherever possible).
        '''
        # Create the directory the file needs to live under
        path = self.get_saved_link_dir_prefix()

        if os.path.exists(path):
            log.debug('Cleaning up stale inodes from previous run')
            shutil.rmtree(path)

    def save_links(self, files):
        '''
        @files: list of files that will be modified by the
         upgrade.

        Create a new (hard) link to all files that will be modified by
        the system-image upgrade.

        Also updates the file_map which maps the paths for the files to
        change to their current inode.

        We cannot just remove files from the root partition blindly
        since the chances are some of the files we wish to remove are
        currently being used by running processes (either directly via
        open(2) calls, or indirectly via the link loader pulling in
        required shared libraries).

        Technically, we *can* remove any file, but we will never be able
        to make the filesystem read-only again as it is now in an
        inconsistent state (since the unlinked inodes are still in use).
        This manifests itself with the dreaded 'mount: / busy' message
        when attempting to flip the rootfs to be read-only once again.

        The solution is to hard-link all the files that we are about to
        either change or delete into '/lost+found/', retaining their
        original directory structure. We can then unlink the master copy,
        but retain another copy of the same inode.

        The kernel is then happy to allow us to remount the rootfs
        read-only once again.
        '''

        for file in files:

            if file.startswith('/dev/') and not os.path.exists(file):
                continue

            if file.startswith('/run/'):
                continue

            if not os.path.exists(file):
                # ignore files that don't exist - they must be new files
                # that are about to be created by the new system image.
                continue

            st = os.stat(file)
            mode = st.st_mode

            if stat.S_ISDIR(mode):
                # ignore directories (and sym-links to directories)
                continue

            if stat.S_ISCHR(mode) or stat.S_ISBLK(mode):
                # ignore devices
                continue

            # save original inode number
            self.file_map[file] = st.st_ino

            if self.options.dry_run:
                continue

            # File is crosses the filesytem boundary,
            # so can only be ignored.
            #
            # It will either be a sym-link or a bind-mount
            # unrelated to those specified by writable-paths(5).
            if st.st_dev != self.mountpoint_stat.st_dev:
                if self.options.debug > 1:
                    log.debug('Ignoring cross-FS file: {}'
                              .format(file))
                continue

            # Create the directory the file needs to live under
            saved_link_dirname = self.get_saved_link_dir(file)

            # First, create the directory structure.
            #
            # Note that we cannot handle the deletion of directories that
            # a process has open (readdir(2)) as you can't create a hard
            # link to a directory.
            #
            # FIXME: consider:
            #  - sym links to files.
            #  - sym links to directories.
            #
            try:
                if self.options.debug > 1:
                    log.debug('creating directory {}'
                              .format(saved_link_dirname))
                os.makedirs(saved_link_dirname,
                            mode=self.DIR_MODE,
                            exist_ok=True)
            except Exception as e:
                log.error('failed to create directory {}: {}'
                          .format(saved_link_dirname, e))
                sys.exit(1)

            link_path = self.get_saved_link_path(file)

            # Now, create a hard link to the original file in the
            # directory.
            log.debug('linking {} to {}'.format(file, link_path))

            try:
                os.link(file, link_path)
            except Exception as e:
                log.error('failed to create link for file {}: {}'
                          .format(file, e))
                sys.exit(1)

    def remove_links(self, exclude_list):
        '''
        @exclude_list: list of files whose hard links below /lost+found should
        NOT be removed. Note the each element is a full rootfs path.

        Remove all hard links to inodes below /lost+found except those
        specified by the files in "exclude_list.
        '''

        if self.options.dry_run:
            return

        prefix = self.get_saved_link_dir_prefix()

        # Remove all the files that are not in @exclude_list
        for root, dirs, files in os.walk(self.get_saved_link_dir_prefix(),
                                         topdown=False):
            for file in files:
                link_path = os.path.join(root, file)

                path = link_path[len(prefix):]

                if path in exclude_list:
                    log.debug('not removing in-use file {}'
                              .format(file))
                    continue

                log.debug('removing link {}'.format(link_path))
                os.unlink(link_path)

        # Now, remove all the directories we can.
        # We KISS by just attempting to remove every directory. Removals
        # will fail if the directory is not empty (as will happen when
        # they contain a link to a file that is still in use), so we
        # ignore such errors but any other is fatal.
        for root, dirs, files in os.walk(self.get_saved_link_dir_prefix(),
                                         topdown=False):
            for dir in dirs:
                try:
                    os.rmdir(os.path.join(root, dir))
                except OSError as e:
                    if e.errno != errno.ENOTEMPTY:
                        raise e

    def tar_generator(self, members):
        '''
        Generator function to handle extracting members from the system
        image tar file.
        '''
        for member in members:

            # Restrictions:
            #
            # - Don't unpack the removed file.
            # - Don't unpack device files *iff* they already exist.
            # - Don't unpack files that are not below
            #   TAR_FILE_SYSTEM_PREFIX or TAR_FILE_ASSETS_PREFIX.
            #
            # - XXX: This is a temporary function to work-around for
            #   LP: #1381121: we shouldn't need to filter which files are
            #   extracted!
            device_prefix = '{}dev/'.format(TAR_FILE_SYSTEM_PREFIX)

            mount_path = '{}/{}'.format(self.get_cache_dir(), member.name)

            unpack = True

            if member.name == self.removed_file:
                # already handled
                unpack = False

            if member.name.startswith(device_prefix) and \
                    os.path.exists(mount_path):
                # device already exists
                unpack = False

            if not (member.name.startswith(TAR_FILE_SYSTEM_PREFIX) or
                    member.name.startswith(TAR_FILE_ASSETS_PREFIX)):
                # unrecognised prefix directory
                unpack = False

            if not unpack:
                log.debug('not unpacking file {}'.format(member.name))
            else:
                # A modified root directory requires convering
                # absolute paths to be located below the modified root
                # directory.
                if self.options.root_dir != '/':
                    base = remove_prefix(member.name)
                    member.name = '{}{}'.format(self.options.root_dir, base)

                    if member.type in (tarfile.LNKTYPE, tarfile.SYMTYPE) \
                            and member.linkname.startswith('/'):
                        # Hard and symbolic links also need their
                        # 'source' updated to take account of the root
                        # directory.
                        #
                        # But rather than remove the prefix, we add the
                        # root directory as a prefix to contain the link
                        # within that root.
                        base = os.path.join(self.options.root_dir,
                                            member.linkname)

                        member.linkname = '{}{}'.format(self.options.root_dir,
                                                        base)
                else:
                    path = mount_path

                log.debug('unpacking file {}'.format(member.name))

                # If the file is a binary and that binary is currently
                # being executed by a process, attempting to unpack it
                # will result in ETXTBSY (OSError: [Errno 26] Text file
                # busy). The simplest way around this issue is to unlink
                # the file just before unpacking it (ideally, we'd catch
                # the exception and handle it separately). This allows
                # the unpack to continue, and the running process to
                # continue to use it's (old) version of the binary until
                # it's corresponding service is restarted.
                #
                # Note that at this point, we already have another copy
                # of the inode below /lost+found/.
                if not member.isdir() and os.path.lexists(path):
                    log.debug('removing file {}'.format(path))
                    os.unlink(path)

                yield member

    def _cmd_update(self, args):
        '''
        Unpack a new system image.

        Note that this method will be called multiple times, once for
        each file that comprises the upgrade.
        '''
        try:
            file = args[0]
            signature = args[1]
        except:
            log.warning('expected file and signature')
            return

        file = os.path.join(self.file_dir, file)
        signature = os.path.join(self.file_dir, signature)

        for f in (file, signature):
            if not os.path.exists(f):
                log.warning('ignoring missing file {}'
                            .format(f))
                return

            self.remove_list.append(f)

        log.info('applying update: {}'.format(file))
        tar = tarfile.open(file)

        found_removed_file = False

        # start with a list of all the files in the tarfile
        all_files = tar.getnames()

        if self.removed_file in all_files:
            # Exclude the removed file from the normal list as it is not
            # to be unpacked on the normal filesystem.
            all_files.remove(self.removed_file)
            found_removed_file = True

        # convert the paths back into real filesystem paths
        # (by dropping the 'system/' prefix).
        all_files = list(map(remove_prefix, all_files))

        # remove entries for files which failed the remove_prefix() check
        all_files = [x for x in all_files if x is not None]

        if found_removed_file:
            to_remove = self.get_file_contents(tar, self.removed_file)
        else:
            to_remove = []

        if self.upgrade_type == self.upgrade_types.UPGRADE_IN_PLACE:
            # Add all files to remove not already in the list of all
            # files that will be affected by the upgrade. This overall
            # list allows the upgrader to backup files in the case of
            # in-place upgrades.
            for f in to_remove:

                system_path = remove_prefix(f)

                # path should now be absolute
                if not system_path.startswith('/'):
                    continue

                # ignore relative paths
                if '../' in system_path:
                    continue

                if system_path not in all_files:
                    all_files.append(f)

            log.debug('saving inode details')
            self.save_links(all_files)

        if self.upgrade_type == self.upgrade_types.UPGRADE_AB_PARTITIONS:
            # An update to the "other" partition should always flag
            # a reboot since the image on that partition is newer.
            self.reboot_reason = self.reboot_reasons.REBOOT_NEW_IMAGE

        if not self.full_image and not self.options.check_reboot:

            if found_removed_file:
                log.debug('processing {} file'
                          .format(self.removed_file))

                # process backwards to work around bug LP:#1381134.
                for remove in sorted(to_remove, reverse=True):
                    remove = remove.strip()

                    # don't remove devices
                    if remove.startswith('{}dev/'
                                         .format(TAR_FILE_SYSTEM_PREFIX)):
                        continue

                    # The upgrader runs as root so can modify any
                    # file. However, we should still check to ensure the
                    # paths look "reasonable". Since the server
                    # should only ever specify absolute paths,
                    # ignore anything that isn't.
                    if '../' in remove:
                        continue

                    if self.options.root_dir == '/':
                        final = os.path.join(self.get_cache_dir(), remove)
                    else:
                        # os.path.join() refuses to work if the file
                        # begins with a slash.
                        base = remove_prefix(remove)
                        final = '{}{}'.format(self.options.root_dir,
                                              base)

                    if not os.path.exists(final):
                        # This scenario can only mean there is a bug
                        # with system-image generation (or someone made
                        # the image writable and removed some files
                        # manually).
                        log.debug('ignoring non-existent file {}'
                                  .format(final))
                        continue

                    if self.options.dry_run:
                        log.info('DRY-RUN: would remove file {}'.format(final))
                        continue

                    log.debug('removing file {}'.format(final))
                    try:
                        if os.path.isdir(final) and not os.path.islink(final):
                            shutil.rmtree(final)
                        else:
                            os.remove(final)
                    except Exception as e:
                        log.warning('failed to remove {}: {}'
                                    .format(final, e))

        if self.options.dry_run:
            if not self.options.check_reboot:
                log.info('DRY-RUN: would apply the following files:')
                tar.list(verbose=True)
        else:
            log.debug('starting unpack')
            # see bug #1408579, this forces tarfile to use tarinfo.gid
            # instead of looking up the gid by local name which might
            # be incorrect
            from unittest.mock import patch
            with patch("grp.getgrnam") as m:
                m.side_effect = KeyError()
                tar.extractall(path=self.get_cache_dir(),
                               members=self.tar_generator(tar))
        tar.close()

        if self.upgrade_type == self.upgrade_types.UPGRADE_AB_PARTITIONS:
            # Nothing further to do
            return

        # Look for pids that still have the original inodes for the old
        # file versions open.
        self.pid_file_map = get_affected_pids(list(self.file_map.values()))

        if len(self.pid_file_map):
            log.debug('processes are using new image files from {}'
                      .format(file))
            self.set_reboot_reason(self.reboot_reasons.REBOOT_OPEN_FILE)
        else:
            log.debug('no processes are using new image files from {}'
                      .format(file))

        # Remove what hard links we can (namely all those that do not
        # relate to running processes that have the files open).
        if self.upgrade_type == self.upgrade_types.UPGRADE_IN_PLACE:
            self.remove_links(list(self.pid_file_map.keys()))

        if len(self.pid_file_map.keys()) == 0:
            # there were no processes holding inodes open, so nothing
            # more to be done.
            return

        if self.options.check_reboot:
            return

        if not self.restart_associated_services():
            self.set_reboot_reason(self.reboot_reasons.REBOOT_SERVICE)
        else:
            # if all services were restarted, all remaining hard-links can
            # be removed, hence pass an empty exclude list.
            self.remove_links([])

    def sync_partitions(self):
        '''
        Copy all rootfs data from the current partition to the other
        partition.

        XXX: Assumes that the other rootfs is already mounted to
        mountpoint get_mount_target().
        '''
        target = self.get_mount_target()
        bindmount_rootfs_dir = tempfile.mkdtemp(prefix=script_name,
                                                dir=self.get_cache_dir())
        bind_mount("/", bindmount_rootfs_dir)

        cwd = os.getcwd()
        os.chdir(bindmount_rootfs_dir)

        cmd = '/bin/cp'
        args = [cmd, '-a', '.', target]

        log.debug('running (from directory {}): {}'
                  .format(bindmount_rootfs_dir, args))

        proc = subprocess.Popen(args,
                                stdout=subprocess.DEVNULL,
                                stderr=subprocess.DEVNULL,
                                universal_newlines=True)
        if proc.wait() != 0:
            os.chdir(cwd)
            log.error('failed to sync partitions')
            sys.exit(1)

        os.chdir(cwd)

        unmount(bindmount_rootfs_dir)
        os.rmdir(bindmount_rootfs_dir)
        self.other_is_empty = False

    def restart_associated_service(self, pid, file):
        '''
        Lookup the service associated with @pid and restart.

        Returns True on success and False if either service could not be
        established, or the restart failed.
        '''

        if pid == 1:
            # As always, this is special :)
            if self.options.dry_run:
                log.info('DRY-RUN: would restart systemd (pid {})'
                         .format(pid))
                return True
            else:
                log.debug('restarting systemd (pid {})' .format(pid))
            return self.systemd.restart_manager()
        else:
            service = self.systemd.find_unit(pid)
            if not service:
                log.debug('cannot determine service for pid {}'
                          .format(pid))
                # trigger a reboot
                return False

            if self.options.dry_run:
                log.info('DRY-RUN: would restart service {} (pid {})'
                         .format(service, pid))
                return True
            else:
                cmd = get_command(pid) or '<<UNKNOWN>>'
                log.debug('restarting service {} ({}, pid {}) '
                          'holding file {} open'
                          .format(service, cmd, pid, file))

            return self.systemd.restart_service(service)

    def restart_associated_services(self):
        '''
        Restart all services associated with the pids in self.pid_file_map.

        Returns False if any service failed to start, else True.
        '''
        failed = False

        for file in self.pid_file_map:
            for pid in self.pid_file_map[file]:
                ret = self.restart_associated_service(pid, file)
                if not ret:
                    failed = True

        return not failed
