import json
import os
import select
import subprocess
import time

from systemimage.config import config


class ApplyUpgradeError(Exception):
    pass


def is_sym_link_broken(path):
    '''
    Returns True if the specified path is a broken symbolic link, else
    False.
    '''
    try:
        os.lstat(path)
        os.stat(path)
    except:
        return True
    return False


def newest_mtime(directory):
    """
    Find the most recently modified file in a directory.
    Ignores broken symlinks.
    """
    files = [f for f in os.listdir(directory)
             if not is_sym_link_broken(os.path.join(directory, f))]

    return max([os.path.getmtime(os.path.join(directory, f)) for f in files])


class ApplyUpgrade:
    """Hook for system-image-cli"""

    UPGRADER = ["ubuntu-core-upgrade"]
    UPGRADER_ARGS = ["/writable/cache/ubuntu_command"]

    def apply_update(self):
        p = subprocess.Popen(
            self.UPGRADER + self.UPGRADER_ARGS,
            stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
        # collect the output while iterating to avoid hitting the 64k
        # pipe size limit (unlikely but...)
        stdout = ""
        while p.poll() is None:
            # send alive ping
            if config.dbus_service:
                config.dbus_service.UpdateProgress(-1, -1)
            if getattr(config, "machine_readable", False):
                print("SPINNER:"+json.dumps({}))
            time.sleep(0.1)
            if select.select([p.stdout.fileno()], [], []) != ([], [], []):
                stdout += p.stdout.read().decode("utf-8", errors="replace")
        # get return code
        ret_code = p.poll()
        # close the pipes
        p.communicate()
        if ret_code != 0:
            raise ApplyUpgradeError(
                "%s exited with %s:\n%s", self.UPGRADER, ret_code, stdout)
        return True

    # FIXME: this is the entry point that system-image-cli uses, we need
    #        to teach it that the name should be something more generic
    #        than reboot
    #
    # Note that this reboot logic is specific to in-place upgrades.
    def reboot(self):
        old_mtime_in_boot = newest_mtime("/boot")
        self.apply_update()
        if newest_mtime("/boot") > old_mtime_in_boot:
            print("Need reboot")
            subprocess.check_call(["/sbin/reboot"])
