# FIXME: ignore our own pid (and our parents).

# --------------------------------------------------------------------
# Copyright © 2014 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# --------------------------------------------------------------------

'''
Apply an Ubuntu Core system image update to the running system.

WARNING : Should be run within a new mount namespace to avoid
          disturbing the existing (read-only) root filesystem.

NOTE    : The signature files associated with each image file is *NOT*
          re-verified, since this must already have been handled by
          system-image-cli(1).
'''

import sys
import os
import logging
import shutil
import subprocess
import tempfile
import tarfile
import stat
import errno

from enum import Enum
from time import sleep

script_name = os.path.basename(__file__)

log = logging.getLogger()

# Name of user-data partition label as created by
# ubuntu-device-flash(1).
USER_DATA_LABEL = 'user-data'

USER_DATA_MOUNTPOINT = '/userdata'

# The tar file contains a 'system/' directory with the new files
# to apply to the real system. It may also contain a top-level
# file called 'removed' that lists files relative to the system
# mount for files on the system that should be removed before
# unpacking the rest of the archive.
#
# Examples:
#
# - if the file '/etc/foo' should be removed, it
#   would be specified in the removed_file as 'system/etc/foo'.
#
# - if the file '/etc/bar' should be modified, it
#   would be specified in the tar file as member 'system/etc/bar'.
TAR_FILE_SYSTEM_PREFIX = 'system/'
TAR_FILE_REMOVED_FILE = 'removed'


def remove_prefix(path, prefix=TAR_FILE_SYSTEM_PREFIX):
    '''
    Remove specified prefix from path and return the result.

    Prefix must end with a slash.

    If @prefix is not a prefix of @path, returns None.
    '''

    assert(prefix.endswith('/'))

    i = path.find(prefix)

    if i < 0:
        return None

    # ensure that the returned path has a leading slash
    return path[len(prefix)-1:]


def get_command(pid):
    '''
    Returns full path to binary associated with @pid, or None if the
    lookup failed.
    '''
    try:
        exe = os.readlink('/proc/' + str(pid) + '/exe')
    except:
        # pid probably went away
        return None

    return exe


def get_userdata_disk():
    '''
    Establish the disk partition for the writable user-data partition.
    '''
    cmd = "blkid -L '{}'".format(USER_DATA_LABEL)

    output = subprocess.getoutput(cmd)
    return output.rstrip()


def bind_mount(source, target):
    '''
    Bind mount @source to @target.
    '''

    args = ['mount', '-o', 'bind', source, target]
    proc = subprocess.Popen(args,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)
    if proc.wait() != 0:
        stderr = proc.communicate()[1]
        log.error('failed to mount ({}): {}'
                  .format(args, stderr))
        sys.exit(1)


def restart_systemd():
    '''
    Cause systemd to re-exec itself.
    '''

    args = ['systemctl', 'daemon-reexec']
    proc = subprocess.Popen(args,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)
    if proc.wait() != 0:
        stderr = proc.communicate()[1]
        log.error('failed to restart systemd ({}): {}'
                  .format(args, stderr))
        sys.exit(1)


def restart_service(name):
    '''
    Restart the specified service.

    Returns True on success, else False.
    '''

    args = ['systemctl', 'restart', name]
    proc = subprocess.Popen(args,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)
    if proc.wait() != 0:
        stderr = proc.communicate()[1]
        log.error('failed to restart service ({}): {}'
                  .format(args, stderr))
        return False

    return True


def mount_all():
    '''
    Mount all filesystems if not already mounted.
    '''

    args = ['mount', '-a']
    proc = subprocess.Popen(args,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)

    if proc.wait() != 0:
        stderr = proc.communicate()[1]
        log.error('failed to mount all ({}): {}'
                  .format(args, stderr))


def unmount(target):
    '''
    Unmout the specified mount target
    '''

    log.debug('unmounting {}'.format(target))

    args = ['umount', '--lazy', target]
    proc = subprocess.Popen(args,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)

    if proc.wait() != 0:
        stderr = proc.communicate()[1]
        log.error('failed to unmount ({}): {}'
                  .format(args, stderr))
        # its a lazy umount, do not fail hard just now
        # sys.exit(1)


def make_mount_private(target):
    '''
    Make the specified filesystem private.
    '''

    args = ['mount', '--make-rprivate', target]
    proc = subprocess.Popen(args,
                            stderr=subprocess.PIPE,
                            universal_newlines=True)

    if proc.wait() != 0:
        stderr = proc.communicate()[1]
        log.error('failed to make {} private ({}): {}'
                  .format(target, args, stderr))


def unmount_specified(mounts):
    '''
    Unmount all targets specified by @mounts.
    '''

    for mount in mounts:
        unmount(mount)


def find_service(pid):
    '''
    Find the service associated with @pid and return its name.
    '''

    # XXX: note that the service is looked-up using the command-line
    # tool since this uses the private socket and does not require the
    # dbus-daemon to be running.
    args = ['systemctl', 'status', str(pid)]

    log.debug('Looking for service associated with pid {}' .format(pid))

    proc = subprocess.Popen(args,
                            stdout=subprocess.PIPE,
                            stderr=subprocess.DEVNULL,
                            universal_newlines=True)
    if proc.wait() != 0:
        # No systemd service associated with this pid
        return None

    stdout = proc.communicate()[0]

    lines = stdout.split('\n')
    # first line has the format:
    #   "unit.name - some description" (209)
    # or
    #   "* unit.name - some description" (215)
    first_line = lines[0].strip("*")
    unit = first_line.split()[1]

    return unit


def restart_associated_service(pid, dry_run):
    '''
    Lookup the service associated with @pid and restart.

    Returns True on success and False if either service could not be
    established, or the restart failed.
    '''

    if pid == 1:
        # As always, this is special :)
        if dry_run:
            log.info('DRY-RUN: would restart systemd (pid {})'
                     .format(pid))
            return True
        else:
            log.debug('restarting systemd (pid {})' .format(pid))
        restart_systemd()
        return True
    else:
        service = find_service(pid)
        if not service:
            log.debug('Cannot determine service for pid {}'
                      .format(pid))
            # trigger a reboot
            return False

        if dry_run:
            log.info('DRY-RUN: would restart service {} (pid {})'
                     .format(service, pid))
            return True
        else:
            log.debug('restarting service {} (pid {})'
                      .format(service, pid))

        return restart_service(service)


def restart_services(pid_file_map, dry_run):
    '''
    Restart all services associated with the pids in @pid_file_map,
    which is a map of filenames to an array of pids holding each file
    open.

    Returns False if any service failed to start, else True.
    '''
    failed = False

    for file in pid_file_map:
        for pid in pid_file_map[file]:
            ret = restart_associated_service(pid, dry_run)
            if not ret:
                failed = True

    return not failed


def get_userdata_mounts():
    '''
    Returns a list of (bind) mounts whose source location is the
    userdata partition.

    Note that the list of bind mounts is derived from the current
    mounts. This is safer than simply checking fstab since it is
    guaranteed correct.

    Note also that the list is reverse sorted so that assuming the list
    is processed in order, child mounts will be handled before parent
    mounts.
    '''
    disk = get_userdata_disk()
    file = '/proc/mounts'

    mounts = []

    try:
        with open(file, 'r') as fh:
            for mount in fh.readlines():
                mount = mount.strip()
                fields = mount.split()

                # only consider user data mounts
                if fields[0] != disk:
                    continue

                # ignore the primary mount
                if fields[1] == USER_DATA_MOUNTPOINT:
                    continue

                mounts.append(fields[1])
    except:
        sys.exit('Failed to read command file: {}'.format(file))

    # Reverse sort to ensure each child mount is handled before its
    # parent.
    return sorted(mounts, reverse=True)


def get_affected_pids(unpack_inodes):
    '''
    @unpack_inodes: list of inodes representing the files the system image
    upgrade will modify.

    Returns a dict keyed by filename whose value is an array of pids
    that are using the file currently.

    '''

    # Command to list all open files as quickly as possible.
    #
    # XXX: Note that although we nominally only care about files open on
    # the rootfs, we don't constrain lsof to only consider such files
    # since the system-images contain files that are mounted in writable
    # partitions (examples being /var/log/dpkg.log and
    # /var/log/apt/history.log).
    #
    # By considering all files, we increase the likelihood of a reboot
    # but in doing avoid unexpected system behaviour (where a process is
    # seeking through an old copy of a config file for example).
    #
    # Discard stderr to avoid the annoying
    # 'lsof: WARNING can't stat() ...' messages.
    cmd = 'lsof -lnPR 2>/dev/null'

    output = subprocess.getoutput(cmd)

    # key: filename.
    # value: array of pids.
    pid_file_map = {}

    # Read the lsof output. Note that we do _not_ ignore deleted files
    # since in fact the files we are looking for have been deleted
    # (unlinked, but not fully).
    for line in output.split('\n'):
        fields = line.split()

        # ignore header
        if line.startswith('COMMAND'):
            continue

        # ignore files with no inode
        if fields[5] in ('netlink', 'unknown'):
            continue

        pid = int(fields[1])

        # Deleted files have one less field (size/offset).
        if len(fields) == 9 and fields[4] == 'DEL':

            potential_inode = fields[7]

            if not potential_inode.isdigit():
                continue

            inode = int(potential_inode)
            file = fields[8]
        else:
            potential_inode = fields[8]

            if not potential_inode.isdigit():
                continue

            inode = int(potential_inode)
            file = fields[9]

        # ignore kernel threads
        if file == '/':
            continue

        # ignore anything that doesn't look like a file
        if file[0] != '/':
            continue

        # ignore files that don't relate to files the upgrade will
        # change.
        if inode not in unpack_inodes:
            continue

        # create a hash of arrays / dict of lists
        if file not in pid_file_map:
            pid_file_map[file] = []

        pid_file_map[file].append(pid)

    return pid_file_map


class Parser():

    # FIXME: Should query system-image-cli (see bug LP:#1380574).
    DEFAULT_CACHE_DIR = '/userdata/cache'

    # Directory to mount writable root filesystem below the cache
    # diretory.
    MOUNT_TARGET = 'system'

    # Magic file that records time last system image was successfully applied.
    TIMESTAMP_FILE = '.last_update'

    DIR_MODE = 0o750

    def update_timestamp(self):
        '''
        Update the timestamp file to record the time the last upgrade
        completed successfully.
        '''
        file = os.path.join(self.get_cache_dir(), self.TIMESTAMP_FILE)
        open(file, 'w').close()

    def get_cache_dir(self):
        '''
        Returns the full path to the cache directory, which is used as a
        scratch pad, for downloading new images to and bind mounting the
        rootfs.
        '''
        return self.options.tmpdir \
            if self.options.tmpdir \
            else self.DEFAULT_CACHE_DIR

    def get_mount_target(self):
        '''
        Get the full path to the mount target directory.
        '''
        return os.path.join(self.get_cache_dir(), self.MOUNT_TARGET)

    def __init__(self, options, commands, remove_list):
        self.dispatcher = {
            'format': self._format,
            'load_keyring': self._load_keyring,
            'mount': self._mount,
            'unmount': self._unmount,
            'update': self._update,
        }

        self.reboot_reasons = Enum('REBOOT_REASON',
                                   'REBOOT_NONE ' +
                                   'REBOOT_BOOTME ' +
                                   'REBOOT_SERVICE ' +
                                   'REBOOT_OPEN_FILE ')

        self.options = options

        # array of imperative commands to run, as generated by
        # system-image-cli(1).
        self.commands = commands

        # files to remove before unpacking new image.
        self.remove_list = remove_list
        self.full_image = False

        # path => inode map
        self.file_map = {}

        self.lost_found = '/lost+found'

        # Set on any of the following:
        #
        # - Failure to restart all services.
        # - Failure to determine service associated with a pid.
        # - System-image requested it ('bootme' flag).
        self.reboot_reason = self.reboot_reasons.REBOOT_NONE

        self.removed_file = TAR_FILE_REMOVED_FILE

        # Identify the directory the files the command file refers to
        # live in.
        self.file_dir = os.path.dirname(self.options.cmdfile)

    def prepare(self):
        '''
        Required setup.
        '''

        if self.options.dry_run:
            return

        if self.options.clean_only:
            self.cleanup_inodes()
            return

        if self.options.root_dir != '/':
            # Don't modify root when running in test mode
            return

        # Necessary since systemd makes the rootfs shared, which allows
        # mount operations visible across all mount namespaces.
        log.debug('making {} private'.format(self.options.root_dir))

        make_mount_private(self.options.root_dir)

        # Unmount all the bind mounts to avoid any possibility
        # of writing to the userdata partition.
        mounts = get_userdata_mounts()

        log.debug('unmounting writable partitions')

        unmount_specified(mounts)

        self.cleanup_inodes()

    def run(self):
        '''
        Execute the commands in the command file
        '''
        self.prepare()

        if self.options.clean_only:
            return

        for cmdline in self.commands:
            cmdline = cmdline.strip()

            args = cmdline.split()
            cmd = args[0]
            args = args[1:]

            if self.dispatcher[cmd]:
                log.debug('running dispatcher {} ({})'.format(cmd, args))
                self.dispatcher[cmd](args)
            else:
                log.warning('ignoring bogus input line: {}'
                            .format(cmdline))
        self.finish()

    def finish(self):
        '''
        Final tidy-up.
        '''

        if self.options.dry_run:
            self.show_outcome()
            return

        if self.options.leave_files:
            log.debug('not removing files')
        else:
            for file in self.remove_list:
                log.debug('removing file {}'.format(file))
                os.remove(file)

        # Don't remount or reboot in test mode.
        if self.options.root_dir != '/':
            return

        log.debug('remounting all writable partitions')
        mount_all()

        try:
            prefix = self.get_saved_link_dir_prefix()
            os.rmdir(prefix)
        except:
            # there must be remaining links, so leave them to be cleaned
            # up via the initramfs.
            pass

        self.show_outcome()

        self.update_timestamp()

        if self.reboot_reason == self.reboot_reasons.REBOOT_NONE:
            return

        if self.options.no_reboot:
            log.warning('Not rebooting at user request')
            return

        # give the admin a chance to see the message
        log.debug('Waiting for {} seconds before rebooting'
                  .format(self.options.reboot_delay))
        sleep(self.options.reboot_delay)
        os.system('/sbin/reboot')

    def _format(self, args):
        try:
            target = args[0]
        except:
            log.warning('expected target')
            return

        if target == 'system':
            self.full_image = True

        # Don't modify the system state
        if self.options.dry_run or self.options.root_dir != '/':
            return

        log.warning('ingoring format target {} (unsupported operation)'
                    .format(target))
        return

    def _load_keyring(self, args):
        try:
            keyring = args[0]
            signature = args[1]
        except:
            log.warning('expected keyring and signature')
            return

        keyring = os.path.join(self.file_dir, keyring)
        signature = os.path.join(self.file_dir, signature)

        self.remove_list.append(keyring)
        self.remove_list.append(signature)

        # already handled by system-image-cli(1) on image download
        log.info('ignoring keyring {} and signature {}'
                 ' (already handled)'
                 .format(keyring, signature))

    def _mount(self, args):
        try:
            target_type = args[0]
        except:
            log.warning('expected target type')
            return

        # Don't modify the system state
        if self.options.dry_run:
            return

        # the mount argument is imperative - this script decides what to
        # mount and where :)
        if target_type != 'system':
            log.warning('unknown mount target type: {}'
                        .format(target_type))
            return

        target = self.get_mount_target()

        try:
            os.makedirs(target, mode=self.DIR_MODE, exist_ok=True)
        except Exception as e:
            st = os.stat(target)
            if st.mode == self.DIR_MODE:
                log.error('failed to create directory {}: {}'
                          .format(target, e))
                sys.exit(1)

        bind_mount(self.options.root_dir, target)

    def _unmount(self, args):
        try:
            target_type = args[0]
        except:
            log.warning('expected target type')
            return

        # Don't modify the system state
        if self.options.dry_run:
            return

        # the unmount argument is imperative - this script decides what to
        # mount and where :)
        if target_type != 'system':
            log.warning('unknown mount target type: {}'
                        .format(target_type))
            return

        target = self.get_mount_target()
        unmount(target)

        try:
            os.rmdir(target)
        except Exception as e:
            log.error('failed to remove directory {}: {}'
                      .format(target, e))
            sys.exit(1)

    def show_outcome(self):
        '''
        Show the user whether a reboot is required and if so, why.
        '''
        if self.reboot_reason == self.reboot_reasons.REBOOT_NONE:
            log.info('System update completed - no reboot required')
        else:
            log.warning('System update requires a reboot to finalise')
            log.warning('(Reboot reason: {})'
                        .format(self.reboot_reason))

    def get_saved_link_path(self, file):
        '''
        Returns the full path to the hard-link copy of the inode
        associated with @file.
        '''
        prefix = self.get_saved_link_dir_prefix()
        link_path = os.path.normpath('{}/{}'.format(prefix, file))

        return link_path

    def get_saved_link_dir(self, file):
        '''
        Returns the full path to the directory where the hard-link inode
        copy for file @file is located.
        '''
        prefix = self.get_saved_link_dir_prefix()
        dirname = os.path.dirname(file)
        saved_link_dirname = os.path.normpath('{}{}'.format(prefix, dirname))

        return saved_link_dirname

    def get_saved_link_dir_prefix(self):
        '''
        Returns the full path to the directory under which the
        hard-link inode file copies are stored.
        '''
        self.make_lost_and_found()

        return os.path.join(self.lost_found,
                            'ubuntu-core-upgrader')

    def make_lost_and_found(self):
        '''
        Create a lost+found directory on the root filesystem.
        '''
        if os.path.exists(self.lost_found) and \
            os.path.isdir(self.lost_found) and \
                not os.path.islink(self.lost_found):
                return

        cwd = os.getcwd()
        os.chdir('/')

        if os.path.islink(self.lost_found) or os.path.isfile(self.lost_found):
                os.remove(self.lost_found)

        cmd = 'mklost+found'
        args = [cmd]
        proc = subprocess.Popen(args,
                                stderr=subprocess.PIPE,
                                universal_newlines=True)
        if proc.wait() != 0:
            stderr = proc.communicate()[1]
            log.error('failed to run {}: {}'.format(cmd, stderr))
            sys.exit(1)

        os.chdir(cwd)

        # create a subdirectory to work in
        dir = os.path.join(self.lost_found, 'ubuntu-core-upgrader')
        try:
            os.makedirs(dir, mode=self.DIR_MODE, exist_ok=True)
        except Exception as e:
            log.error('cannot create directory {}: {}'
                      .format(dir, e))
            sys.exit(1)

    def get_file_contents(self, tar, file):
        '''
        @tar: tarfile object.
        @file: full path to file within @tar to extract.

        Returns: contents of @file from within @tar.
        '''
        tmpdir = tempfile.mkdtemp(prefix=script_name)
        tar.extract(path=tmpdir, member=tar.getmember(file))

        path = os.path.join(tmpdir, file)
        lines = [line.rstrip() for line in open(path, 'r')]

        shutil.rmtree(tmpdir)

        return lines

    def cleanup_inodes(self):
        '''
        Remove stale inodes from below '/lost+found'.

        If the upgrader has already been run and forced a reboot
        due to unknown processes holding open inodes, hard links will
        still exist below '/lost+found'.

        These need to be removed before the current upgrade process can
        continue since it the next upgrade may affect the same files as
        last time.

        Cleanup up via the upgrader isn't ideal since inodes are being
        wasted in the time window between calls to the upgrader.
        However, the only other alternatives are to perform the
        cleanup at boot (initramfs) or shutdown and these options are
        not ideal as problems could result in an unbootable system (they
        would also require yet more toggling of the root FS to rw and we
        try to avoid that wherever possible).
        '''
        # Create the directory the file needs to live under
        path = self.get_saved_link_dir_prefix()

        if os.path.exists(path):
            log.debug('Cleaning up stale inodes from previous run')
            shutil.rmtree(path)

    def save_inodes(self, all_files):
        '''
        @all_files: list of all files that will be modified by the
         upgrade.

        Create a new (hard) link to all files that will be modified by
        the system-image upgrade.

        Also updates the file_map which maps the paths for the files to
        change to their current inode.

        We cannot just remove files from the root partition blindly
        since the chances are some of the files we wish to remove are
        currently being used by running processes (either directly via
        open(2) calls, or indirectly via the link loader pulling in
        required shared libraries).

        Technically, we *can* remove any file, but we will never be able
        to make the filesystem read-only again as it is now in an
        inconsistent state (since the unlinked inodes are still in use).
        This manifests itself with the dreaded 'mount: / busy' message
        when attempting to flip the rootfs to be read-only once again.

        The solution is to hard-link all the files that we are about to
        either change or delete into '/lost+found/', retaining their
        original directory structure. We can then unlink the master copy,
        but retain another copy of the same inode.

        The kernel is then happy to allow us to remount the rootfs
        read-only once again.
        '''
        for file in all_files:

            if file.startswith('/dev/') and not os.path.exists(file):
                continue

            if file.startswith('/run/'):
                continue

            if not os.path.exists(file):
                # ignore files that don't exist - they must be new files
                # that are about to be created by the new system image.
                continue

            st = os.stat(file)
            mode = st.st_mode

            if stat.S_ISDIR(mode):
                # ignore directories (and sym-links to directories)
                continue

            if stat.S_ISCHR(mode) or stat.S_ISBLK(mode):
                # ignore devices
                continue

            # save original inode number
            self.file_map[file] = st.st_ino

            if self.options.dry_run:
                continue

            # Create the directory the file needs to live under
            saved_link_dirname = self.get_saved_link_dir(file)

            # First, create the directory structure.
            #
            # Note that we cannot handle the deletion of directories that
            # a process has open (readdir(2)) as you can't create a hard
            # link to a directory.
            #
            # FIXME: consider:
            #  - sym links to files.
            #  - sym links to directories.
            #
            try:
                if self.options.debug > 1:
                    log.debug('creating directory {}'
                              .format(saved_link_dirname))
                os.makedirs(saved_link_dirname,
                            mode=self.DIR_MODE,
                            exist_ok=True)
            except Exception as e:
                log.error('failed to create directory {}: {}'
                          .format(saved_link_dirname, e))
                sys.exit(1)

            link_path = self.get_saved_link_path(file)

            # Now, create a hard link to the original file in the
            # directory.
            log.debug('linking {} to {}'.format(file, link_path))

            try:
                os.link(file, link_path)
            except Exception as e:
                log.error('failed to create link for file {}: {}'
                          .format(file, e))
                sys.exit(1)

    def remove_links(self, exclude_list):
        '''
        @exclude_list: list of files whose hard links below /lost+found should
        NOT be removed. Note the each element is a full rootfs path.

        Remove all hard links to inodes below /lost+found except those
        specified by the files in "exclude_list.
        '''

        if self.options.dry_run:
            return

        prefix = self.get_saved_link_dir_prefix()

        # Remove all the files that are not in @exclude_list
        for root, dirs, files in os.walk(self.get_saved_link_dir_prefix(),
                                         topdown=False):
            for file in files:
                link_path = os.path.join(root, file)

                path = link_path[len(prefix):]

                if path in exclude_list:
                    log.debug('not removing in-use file {}'
                              .format(file))
                    continue

                log.debug('removing link {}'.format(link_path))
                os.unlink(link_path)

        # Now, remove all the directories we can.
        # We KISS by just attempting to remove every directory. Removals
        # will fail if the directory is not empty (as will happen when
        # they contain a link to a file that is still in use), so we
        # ignore such errors but any other is fatal.
        for root, dirs, files in os.walk(self.get_saved_link_dir_prefix(),
                                         topdown=False):
            for dir in dirs:
                try:
                    os.rmdir(os.path.join(root, dir))
                except OSError as e:
                    if e.errno != errno.ENOTEMPTY:
                        raise e

    def tar_generator(self, members):
        '''
        Generator function to handle extracting members from the system
        image tar file.
        '''
        for member in members:

            # Restrictions:
            #
            # - Don't unpack the removed file.
            # - Don't unpack device files *iff* they already exist.
            # - Don't unpack files that are not below
            #   TAR_FILE_SYSTEM_PREFIX.
            #
            # - XXX: This is a temporary function to work-around for
            #   LP: #1381121: we shouldn't need to filter which files are
            #   extracted!
            prefix = '{}dev/'.format(TAR_FILE_SYSTEM_PREFIX)

            mount_path = '{}/{}'.format(self.get_cache_dir(), member.name)

            if (member.name == self.removed_file) or \
               (member.name.startswith(prefix) and
                   os.path.exists(mount_path) or
                   not member.name.startswith(TAR_FILE_SYSTEM_PREFIX)):
                log.debug('not unpacking file {}'.format(member.name))
            else:
                # A modified root directory requires convering
                # absolute paths to be located below the modified root
                # directory.
                if self.options.root_dir != '/':
                    base = remove_prefix(member.name)
                    member.name = '{}{}'.format(self.options.root_dir, base)

                    if member.type in (tarfile.LNKTYPE, tarfile.SYMTYPE) \
                            and member.linkname.startswith('/'):
                        # Hard and symbolic links also need their
                        # 'source' updated to take account of the root
                        # directory.
                        #
                        # But rather than remove the prefix, we add the
                        # root directory as a prefix to contain the link
                        # within that root.
                        base = os.path.join(self.options.root_dir,
                                            member.linkname)

                        member.linkname = '{}{}'.format(self.options.root_dir,
                                                        base)

                log.debug('unpacking file {}'.format(member.name))
                yield member

    def _update(self, args):
        '''
        Unpack the new system image.
        '''
        try:
            file = args[0]
            signature = args[1]
        except:
            log.warning('expected file and signature')
            return

        file = os.path.join(self.file_dir, file)
        signature = os.path.join(self.file_dir, signature)

        for f in (file, signature):
            if not os.path.exists(f):
                log.warning('ignoring missing file {}'
                            .format(f))
                return

            self.remove_list.append(f)

        log.info('applying update: {}'.format(file))
        tar = tarfile.open(file)

        found_removed_file = False

        # start with a list of all the files in the tarfile
        all_files = tar.getnames()

        if self.removed_file in all_files:
            # Exclude the removed file from the normal list as it is not
            # to be unpacked on the normal filesystem.
            all_files.remove(self.removed_file)
            found_removed_file = True

        # convert the path back into a real system path
        all_files = list(map(remove_prefix, all_files))

        # remove entries for files which failed the remove_prefix() check
        all_files = [x for x in all_files if x is not None]

        if found_removed_file:
            to_remove = self.get_file_contents(tar, self.removed_file)

            # convert the path back into a real system path
            to_remove = list(map(remove_prefix, to_remove))

            # remove entries for files which failed the remove_prefix() check
            to_remove = [x for x in to_remove if x is not None]

            # add all files not already in the list
            for file in to_remove:

                # all files should now be absolute
                if not file.startswith('/'):
                    continue

                # ignore relative paths
                if '../' in file:
                    continue

                if file not in all_files:
                    all_files.append(file)

        log.debug('saving inode details')
        self.save_inodes(all_files)

        if not self.full_image and not self.options.check_reboot:

            if found_removed_file:
                log.debug('processing {} file'
                          .format(self.removed_file))

                # process backwards to work around bug LP:#1381134.
                for remove in sorted(to_remove, reverse=True):
                    remove = remove.strip()

                    # don't remove devices
                    if remove.startswith('/dev/'):
                        continue

                    # The upgrader runs as root and can modify any
                    # file. However, we should still check to ensure the
                    # paths look "reasonable". Since the server
                    # should only ever specify absolute paths,
                    # ignore anything that isn't.
                    if '../' in remove:
                        continue

                    if self.options.root_dir == '/':
                        final = os.path.join(self.get_cache_dir(), remove)
                    else:
                        # os.path.join() refuses to work if the file
                        # begins with a slash.
                        final = '{}{}'.format(self.options.root_dir, remove)

                    if not os.path.exists(final):
                        # This scenario can only mean there is a bug
                        # with system-image generation (or someone made
                        # the image writable and removed some files
                        # manually).
                        log.debug('ignoring non-existent file {}'
                                  .format(final))
                        continue

                    if self.options.dry_run:
                        log.info('DRY-RUN: would remove file {}'.format(final))
                        continue

                    log.debug('removing file {}'.format(final))
                    try:
                        if os.path.isdir(final) and not os.path.islink(final):
                            shutil.rmtree(final)
                        else:
                            os.remove(final)
                    except Exception as e:
                        log.warning('failed to remove {}: {}'
                                    .format(final, e))

        if self.options.dry_run:
            if not self.options.check_reboot:
                log.info('DRY-RUN: would apply the following files:')
                tar.list(verbose=True)
        else:
            log.debug('starting unpack')
            tar.extractall(path=self.get_cache_dir(),
                           members=self.tar_generator(tar))

        tar.close()

        # Look for pids that still have the original inodes for the old
        # file versions open.
        pid_file_map = get_affected_pids(list(self.file_map.values()))

        if len(pid_file_map):
            # only reset the reboot reason if it hasn't already been set
            if self.reboot_reason == self.reboot_reasons.REBOOT_NONE:
                self.reboot_reason = self.reboot_reasons.REBOOT_OPEN_FILE

            for file in pid_file_map:

                if not self.options.debug:
                    continue

                log.warning(
                    'file {} is held open by the following pids:'
                    .format(file))
                for pid in pid_file_map[file]:
                    cmd = get_command(pid)
                    log.warning('   {} ({})'
                                .format(pid,
                                        cmd if cmd else '<<UNKNOWN>>'))
        else:
            log.debug('no processes are using new image files from {}'
                      .format(file))

        # Remove what hard links we can (namely all those that do not
        # relate to running processes that have the files open).
        self.remove_links(list(pid_file_map.keys()))

        if len(pid_file_map.keys()) == 0:
            # there are no processes holding inodes open so nothing more
            # to be done.
            return

        if self.options.check_reboot:
            return

        if not restart_services(pid_file_map,
                                self.options.dry_run):
            self.reboot_reason = self.reboot_reasons.REBOOT_SERVICE
        else:
            # if all services were restarted, all remaining hard-links can
            # be removed, hence pass an empty exclude list.
            self.remove_links([])
