# Copyright (c) 2007 Andrew Price
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. The name of the author may not be used to endorse or promote products
#    derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import sys
import os
import stat
import datetime
from ConfigParser import SafeConfigParser, NoOptionError, NoSectionError

class User:

	def __init__(self):

		homedir = os.environ['HOME']
		self.configpath = os.path.join(homedir, '.twytrc')

		self.config = self.initialconfig()
		self.lasttimes = self.gettimes()
		self.timesdirty = False
		self.password = ""

	def initialconfig(self):

		conf = SafeConfigParser()
		if not os.path.exists(self.configpath):
			fd = open(self.configpath, "w")
			fd.close()
			os.chmod(self.configpath, stat.S_IRUSR|stat.S_IWUSR)
		else:
			fd = open(self.configpath, "r")
			conf.readfp(fd)
			fd.close()

		return conf

	def gettimes(self):

		times = []
		for i in ('friendstl', 'usertl', 'publictl'):
			try:
				for j in self.config.options(i):
					times.append((i,j,self.config.get(i,j)))
			except NoSectionError:
				pass

		return times
			

	def setconfig(self, conflist): # conflist = [(section, key, val)]

		dirty = False
		for sec, key, val in conflist:
			if not self.config.has_section(sec):
				self.config.add_section(sec)
				self.config.set(sec, key, val)
				dirty = True
			elif not self.config.has_option(sec, key):
				self.config.set(sec, key, val)
				dirty = True
			elif self.config.get(sec, key) != val:
				self.config.set(sec, key, val)
				dirty = True

		# Only do file I/O if we really need to
		if dirty:
			fd = open(self.configpath, "w")
			self.config.write(fd)
			fd.close()
			

	def getconfig(self, section, key):
		
		try:
			value = self.config.get(section, key)
			return value
		except NoOptionError:
			pass
		except NoSectionError:
			pass

		return ""

	def getusername(self):
		
		return self.getconfig('user','name')

	def getpassword(self):
		
		if self.password:
			return self.password
		return self.getconfig('user','pass')

	def setusername(self, user):

		self.setconfig([('user','name', user)])

	def setpassword(self, pwd):

		self.password = pwd

	def savepass(self, user, pwd):
		
		self.setconfig([("user", "name", user),
				("user", "pass", pwd)])
		self.setusername(user)
		self.setpassword(pwd)

	def getlastread(self, tl, usr):

		if not usr:
			usr = '%self%'

		for s, e, x in self.lasttimes:
			if tl == s and usr == e:
				return x

		return None


	def setlastread(self, tl, usr, readtime):

		if not usr:
			usr = '%self%'

		if tl in ('publictl', 'friendstl', 'usertl'):
			i = 0
			while i < len(self.lasttimes):
				t = self.lasttimes[i]
				if t[0] == tl and t[1] == usr and (t[2] != readtime):
					del self.lasttimes[i]
					break
				i += 1

			self.lasttimes.append((tl, usr, readtime))
			self.timesdirty = True


	def synctimes(self):

		if self.timesdirty:
			self.setconfig(self.lasttimes)
