/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 20/08/2001
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "tulip/SuperGraphView.h"
#include "tulip/PropertyManager.h"
#include "tulip/SelectionProxy.h"
#include "tulip/SuperGraph.h"
#include "tulip/GraphIterator.h"
#include "tulip/StableIterator.h"
#include "tulip/SelectionProxy.h"

using namespace std;
//----------------------------------------------------------------
SuperGraphView::SuperGraphView(SuperGraph *father,SelectionProxy *filter):
  SuperGraphAbstract(father),
  nNodes(0),
  nEdges(0) {
  nodeAdaptativeFilter.setAll(false);
  edgeAdaptativeFilter.setAll(false);
  //  inDegree.setAll(0);
  //  outDegree.setAll(0);
  if (filter == 0) return;
  Iterator<unsigned int> *it=0;
  try {
    it = filter->nodeProperties.findAll(true);
  } catch (ImpossibleOperation &e) {
    it=0;
  }
  Iterator<node> *iteN;
  if (it==0)
    iteN =getFather()->getNodes();
  else
    iteN = new UINTIterator<node>(it);
  
  while (iteN->hasNext()) {
    node n=iteN->next();
    if (filter->getNodeValue(n)) addNode(n);
  } delete iteN;

  try {
    it = filter->edgeProperties.findAll(true);
  } catch (ImpossibleOperation &e) {
    it=0;
  }
  Iterator<edge> *iteE;
  if (it==0)
    iteE = getFather()->getEdges();
  else
    iteE = new UINTIterator<edge>(it);

  while (iteE->hasNext()) {
    edge e=iteE->next();
    if (filter->getEdgeValue(e)) addEdge(e);
  } delete iteE;
}
//----------------------------------------------------------------
SuperGraphView::~SuperGraphView() {
  notifyDestroy(this);
}
//----------------------------------------------------------------
bool SuperGraphView::isElement(const node n) 
{return nodeAdaptativeFilter.get(n.id);}
//----------------------------------------------------------------
bool SuperGraphView::isElement(const edge e)  {
  return edgeAdaptativeFilter.get(e.id);
}
//----------------------------------------------------------------
//unsigned int SuperGraphView::deg(const node n) const {
//  return inDegree.get(n.id)+outDegree.get(n.id);
//}
//----------------------------------------------------------------
//unsigned int SuperGraphView::indeg(const node n) const {
//  return inDegree.get(n.id);
//}
//----------------------------------------------------------------
//unsigned int SuperGraphView::outdeg(const node n) const {
//  return outDegree.get(n.id);
//}
//----------------------------------------------------------------
/*
  void SuperGraphView::reverse(const edge e) {
  notifyReverseEdge(this,e);
  node src = source(e);
  node tgt = target(e);
  outDegree.set(src.id, outDegree.get(src.id)-1);
  inDegree.set(tgt.id, inDegree.get(tgt.id)-1);
  inDegree.set(src.id, inDegree.get(src.id)+1);
  outDegree.set(tgt.id, outDegree.get(tgt.id)+1);
  getFather()->reverse(e);
  }
*/
//----------------------------------------------------------------
node SuperGraphView::addNode() {
  node tmp=getFather()->addNode();
  nodeAdaptativeFilter.set(tmp.id,true);
  ++nNodes;
  //  inDegree.set(tmp.id,0);
  //  outDegree.set(tmp.id,0);
  notifyAddNode(this, tmp);
  return tmp;
}
//----------------------------------------------------------------
void SuperGraphView::addNode(const node n) {
  assert(getRoot()->isElement(n));
  if (!isElement(n)) {
    if (!getFather()->isElement(n)) getFather()->addNode(n);
    nodeAdaptativeFilter.set(n.id,true);
    ++nNodes;
    //    inDegree.set(n.id,0);
    //    outDegree.set(n.id,0);
    notifyAddNode(this, n);
  }
}
//----------------------------------------------------------------
edge SuperGraphView::addEdge(const node n1,const node n2) {
  assert(isElement(n1));
  assert(isElement(n2));
  edge tmp = getFather()->addEdge(n1,n2);
  edgeAdaptativeFilter.set(tmp.id,true);
  ++nEdges;
  //  outDegree.set(n1.id, outDegree.get(n1.id)+1);
  //  inDegree.set(n2.id, inDegree.get(n2.id)+1);
  notifyAddEdge(this, tmp);
  return tmp;
}
//----------------------------------------------------------------
void SuperGraphView::addEdge(const edge e) {
  assert(getRoot()->isElement(e));
  assert(isElement(source(e)));
  assert(isElement(target(e)));
  if (!edgeAdaptativeFilter.get(e.id)) {
    if (!getFather()->isElement(e)) getFather()->addEdge(e);
    edgeAdaptativeFilter.set(e.id,true);
    ++nEdges;
    //    outDegree.set(source(e).id, outDegree.get(source(e).id)+1);
    //    inDegree.set(target(e).id, inDegree.get(target(e).id)+1);
    notifyAddEdge(this, e);
  }
}
//----------------------------------------------------------------
void SuperGraphView::delNode(const node n) {
  if (isElement(n)) {
    notifyDelNode(this, n);
    Iterator<SuperGraph *>*itS = getSubGraphs();
    while (itS->hasNext())
      itS->next()->delNode(n);
    delete itS;
    StableIterator<edge> itE(getInOutEdges(n));
    while (itE.hasNext()) delEdge(itE.next());
    nodeAdaptativeFilter.set(n.id,false);
    getPropertyManager()->erase(n);
    --nNodes;
  }
}
//----------------------------------------------------------------
void SuperGraphView::delEdge(const edge e) {
  if (isElement(e)) {
    notifyDelEdge(this,e);
    Iterator<SuperGraph *>*itS=getSubGraphs();
    while (itS->hasNext()) 
      itS->next()->delEdge(e);
    delete itS;
    edgeAdaptativeFilter.set(e.id,false);
    getPropertyManager()->erase(e);
    --nEdges;
    //    outDegree.set(source(e).id, outDegree.get(source(e).id)-1);
    //    inDegree.set(target(e).id, inDegree.get(target(e).id)-1);
  }
}
//----------------------------------------------------------------
void SuperGraphView::delAllNode(const node n){
  getFather()->delAllNode(n);
}
//----------------------------------------------------------------
void SuperGraphView::delAllEdge(const edge e){
  getFather()->delAllEdge(e);
}
//----------------------------------------------------------------
void SuperGraphView::setEdgeOrder(const node n,const std::vector<edge> &v ) {
  getFather()->setEdgeOrder(n,v);
}
//----------------------------------------------------------------
void SuperGraphView::swapEdgeOrder(const node n,const edge e1 , const edge e2) {
  getFather()->swapEdgeOrder(n,e1,e2);
}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getNodes() const {
  Iterator<unsigned int> *it=0;
  try {
    it = nodeAdaptativeFilter.findAll(true);
  } catch (ImpossibleOperation &e) {
    it=0;
    //    if (numberOfNodes()<getFather()->numberOfNodes() / 2)
    //      cerr << "SuperGraphView : Optimization problem: " << numberOfNodes() << "/" << getFather()->numberOfNodes() << endl;
  }
  if (it==0)
    return (new SGraphNodeIterator(this, nodeAdaptativeFilter));
  else
    return (new UINTIterator<node>(it));
}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getInNodes(const node n) const 
{return (new InNodesIterator(this, edgeAdaptativeFilter,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getOutNodes(const node n)const 
{return (new OutNodesIterator(this, edgeAdaptativeFilter,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getInOutNodes(const node n)const 
{return (new InOutNodesIterator(this, edgeAdaptativeFilter,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getEdges()const {
  Iterator<unsigned int> *it=0;
  try {
    it = edgeAdaptativeFilter.findAll(true);
  } catch (ImpossibleOperation &e) {
    it=0;
  }
  if (it==0)
    return (new SGraphEdgeIterator(this, edgeAdaptativeFilter));
  else
    return (new UINTIterator<edge>(it));
}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getInEdges(const node n)const 
{return (new InEdgesIterator(this, edgeAdaptativeFilter,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getOutEdges(const node n)const 
{return (new OutEdgesIterator(this, edgeAdaptativeFilter,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getInOutEdges(const node n)const
{return (new InOutEdgesIterator(this, edgeAdaptativeFilter,n));}
//----------------------------------------------------------------
unsigned int SuperGraphView::numberOfNodes()const {
  return nNodes;
}
//----------------------------------------------------------------
unsigned int SuperGraphView::numberOfEdges() const {
  return nEdges;
}
//----------------------------------------------------------------
