//-*-c++-*-
#include <cmath>
#include <fstream>
#include "tulip/LayoutProxy.h"
#include "tulip/TemplateFactory.h"
#include "tulip/Coord.h"
#include "tulip/MethodFactory.h"
#include "tulip/Metric.h"
#include "tulip/PluginContext.h"

using namespace std;

inline double sqr(double x){return (x*x);}

//======================================================
LayoutProxy::LayoutProxy (SuperGraph *graph):PropertyProxy<PointType,LineType,Layout>(graph) {
  minMaxOk[(int)superGraph]=false;
}
//======================================================
LayoutProxy::~LayoutProxy() {
}
//======================================================
Coord LayoutProxy::getMax(SuperGraph *sg) {
  if (sg==0) sg=superGraph;
  int sgi=(int)sg;
  if (minMaxOk.find(sgi)==minMaxOk.end()) minMaxOk[sgi]=false;
  if (!minMaxOk[sgi]) computeMinMax(sg);
  return max[sgi];
}
//======================================================
Coord  LayoutProxy::getMin(SuperGraph *sg) {
  if (sg==0) sg=superGraph;
  int sgi=(int)sg;
  if (minMaxOk.find(sgi)==minMaxOk.end()) minMaxOk[sgi]=false;
  if (!minMaxOk[sgi]) computeMinMax(sg);
  return min[sgi];
}
//=================================================================================
void LayoutProxy::scale(const tlp::Vector<float,3>& v, SuperGraph *graph) {
  if (graph==0) graph = superGraph;
  if (graph->numberOfNodes()==0) return;
  Observable::holdObservers();
  Iterator<node> *itN=graph->getNodes();
  while (itN->hasNext()) {
     node itn = itN->next();
    Coord tmpCoord(getNodeValue(itn));
    tmpCoord *= v;
    setNodeValue(itn,tmpCoord);
  } delete itN;
  Iterator<edge> *itE=graph->getEdges();
  while (itE->hasNext()) {
    edge ite=itE->next();
    if (!getEdgeValue(ite).empty()) {
      LineType::RealType tmp = getEdgeValue(ite);
      LineType::RealType::iterator itCoord;
      itCoord=tmp.begin();
      while(itCoord!=tmp.end()) {
	*itCoord *= v;
	++itCoord;
      }
      setEdgeValue(ite,tmp);
    }
  } delete itE;
  Observable::unholdObservers();
}
//=================================================================================
void LayoutProxy::translate(const tlp::Vector<float,3>& v, SuperGraph *graph) {
  if (graph==0) graph = superGraph;
  if (graph->numberOfNodes()==0) return;
  Observable::holdObservers();
  Iterator<node> *itN=graph->getNodes();
  while (itN->hasNext()) {
     node itn = itN->next();
    Coord tmpCoord(getNodeValue(itn));
    tmpCoord += v;
    setNodeValue(itn,tmpCoord);
  } delete itN;
  Iterator<edge> *itE=graph->getEdges();
  while (itE->hasNext()) {
    edge ite=itE->next();
    if (!getEdgeValue(ite).empty()) {
      LineType::RealType tmp = getEdgeValue(ite);
      LineType::RealType::iterator itCoord;
      itCoord=tmp.begin();
      while(itCoord!=tmp.end()) {
	*itCoord += v;
	++itCoord;
      }
      setEdgeValue(ite,tmp);
    }
  } delete itE;
  resetBoundingBox();
  notifyObservers();
  Observable::unholdObservers();
}
//=================================================================================
void LayoutProxy::center(SuperGraph *graph) {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__  << endl;
#endif
  if (graph==0) graph = superGraph;
  if (graph->numberOfNodes()==0) return;
  Observable::holdObservers();
  Coord tr=getMax(graph)+getMin(graph);
  tr /= -2.0;
  translate(tr, graph);
  resetBoundingBox();
  notifyObservers();
  Observable::unholdObservers();
}
//=================================================================================
void LayoutProxy::normalize(SuperGraph *graph) {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << endl;
#endif
  if (graph==0) graph = superGraph;
  if (graph->numberOfNodes()==0) return;
  Observable::holdObservers();
  center();
  double dtmpMax = 1.0;
  Iterator<node> *itN=graph->getNodes();
  while (itN->hasNext())  {
    node itn=itN->next();
    Coord tmpCoord(getNodeValue(itn));
    dtmpMax >?= sqr(tmpCoord[0])+sqr(tmpCoord[1])+sqr(tmpCoord[2]);
  } delete itN;
  dtmpMax = 1.0 / sqrt(dtmpMax);
  scale(Coord(dtmpMax,dtmpMax,dtmpMax), graph);  
  resetBoundingBox();
  notifyObservers();
  Observable::unholdObservers();
}
//=================================================================================
void LayoutProxy::perfectAspectRatio() {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << endl;
#endif
  if (superGraph->numberOfNodes()==0) return;
  Observable::holdObservers();
  center();
  double scaleX,scaleY,scaleZ;
  double deltaX,deltaY,deltaZ;
  deltaX = (double )getMax()[0]-(double )getMin()[0];
  deltaY = (double )getMax()[1]-(double )getMin()[1];
  deltaZ = (double )getMax()[2]-(double )getMin()[2];
  double delta= deltaX >? deltaY;
  delta >?= deltaZ;
  if (delta<0.001) return;
  if (deltaX<0.001) deltaX=delta;
  if (deltaY<0.001) deltaY=delta;
  if (deltaZ<0.001) deltaZ=delta;
  scaleX = delta / deltaX;
  scaleY = delta / deltaY;
  scaleZ = delta / deltaZ;
  scale(Coord(scaleX,scaleY,scaleZ));
  notifyObservers();
  Observable::unholdObservers();
}
//=================================================================================
void LayoutProxy::computeMinMax(SuperGraph *sg) {
#ifndef NDEBUG
  cerr << "LayoutProxy::computeMinMax begin" << endl;
#endif
  Coord tmpCoord;
  double maxX=0;
  double minX=0;
  double maxY=0;
  double minY=0;
  double maxZ=0;
  double minZ=0;
  if (sg==0) sg=superGraph;
  Iterator<node> *itN=sg->getNodes();
  if  (itN->hasNext()) {
    node itn=itN->next();
    tmpCoord=getNodeValue(itn);
    maxX=tmpCoord[0];
    minX=tmpCoord[0];
    maxY=tmpCoord[1];
    minY=tmpCoord[1];
    maxZ=tmpCoord[2];
    minZ=tmpCoord[2];
  }
  while (itN->hasNext()) {
    node itn=itN->next();
    tmpCoord=getNodeValue(itn);
    maxX >?= tmpCoord[0];
    minX <?= tmpCoord[0];
    maxY >?= tmpCoord[1];
    minY <?= tmpCoord[1];
    maxZ >?= tmpCoord[2];
    minZ <?= tmpCoord[2];
  } delete itN;
  Iterator<edge> *itE=sg->getEdges();
  while (itE->hasNext()) {
    edge ite=itE->next();
    LineType::RealType::const_iterator itCoord;
    for (itCoord=getEdgeValue(ite).begin();itCoord!=getEdgeValue(ite).end();++itCoord) {
      tmpCoord = *itCoord;
      maxX >?= tmpCoord[0];
      minX <?= tmpCoord[0];
      maxY >?= tmpCoord[1];
      minY <?= tmpCoord[1];
      maxZ >?= tmpCoord[2];
      minZ <?= tmpCoord[2];
    }
  } delete itE;
  int sgi=(int)sg;
  minMaxOk[sgi]=true;  
  min[sgi][0]=minX;
  min[sgi][1]=minY;
  min[sgi][2]=minZ;
  max[sgi][0]=maxX;
  max[sgi][1]=maxY;
  max[sgi][2]=maxZ;
  //  cerr << "LayoutProxy::computeMinMax end" << endl;
}
//=================================================================================
void LayoutProxy::reset_handler() {
  resetBoundingBox();
}
//=================================================================================
void LayoutProxy::clone_handler(PropertyProxy<PointType,LineType> &proxyC) {
  if (typeid(this)==typeid(&proxyC)) {
    LayoutProxy *proxy=(LayoutProxy *)&proxyC;
    minMaxOk = proxy->minMaxOk;
    min = proxy->min;
    max = proxy->max;
  }
}
//=================================================================================
void LayoutProxy::resetBoundingBox() {
  minMaxOk.clear();
  min.clear();
  max.clear();
}
//================================================================================
void LayoutProxy::setNodeValue_handler(const node n){resetBoundingBox();}
void LayoutProxy::setEdgeValue_handler(const edge e){resetBoundingBox();}
void LayoutProxy::setAllNodeValue_handler(){resetBoundingBox();}
void LayoutProxy::setAllEdgeValue_handler(){resetBoundingBox();}
//=================================================================================
double LayoutProxy::averageAngularResolution(SuperGraph *sg) {
  if (sg==0) sg=superGraph;
  Iterator<node> *itN=sg->getNodes();
  double result=0;
  while (itN->hasNext()) {
    result+=averageAngularResolution(itN->next(),sg);
  } delete itN;
  return result/(double)sg->numberOfNodes();
}
//=================================================================================
struct AngularOrder {
  bool operator() (const Coord &c1, const Coord &c2) {
    //if the vectors have not the same direction on y-coordiantes
    //the result is direct.
    if (c1[1]>=0 && c2[1]<0) return false;  
    if (c2[1]>=0 && c1[1]<0) return true; 
    //If the vectors have the same size on the y-coordinates, we compare
    //their x-coordinates
    if (c2[1]>=0 && c1[1]>=0) 
      return c1[0]>c2[0];
    else
      return c1[0]<c2[0];
  }
};
//=================================================================================
vector<double> LayoutProxy::angularResolutions(const node n, SuperGraph *sg) {
  vector<double> result;
  if (sg==0) sg=superGraph;
  double degree=sg->deg(n);
  if (sg->deg(n)==0) return result;
  if (sg->deg(n)==1) {
    result.push_back(0.0);
    return result;
  }
  //===========
  list<Coord> adjCoord;
  //Extract all adjacent edges, the bends are taken
  //into account.
  Iterator<edge> *itE=sg->getInOutEdges(n);
  for (unsigned int i=0;itE->hasNext();++i) {
    edge ite=itE->next();
    if (getEdgeValue(ite).size()>0) {
      if (sg->source(ite)==n)
	adjCoord.push_back(getEdgeValue(ite).front());
      else
	adjCoord.push_back(getEdgeValue(ite).back());
    }
    else {
      adjCoord.push_back(getNodeValue(sg->opposite(ite,n)));
    }
  } delete itE;
  
  //Compute normalized vectors associated to incident edges.
  Coord center=getNodeValue(n);
  list<Coord>::iterator it;
  for (it=adjCoord.begin();it!=adjCoord.end();++it) {
    (*it)-=center;
    (*it)/=(*it).norm();
  }
  //Sort the vector to compute angles between two edges
  //Correctly.
  adjCoord.sort(AngularOrder());
  //Compute the angles
  it=adjCoord.begin();
  Coord first=(*it);
  Coord current=first;
  ++it;

  int stop=2;
  for (;stop>0;) {
    Coord next=*it;
    double cosTheta = current.dotProduct(next);//current * next;
    double sinTheta = (current^next)[2];
    if (cosTheta+0.0001>1) cosTheta-=0.0001;
    if (cosTheta-0.0001<-1) cosTheta+=0.0001;
    if (sinTheta+0.0001>1) sinTheta-=0.0001;
    if (sinTheta-0.0001<-1) sinTheta+=0.0001;
    if (sinTheta>=0) {
      result.push_back(2.0*M_PI/degree-acos(cosTheta));
    }
    else {
      result.push_back(2.0*M_PI/degree-(2.0*M_PI-acos(cosTheta)));
    }
    current=next;
    ++it;
    if (stop<2) stop=0;
    if (it==adjCoord.end()){
      it=adjCoord.begin();
      stop--;
    }
  }
  return result;
}
//=================================================================================
double LayoutProxy::averageAngularResolution(const node n, SuperGraph *sg) {
  if (sg==0) sg=superGraph;
  double degree=sg->deg(n);
  if (sg->deg(n)<2) return 0;

  list<Coord> adjCoord;
  //Extract all adjacent edges, the bends are taken
  //into account.
  Iterator<edge> *itE=sg->getInOutEdges(n);
  for (unsigned int i=0;itE->hasNext();++i) {
    edge ite=itE->next();
    if (getEdgeValue(ite).size()>0) {
      if (sg->source(ite)==n)
	adjCoord.push_back(getEdgeValue(ite).front());
      else
	adjCoord.push_back(getEdgeValue(ite).back());
    }
    else {
      adjCoord.push_back(getNodeValue(sg->opposite(ite,n)));
    }
  } delete itE;
  
  //Compute normalized vectors associated to incident edges.
  Coord center=getNodeValue(n);
  list<Coord>::iterator it;
  for (it=adjCoord.begin();it!=adjCoord.end();++it) {
    (*it)-=center;
    (*it)/=(*it).norm();
  }
  //Sort the vector to compute angles between two edges
  //Correctly.
  adjCoord.sort(AngularOrder());
  //Compute the angles
  double result=0;
  it=adjCoord.begin();
  Coord first=(*it);
  Coord current=first;
  ++it;

  int stop=2;
  for (;stop>0;) {
    Coord next=*it;
    double cosTheta = current.dotProduct(next); //current *  next;
    double sinTheta=(current^next)[2];
    if (cosTheta+0.0001>1) cosTheta-=0.0001;
    if (cosTheta-0.0001<-1) cosTheta+=0.0001;
    if (sinTheta+0.0001>1) sinTheta-=0.0001;
    if (sinTheta-0.0001<-1) sinTheta+=0.0001;
    if (sinTheta>=0) {
      result+=fabs(2.0*M_PI/degree-acos(cosTheta));
    }
    else {
      result+=fabs(2.0*M_PI/degree-(2.0*M_PI-acos(cosTheta)));
    }
    current=next;
    ++it;
    if (stop<2) stop=0;
    if (it==adjCoord.end()){
      it=adjCoord.begin();
      stop--;
    }
  }
  return result/degree;  
}
//=================================================================================
double LayoutProxy::edgeLength(edge e) {
  Coord start=getNodeValue(superGraph->source(e));
  Coord end=getNodeValue(superGraph->target(e));
  double result=0;
  const vector<Coord> & tmp=getEdgeValue(e);
  for (unsigned int i=0;i<tmp.size();++i) {
    result+=(tmp[i]-start).norm();
    start=tmp[i];
  }
  result+=(end-start).norm();
  return result;
}
//=================================================================================
unsigned int LayoutProxy::crossingNumber() {
  cerr << "!!! Warning: Not Implemented function :" << endl;
  cerr << __PRETTY_FUNCTION__ << endl;
  return 0;
}
//=================================================================================
PProxy* LayoutProxy::clonePrototype(SuperGraph * g, std::string n) {
  if( !g )
    return 0;
  LayoutProxy * p = g->getLocalProperty<LayoutProxy>( n );
  p->setAllNodeValue( getNodeDefaultValue() );
  p->setAllEdgeValue( getEdgeDefaultValue() );
  return p;
}
//=============================================================
void LayoutProxy::copy( const node n0, const node n1, PProxy * p ) {
  if( !p )
    return;
  LayoutProxy * tp = dynamic_cast<LayoutProxy*>(p);
  assert( tp );
  setNodeValue( n0, tp->getNodeValue(n1) );
}
//=============================================================
void LayoutProxy::copy( const edge e0, const edge e1, PProxy * p ) {
  if( !p )
    return;
  LayoutProxy * tp = dynamic_cast<LayoutProxy*>(p);
  assert( tp );
  setEdgeValue( e0, tp->getEdgeValue(e1) );
}
//======================================================
