#include "calc_decomp_cuts.h"
#include <math.h>
#include <stdlib.h>
#include "uns_inline_decomp.h"
#include "inline_mesh_desc.h"
#include <sstream>
#include <set>
#include "Random.h"
#include <iostream>
#include <algorithm>
#include <limits.h>
//#pragma warning(disable:981)
//#pragma warning(disable:383)
namespace PAMGEN_NEVADA {

  Inline_Mesh_Desc * Inline_Mesh_Desc::static_storage = NULL;
  std::stringstream Inline_Mesh_Desc::echo_stream;



bool part_compare_size(const Partition *a, const Partition *b) {
  if(a->numels < b->numels)
    return true;
  if(a->numels == b->numels)
    return (a->unique_id < b->unique_id);
  return false;
}

bool part_compare_centroid(const Partition *a, const Partition *b) {
  if(a->centroid < b->centroid)
    return true;
  if(a->centroid == b->centroid)
    return (a->unique_id < b->unique_id);
  return false;
}




/*****************************************************************************/
Inline_Mesh_Desc::~Inline_Mesh_Desc()
/*****************************************************************************/
{
  for(int i = 0; i < 3; i ++){
    if(block_dist[i])delete []  block_dist[i];
    if(c_block_dist[i])delete []  c_block_dist[i];
    if(first_size[i])delete  [] first_size[i];
    if(last_size[i])delete  [] last_size[i];
    if(interval[i])delete  [] interval[i];
  }
  delete [] block_dist;
  delete [] c_block_dist;
  delete [] first_size;
  delete [] last_size;
  delete [] interval;

  delete [] a_inline_nx;
  delete [] a_inline_ny;
  delete [] a_inline_nz;

  delete [] c_inline_nx;
  delete [] c_inline_ny;
  delete [] c_inline_nz;

  if(cum_block_totals)delete [] cum_block_totals;
  if(els_in_block) delete [] els_in_block;

  if(Element_Density_Functions[0])delete Element_Density_Functions[0];
  if(Element_Density_Functions[1])delete Element_Density_Functions[1];
  if(Element_Density_Functions[2])delete Element_Density_Functions[2];
  if(Geometry_Transform_Function)delete Geometry_Transform_Function;
  if(base_partition){
    base_partition->empty();
    delete base_partition;
    base_partition = NULL;
  }
  if (Icoors)delete [] Icoors;
  if (Jcoors)delete [] Jcoors;
  if (Kcoors)delete [] Kcoors;
  if(element_block_lists)delete [] element_block_lists;
  if(sideset_list.size())delete [] sideset_vectors;
  if(nodeset_list.size())delete [] nodeset_vectors;

  std::list < PG_BC_Specification * > :: iterator it;
  for(it = nodeset_list.begin(); it != nodeset_list.end(); it ++){
    PG_BC_Specification * bcs = *it;
    delete bcs;
  }
  for(it = sideset_list.begin(); it != sideset_list.end(); it ++){
    PG_BC_Specification * bcs = *it;
    delete bcs;
  }
}

/****************************************************************************/
int Inline_Mesh_Desc::reportSize(const long long & total_el_count, 
				  const long long & total_node_count, 
				  const long long & total_edge_count,
				  std::stringstream & es)
/****************************************************************************/
{
  int status = 0;
  if(total_el_count > INT_MAX){
    es << "Terminating from Inline_Mesh_Desc, ";
    es << total_el_count ;
    es << " elements requested, max ";
    es << INT_MAX;
    es << " elements permitted.";
    status = 1;
  }

  if(total_node_count > INT_MAX){
    es << "Terminating from Inline_Mesh_Desc,";
    es << total_node_count ;
    es << " nodes requested, max ";
    es << INT_MAX;
    es << " nodes permitted.";
    status = 1;
  }

  if(total_edge_count > INT_MAX){
    es << "Terminating from Inline_Mesh_Desc,";
    es << total_edge_count ;
    es << " edges requested, max ";
    es << INT_MAX;
    es << " edges permitted.";
    status = 1;
  }
  return status;
}

//! This is an access function to protect agains the stl maps
//! behaviour of always returning an entry for every [] operator
//! query.
/****************************************************************************/
int Inline_Mesh_Desc::get_map_entry(std::map < int, int > & the_map, const int & key)
/****************************************************************************/
{
  std::map <int, int > ::iterator foo;
  foo = the_map.find(key);
  if(foo == the_map.end()){
    error_stream << "Looking for but not finding key entry " << key << "\n";
  }
  return (*foo).second;
}

/****************************************************************************/
int Inline_Mesh_Desc::get_block_index(int ordinal_val, 
				   int count ,
				   int * cumulative)//c_inline_nz);
/****************************************************************************/

{
  int i = 0;
  while(i < count-1){
    if(ordinal_val >= cumulative[i] && ordinal_val < cumulative[i+1])return i;
    i ++;
  }
  return i;
}


//! Queries which processor an element lies on.
//! Calls the recursive Partition::Element_Proc function.
/****************************************************************************/
int Inline_Mesh_Desc::Element_Proc(int global_element_id)
/****************************************************************************/
{
  unsigned proc = 0;
  if(inline_decomposition_type == SEQUENTIAL){
    int total = kestride * nelz_tot;
    int num_per_proc = total/num_processors;
    proc = global_element_id/num_per_proc;
    if(proc >= num_processors)proc = num_processors-1;
  }
  else if(inline_decomposition_type == RANDOM){
    SRANDOM(global_element_id);
    int rand_num = RANDOM();
    proc = rand_num%num_processors;
  }
  else if((inline_decomposition_type == BISECTION) || (inline_decomposition_type == PROCESSOR_LAYOUT)){
    int l,i,j,k;
    get_l_i_j_k_from_element_number(global_element_id,l,i,j,k);
    int ginds[4];
    ginds[0] = i;
    ginds[1] = j;
    ginds[2] = k;
    ginds[3] = l;
    return base_partition->Element_Proc(ginds);
  }

  return proc;
}


//! Partitions all elements by a recursive bisection into 
//! rectangular chunks. Alternative decompositions are possible.
//! This one is simple to implement and easy to run on all processors.
//! Zoltan could be used for this. The elements on the processor are
//! packed into the stl list.
/****************************************************************************/
Partition * Inline_Mesh_Desc::Decompose(std::list <int> & global_el_ids,int & err_code)
/****************************************************************************/
{
  //Recursive Bisection decomposition
  //Create a partition object having the entire domain
  //Place it in a list sorted by number of elements


  //While sorted_list.size < num_procs
  //  get first entry in list and the partition object it points to
  //  bisect the partition object
  //  remove the entry from the sorted_list
  //  add the children of the partition object to the sorted list
  //  re-sort the list

  //Then put the partition objects into a list sorted by centroid distance from bottom left corner
  //Assign proc id numbers to the centroid sorted list entries

  //Then loop over the elements for the current processor and push them into
  //the global el_ids list

  /**********************************************/
  //Recursive Bisection decomposition
  //Create a partition object having the entire domain

  if( Debug_Location())
    std::cout << "Inline_Mesh_Desc::Decompose()" << std::endl;
  

  std::vector <Partition *> sorted_partition_list;

if(inline_decomposition_type == PROCESSOR_LAYOUT){

  int remaining_cuts[3];
  remaining_cuts[0] = inline_nprocs[0];
  remaining_cuts[1] = inline_nprocs[1];
  remaining_cuts[2] = inline_nprocs[2];

  base_partition  = new Partition(0,0,0,0,1,nelx_tot,nely_tot,nelz_tot,inline_decomposition_type,remaining_cuts);
  //Place it in a list sorted by number of elements
  sorted_partition_list.push_back(base_partition);
  Partition* biggest;

  if(num_processors != (unsigned)inline_nprocs[0]*inline_nprocs[1]*inline_nprocs[2]){
    error_stream << "Inline_Mesh_Desc::Decompose "
		 << "The specified inline processor layout " 
		 << inline_nprocs[0] << " X " 
		 << inline_nprocs[1] << " X " 
		 << inline_nprocs[2] << " does not correspond to the number of processors " << num_processors << "." ;
    err_code = 1;
    return NULL;
   }
    inc_nels[0] = nelx_tot/inline_nprocs[0];
    inc_nels[1] = nely_tot/inline_nprocs[1];
    inc_nels[2] = nelz_tot/inline_nprocs[2];


    if(inc_nels[0] == 0 ){
      error_stream << "Inline_Mesh_Desc::Decompose"
		   << " Value for numprocs specified in I direction " << inline_nprocs[0] 
		   << " is greater than the number of elements in I " << nelx_tot << ".";
      err_code = 1;
      return NULL;
    }

    if(inc_nels[1] == 0 ){
      error_stream << "Inline_Mesh_Desc::Decompose"
		   << " Value for numprocs specified in J direction " << inline_nprocs[1] 
		   << " is greater than the number of elements in J " << nely_tot << ".";
      err_code = 1;
      return NULL;
    }

    if(inc_nels[2] == 0 ){
      error_stream << "Inline_Mesh_Desc::Decompose"
		   << " Value for numprocs specified in K direction " << inline_nprocs[2] 
		   << " is greater than the number of elements in K " << nelz_tot << ".";
      err_code = 1;
      return NULL;
    }
    
    info_stream << "Using PROCESSOR LAYOUT decomposition.\n";
    info_stream << "Number of elements/segment in directions I/X/R \t\t" << inc_nels[0] << "\n";
    info_stream << "Number of elements/segment in directions J/Y/THETA \t" << inc_nels[1] << "\n";
    info_stream << "Number of elements/segment in directions K/Z/PHI \t" << inc_nels[2] << "\n";
    info_stream << "Number of mesh segments in directions I/X/R \t\t" << remaining_cuts[0] << "\n";
    info_stream << "Number of mesh segments in directions J/Y/THETA \t" << remaining_cuts[1] << "\n";
    info_stream << "Number of mesh segments in directions K/Z/PHI \t" << remaining_cuts[2] << "\n";


    while(sorted_partition_list.size() < num_processors){
      //  get first entry in list and the partition object it points to
      biggest = sorted_partition_list.back();
      //  remove the entry from the sorted_list
      sorted_partition_list.pop_back();
      
      //  bisect the partition object and
      //  add the children of the partition object to the sorted list
      biggest->Processor_Partition(sorted_partition_list,inc_nels);
      //  re-sort the list
      std::sort(sorted_partition_list.begin(),sorted_partition_list.end(),part_compare_size);
    }
  }
  else if (inline_decomposition_type == BISECTION){

    //SETUP for bisection
  int remaining_cuts[3];
  int decomp_result = 0;
  if(dimension == 3){
    decomp_result = dom_decomp_3d(nelx_tot,nely_tot,nelz_tot,num_processors,&(inc_nels[0]),&(inc_nels[1]),&(inc_nels[2]));
  }
  else{
    decomp_result = dom_decomp_2d(nelx_tot,nely_tot,num_processors,&(inc_nels[0]),&(inc_nels[1]));
  }

    if(decomp_result != 0){
      error_stream << "Terminating from Inline_Mesh_Desc::Decompose, ";
      error_stream << "non-zero return value from dom_decomp_2/3d ";
      error_stream << decomp_result;
      err_code = 1;
      return NULL;
    }

  if(dimension == 3){
    remaining_cuts[0] = inc_nels[0];
    remaining_cuts[1] = inc_nels[1];
    remaining_cuts[2] = inc_nels[2];
    inc_nels[0] = nelx_tot/inc_nels[0];
    inc_nels[1] = nely_tot/inc_nels[1];
    inc_nels[2] = nelz_tot/inc_nels[2];
  }
  else{
    remaining_cuts[0] = inc_nels[0];
    remaining_cuts[1] = inc_nels[1];
    remaining_cuts[2] = 1;
    inc_nels[0] = nelx_tot/inc_nels[0];
    inc_nels[1] = nely_tot/inc_nels[1];
    inc_nels[2] = 1;
  }

    {
      info_stream << "Using BISECTION LAYOUT decomposition.\n";
      info_stream << "Number of elements/segment in directions I/X/R \t\t" << inc_nels[0] << "\n";
      info_stream << "Number of elements/segment in directions J/Y/THETA \t" << inc_nels[1] << "\n";
      info_stream << "Number of elements/segment in directions K/Z/PHI \t" << inc_nels[2] << "\n";
      info_stream << "Number of mesh segments in directions I/X/R \t\t" << remaining_cuts[0] << "\n";
      info_stream << "Number of mesh segments in directions J/Y/THETA \t" << remaining_cuts[1] << "\n";
      info_stream << "Number of mesh segments in directions K/Z/PHI \t" << remaining_cuts[2] << "\n";
    }

    base_partition  = new Partition(0,0,0,0,1,nelx_tot,nely_tot,nelz_tot,inline_decomposition_type,remaining_cuts);
  //Place it in a list sorted by number of elements
  sorted_partition_list.push_back(base_partition);
  Partition* biggest;

    while(sorted_partition_list.size() < num_processors){
      //  get first entry in list and the partition object it points to
      biggest = sorted_partition_list.back();
      //  remove the entry from the sorted_list
      sorted_partition_list.pop_back();

      {
        biggest->Processor_Partition(sorted_partition_list, inc_nels);
      }
      //  re-sort the list
      std::sort(sorted_partition_list.begin(),sorted_partition_list.end(),part_compare_size);//sorted_partition_list.sort();
    }
  }
  else if(inline_decomposition_type == RANDOM){
    int total = kestride * nelz_tot;
    for(int i = 0; i < total; i ++){
      SRANDOM(i);
      int rand_num = RANDOM();
      int proc = rand_num%num_processors;
      if(proc == (int)my_rank)global_el_ids.push_back(i);
    }
    return base_partition;
  }
  else if(inline_decomposition_type == SEQUENTIAL){
    int total = kestride * nelz_tot;
    int num_per_proc = total/num_processors;
    int remainder = total - num_per_proc*num_processors;
    int my_start = my_rank * num_per_proc;
    int my_end = my_start + num_per_proc;
    if(my_rank == num_processors-1)my_end +=remainder;

    for(int mtotal = my_start; mtotal < my_end; mtotal ++){
      global_el_ids.push_back(mtotal);
    }
    return base_partition;
  }  

  std::sort(sorted_partition_list.begin(),sorted_partition_list.end(),part_compare_centroid);//sorted_partition_list.sort();


  //Assign proc id numbers to the centroid sorted list entries
  std::vector < Partition * > :: iterator citer;
  unsigned proc_cnt = 0;
  Partition *my_part = NULL;
  for(citer = sorted_partition_list.begin();citer != sorted_partition_list.end();citer ++,proc_cnt++){
    (*citer)->proc_id = proc_cnt;
    if(proc_cnt == my_rank)my_part = (*citer);
  }

  //Then loop over the elements for the current processor and push them into
  //the global el_ids list
  for(int k = my_part->lows[2]; k < my_part->highs[2]; k++){
    for(int j = my_part->lows[1]; j < my_part->highs[1]; j++){
      for(int i = my_part->lows[0]; i < my_part->highs[0]; i++){
        int elnum = iestride*i+jestride*j+kestride*k;
        global_el_ids.push_back(elnum);
      }
    }
  }

  if( Debug_Location())
    std::cout << "Inline_Mesh_Desc::Leaving-Decompose()" << std::endl;

  return my_part;
}

//! A utility function to build up required bookkeeping objects.
/****************************************************************************/
void Inline_Mesh_Desc::Build_Global_Lists(std::list <int> & element_list,
                                       std::vector <int> & element_vector,
                                       std::list <int> & global_node_list,
                                       std::vector <int> & global_node_vector,
                                       std::map <int, int> & global_node_map,
                                       std::map <int, int> & global_element_map)
/****************************************************************************/
{
  if( Debug_Location()) 
    std::cout << "Inline_Mesh_Desc::Build_Global_Lists()" << std::endl;

  for(std::list <int>::iterator the_it = element_list.begin();the_it != element_list.end();the_it++){
    element_vector.push_back(*the_it);
  }
  element_block_lists = new std::vector <int> [numBlocks()];

  std::list <int> ::iterator lit;
  for(lit = element_list.begin();lit != element_list.end();lit ++){
    int the_element = *lit;
    // These are the indices of the element in the entire domain
    int global_k = the_element/(kestride);
    int global_j = (the_element - global_k*(kestride))/(jestride);
    int global_i = the_element - global_k*(kestride)-global_j*(jestride);

    // these are the indices of the block in which the element resides
    //       int block_k = global_k/(inline_nz);
    //       int block_j = global_j/(inline_ny);
    //       int block_i = global_i/(inline_nx);
    int block_k = get_block_index(global_k,inline_bz,c_inline_nz);
    int block_j = get_block_index(global_j,inline_by,c_inline_ny);
    int block_i = get_block_index(global_i,inline_bx,c_inline_nx);

    // This is the ordinal number of the block the element resides in
    int local_block = block_i + block_j*(inline_bx)+ block_k*(blockKstride());
    element_block_lists[local_block].push_back(the_element);
    int nn;

    if(periodic_j && (block_j == (inline_by-1)) && (global_j == (nely_tot-1))){
      if(dimension == 2){
      nn = (global_i+0)*instride + (global_j+0)*jnstride;                         global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (global_j+0)*jnstride;                         global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (0+0)*jnstride+(global_k+0)*knstride;      global_node_list.push_back(nn);
      nn = (global_i+0)*instride + (0+0)*jnstride+(global_k+0)*knstride;      global_node_list.push_back(nn);
      }
      else{
      nn = (global_i+0)*instride + (global_j+0)*jnstride + (global_k+0)*knstride; global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (global_j+0)*jnstride + (global_k+0)*knstride; global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (0+0)*jnstride+(global_k+0)*knstride;      global_node_list.push_back(nn);
      nn = (global_i+0)*instride + (0+0)*jnstride+(global_k+0)*knstride;      global_node_list.push_back(nn);

      nn = (global_i+0)*instride + (global_j+0)*jnstride + (global_k+1)*knstride; global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (global_j+0)*jnstride + (global_k+1)*knstride; global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (0+0)*jnstride+(global_k+1)*knstride;      global_node_list.push_back(nn);
      nn = (global_i+0)*instride + (0+0)*jnstride+(global_k+1)*knstride;      global_node_list.push_back(nn);
      }
    }
    else{
      if(dimension == 2){
      nn = (global_i+0)*instride + (global_j+0)*jnstride; global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (global_j+0)*jnstride; global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (global_j+1)*jnstride; global_node_list.push_back(nn);
      nn = (global_i+0)*instride + (global_j+1)*jnstride; global_node_list.push_back(nn);
      }
      else{
      nn = (global_i+0)*instride + (global_j+0)*jnstride + (global_k+0)*knstride; global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (global_j+0)*jnstride + (global_k+0)*knstride; global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (global_j+1)*jnstride + (global_k+0)*knstride; global_node_list.push_back(nn);
      nn = (global_i+0)*instride + (global_j+1)*jnstride + (global_k+0)*knstride; global_node_list.push_back(nn);

      nn = (global_i+0)*instride + (global_j+0)*jnstride + (global_k+1)*knstride; global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (global_j+0)*jnstride + (global_k+1)*knstride; global_node_list.push_back(nn);
      nn = (global_i+1)*instride + (global_j+1)*jnstride + (global_k+1)*knstride; global_node_list.push_back(nn);
      nn = (global_i+0)*instride + (global_j+1)*jnstride + (global_k+1)*knstride; global_node_list.push_back(nn);
      }
    }
  }
  global_node_list.sort();
  global_node_list.unique();


  // Create the global_node_map
  std::list <int> ::iterator nit;
  int total = 0;
  for(nit = global_node_list.begin();nit != global_node_list.end();nit ++,total++){
    global_node_vector.push_back(*nit);
    global_node_map[*nit] = total;
  }


  // Create the global_element_map
  int total_element_count = 0;
  for(int bct = 0; bct < numBlocks();bct ++ ){
    for(unsigned elct = 0;elct < element_block_lists[bct].size();elct++,total_element_count++){
      int the_el = element_block_lists[bct][elct];
      global_element_map[the_el] = total_element_count;
    }
  }

  if( Debug_Location())
    std::cout << "Inline_Mesh_Desc::Leaving - Build_Global_Lists()" << std::endl;
}

/****************************************************************************/
void Inline_Mesh_Desc::Populate_Nodeset_Info(int * const * node_set_nodes,
					     std::map <int, int> & global_node_map)
/****************************************************************************/
{
  std::list < PG_BC_Specification *> ::iterator setit;
  int nsct = 0;
  for(setit = nodeset_list.begin(); setit != nodeset_list.end();setit++,nsct ++){
    int * the_nodes = node_set_nodes[nsct];
    for(unsigned elct = 0; elct < nodeset_vectors[nsct].size();elct++){
      the_nodes[elct] = get_map_entry(global_node_map,nodeset_vectors[nsct][elct])+1;// add one for exodus convention
    }
  }
}

/****************************************************************************/
int Inline_Mesh_Desc::Populate_Sideset_Info(std::map <int, int> & global_element_map,
					  std::map <int, int> & global_node_map,
					  int * const * side_set_elements,
					  int * const * side_set_faces,
					  int * const * side_set_nodes,
					  int * const * side_set_node_counter)
/****************************************************************************/
{
  int num_nodes_per_face = 2;
  if(dimension == 3){
    num_nodes_per_face = 4;
  }

  int nsct = 0;
   std::list < PG_BC_Specification *> ::iterator setit;

  for(setit = sideset_list.begin(); setit != sideset_list.end();setit++,nsct ++){

    int * the_elements = side_set_elements[nsct];
    int * the_faces = side_set_faces[nsct];
    int * the_nodes = side_set_nodes[nsct];
    int * the_node_counter = side_set_node_counter[nsct];

    Topo_Loc the_location = (*setit)->location;
    //Sidesets allowed only on faces of block, not on edges or corners


    for(unsigned elct = 0; elct < sideset_vectors[nsct].size();elct ++){
      int the_element = sideset_vectors[nsct][elct].first;
      // These are the indices of the element in the entire domain
      int gl_k = the_element/(kestride);
      int gl_j = (the_element-gl_k*(kestride))/(jestride);
      int gl_i = the_element - gl_k*(kestride)-gl_j*(jestride);

      //The result of the calculation is the id of the element in the block as found in the connectivity array
      the_elements[elct] = get_map_entry(global_element_map,the_element)+1;
      if(!getErrorString().empty()){return 1;}

      the_faces[elct] = topo_loc_to_exo_face[the_location];
      the_node_counter[elct] = num_nodes_per_face;
      //Since the nodes are numbered across blocks, the global indices are the actual indices of the node
      //It is only required to permute the indices to collect the appropriate nodes for the indicated face and 
      //calculated the node index using the nodal stride values for the entire mesh and the indices

      // adjust for periodicity in j 
      int glj_plus1 = gl_j + 1;
      if(periodic_j){
        if(glj_plus1 == nely_tot){
          glj_plus1 = 0;
        } 
      }

      switch(the_location) {

        case MINUS_I:{
      if(dimension == 2){
	  the_nodes[elct*num_nodes_per_face + 0] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (glj_plus1)*jnstride + (gl_k + 0)*knstride);
	  the_nodes[elct*num_nodes_per_face + 1] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (gl_j + 0)*jnstride + (gl_k + 0)*knstride);
      }
      else{
	  the_nodes[elct*num_nodes_per_face + 0] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (gl_j + 0)*jnstride + (gl_k + 0)*knstride);
	  the_nodes[elct*num_nodes_per_face + 1] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (gl_j + 0)*jnstride + (gl_k + 1)*knstride);
	  the_nodes[elct*num_nodes_per_face + 2] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (glj_plus1)*jnstride + (gl_k + 1)*knstride);
	  the_nodes[elct*num_nodes_per_face + 3] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (glj_plus1)*jnstride + (gl_k + 0)*knstride);
      }
	  break;
	}
      case PLUS_I:{
      if(dimension == 2){
	the_nodes[elct*num_nodes_per_face + 0] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (gl_j + 0)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 1] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (glj_plus1)*jnstride + (gl_k + 0)*knstride);
      }
      else{
	the_nodes[elct*num_nodes_per_face + 0] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (gl_j + 0)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 1] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (glj_plus1)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 2] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (glj_plus1)*jnstride + (gl_k + 1)*knstride);
	the_nodes[elct*num_nodes_per_face + 3] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (gl_j + 0)*jnstride + (gl_k + 1)*knstride);
      }
	break;
      }
      case MINUS_J:{
      if(dimension == 2){
	the_nodes[elct*num_nodes_per_face + 0] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (gl_j + 0)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 1] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (gl_j + 0)*jnstride + (gl_k + 0)*knstride);
      }
      else{
	the_nodes[elct*num_nodes_per_face + 0] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (gl_j + 0)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 1] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (gl_j + 0)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 2] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (gl_j + 0)*jnstride + (gl_k + 1)*knstride);
	the_nodes[elct*num_nodes_per_face + 3] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (gl_j + 0)*jnstride + (gl_k + 1)*knstride);
      }
	break;
      }
      case PLUS_J:{
      if(dimension == 2){
	the_nodes[elct*num_nodes_per_face + 0] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (glj_plus1)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 1] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (glj_plus1)*jnstride + (gl_k + 0)*knstride);
      }
      else{
	the_nodes[elct*num_nodes_per_face + 0] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (glj_plus1)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 1] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (glj_plus1)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 2] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (glj_plus1)*jnstride + (gl_k + 1)*knstride);
	the_nodes[elct*num_nodes_per_face + 3] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (glj_plus1)*jnstride + (gl_k + 1)*knstride);
      }
	break;
      }
      case MINUS_K:{
      if(dimension == 2){
      }
      else{
	the_nodes[elct*num_nodes_per_face + 0] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (gl_j + 0)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 1] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (glj_plus1)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 2] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (glj_plus1)*jnstride + (gl_k + 0)*knstride);
	the_nodes[elct*num_nodes_per_face + 3] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (gl_j + 0)*jnstride + (gl_k + 0)*knstride);
      }
	break;
      }
      case PLUS_K:{
      if(dimension == 2){
      }
      else{
	the_nodes[elct*num_nodes_per_face + 0] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (gl_j + 0)*jnstride + (gl_k + 1)*knstride);
	the_nodes[elct*num_nodes_per_face + 1] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (gl_j + 0)*jnstride + (gl_k + 1)*knstride);
	the_nodes[elct*num_nodes_per_face + 2] =1+ get_map_entry(global_node_map,(gl_i+1)*instride + (glj_plus1)*jnstride + (gl_k + 1)*knstride);
	the_nodes[elct*num_nodes_per_face + 3] =1+ get_map_entry(global_node_map,(gl_i+0)*instride + (glj_plus1)*jnstride + (gl_k + 1)*knstride);
      }
	break;		     
      }
      default:
	error_stream << "Inline_Mesh_Desc::Read_mesh(): "
		     << "Sideset applied to unknown Topo Location.";
	return 1;
	break;
      }   
    }
  }
  //END of SIDESETS
  return 0;
}




/****************************************************************************/
void Inline_Mesh_Desc::Populate_Connectivity(int * const * conn_array,                                         
					  std::map <int, int> & global_node_map
)
/****************************************************************************/
{
  int num_nodes_per_element = 4;
  if(dimension == 3){
    num_nodes_per_element = 8;
  }
  //Nodes are ordered across the entire block domain.
  //To calculate connectivity for a given element in a given block.
  //It is necessary to calculate the global element indices in i,j,k
  //that identify that element in the global space.
  int total_element_count = 0;
  for(int bct = 0; bct < numBlocks(); bct ++ ){
    int * conn = conn_array[bct];
    //build connectivity for each element
    //nodes are numbered 1-tot_num_nodes+1 across all blocks
    //incrementing i fastest
    for(unsigned elct = 0;elct < element_block_lists[bct].size();elct++,total_element_count++){
      int the_el = element_block_lists[bct][elct];
      int Kg = the_el/(nelx_tot*nely_tot);
      int Jg = (the_el - Kg*nelx_tot*nely_tot)/(nelx_tot);
      int Ig = the_el  - Kg*nelx_tot*nely_tot - Jg*nelx_tot;

      if(periodic_j){
        conn[elct*num_nodes_per_element + 0] = get_map_entry(global_node_map,(Ig+0)*instride + (Jg+0)*jnstride+(Kg+0)*knstride)+1;
        conn[elct*num_nodes_per_element + 1] = get_map_entry(global_node_map,(Ig+1)*instride + (Jg+0)*jnstride+(Kg+0)*knstride)+1;
	if(dimension == 3){
	  conn[elct*num_nodes_per_element + 0 + 4] = get_map_entry(global_node_map,(Ig+0)*instride + (Jg+0)*jnstride+(Kg+1)*knstride)+1;
	  conn[elct*num_nodes_per_element + 1 + 4] = get_map_entry(global_node_map,(Ig+1)*instride + (Jg+0)*jnstride+(Kg+1)*knstride)+1;
	}
        int Kblock = bct/(blockKstride());
        int Jblock = (bct-Kblock*blockKstride())/inline_bx;
        //last j block
        if((Jblock == (inline_by-1)) && (Jg == (nely_tot-1))){
          // last element in j direction in block
          conn[elct*num_nodes_per_element + 2] = get_map_entry(global_node_map,(Ig+1)*instride + (0+0)*jnstride+(Kg+0)*knstride) + 1;
          conn[elct*num_nodes_per_element + 3] = get_map_entry(global_node_map,(Ig+0)*instride + (0+0)*jnstride+(Kg+0)*knstride) + 1;
	  if(dimension == 3){
	    conn[elct*num_nodes_per_element + 2 + 4] = get_map_entry(global_node_map,(Ig+1)*instride + (0+0)*jnstride+(Kg+1)*knstride) + 1;
	    conn[elct*num_nodes_per_element + 3 + 4] = get_map_entry(global_node_map,(Ig+0)*instride + (0+0)*jnstride+(Kg+1)*knstride) + 1;
	  }
        }
        else{
          conn[elct*num_nodes_per_element + 2] = get_map_entry(global_node_map,(Ig+1)*instride + (Jg+1)*jnstride+(Kg+0)*knstride)+1;
          conn[elct*num_nodes_per_element + 3] = get_map_entry(global_node_map,(Ig+0)*instride + (Jg+1)*jnstride+(Kg+0)*knstride)+1;
	  if(dimension == 3){
	    conn[elct*num_nodes_per_element + 2 + 4] = get_map_entry(global_node_map,(Ig+1)*instride + (Jg+1)*jnstride+(Kg+1)*knstride)+1;
	    conn[elct*num_nodes_per_element + 3 + 4] = get_map_entry(global_node_map,(Ig+0)*instride + (Jg+1)*jnstride+(Kg+1)*knstride)+1;
	  }
        }
      }
      else{
        conn[elct*num_nodes_per_element + 0] = get_map_entry(global_node_map,(Ig+0)*instride + (Jg+0)*jnstride+(Kg+0)*knstride)+1;
        conn[elct*num_nodes_per_element + 1] = get_map_entry(global_node_map,(Ig+1)*instride + (Jg+0)*jnstride+(Kg+0)*knstride)+1;
        conn[elct*num_nodes_per_element + 2] = get_map_entry(global_node_map,(Ig+1)*instride + (Jg+1)*jnstride+(Kg+0)*knstride)+1;
        conn[elct*num_nodes_per_element + 3] = get_map_entry(global_node_map,(Ig+0)*instride + (Jg+1)*jnstride+(Kg+0)*knstride)+1;
	if(dimension == 3){
	  conn[elct*num_nodes_per_element + 0 + 4] = get_map_entry(global_node_map,(Ig+0)*instride + (Jg+0)*jnstride+(Kg+1)*knstride)+1;
	  conn[elct*num_nodes_per_element + 1 + 4] = get_map_entry(global_node_map,(Ig+1)*instride + (Jg+0)*jnstride+(Kg+1)*knstride)+1;
	  conn[elct*num_nodes_per_element + 2 + 4] = get_map_entry(global_node_map,(Ig+1)*instride + (Jg+1)*jnstride+(Kg+1)*knstride)+1;
	  conn[elct*num_nodes_per_element + 3 + 4] = get_map_entry(global_node_map,(Ig+0)*instride + (Jg+1)*jnstride+(Kg+1)*knstride)+1;
	}
      }
    }
  }
}

/****************************************************************************/
void Inline_Mesh_Desc::Populate_Map_and_Global_Element_List(int * the_map, 
							 int * global_element_numbers)
/****************************************************************************/
{
  int total_count = 0;
  for(int bct = 0; bct < numBlocks();bct ++ ){
    for(unsigned ect = 0; ect < element_block_lists[bct].size();ect ++){
      int the_el = element_block_lists[bct][ect];
      //global element indices
      int Kg = the_el/kestride;
      int Jg = (the_el - Kg*kestride)/(jestride);
      int Ig = the_el  - Kg*kestride - Jg*jestride;

      int Kbg = get_block_index(Kg,inline_bz,c_inline_nz);
      int Jbg = get_block_index(Jg,inline_by,c_inline_ny);
      int Ibg = get_block_index(Ig,inline_bx,c_inline_nx);


      //ordinal of the block
      int the_block = Ibg + Jbg*inline_bx + Kbg*blockKstride();

      //indices inside the block
      int Kblock = Kg-c_inline_nz[Kbg];
      int Jblock = Jg-c_inline_ny[Jbg];
      int Iblock = Ig-c_inline_nx[Ibg];


      //product
      //       int elid = the_block*inline_nx*inline_ny*inline_nz + Iblock + Jblock*inline_nx + Kblock*inline_nx*inline_ny;
      int elid = cum_block_totals[the_block] +
        Iblock + Jblock*a_inline_nx[Ibg] + Kblock*a_inline_nx[Ibg]*a_inline_ny[Jbg];

      the_map[total_count] = elid + 1;
      global_element_numbers[total_count] = elid + 1;
      total_count ++;
    }
  }
}

/****************************************************************************/
int Inline_Mesh_Desc::Check_Spans()
/****************************************************************************/
{
  if((inline_gmaxx - inline_gminx) <= 0.){
    error_stream << "Invalid span for 'X' range of inline mesh specification. The span must be positive";
    return 1;
  }
  
  if((inline_gmaxy - inline_gminy) <= 0.){
    error_stream << "Invalid span for 'Y' range of inline mesh specification. The span must be positive";
    return 1;
  }
  if(dimension == 3){
    if((inline_gmaxz-inline_gminz) <= 0.){
      error_stream << "Invalid span for 'Z' range of inline mesh specification. The span must be positive";
      return 1;
    }
  }
  return 0;
}

/****************************************************************************/
int Inline_Mesh_Desc::Check_Block_BC_Sets()
/****************************************************************************/
{
  std::list < PG_BC_Specification * > ::iterator setit;
  for(setit = nodeset_list.begin(); setit != nodeset_list.end();setit++){
    if((*setit)->block_boundary_set){
      int bid = (*setit)->block_id;
      int bmax = numBlocks();
      if (bid < 1 || bid > bmax){
        error_stream << "Terminating from Inline_Mesh_Desc::Check_Block_BC_Sets,block index ";
        error_stream << bid ;
        error_stream << " is outside the range of blocks present in the mesh  1 to ";
        error_stream << bmax;
        error_stream << ".";
	return 1;
      }
    }
  }
  for(setit = sideset_list.begin(); setit != sideset_list.end();setit++){
    if((*setit)->block_boundary_set){
      int bid = (*setit)->block_id;
      int bmax = numBlocks();
      if (bid < 1 || bid > bmax){
        error_stream << "Terminating from Inline_Mesh_Desc::Check_Block_BC_Sets,block index ";
        error_stream << bid ;
        error_stream << " is outside the range of blocks present in the mesh  1 to ";
        error_stream << bmax;
        error_stream << ".";
	return 1;
       }
    }
  }
  return Rename_Block_BC_Sets();
}

/****************************************************************************/
void Inline_Mesh_Desc::Size_BC_Sets(int nnx, 
				 int nny, 
				 int nnz)
/****************************************************************************/
{
  //Nodesets
  std::list < PG_BC_Specification * > ::iterator setit;
  for(setit = nodeset_list.begin(); setit != nodeset_list.end();setit++){
    Topo_Loc the_location = (*setit)->location;
    if((*setit)->block_boundary_set){
      int bid = (*setit)->block_id-1;
      int kind = bid/(inline_bx*inline_by);
      int jind = (bid-kind*(inline_bx * inline_by))/inline_bx;
      int iind = bid - jind *(inline_bx) - kind*(inline_bx * inline_by);
      
      (*setit)->limits = getLimits(the_location,
				   c_inline_nx[iind],c_inline_nx[iind+1]+1,
				   c_inline_ny[jind],c_inline_ny[jind+1]+1,
				   c_inline_nz[kind],c_inline_nz[kind+1]+1,
				   nnx, 
				   nny);
    }
    else{
      (*setit)->limits = getLimits(the_location,
                                   0,nnx,
                                   0,nny,
                                   0,nnz,
				   nnx,
				   nny);
    }
  }
  //Sidesets
  for(setit = sideset_list.begin(); setit != sideset_list.end();setit++){
    Topo_Loc the_location = (*setit)->location;
    if((*setit)->block_boundary_set){
      int bid = (*setit)->block_id-1;
      int kind = bid/(inline_bx*inline_by);
      int jind = (bid-kind*(inline_bx * inline_by))/inline_bx;
      int iind = bid - jind *(inline_bx) - kind*(inline_bx * inline_by);
      (*setit)->limits = getLimits(the_location,
                                   c_inline_nx[iind],c_inline_nx[iind+1],
                                   c_inline_ny[jind],c_inline_ny[jind+1],
                                   c_inline_nz[kind],c_inline_nz[kind+1],
                                   nelx_tot, 
				   nely_tot);
    }
    else{
      (*setit)->limits = getLimits(the_location,
                                   0,nelx_tot,
                                   0,nely_tot,
                                   0,nelz_tot,
				   nelx_tot,
				   nely_tot);
    } 
  }
}

/******************************************************************************/
LoopLimits Inline_Mesh_Desc::getLimits( Topo_Loc the_set_location,
				     int sx, int nx, 
				     int sy, int ny, 
				     int sz, int nz, 
				     int irange, int jrange)
/******************************************************************************/
{
  LoopLimits ll;
  int istart = sx;
  int iend = nx;
  int jstart = sy;
  int jend = ny;
  int kstart = sz;
  int kend = nz;

  switch(the_set_location) {
  case MINUS_I:{
    iend = istart + 1;
    break;
  }
  case PLUS_I:{
    istart = iend - 1;
    break;
  }
  case MINUS_J:{
    jend = jstart + 1;
    break;
  }
  case PLUS_J:{
    jstart = jend -1;
    break;
  }
  case MINUS_K:{
    kend = kstart + 1;
    break;
  }
  case PLUS_K:{
    kstart = kend -1;
    break;		     
  }
  case EDGE0:{
    jend = jstart + 1;
    kend = kstart + 1;
    break;		     
  }
  case EDGE1:{
    istart = iend - 1;
    kend = kstart + 1;
    break;		     
  }
  case EDGE2:{
    jstart = jend - 1;
    kend = kstart + 1;
    break;		     
  }
  case EDGE3:{
    iend = istart + 1;
    kend = kstart + 1;
    break;		     
  }
  case EDGE4:{
    iend = istart + 1;
    jend = jstart + 1;
    break;		     
  }
  case EDGE5:{
    istart = iend - 1;
    jend = jstart + 1;
    break;		     
  }
  case EDGE6:{
    istart = iend - 1;
    jstart = jend - 1;
    break;		     
  }
  case EDGE7:{
    iend = istart + 1;
    jstart = jend - 1;
    break;		     
  }
  case EDGE8:{
    jend = jstart + 1;
    kstart = kend - 1;
    break;		     
  }
  case EDGE9:{
    kstart = kend - 1;
    istart = iend - 1;
    break;		     
  }
  case EDGE10:{
    jstart = jend - 1;
    kstart = kend - 1;
    break;		     
  }
  case EDGE11:{
    iend = istart + 1;
    kstart = kend - 1;
    break;		     
  }
  case VERTEX0:{
    iend = istart + 1;
    jend = jstart + 1;
    kend = kstart + 1;
    break;		     
  }
  case VERTEX1:{
    istart = iend - 1;
    jend = jstart + 1;
    kend = kstart + 1;
    break;		     
  }
  case VERTEX2:{
    istart = iend - 1;
    jstart = jend - 1;
    kend = kstart + 1;
    break;		     
  }
  case VERTEX3:{
    iend = istart + 1;
    jstart = jend - 1;
    kend = kstart + 1;
    break;		     
  }
  case VERTEX4:{
    iend = istart + 1;
    jend = jstart + 1;
    kstart = kend - 1;
    break;		     
  }
  case VERTEX5:{
    istart = iend - 1;
    jend = jstart + 1;
    kstart = kend - 1;
    break;		     
  }
  case VERTEX6:{
    istart = iend - 1;
    jstart = jend - 1;
    kstart = kend - 1;
    break;		     
  }
  case VERTEX7:{
    iend = istart + 1;
    jstart = jend - 1;
    kstart = kend - 1;
    break;		     
  }
  case NUM_TOPO_CONNECTIONS:{
    break;		     
  }
  case PROCESSOR_NODE:{
    break;		     
  }
  default:
    iend = istart;
    jend = jstart;
    kend = kstart;
    break;
  }
  ll.is = istart;
  ll.ie = iend;
  ll.js = jstart;
  ll.je = jend;
  ll.jstride = irange;
  ll.total = (iend-istart)*(jend-jstart);
  if(dimension == 3){
    ll.ks = kstart;
    ll.ke = kend;
    ll.kstride = irange*jrange;
    ll.total = (iend-istart)*(jend-jstart)*(kend-kstart);
  }
  return ll;
}




/*****************************************************************************/
void Inline_Mesh_Desc::ZeroSet()
/*****************************************************************************/
{ 


  inline_geometry_type = UNKNOWN;
  inline_decomposition_type = BISECTION;
  trisection_blocks = 0;
  inline_bx = 1;
  inline_by = 1;
  inline_bz = 1;
  inline_nx = 1;
  inline_ny = 1;
  inline_nz = 1;
  inline_gminx = 0.;
  inline_gminy = 0.;
  inline_gminz = 0.;
  inline_gmaxx = 0.;
  inline_gmaxy = 0.;
  inline_gmaxz = 0.;
  inc_nels[0] = 0;
  inc_nels[1] = 0;
  inc_nels[2] = 0;
  inc_nocuts[0] = false;
  inc_nocuts[1] = false;
  inc_nocuts[2] = false;
  inline_nprocs[0] = 1;
  inline_nprocs[1] = 1;
  inline_nprocs[2] = 1;
  periodic_i = false;
  periodic_j = false;
  periodic_k = false;
  try_squared = false;
  enforce_periodic = false;
  Element_Density_Functions[0] = NULL;
  Element_Density_Functions[1] = NULL;
  Element_Density_Functions[2] = NULL;
  Geometry_Transform_Function = NULL;
  a_inline_nx = NULL;
  a_inline_ny = NULL;
  a_inline_nz = NULL;
  c_inline_nx = NULL;
  c_inline_ny = NULL;
  c_inline_nz = NULL;
  cum_block_totals = NULL;
  els_in_block = NULL;
  nelx_tot = 0;
  nely_tot = 0;
  nelz_tot = 0;
  block_dist = new double * [3];
  c_block_dist = new double * [3];
  first_size = new double * [3];
  last_size = new double * [3];
  interval = new int * [3];
  for(int i = 0; i < 3; i ++){
    block_dist[i] = NULL;
    c_block_dist[i] = NULL;
    first_size[i] = NULL;
    last_size[i] = NULL;
    interval[i] = NULL;
  }

    Icoors = NULL;
    Jcoors = NULL;
    Kcoors = NULL;
    base_partition = NULL;
    
    transition_radius = -1;

  my_rank = 0;
  num_processors = 1;

  topo_loc_to_exo_face[MINUS_I] = 4;
  topo_loc_to_exo_face[PLUS_I] = 2;
  topo_loc_to_exo_face[MINUS_J] = 1;
  topo_loc_to_exo_face[PLUS_J] = 3;
  topo_loc_to_exo_face[MINUS_K] = 5;
  topo_loc_to_exo_face[PLUS_K] = 6;

  sideset_vectors = NULL;
  nodeset_vectors = NULL;

  debug_mode = false;

}


/*****************************************************************************/
void Inline_Mesh_Desc::Display_Class(std::ostream& s, const std::string &indent)
/*****************************************************************************/
{
  for(int i = 0; i < dimension; i++){
    if(Element_Density_Functions[i])Element_Density_Functions[i]->Display_Class(s,indent);
  }
}

/*****************************************************************************/
void Inline_Mesh_Desc::Populate_Border_Nodes_Elements( int * internal_elements,
						       int * internal_nodes,
						       int * border_elements,
						       int * border_nodes,
						       std::list   <int> & internal_node_list,	
						       std::list   <int> & border_nodes_list,
						       std::list   <int> & internal_element_list,
						       std::list   <int> & border_elements_list,
						       std::map <int, int> & global_node_map,
						       std::map <int, int> & global_element_map)
/*****************************************************************************/
{
  std::list <int> :: iterator eit;
  int the_count = 0;
  for(eit = internal_element_list.begin();eit != internal_element_list.end();eit ++,the_count++)
    internal_elements[the_count]= get_map_entry(global_element_map,(*eit))+1;
  the_count = 0;
  for(eit = internal_node_list.begin();eit != internal_node_list.end();eit ++,the_count++)
    internal_nodes[the_count]= get_map_entry(global_node_map,(*eit))+1;
  the_count = 0;
  for(eit = border_elements_list.begin();eit != border_elements_list.end();eit ++,the_count++)
    border_elements[the_count]= get_map_entry(global_element_map,(*eit))+1;
  the_count = 0;
  for(eit = border_nodes_list.begin();eit != border_nodes_list.end();eit ++,the_count++)
    border_nodes[the_count]= get_map_entry(global_node_map,(*eit))+1;
}


/*****************************************************************************/
void Inline_Mesh_Desc::Populate_Cmap( int * node_cmap_node_cnts,
				      int * node_cmap_ids,
				      int * elem_cmap_elem_cnts,
				      int * elem_cmap_ids,
				      std::vector <int> & node_neighbor_vector,
				      std::vector <int> & element_neighbor_vector,
				      std::list <int>  * & boundary_node_list,                   
				      std::list < std::pair <int ,Topo_Loc > > * & boundary_element_list)
/*****************************************************************************/
{
  for(unsigned i = 0; i < node_neighbor_vector.size();i++){
    node_cmap_node_cnts[i] = boundary_node_list[i].size();
    node_cmap_ids[i] = node_neighbor_vector[i];
  }
  for(unsigned i = 0; i < element_neighbor_vector.size();i++){
    elem_cmap_elem_cnts[i] = boundary_element_list[i].size();
    elem_cmap_ids[i] = element_neighbor_vector[i];
  }
}

/*****************************************************************************/
void Inline_Mesh_Desc::Populate_Parallel_Info( int* const * comm_node_ids ,
					       int* const * comm_node_proc_ids,
					       int* const * comm_elem_ids,
					       int* const * comm_side_ids,
					       int* const * comm_elem_proc_ids,
					       std::vector <int> & node_neighbor_vector,
					       std::vector <int> & element_neighbor_vector,		
					       std::list <int>  * & boundary_node_list,
					       std::map <int, int> & global_node_map,
					       std::list <std::pair <int ,Topo_Loc > > * & boundary_element_list,
					       std::map <int, int> & global_element_map)
/*****************************************************************************/
{
  for(unsigned i = 0; i < node_neighbor_vector.size();i++){
    int * comm_nodes = comm_node_ids[i];
    int * comm_node_procs = comm_node_proc_ids[i];
    std::list <int> :: iterator nlit;
    int nct = 0;
    for(nlit = boundary_node_list[i].begin();nlit != boundary_node_list[i].end();nlit++,nct ++){
      comm_nodes[nct] = get_map_entry(global_node_map,(*nlit))+1;
      comm_node_procs[nct] = node_neighbor_vector[i];// is this right?
    }
  }
  for(unsigned i = 0; i < element_neighbor_vector.size();i++){
    int * comm_elements = comm_elem_ids[i];
    int * comm_sides = comm_side_ids[i];
    int * comm_elem_procs = comm_elem_proc_ids[i];
    std::list < std::pair <int ,Topo_Loc > > :: iterator nlit;
    int nct = 0;
    for(nlit = boundary_element_list[i].begin();nlit != boundary_element_list[i].end();nlit++,nct ++){
      comm_elements[nct] = get_map_entry(global_element_map,(*nlit).first)+1;//foo bar
      comm_sides[nct] = topo_loc_to_exo_face[(*nlit).second];
      comm_elem_procs[nct] = element_neighbor_vector[i];// is this right?
    }
  }
}

/*****************************************************************************/
void Inline_Mesh_Desc::setStrides()
/*****************************************************************************/
{
  instride = 1;
  jnstride = nelx_tot+1;
  knstride = (nelx_tot+1)*(nely_tot+1);
  
  if(inline_geometry_type == RADIAL && periodic_j){
    instride = 1;
    jnstride = nelx_tot+1;
    knstride = (nelx_tot+1)*(nely_tot);
  }
  
  iestride = 1;
  jestride = nelx_tot;
  kestride = (nelx_tot)*(nely_tot);

}

//! A utility function to build up required bookkeeping objects.
/****************************************************************************/
void Inline_Mesh_Desc::get_l_i_j_k_from_element_number(int el,
								     int & l,
								     int & i,
								     int & j,
								     int & k)
/****************************************************************************/
{
  l = 0;
  k = el/kestride;
  int remainder = el-k*kestride;
  
  j = (remainder)/(jestride);
  i = remainder - j*(jestride);

}

//! A utility function to build up required bookkeeping objects.
/****************************************************************************/
void Inline_Mesh_Desc::get_l_i_j_k_from_node_number(int nn,
								 int & l,
								 int & i,
								 int & j,
								 int & k)
/****************************************************************************/
{
  k = nn/knstride;
  int remainder = nn - k*knstride;  
  j = remainder / (nelx_tot+1);
  i = remainder - j*(nelx_tot+1);
}

/****************************************************************************/
int Inline_Mesh_Desc::get_element_number_from_l_i_j_k( int  l,
						       int  i,
						       int  j,
						       int  k)
/****************************************************************************/
{
  //adding bounds checking to deal with neighbor calcultion
  // return -1 if off mesh
  // adjust and recurse if i or j throw element onto adjacent block
  if(k < 0) return -1;
  if(k >= nelz_tot) return -1;
  if(periodic_j){
    if(j < 0) j = nely_tot-1;
    if(j >= nely_tot)j = 0;
  }
  else{
  if(j < 0) return -1;
  if(j >= nely_tot)return -1;
  }

  if(i<0)return -1;
  if(i >= nelx_tot)return -1;

  int elno;
  elno = k*kestride;
  elno += j*jestride;
  elno += i;

  return elno;
}

/****************************************************************************/
int Inline_Mesh_Desc::get_node_number_from_l_i_j_k( int  l,
								 int  i,
								 int  j,
								 int  k)
  /****************************************************************************/
{
  if(periodic_j){
    if( j == nely_tot)j = 0;
  }
  int nno = k*knstride;
  nno += j*jnstride;
  nno += i;
  return nno;
}


/****************************************************************************/
int Inline_Mesh_Desc::get_neighbor(Topo_Loc tl,
				   int ll, 
				   int li, 
				   int lj, 
				   int lk)
/****************************************************************************/
{
  switch(tl) {
  
  case MINUS_I:{
    return get_element_number_from_l_i_j_k(ll,li-1,lj,lk);
  }
  case PLUS_I:{
    return get_element_number_from_l_i_j_k(ll,li+1,lj,lk);
  }
  case MINUS_J:{
    return get_element_number_from_l_i_j_k(ll,li,lj-1,lk);
  }
  case PLUS_J:{
    return get_element_number_from_l_i_j_k(ll,li,lj+1,lk);
  }
  case MINUS_K:{
    return get_element_number_from_l_i_j_k(ll,li,lj,lk-1);
  }
  case PLUS_K:{
    return get_element_number_from_l_i_j_k(ll,li,lj,lk+1);
  }
  case EDGE0:{
    return get_element_number_from_l_i_j_k(ll,li,lj-1,lk-1);
  }
  case EDGE1:{
    return get_element_number_from_l_i_j_k(ll,li+1,lj,lk-1);
  }
  case EDGE2:{
    return get_element_number_from_l_i_j_k(ll,li,lj+1,lk-1);
  }
  case EDGE3:{
    return get_element_number_from_l_i_j_k(ll,li-1,lj,lk-1);
  }
  case EDGE4:{
    return get_element_number_from_l_i_j_k(ll,li-1,lj-1,lk);
  }
  case EDGE5:{
    return get_element_number_from_l_i_j_k(ll,li+1,lj-1,lk);
  }
  case EDGE6:{
    return get_element_number_from_l_i_j_k(ll,li+1,lj+1,lk);
  }
  case EDGE7:{
    return get_element_number_from_l_i_j_k(ll,li-1,lj+1,lk);
  }
  case EDGE8:{
    return get_element_number_from_l_i_j_k(ll,li,lj-1,lk+1);
  }
  case EDGE9:{
    return get_element_number_from_l_i_j_k(ll,li+1,lj,lk+1);
  }
  case EDGE10:{
    return get_element_number_from_l_i_j_k(ll,li,lj+1,lk+1);
  }
  case EDGE11:{
    return get_element_number_from_l_i_j_k(ll,li-1,lj,lk+1);
  }
  case VERTEX0:{
    return get_element_number_from_l_i_j_k(ll,li-1,lj-1,lk-1);
  }
  case VERTEX1:{
    return get_element_number_from_l_i_j_k(ll,li+1,lj-1,lk-1);
  }
  case VERTEX2:{
    return get_element_number_from_l_i_j_k(ll,li+1,lj+1,lk-1);
  }
  case VERTEX3:{
    return get_element_number_from_l_i_j_k(ll,li-1,lj+1,lk-1);
  }
  case VERTEX4:{
    return get_element_number_from_l_i_j_k(ll,li-1,lj-1,lk+1);
  }
  case VERTEX5:{
    return get_element_number_from_l_i_j_k(ll,li+1,lj-1,lk+1);
  }
  case VERTEX6:{
    return get_element_number_from_l_i_j_k(ll,li+1,lj+1,lk+1);
  }
  case VERTEX7:{
    return get_element_number_from_l_i_j_k(ll,li-1,lj+1,lk+1);
  }
  default:
    break;
  }
  return -1;
}

/****************************************************************************/
void Inline_Mesh_Desc::get_face_nodes(Topo_Loc tl,
				      int global_element_id,
				      int the_nodes[])
/****************************************************************************/
{
  int gl_l,gl_i,gl_j,gl_k;
  get_l_i_j_k_from_element_number(global_element_id,gl_l,gl_i,gl_j,gl_k);
  switch(tl) {
    
  case MINUS_I:{
      if(dimension == 2){
	the_nodes[0] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 1, gl_k + 0);
	the_nodes[1] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 0, gl_k + 0);
      }
      else{
	the_nodes[0] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 0, gl_k + 0);
	the_nodes[1] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 0, gl_k + 1);
	the_nodes[2] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 1, gl_k + 1);
	the_nodes[3] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 1, gl_k + 0);
      }
    break;
  }
  case PLUS_I:{
      if(dimension == 2){
	the_nodes[0] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 0, gl_k + 0);
	the_nodes[1] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 1, gl_k + 0);
      }    
      else{
	the_nodes[0] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 0, gl_k + 0);
	the_nodes[1] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 1, gl_k + 0);
	the_nodes[2] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 1, gl_k + 1);
	the_nodes[3] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 0, gl_k + 1);
      }
    break;
  }
  case MINUS_J:{
      if(dimension == 2){
	the_nodes[0] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 0, gl_k + 0);
	the_nodes[1] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 0, gl_k + 0);
      }
      else{
	the_nodes[0] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 0, gl_k + 0);
	the_nodes[1] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 0, gl_k + 0);
	the_nodes[2] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 0, gl_k + 1);
	the_nodes[3] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 0, gl_k + 1);
      }
    break;
  }
  case PLUS_J:{
      if(dimension == 2){
	the_nodes[0] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 1, gl_k + 0);
	the_nodes[1] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 1, gl_k + 0);
      }
      else{
	the_nodes[0] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 1, gl_k + 0);
	the_nodes[1] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 1, gl_k + 0);
	the_nodes[2] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 1, gl_k + 1);
	the_nodes[3] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 1, gl_k + 1);
      }
    break;
  }
  case MINUS_K:{
      if(dimension == 2){
      }
      else{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 0, gl_k + 0);
    the_nodes[1] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 1, gl_k + 0);
    the_nodes[2] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 1, gl_k + 0);
    the_nodes[3] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 0, gl_k + 0);
      }
    break;
  }
  case PLUS_K:{
      if(dimension == 2){
      }
      else{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 0, gl_k + 1);
    the_nodes[1] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 0, gl_k + 1);
    the_nodes[2] = get_node_number_from_l_i_j_k(gl_l, gl_i + 1, gl_j + 1, gl_k + 1);
    the_nodes[3] = get_node_number_from_l_i_j_k(gl_l, gl_i + 0, gl_j + 1, gl_k + 1);
      }
    break;		     
  }
  case EDGE0:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+0,gl_k+0);
    the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+0,gl_k+0);
    break;		     
  }
  case EDGE1:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+0,gl_k+0);
    the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+1,gl_k+0);
    break;		     
  }
  case EDGE2:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+1,gl_k+0);
    the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+1,gl_k+0);
    break;		     
  }
  case EDGE3:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+0,gl_k+0);
    the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+1,gl_k+0);
    break;		     
  }
  case EDGE4:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+0,gl_k+0);
    if(dimension == 3){
      the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+0,gl_k+1);
    }
    break;		     
  }
  case EDGE5:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+0,gl_k+0);
    if(dimension == 3){
      the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+0,gl_k+1);
    }
    break;		     
  }
  case EDGE6:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+1,gl_k+0);
    if(dimension == 3){
      the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+1,gl_k+1);
    }
    break;		     
  }
  case EDGE7:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+1,gl_k+0);
    if(dimension == 3){
      the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+1,gl_k+1);
    }
    break;		     
  }
  case EDGE8:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+0,gl_k+1);
    the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+0,gl_k+1);
    break;		     
  }
  case EDGE9:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+0,gl_k+1);
    the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+1,gl_k+1);
    break;		     
  }
  case EDGE10:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+1,gl_k+1);
    the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+1,gl_k+1);
    break;		     
  }
  case EDGE11:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+0,gl_k+1);
    the_nodes[1] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+1,gl_k+1);
    break;		     
  }
  case VERTEX0:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+0,gl_k+0);
    break;		     
  }
  case VERTEX1:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+0,gl_k+0);
    break;		     
  }
  case VERTEX2:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+1,gl_k+0);
    break;		     
  }
  case VERTEX3:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+1,gl_k+0);
    break;		     
  }
  case VERTEX4:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+0,gl_k+1);
    break;		     
  }
  case VERTEX5:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+0,gl_k+1);
    break;		     
  }
  case VERTEX6:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+1,gl_j+1,gl_k+1);
    break;		     
  }
  case VERTEX7:{
    the_nodes[0] = get_node_number_from_l_i_j_k(gl_l,gl_i+0,gl_j+1,gl_k+1);
    break;		     
  }
  default:
    
    break;
  }   
}


/****************************************************************************/
void Inline_Mesh_Desc::Calc_Parallel_Info(                                          
					  std::vector <int> & element_vector,
					  std::vector<int> & global_node_vector,   
					  std::map <int, int> & global_node_map,                          
					  std::list   <int> & internal_node_list,	
					  std::list   <int> & border_nodes_list,
					  std::list   <int> & internal_element_list,
					  std::list   <int> & border_elements_list,
					  std::list   <int> & node_proc_id_list,
					  std::list   <int> & element_proc_id_list,
					  std::vector <int> & node_neighbor_vector,
					  std::list <int> * &  boundary_node_list,
					  std::vector <int> & element_neighbor_vector,
					  std::list <std::pair <int ,Topo_Loc > > * & boundary_element_list)
  /****************************************************************************/
{

  Tel *el_array = NULL;
  if(element_vector.size()>0){
    el_array = new Tel[element_vector.size()];
    for(unsigned gev = 0; gev < element_vector.size(); gev ++){
      el_array[gev].global_id = element_vector[gev];
      el_array[gev].real_element = true;
    }
  }
  Tel * node_array = NULL;
  if(global_node_vector.size()>0){
    node_array = new Tel[global_node_vector.size()];
    
    for(unsigned gnv = 0;gnv < global_node_vector.size();gnv ++){
      node_array[gnv].global_id = global_node_vector[gnv];
    }
  }

  // walk all local elements
  // peer through their neighboring faces and ask if that neighbor is on my processor
  // if it is do nothing
  // if it is not
  //   I am a border
  //   all nodes on that face are border nodes 
  //   expand node_array and el_array objects with neighbor proc ids and topology directions

  // walk all local elements

  int nfn = 2;
  int nfaces = 4;
  int nen = 1;
  
  if(dimension == 3){
    nfn = 4;
    nfaces = 6;
    nen = 2;
  }




  for(unsigned gev = 0; gev < element_vector.size(); gev ++){
    int face_nodes_array[4];
    int my_id = el_array[gev].global_id;
    int ll,li,lj,lk;
    get_l_i_j_k_from_element_number(my_id,ll,li,lj,lk);
    el_array[gev].visits ++;
    
    for(int face_count = 0; face_count < nfaces; face_count ++){
      Topo_Loc tl = (Topo_Loc)face_count;
      int neighbor = get_neighbor(tl,ll,li,lj,lk);
      if(neighbor >= 0){
	unsigned neighbor_proc_id = Element_Proc(neighbor);
	if(neighbor_proc_id != my_rank){
	  std::pair < int,Topo_Loc> conn_pair(neighbor_proc_id,tl);
	  el_array[gev].conn_connections.push_back(conn_pair);

	  el_array[gev].visits ++;
	  get_face_nodes(tl,my_id,face_nodes_array);
     
	  for(int fnc = 0; fnc < nfn; fnc ++){
	    node_array[get_map_entry(global_node_map,face_nodes_array[fnc])].visits ++;
	    node_array[get_map_entry(global_node_map,face_nodes_array[fnc])].proc_neighbors.push_back(neighbor_proc_id);
	  }
	}
      }
    }


    // need to do edges and vertex neighbors for nodes
    int edge_start = EDGE4;
    int edge_end = EDGE8;
    if(dimension == 3){
      edge_start = EDGE0;
      edge_end = VERTEX0;
    }
    for(int edge_count = edge_start; edge_count < edge_end; edge_count ++){
      Topo_Loc tl = (Topo_Loc)edge_count;
      int neighbor = get_neighbor(tl,ll,li,lj,lk);
      if(neighbor >= 0){
	unsigned neighbor_proc_id = Element_Proc(neighbor);
	if(neighbor_proc_id != my_rank){
	  get_face_nodes(tl,my_id,face_nodes_array);
	  
	  for(int fnc = 0; fnc < nen; fnc ++){
	    node_array[get_map_entry(global_node_map,face_nodes_array[fnc])].visits ++;
	    node_array[get_map_entry(global_node_map,face_nodes_array[fnc])].proc_neighbors.push_back(neighbor_proc_id);
	  }
	}
      }
    }
    if(dimension == 3){
      // need to do vertices and vertex neighbors for nodes
      for(int vertex_count = EDGE11; vertex_count < NUM_TOPO_CONNECTIONS; vertex_count ++){
	Topo_Loc tl = (Topo_Loc)vertex_count;
	int neighbor = get_neighbor(tl,ll,li,lj,lk);
	if(neighbor >= 0){
	  unsigned neighbor_proc_id = Element_Proc(neighbor);
	  if(neighbor_proc_id != my_rank){
	    get_face_nodes(tl,my_id,face_nodes_array);
	    for(int fnc = 0; fnc < 1; fnc ++){
	      node_array[get_map_entry(global_node_map,face_nodes_array[fnc])].visits ++;
	      node_array[get_map_entry(global_node_map,face_nodes_array[fnc])].proc_neighbors.push_back(neighbor_proc_id);
	    }
	  }
	}
      }
    }
  }

  for(unsigned i = 0; i < element_vector.size();i ++){
    if(el_array[i].visits > 1){
      // loop over all conn_connections
      std::list < std::pair < int , Topo_Loc > > ::iterator conit;
      for(conit  = el_array[i].conn_connections.begin();
          conit != el_array[i].conn_connections.end();
          conit ++){
        element_proc_id_list.push_back((*conit).first);
      }
    }
  }
  // sort and uniq element_proc_id_list
  element_proc_id_list.sort();
  element_proc_id_list.unique();

  if(element_proc_id_list.size()){
    boundary_element_list = new std::list < std::pair <int ,Topo_Loc > > [element_proc_id_list.size()];
  }

  std::map <int,int> element_neighbor_proc_map; //key is proc_id value is ordinal
  std::list <int> ::iterator listit;
  int the_count = 0;
  for(listit = element_proc_id_list.begin(); listit != element_proc_id_list.end(); listit++,the_count ++){
    element_neighbor_proc_map[*listit] = the_count;
    element_neighbor_vector.push_back(*listit);
  }

  // now populate the maps

  for(unsigned i = 0; i < element_vector.size();i ++){
    int the_element = element_vector[i];
    if(el_array[i].visits == 1){
      internal_element_list.push_back(the_element);
    }
    if(el_array[i].visits > 1){
      // loop over all conn_connections
      
      border_elements_list.push_back(the_element);
      std::list < std::pair < int , Topo_Loc > > ::iterator conit;
      for(conit  = el_array[i].conn_connections.begin();
          conit != el_array[i].conn_connections.end();
          conit ++){
	
        int index = get_map_entry(element_neighbor_proc_map,(*conit).first);
        boundary_element_list[index].push_back( std::pair <int ,Topo_Loc >(the_element,(*conit).second));
      }
    }
  }

  border_elements_list.sort();
  border_elements_list.unique();


  for(unsigned gnv = 0;gnv < global_node_vector.size();gnv ++){
    if(node_array[gnv].visits > 0){
      // loop over all conn_connections
      std::list < int > ::iterator conit;
      for(conit  = node_array[gnv].proc_neighbors.begin();
	  conit != node_array[gnv].proc_neighbors.end();
	  conit ++){
	node_proc_id_list.push_back((*conit));
      }
    }
  }
    
  node_proc_id_list.sort();
  node_proc_id_list.unique();

  std::map <int,int> node_neighbor_proc_map; //key is proc_id value is ordinal
  std::list <int> ::iterator nlistit;
  the_count = 0;
  for(nlistit = node_proc_id_list.begin(); nlistit != node_proc_id_list.end(); nlistit++,the_count ++){
    node_neighbor_proc_map[*nlistit] = the_count;
    node_neighbor_vector.push_back(*nlistit);
  }

  if(node_proc_id_list.size()){
    boundary_node_list = new std::list <int> [node_proc_id_list.size()];
  }



  //node array needs global_id!!!!
  for(unsigned i = 0;i < global_node_vector.size();i ++){
    if(node_array[i].visits == 0){
      int the_node = node_array[i].global_id;
      internal_node_list.push_back(the_node);
    }
    else if(node_array[i].visits > 0){
      int the_node = node_array[i].global_id;
      // loop over all conn_connections
      std::list < int > ::iterator conit;
      for(conit  = node_array[i].proc_neighbors.begin();
	  conit != node_array[i].proc_neighbors.end();
	  conit ++){
	int index = get_map_entry(node_neighbor_proc_map,(*conit));
	boundary_node_list[index].push_back(the_node);
	border_nodes_list.push_back(the_node);
      }
    }
  }
  // sort the boundary_node_list
  for(unsigned i = 0; i < node_proc_id_list.size();i++){
    boundary_node_list[i].sort();
    boundary_node_list[i].unique();    
  }
  border_nodes_list.sort();
  border_nodes_list.unique();

  
  
  
  delete [] el_array;
  delete[] node_array;

  //check number node comm_maps
}

//! Reads in/creates the serial component of the unstructured mesh in parallel
/****************************************************************************/
void Inline_Mesh_Desc::Calc_Serial_Component(Partition * my_part,
                                          std::vector <int> & element_vector,
                                          std::list <int> & global_node_list,
                                          std::vector<int> & global_node_vector,
                                          std::map <int, int> & global_node_map,
                                          std::map <int, int> & global_element_map)
/****************************************************************************/
{
  //NODESETS
  // Nodesets are simple because we are numbering nodes across the entire domain
  // sequentially in i,j,k
  // The loop limits are the index bounds that allow traversal of the nodes of interest.
  //DMHMOD

  if(nodeset_list.size() > 0)nodeset_vectors = new std::vector <int> [nodeset_list.size()];

  std::list < PG_BC_Specification *> ::iterator setit;
  int nsct = 0;
  for(setit = nodeset_list.begin(); setit != nodeset_list.end();setit++,nsct ++){
    LoopLimits ll = (*setit)->limits;
    std::list < int > nodes_vector;

    // SET_INTERSECT
    std::list < int > tnodes_vector;
    std::list < int > tglobal_nodes_vector;


    for(unsigned the_nct = 0; the_nct < global_node_vector.size();the_nct++){
      tglobal_nodes_vector.push_back(global_node_vector[the_nct]);
    }

    tglobal_nodes_vector.sort();
    tglobal_nodes_vector.unique();

    if(dimension == 3){
      for ( int _nk_ = ll.ks; _nk_ < ll.ke; _nk_ ++){ 
	for ( int _nj_ = ll.js; _nj_ < ll.je; _nj_ ++){ 
	  for ( int _ni_ = ll.is; _ni_ < ll.ie; _ni_ ++){ 
	    int global_node_id = get_node_number_from_l_i_j_k(trisection_blocks,_ni_,_nj_,_nk_);
	    tnodes_vector.push_back(global_node_id);
	  }
	}
      }
    }
    else{
      int _nk_ = 0;
      for ( int _nj_ = ll.js; _nj_ < ll.je; _nj_ ++){ 
 	for ( int _ni_ = ll.is; _ni_ < ll.ie; _ni_ ++){ 
	  int global_node_id = get_node_number_from_l_i_j_k(trisection_blocks,_ni_,_nj_,_nk_);
	  tnodes_vector.push_back(global_node_id);
	}
      }
    }

    tnodes_vector.sort();
    tnodes_vector.unique();

    std::set_intersection(tnodes_vector.begin(),
		     tnodes_vector.end(),
		     tglobal_nodes_vector.begin(),
		     tglobal_nodes_vector.end(),
		     front_inserter(nodes_vector));

    nodes_vector.sort();
    nodes_vector.unique();

    std::list < int > :: iterator nit;
    for(nit = nodes_vector.begin(); nit != nodes_vector.end(); nit ++){
      for(unsigned the_nct = 0; the_nct < global_node_vector.size();the_nct++){
        if((*nit) == global_node_vector[the_nct])nodeset_vectors[nsct].push_back((*nit));
      }
    }
  }
  // END OF NODESETS

  //SIDESETS
  // Sidesets are defined by the element id and the id of the face to which the set
  // applies.

  // For the purposes of calculation the sideset definition process starts by 
  // considering all elements to be numbered sequentially in i then j then k
  // across all blocks. This is not actually the case since exodus requires elements
  // to be numbered sequentially within a block.

  // If we consider the elements to be numbered sequentially across blocks in i,j, and k
  // then we can use a loop limits type call to get the i,j,and k index limits of the
  // elements in the sideset. We can then loop over these limits to calculate the 
  // index of the element if it were numbered across blocks. It is then left to 
  // calculate the index of this element if it were numbered sequentially within blocks.
  // This is done by calculating the global i,j,and k indices of the element within the 
  // entire domain, calculating the indices of the block within which the element resides,
  // calculating the ordinal number of the block in which the element resides, and the 
  // local i,j,and k indices of the element within the block it resides.

  //  These values are combined to calculate the index of the element that corresponds
  // to numbering the elements sequentially within blocks. 
  nsct = 0;
  if(sideset_list.size() > 0)sideset_vectors = new std::vector < std::pair <int ,Topo_Loc > > [sideset_list.size()];  

  for(setit = sideset_list.begin(); setit != sideset_list.end();setit++,nsct ++){
     Topo_Loc the_location = (*setit)->location;
    //Sidesets allowed only on faces of block, not on edges or corners

    LoopLimits ll = (*setit)->limits;

    if(dimension == 3){
      for ( int _nk_ = ll.ks; _nk_ < ll.ke; _nk_ ++){ 
	for ( int _nj_ = ll.js; _nj_ < ll.je; _nj_ ++){ 
	  for ( int _ni_ = ll.is; _ni_ < ll.ie; _ni_ ++) {
	    int elnumber = get_element_number_from_l_i_j_k(trisection_blocks,_ni_,_nj_,_nk_);
	    unsigned the_proc_id = Element_Proc(elnumber);
	    if(the_proc_id == my_rank){// add only if on this proc	  
	      std::pair <int ,Topo_Loc > el_loc_pair(elnumber,the_location);
	      sideset_vectors[nsct].push_back(el_loc_pair);
	    }
	  }
	}
      }
    }
    else{
      int _nk_ = 0;
      for ( int _nj_ = ll.js; _nj_ < ll.je; _nj_ ++){ 
	for ( int _ni_ = ll.is; _ni_ < ll.ie; _ni_ ++) {
	  int elnumber = get_element_number_from_l_i_j_k(trisection_blocks,_ni_,_nj_,_nk_);
	  unsigned the_proc_id = Element_Proc(elnumber);
	  if(the_proc_id == my_rank){// add only if on this proc	  
	    std::pair <int ,Topo_Loc > el_loc_pair(elnumber,the_location);
	    sideset_vectors[nsct].push_back(el_loc_pair);
	  }
	}
      }
    }
  }
  //END of SIDESETS
}

/****************************************************************************/
void Inline_Mesh_Desc::getGlobal_Element_Block_Totals(int * totals_array)
/****************************************************************************/
{
  for(int bct = 0; bct < numBlocks();bct ++ ){
    int kind = bct/blockKstride();
    int jind = (bct - kind * blockKstride())/inline_bx;
    int iind = bct - jind * inline_bx - kind * blockKstride();
    totals_array[bct] = a_inline_nx[iind]*a_inline_ny[jind]*a_inline_nz[kind];
  } 
}


//! Queries which processor an element lies on.
//! Calls the recursive Partition::Element_Proc function.
/****************************************************************************/
void Inline_Mesh_Desc::Customize_Coords(double * coords, int num_nodes,int dim)
/****************************************************************************/
{
  if(!Geometry_Transform_Function)return;
  Geometry_Transform_Function->Operate(coords,num_nodes,dim);
}

}//end namespace PAMGEN_NEVADA
