/* Copyright (C) 2006,2007 Daiki Ueno <ueno@unixuser.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA */

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <stdio.h>
#include <assert.h>
#include "config.h"
#include "treil.h"
#if HAVE_EXPAT_H
#include <expat.h>
#endif

void
treil_tree_free (treil_tree_t tree)
{
  if (tree->sibling)
    treil_tree_free (tree->sibling);
  if (tree->children)
    treil_tree_free (tree->children);
  free (tree->name);
  free (tree);
}

#if HAVE_EXPAT_H
struct tree_xml_ctx {
  treil_tree_t tree;
  char *value;
  XML_Parser parser;
};

static void XMLCALL
start_element (void *user_data, const char *name, const char **atts)
{
  struct tree_xml_ctx *ctx = user_data;
  treil_tree_t tree;
  char **p;

  if (ctx->value)
    {
      free (ctx->value);
      ctx->value = NULL;
    }

  for (p = (char **)atts; p && *p; p++)
    if (!strcmp (*p, "label"))
      name = *++p;
    else if (!strcmp (*p, "value"))
      ctx->value = strdup (*++p);
    else
      ++p;

  tree = malloc (sizeof(struct treil_tree));
  if (!tree)
    {
      XML_StopParser (ctx->parser, 0);
      return;
    }

  tree->size = 0;
  tree->name = strdup (name);
  if (ctx->tree)
    {
      tree->depth = ctx->tree->depth + 1;
      tree->sibling = ctx->tree->children;
      ctx->tree->children = tree;
      tree->children = NULL;
      tree->parent = ctx->tree;
    }
  else
    {
      tree->depth = 0;
      tree->sibling = NULL;
      tree->children = NULL;
      tree->parent = NULL;
    }
  ctx->tree = tree;
}

static void XMLCALL
end_element (void *user_data, const char *name)
{
  struct tree_xml_ctx *ctx = user_data;

  assert (ctx);
  assert (ctx->tree);

  if (!ctx->tree->children && ctx->value)
    {
      char *endptr;
      long value = strtol (ctx->value, &endptr, 10);
      if ((errno == ERANGE && (value == LONG_MAX || value == LONG_MIN))
	  || (errno != 0 && value == 0))
	{
	  XML_StopParser (ctx->parser, 0);
	  return;
	}
      ctx->tree->size = value;
    }
  if (ctx->tree->parent)
    {
      if (ctx->tree->size)
	ctx->tree->parent->size += ctx->tree->size;
      ctx->tree = ctx->tree->parent;
    }
}

treil_tree_t
treil_tree_xml (int fd, const char *encoding)
{
  XML_Parser parser;
  struct tree_xml_ctx *ctx = NULL;
  treil_tree_t tree = NULL;
  char buf[BUFSIZ];
  int is_final = 0;

  parser = XML_ParserCreate (encoding);
  if (!parser)
    goto end;

  ctx = malloc (sizeof(struct tree_xml_ctx));
  if (!ctx)
    goto end;
  ctx->tree = NULL;
  ctx->value = NULL;

  XML_SetUserData (parser, ctx);
  XML_SetElementHandler (parser, start_element, end_element);

  while (!is_final)
    {
      enum XML_Status status;
      ssize_t len = read (fd, buf, BUFSIZ);
      if (len < 0)
	goto end;
      is_final = len < BUFSIZ;
      if (XML_Parse (parser, buf, len, is_final) == XML_STATUS_ERROR)
	{
	  fputs (XML_ErrorString (XML_GetErrorCode (parser)), stderr);
	  goto end;
	}
    }
  tree = ctx->tree;
 end:
  if (parser)
    XML_ParserFree (parser);
  if (ctx)
    {
      if (ctx->value)
	free (ctx->value);
      free (ctx);
    }

  return tree;
}
#endif
