# Copyright (C) 2006,2007 Daiki Ueno <ueno@unixuser.org>

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

module Treil
  class Config
    def initialize(hash)
      @hash = hash
    end

    def [](key)
      @hash[key]
    end

    def []=(key, val)
      @hash[key] = val
    end

    def config_value(rect, name)
      val = @hash[rect.tree.escape_path] || @hash['']
      if val
        val = val[name]
        return val if val
      end
    end

    def config_has_value?(rect, name)
      (@hash[rect.tree.escape_path] || @hash['']).include?(name)
    end

    def border_width(rect)
      val = config_value(rect, 'border_width')
      return val if val
      5.0 * (0.5 ** (1 + rect.tree.depth))
    end

    def border_color(rect)
      val = config_value(rect, 'border_color')
      return val if val
      [0.0, 0.0, 0.0]
    end

    def background_color(rect)
      val = config_value(rect, 'background_color')
      return val if val
      [0.7, 0.7, 0.7]
    end

    def rect_visible?(rect)
      if config_has_value?(rect, 'rect_visible')
        return config_value(rect, 'rect_visible')
      end
      return false if rect.tree.depth == 0
      max_depth = @hash['']['max_depth']
      return false if max_depth && r.tree.depth > max_depth
      true
    end

    def rect_fill?(rect)
      max_depth = @hash['']['max_depth']
      return true if rect.tree.leaf? || (max_depth && rect.tree.depth == max_depth)
    end

    def name_visible?(rect)
      if config_has_value?(rect, 'name_visible')
        return config_value(rect, 'name_visible')
      end
      return false if rect.tree.leaf?
      return false if rect.tree.depth == 0
      max_depth = @hash['']['max_depth']
      next if max_depth && rect.tree.depth > max_depth
      true
    end

    def name_color(rect)
      val = config_value(rect, 'name_color')
      return val if val
      [1.0, 1.0, 1.0] + [0.75 ** rect.tree.depth]
    end

    def name_font_face(rect)
      val = config_value(rect, 'name_font_face')
      return val if val
      {'family' => 'BoxFont 2'}
    end

    def name_font_size(rect)
      val = config_value(rect, 'name_font_size')
      return val if val
      64
    end
  end
end
