/* Copyright (C) 2007 Daiki Ueno <ueno@unixuser.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA */

#include "ruby.h"
#include "treil.h"

/* RARRAY_LEN is not available in 1.8. */
#ifndef RARRAY_LEN
#define RARRAY_LEN(a) RARRAY(a)->len
#endif

/* RARRAY_PTR is not available in 1.8. */
#ifndef RARRAY_PTR
#define RARRAY_PTR(a) RARRAY(a)->ptr
#endif

static VALUE cTree, cRect;

static treil_tree_t
rb_treil_tree_s_copy (VALUE v_tree, treil_tree_t parent, treil_tree_t sibling)
{
  treil_tree_t tree, children = NULL;
  VALUE v_name, v_size, v_children;
  long i;
  
  v_name = rb_funcall (v_tree, rb_intern ("name"), 0);
  v_size = rb_funcall (v_tree, rb_intern ("size"), 0);
  v_children = rb_funcall (v_tree, rb_intern ("children"), 0);

  tree = ALLOC(struct treil_tree);
  tree->size = NUM2LONG(v_size);
  tree->name = strdup (StringValueCStr(v_name));
  tree->depth = parent ? parent->depth + 1 : 0;
  tree->sibling = sibling;

  for (i = 0; i < RARRAY_LEN(v_children); i++)
    children = rb_treil_tree_s_copy (RARRAY_PTR(v_children)[i], tree,
				     children);
  tree->children = children;
  tree->parent = parent;

  return tree;
}

static VALUE
rb_treil_tree_s_new (VALUE klass, VALUE v_tree)
{
  treil_tree_t tree;

  tree = rb_treil_tree_s_copy (v_tree, NULL, NULL);
  v_tree = Data_Wrap_Struct(cTree, 0, treil_tree_free, tree);
  return v_tree;
}

static VALUE
rb_treil_tree_path (VALUE v_tree)
{
  treil_tree_t tree;
  VALUE v_path;

  Data_Get_Struct(v_tree, struct treil_tree, tree);
  for (v_path = rb_ary_new (); tree; tree = tree->parent)
    rb_ary_unshift (v_path, rb_str_new2 (tree->name));
  return v_path;
}

static VALUE
rb_treil_tree_is_leaf (VALUE v_tree)
{
  treil_tree_t tree;

  Data_Get_Struct(v_tree, struct treil_tree, tree);
  return tree->children ? Qfalse : Qtrue;
}

static VALUE
rb_treil_tree_size (VALUE v_tree)
{
  treil_tree_t tree;

  Data_Get_Struct(v_tree, struct treil_tree, tree);
  return INT2NUM(tree->size);
}

static VALUE
rb_treil_tree_name (VALUE v_tree)
{
  treil_tree_t tree;

  Data_Get_Struct(v_tree, struct treil_tree, tree);
  return rb_str_new2 (tree->name);
}

static VALUE
rb_treil_tree_depth (VALUE v_tree)
{
  treil_tree_t tree;

  Data_Get_Struct(v_tree, struct treil_tree, tree);
  return INT2NUM(tree->depth);
}

static void
_rb_treil_rect_each (treil_rect_t rect)
{
  if (rect->rect0)
    _rb_treil_rect_each (rect->rect0);
  if (rect->rect1)
    _rb_treil_rect_each (rect->rect1);
  if (rect->tree)
    {
      VALUE v_rect;

      v_rect = Data_Wrap_Struct(cRect, 0, 0, rect);
      rb_yield(v_rect);
    }
}

static VALUE
rb_treil_rect_each (VALUE v_rect)
{
  treil_rect_t rect;

  Data_Get_Struct(v_rect, struct treil_rect, rect);
  _rb_treil_rect_each (rect);
  return Qnil;
}

static VALUE
rb_treil_rect_fill (VALUE v_rect, VALUE v_tree)
{
  treil_rect_t rect;
  treil_tree_t tree;

  Data_Get_Struct(v_rect, struct treil_rect, rect);
  Data_Get_Struct(v_tree, struct treil_tree, tree);

  if (treil_rect_fill (rect, tree) < 0)
    rb_raise (rb_eRuntimeError, "Failed to fill rectangle");
  return v_rect;
}

static VALUE
rb_treil_rect_s_new (VALUE klass, VALUE v_x, VALUE v_y, VALUE v_width, VALUE v_height)
{
  treil_rect_t rect;
  VALUE v_rect;

  Check_Type(v_x, T_FLOAT);
  Check_Type(v_y, T_FLOAT);
  Check_Type(v_width, T_FLOAT);
  Check_Type(v_height, T_FLOAT);

  rect = treil_rect_new (RFLOAT(v_x)->value, RFLOAT(v_y)->value,
			 RFLOAT(v_width)->value, RFLOAT(v_height)->value);
  if (!rect)
    rb_raise (rb_eRuntimeError, "Can't allocate a rectangle");

  v_rect = Data_Wrap_Struct(cRect, 0, treil_rect_free, rect);
  return v_rect;
}

static VALUE
rb_treil_rect_x (VALUE v_rect)
{
  treil_rect_t rect;

  Data_Get_Struct(v_rect, struct treil_rect, rect);
  return rb_float_new (rect->x);
}

static VALUE
rb_treil_rect_y (VALUE v_rect)
{
  treil_rect_t rect;

  Data_Get_Struct(v_rect, struct treil_rect, rect);
  return rb_float_new (rect->y);
}

static VALUE
rb_treil_rect_width (VALUE v_rect)
{
  treil_rect_t rect;

  Data_Get_Struct(v_rect, struct treil_rect, rect);
  return rb_float_new (rect->width);
}

static VALUE
rb_treil_rect_height (VALUE v_rect)
{
  treil_rect_t rect;

  Data_Get_Struct(v_rect, struct treil_rect, rect);
  return rb_float_new (rect->height);
}

static VALUE
rb_treil_rect_tree (VALUE v_rect)
{
  treil_rect_t rect;
  VALUE v_tree;

  Data_Get_Struct(v_rect, struct treil_rect, rect);
  if (!rect->tree)
    return Qnil;

  v_tree = Data_Wrap_Struct(cTree, 0, 0, rect->tree);
  return v_tree;
}

void
Init_treil (void)
{
  VALUE mTreil;

  mTreil = rb_define_module ("Treil");
  cTree = rb_define_class_under (mTreil, "Tree", rb_cObject);
  rb_define_singleton_method (cTree, "new", rb_treil_tree_s_new, 1);
  rb_define_method (cTree, "size", rb_treil_tree_size, 0);
  rb_define_method (cTree, "name", rb_treil_tree_name, 0);
  rb_define_method (cTree, "depth", rb_treil_tree_depth, 0);
  rb_define_method (cTree, "path", rb_treil_tree_path, 0);
  rb_define_method (cTree, "leaf?", rb_treil_tree_is_leaf, 0);

  cRect = rb_define_class_under (mTreil, "Rect", rb_cObject);
  rb_define_singleton_method (cRect, "new", rb_treil_rect_s_new, 4);
  rb_define_method (cRect, "each", rb_treil_rect_each, 0);
  rb_define_method (cRect, "fill", rb_treil_rect_fill, 1);
  rb_define_method (cRect, "x", rb_treil_rect_x, 0);
  rb_define_method (cRect, "y", rb_treil_rect_y, 0);
  rb_define_method (cRect, "width", rb_treil_rect_width, 0);
  rb_define_method (cRect, "height", rb_treil_rect_height, 0);
  rb_define_method (cRect, "tree", rb_treil_rect_tree, 0);
}
