/* Copyright (C) 2006,2007 Daiki Ueno <ueno@unixuser.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA */

#include <sys/types.h>
#include <sys/stat.h>
#include "fnmatch.h"
#include "fts_.h"
#include <limits.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <errno.h>
#include "config.h"
#include "treil.h"

char *program_name;

static inline int
exclude_p (const char *name, const char **excludes)
{
  while (*excludes)
    {
      if (!fnmatch (*excludes, name, 0))
	return true;
      excludes++;
    }
  return false;
}

static treil_tree_t
treil_tree_dir (const char *dir, const char **excludes)
{
  FTS *fts;
  treil_tree_t parent = NULL, tree;
  char *files[2];
  int parent_excluded = 0;

  files[0] = (char *)dir;
  files[1] = NULL;

  fts = fts_open (files, FTS_LOGICAL, NULL);
  if (!fts)
    return NULL;
  while (1)
    {
      FTSENT *entry = fts_read (fts);

      if (!entry)
	goto ok;
      switch (entry->fts_info)
	{
	case FTS_DNR:
	case FTS_ERR:
	case FTS_NS:
	  errno = entry->fts_errno;
	case FTS_NSOK:
	  goto error;
	case FTS_D:
	  if (parent_excluded)
	    continue;
	  if (excludes && exclude_p (entry->fts_name, excludes))
	    {
	      parent_excluded = true;
	      continue;
	    }
	  tree = malloc (sizeof(struct treil_tree));
	  if (!tree)
	    goto error;
	  tree->size = 0;
	  tree->name = strdup (entry->fts_name);
	  tree->depth = entry->fts_level;
	  tree->sibling = NULL;
	  if (parent)
	    {
	      tree->sibling = parent->children;
	      parent->children = tree;
	    }
	  tree->children = NULL;
	  tree->parent = parent;
	  parent = tree;
	  break;
	case FTS_F:
	  if (parent_excluded || (excludes &&
				  exclude_p (entry->fts_name, excludes)))
	    continue;
	  tree = malloc (sizeof(struct treil_tree));
	  if (!tree)
	    goto error;
	  tree->size = entry->fts_statp->st_size;
	  tree->name = strdup (entry->fts_name);
	  tree->depth = entry->fts_level;
	  tree->sibling = NULL;
	  if (parent)
	    {
	      tree->sibling = parent->children;
	      parent->children = tree;
	    }
	  tree->children = NULL;
	  tree->parent = parent;
	  parent->size += tree->size;
	  break;
	case FTS_DP:
	  if (parent_excluded)
	    {
	      if (excludes && exclude_p (entry->fts_name, excludes))
		parent_excluded = 0;
	      continue;
	    }
	  if (parent && parent->parent)
	    {
	      parent->parent->size += parent->size;
	      parent = parent->parent;
	    }
	  break;
	}
    }

 error:
  if (parent)
    treil_tree_free (parent);
  parent = NULL;

 ok:
  fts_close (fts);
  return parent;
}

static char *
escape (char *str)
{
  char *new = NULL;
  int i, j;
  for (i = 0, j = 0; str[i]; i++, j++)
    {
      switch (str[i])
	{
	case '\t': case '\r': case '\n': case '%':
	  if (new)
	    {
	      new = realloc (new, strlen (new) + 3);
	      if (!new)
		return NULL;
	    }
	  else
	    {
	      new = malloc (strlen (str) + 3);
	      if (!new)
		return NULL;
	      strcpy (new, str);
	    }
	  sprintf (&new[j], "%%%02X", str[i]);
	  j += 2;
	  break;
	default:
	  if (new)
	    new[j] = str[i];
	  break;
	}
    }
  if (new)
    {
      new[j] = '\0';
      return new;
    }
  return str;
}

static void
rect_print (treil_rect_t rect, FILE *out)
{
  if (rect->rect0)
    rect_print (rect->rect0, out);
  if (rect->rect1)
    rect_print (rect->rect1, out);
  if (rect->tree)
    {
      treil_tree_t tree;

      fprintf (out, "%#.15lf\t%#.15lf\t%#.15lf\t%#.15lf\t%d\t%c",
	       rect->x, rect->y, rect->width, rect->height,
	       rect->tree->depth, rect->tree->children ? '1' : '0');
      for (tree = rect->tree; tree; tree = tree->parent)
	{
	  char *name = escape (tree->name);
	  if (!name)
	    return;
	  fprintf (out, "\t%s", name);
	  if (name != tree->name)
	    free (name);
	}
      fputc ('\n', out);
    }
}

static void
fail (char *str)
{
  if (errno)
    fprintf (stderr, "%s: %s\n", str, strerror (errno));
  else
    fprintf (stderr, "%s\n", str);
  exit (1);
}

static void
usage (FILE *out)
{
  fprintf (out, "Usage: %s [OPTIONS] [DIRECTORY]\n"
	   "Options are\n"
	   "\t--geometry=WIDTHxHEIGHT, -g\tSpecify size of the outermost rectangle\n"
#if HAVE_EXPAT_H
	   "\t--encoding=ENC, -e\tSpecify encoding of the input XML\n"
#endif
	   "\t--exclude=PAT, -x\tExclude files that match PAT\n"
	   "\t--help, -h\tShow this help\n", program_name);
}

int
main (int argc, char **argv)
{
  treil_tree_t tree;
  treil_rect_t rect;
  int c, nexcludes = 0;
  double width = 800.0, height = 600.0;
  FILE *s;
  char *encoding = NULL, **excludes = NULL;

  struct option long_options[] = {
    {"geometry", 1, 0, 'g'},
#if HAVE_EXPAT_H
    {"encoding", 1, 0, 'e'},
#endif
    {"excludes", 1, 0, 'x'},
    {"help", 0, 0, 'h'},
    {0, 0, 0, 0}
  };

  program_name = argv[0];

  while (1)
    {
      int option_index = 0;

      c = getopt_long (argc, argv, "e:g:hx:", long_options, &option_index);
      if (c == -1)
	break;
      switch (c)
	{
#if HAVE_EXPAT_H
	case 'e':
	  encoding = optarg;
	  break;
#endif
	case 'x':
	  if (!excludes)
	    {
	      excludes = calloc ((argc >> 1) + 1, sizeof(char *));
	      if (!excludes)
		{
		  perror ("calloc");
		  exit (1);
		}
	    }
	  excludes[nexcludes++] = optarg;
	  break;
	case 'g':
	  if (sscanf (optarg, "%lfx%lf", &width, &height) != 2)
	    goto error;
	  break;
	case 'h':
	  usage (stdout);
	  exit (0);
	default:
	  goto error;
	}
    }

  if (optind == argc)
    {
#if HAVE_EXPAT_H
      tree = treil_tree_xml (fileno (stdin), encoding);
#else
      goto error;
#endif
    }
  else
    {
      struct stat st;
      if (stat (argv[optind], &st) < 0)
	{
	  perror ("stat");
	  exit (1);
	}
      if (S_ISDIR(st.st_mode))
	tree = treil_tree_dir (argv[optind], (const char **)excludes);
      else
	{
#if HAVE_EXPAT_H
	  s = fopen (argv[optind], "r");
	  if (!s)
	    {
	      perror ("fopen");
	      exit (1);
	    }
	  tree = treil_tree_xml (fileno (s), encoding);
#else
	  goto error;
#endif
	}
    }
    
  if (!tree)
    fail ("Can't get tree information");

  rect = treil_rect_new (0.0, 0.0, width, height);
  if (!rect)
    fail ("Can't allocate the outermost rectangle");

  if (treil_rect_fill (rect, tree) < 0)
    fail ("Failed to fill rectangle");

  rect_print (rect, stdout);

  treil_tree_free (tree);
  treil_rect_free (rect);

  return 0;

 error:
  usage (stderr);
  exit (1);
}
