/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2013 Martin Brehm
                  2012-2013 Martin Thomas

    This file written by Martin Thomas.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "structurefactor.h"

#include "globalvar.h"
#include "maintools.h"
#include "timestep.h"
#include "xobarray.h"

#define BUF_SIZE 4096

static CxObArray g_isotopes;
static CxObArray g_sFacObserv;

static void createIsotopeList() {
	// Neutron factors from http://www.ncnr.nist.gov/resources/n-lengths/
	// X-Ray factors from http://www.ruppweb.org/new_comp/scattering_factors.htm
	CIsotope *isotope;
	try { isotope = new CIsotope("H", -3.7390f, 0.493f, 0.323f, 0.140f, 0.041f, 10.511f, 26.126f, 3.142f, 57.800f, 0.003f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("1H", -3.7406f, 0.493f, 0.323f, 0.140f, 0.041f, 10.511f, 26.126f, 3.142f, 57.800f, 0.003f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("2H", 6.671f, 0.493f, 0.323f, 0.140f, 0.041f, 10.511f, 26.126f, 3.142f, 57.800f, 0.003f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("3H", 4.792f, 0.493f, 0.323f, 0.140f, 0.041f, 10.511f, 26.126f, 3.142f, 57.800f, 0.003f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("B", 5.30f, 2.055f, 1.333f, 1.098f, 0.707f, 23.219f, 1.021f, 60.350f, 0.140f, -0.193f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("10B", -0.1f, 2.055f, 1.333f, 1.098f, 0.707f, 23.219f, 1.021f, 60.350f, 0.140f, -0.193f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("11B", 6.65f, 2.055f, 1.333f, 1.098f, 0.707f, 23.219f, 1.021f, 60.350f, 0.140f, -0.193f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("C", 6.6460f, 2.310f, 1.020f, 1.589f, 0.865f, 20.844f, 10.208f, 0.569f, 51.651f, 0.216f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("12C", 6.6511f, 2.310f, 1.020f, 1.589f, 0.865f, 20.844f, 10.208f, 0.569f, 51.651f, 0.216f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("13C", 6.19f, 2.310f, 1.020f, 1.589f, 0.865f, 20.844f, 10.208f, 0.569f, 51.651f, 0.216f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("N", 9.36f, 12.213f, 3.132f, 2.013f, 1.166f, 0.006f, 9.893f, 28.997f, 0.583f, -11.529f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("14N", 9.37f, 12.213f, 3.132f, 2.013f, 1.166f, 0.006f, 9.893f, 28.997f, 0.583f, -11.529f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("15N", 6.44f, 12.213f, 3.132f, 2.013f, 1.166f, 0.006f, 9.893f, 28.997f, 0.583f, -11.529f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("O", 5.803f, 3.049f, 2.287f, 1.546f, 0.867f, 13.277f, 5.701f, 0.324f, 32.909f, 0.251f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("16O", 5.803f, 3.049f, 2.287f, 1.546f, 0.867f, 13.277f, 5.701f, 0.324f, 32.909f, 0.251f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("17O", 5.78f, 3.049f, 2.287f, 1.546f, 0.867f, 13.277f, 5.701f, 0.324f, 32.909f, 0.251f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("18O", 5.84f, 3.049f, 2.287f, 1.546f, 0.867f, 13.277f, 5.701f, 0.324f, 32.909f, 0.251f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("F", 5.654f, 3.539f, 2.641f, 1.517f, 1.024f, 10.283f, 4.294f, 0.262f, 26.148f, 0.278f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("S", 2.847f, 6.905f, 5.203f, 1.438f, 1.586f, 1.468f, 22.215f, 0.254f, 56.172f, 0.867f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("32S", 2.804f, 6.905f, 5.203f, 1.438f, 1.586f, 1.468f, 22.215f, 0.254f, 56.172f, 0.867f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("33S", 4.74f, 6.905f, 5.203f, 1.438f, 1.586f, 1.468f, 22.215f, 0.254f, 56.172f, 0.867f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("34S", 3.48f, 6.905f, 5.203f, 1.438f, 1.586f, 1.468f, 22.215f, 0.254f, 56.172f, 0.867f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("36S", 3.1f, 6.905f, 5.203f, 1.438f, 1.586f, 1.468f, 22.215f, 0.254f, 56.172f, 0.867f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("Cl", 9.577f, 11.460f, 7.196f, 6.256f, 1.645f, 0.010f, 1.166f, 18.519f, 47.778f, -9.557f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("35Cl", 11.65f, 11.460f, 7.196f, 6.256f, 1.645f, 0.010f, 1.166f, 18.519f, 47.778f, -9.557f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
	try { isotope = new CIsotope("37Cl", 3.08f, 11.460f, 7.196f, 6.256f, 1.645f, 0.010f, 1.166f, 18.519f, 47.778f, -9.557f); } catch(...) { isotope = NULL; }
	if(isotope == NULL) NewException((double)sizeof(CIsotope), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	g_isotopes.Add(isotope);
}

static void deleteIsotopeList() {
	int i;
	for(i = 0; i < g_isotopes.GetSize(); i++)
		delete (CIsotope *)g_isotopes[i];
}

CIsotope::CIsotope(const char *label, float neutronFactor, float cma1, float cma2, float cma3, float cma4, float cmb1, float cmb2, float cmb3, float cmb4, float cmc) {
	try { _label = new char[strlen(label) + 1]; } catch(...) { _label = NULL; }
	if(_label == NULL) NewException((double)sizeof(char) * (strlen(label) + 1), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	strcpy(_label, label);
	_neutronFactor = neutronFactor;
	_cma[0] = cma1;
	_cma[1] = cma2;
	_cma[2] = cma3;
	_cma[3] = cma4;
	_cmb[0] = cmb1;
	_cmb[1] = cmb2;
	_cmb[2] = cmb3;
	_cmb[3] = cmb4;
	_cmc = cmc;
}

CIsotope::~CIsotope() {
	delete[] _label;
}

float CIsotope::xrayFactor(float q) {
	float x = q * 100.0f / 4.0f / Pi;
	float factor = 0.0f;
	int i;
	for(i = 0; i < 4; i++) {
		factor += _cma[i] * expf(-_cmb[i] * x * x);
	}
	factor += _cmc;
	return factor;
}

CSFacObservation::CSFacObservation(bool global) {
	int i, j, k, l;
	_global = global;
	if(_global) {
		mprintf("    All atoms will be taken.\n\n");
		for(i = 0; i < g_oaMolecules.GetSize(); i++) {
			CAtomGroup *ag;
			try { ag = new CAtomGroup(); } catch(...) { ag = NULL; }
			if(ag == NULL) NewException((double)sizeof(CAtomGroup), __FILE__, __LINE__, __PRETTY_FUNCTION__);
			if(!ag->ParseAtoms((CMolecule *)g_oaMolecules[i], "*")) {
				eprintf("Weird error.\n");
				abort();
			}
			_agList.Add(ag);
		}
	} else {
		while(true) {
			char buf[BUF_SIZE], buf2[BUF_SIZE];
			size_t remaining = BUF_SIZE;
#ifdef TARGET_LINUX
			remaining -= snprintf(buf, remaining, "    Take atoms from which molecule (");
#else
			remaining -= sprintf(buf, "    Take atoms from which molecule (");
#endif
			for(i = 0; i < g_oaMolecules.GetSize(); i++) {
				if(remaining < 1)
					break;
#ifdef TARGET_LINUX
				size_t length = snprintf(buf2, remaining, "%s=%d", ((CMolecule *)g_oaMolecules[i])->m_sName, i+1);
#else
				size_t length = sprintf(buf2, "%s=%d", ((CMolecule *)g_oaMolecules[i])->m_sName, i+1);
#endif
				strncat(buf, buf2, remaining - 1);
				remaining -= length;
				if(i < g_oaMolecules.GetSize() - 1) {
#ifdef TARGET_LINUX
					length = snprintf(buf2, remaining, ", ");
#else
					length = sprintf(buf2, ", ");
#endif
					strncat(buf, buf2, remaining - 1);
					remaining -= length;
				}
			}
			strncat(buf, ")? ", remaining - 1);
			
			int mol = AskRangeInteger_ND(buf, 1, g_oaMolecules.GetSize()) - 1;
			
			CAtomGroup *ag;
			try { ag = new CAtomGroup(); } catch(...) { ag = NULL; }
			if(ag == NULL) NewException((double)sizeof(CAtomGroup), __FILE__, __LINE__, __PRETTY_FUNCTION__);
			
			while(true) {
				if(((CMolecule *)g_oaMolecules[mol])->m_iAtomGes == 3) {
					mprintf("    %s is only one atom, there is no choice.\n", ((CMolecule *)g_oaMolecules[mol])->m_sName);
					ag->Reset();
					ag->m_pMolecule = (CMolecule *)g_oaMolecules[mol];
					ag->AddAtom(0, 0, false);
					ag->SortAtoms();
					ag->BuildName();
				} else {
					mprintf("    Which atom(s) to take from molecule %s (e.g. \"C1,C3-5,H\", \"*\"=all)? [#2] ", ((CMolecule*)g_oaMolecules[mol])->m_sName);
					inpprintf("! Which atom(s) to take from molecule %s (e.g. \"C1,C3-5,H\", \"*\"=all)? [#2]\n", ((CMolecule*)g_oaMolecules[mol])->m_sName);
					char buf[BUF_SIZE];
					myget(buf);
					if(strlen(buf) == 0) {
						if(!ag->ParseAtoms((CMolecule *)g_oaMolecules[mol], "#2")) {
							eprintf("Weird error.\n");
							continue;
						}
					} else if(!ag->ParseAtoms((CMolecule *)g_oaMolecules[mol], buf)) {
						continue;
					}
				}
				break;
			}
			_agList.Add(ag);
			
			if(!AskYesNo("    Add atoms from another molecule (y/n)? [no] ", false))
				break;
		}
		mprintf("\n");
	}
	
	if(_global) {
		try { _name = new char[strlen("global") + 1]; } catch(...) { _name = NULL; }
		if(_name == NULL) NewException((double)sizeof(char) * (strlen("global") + 1), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		sprintf(_name, "global");
	} else {
		char temp[BUF_SIZE];
		temp[0] = 0;
		for(i = 0; i < _agList.GetSize(); i++) {
			if(i > 0)
				strncat(temp, "_", BUF_SIZE - strlen(temp) - 1);
			strncat(temp, ((CAtomGroup *)_agList[i])->m_sName, BUF_SIZE - strlen(temp) - 1);
		}
		
		try { _name = new char[strlen(temp) + 3]; } catch(...) { _name = NULL; }
		if(_name == NULL) NewException((double)sizeof(char) * (strlen(temp) + 3), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		sprintf(_name, "[%s]", temp);
	}
	
	CxObArray isotopeAssignList;
	for(i = 0; i < _agList.GetSize(); i++) {
		CAtomGroup *ag = (CAtomGroup *)_agList[i];
		for(j = 0; j < ag->m_baAtomType.GetSize(); j++) {
			CIsotope *isotope = 0;
			for(k = 0; k < g_isotopes.GetSize(); k++) {
				if(mystricmp(((CAtom *)g_oaAtoms[ag->m_baRealAtomType[j]])->m_sName, ((CIsotope *)g_isotopes[k])->label()) == 0) {
					isotope = (CIsotope *)g_isotopes[k];
					break;
				}
			}
			if(isotope == 0) {
				mprintf(RED, "Isotope data for \"%s\" not found.\n", ((CAtom *)g_oaAtoms[ag->m_baRealAtomType[j]])->m_sName);
				abort();
			}
			for(k = 0; k < ((CxIntArray *)ag->m_oaAtoms[j])->GetSize(); k++) {
				isotopeAssignList.Add(isotope);
			}
		}
	}
	
	if(!AskYesNo("    Use standard atom data (y) or specify isotopes (n)? [yes] ", true)) {
		while(true) {
			char buf[BUF_SIZE], buf2[BUF_SIZE];
			size_t remaining = BUF_SIZE;
#ifdef TARGET_LINUX
			remaining -= snprintf(buf, remaining, "\n    Change isotopes in which molecule (");
#else
			remaining -= sprintf(buf, "\n    Change isotopes in which molecule (");
#endif
			for(i = 0; i < _agList.GetSize(); i++) {
				if(remaining < 1)
					break;
#ifdef TARGET_LINUX
				size_t length = snprintf(buf2, remaining, "%s=%d", ((CMolecule *)((CAtomGroup *)_agList[i])->m_pMolecule)->m_sName, i+1);
#else
				size_t length = sprintf(buf2, "%s=%d", ((CMolecule *)((CAtomGroup *)_agList[i])->m_pMolecule)->m_sName, i+1);
#endif
				strncat(buf, buf2, remaining - 1);
				remaining -= length;
				if(i < _agList.GetSize() - 1) {
#ifdef TARGET_LINUX
					length = snprintf(buf2, remaining, ", ");
#else
					length = sprintf(buf2, ", ");
#endif
					strncat(buf, buf2, remaining - 1);
					remaining -= length;
				}
			}
			strncat(buf, ")? ", remaining - 1);
			
			int mol = AskRangeInteger_ND(buf, 1, _agList.GetSize()) - 1;
			
			int index = 0;
			for(i = 0; i < mol; i++) {
				CAtomGroup *ag = (CAtomGroup *)_agList[i];
				for(j = 0; j < ag->m_baAtomType.GetSize(); j++) {
					index += ((CxIntArray *)ag->m_oaAtoms[j])->GetSize();
				}
			}
			
			while(true) {
				mprintf("\n    The following isotopes are set up:\n");
				CAtomGroup *ag = (CAtomGroup *)_agList[mol];
				int index1 = index;
				for(i = 0; i < ag->m_baAtomType.GetSize(); i++) {
					CxIntArray *a = (CxIntArray *)ag->m_oaAtoms[i];
					for(j = 0; j < a->GetSize(); j++) {
						mprintf("      %s%d: %-4s", ((CAtom *)g_oaAtoms[ag->m_baRealAtomType[i]])->m_sName, a->GetAt(j)+1, ((CIsotope *)isotopeAssignList[index1++])->label());
						if((index1 - index) % 5 == 0)
							mprintf("\n");
					}
				}
				mprintf("\n\n");
				char buf[BUF_SIZE];
				unsigned char ca, cb, cc;
				int index2;
				while(true) {
					do {
						mprintf("    Change isotope for which atom? [done] ");
						inpprintf("! Change isotope for which atom? [done] ");
						myget(buf);
						if(strlen(buf) == 0)
							break;
					} while(!ParseAtom(buf, ((CMolecule *)ag->m_pMolecule)->m_iIndex, ca, cb, cc));
					index2 = index;
					bool contains = false;
					for(i = 0; i < ag->m_baAtomType.GetSize(); i++) {
						if(cb == ag->m_baRealAtomType[i]) {
							CxIntArray *a = (CxIntArray *)ag->m_oaAtoms[i];
							for(j = 0; j < a->GetSize(); j++) {
								if(cc == a->GetAt(j)) {
									contains = true;
									break;
								}
								index2++;
							}
							break;
						} else {
							index2 += ((CxIntArray *)ag->m_oaAtoms[i])->GetSize();
						}
					}
					if(contains)
						break;
					eprintf("There is no atom \"%s\" in this observation.\n", buf);
				}
				if(strlen(buf) == 0)
					break;
				char buf2[BUF_SIZE];
				while(true) {
					mprintf("    Which isotope to set for %s (e.g. \"13C\")? ", buf);
					inpprintf("! Which isotope to set for %s (e.g. \"13C\")? ", buf);
					myget(buf2);
					if(strlen(buf2) == 0) {
						eprintf("There is no default. Please enter a character string.\n");
						continue;
					}
					int isotopeIndex = -1;
					for(i = 0; i < g_isotopes.GetSize(); i++) {
						if(mystricmp(((CIsotope *)g_isotopes[i])->label(), buf2) == 0) {
							isotopeIndex = i;
							break;
						}
					}
					if(isotopeIndex != -1) {
						isotopeAssignList[index2] = (CIsotope *)g_isotopes[isotopeIndex];
						break;
					}
					eprintf("Isotope data for \"%s\" not found.\n", buf2);
				}
			}
			
			if(!AskYesNo("\n    Change isotopes in another molecule (y/n)? [no] ", false))
				break;
		}
		mprintf("\n");
	}
	
	int index = 0;
	for(i = 0; i < _agList.GetSize(); i++) {
		CAtomGroup *ag = (CAtomGroup *)_agList[i];
		CMolecule *mol = ag->m_pMolecule;
		for(j = 0; j < ag->m_baAtomType.GetSize(); j++) {
			CxIntArray *a = (CxIntArray *)ag->m_oaAtoms[j];
			for(k = 0; k < a->GetSize(); k++) {
				CIsotope *isotope = (CIsotope *)isotopeAssignList[index++];
				int isotopeIndex = -1;
				for(l = 0; l < _isotopeTypeList.GetSize(); l++)
					if((CIsotope *)_isotopeTypeList[l] == isotope)
						isotopeIndex = l;
				if(isotopeIndex == -1) {
					isotopeIndex = _isotopeTypeList.GetSize();
					_isotopeTypeList.Add(isotope);
					_isotopeTypeCount.Add(0);
				}
				for(l = 0; l < mol->m_laSingleMolIndex.GetSize(); l++) {
					CSingleMolecule *sm = (CSingleMolecule *)g_oaSingleMolecules[mol->m_laSingleMolIndex[l]];
					_atomIndex.Add(((CxIntArray *)sm->m_oaAtomOffset[ag->m_baAtomType[j]])->GetAt(a->GetAt(k)));
					_isotopeList.Add(isotopeIndex);
					_isotopeTypeCount[isotopeIndex]++;
				}
			}
		}
	}
	
	_rdfMax = AskFloat("    Enter maximum RDF distance to observe (pm): [%d] ", (float)HalfBox(), HalfBox());
	_rdfRes = AskUnsignedInteger("    Enter RDF binning resolution: [%d] ", 2 * HalfBox(), 2 * HalfBox());
	_sfacMax = AskFloat("    Enter maximum wave vector modulus (pm^-1): [0.2] ", 0.2f);
	_sfacRes = AskUnsignedInteger("    Enter Structure Factor binning resolution: [1000] ", 1000);
	_normFFac = AskYesNo("    Normalize by scattering factors (y/n)? [yes] ", true);
	mprintf("\n");
}

CSFacObservation::~CSFacObservation() {
	int i;
	for(i = 0; i < _agList.GetSize(); i++)
		delete (CAtomGroup *)_agList[i];
	for(i = 0; i < _rdfList.GetSize(); i++)
		delete (CDF *)_rdfList[i];
	delete[] _name;
}

void CSFacObservation::initialize() {
	int i;
	for(i = 0; i < _isotopeTypeList.GetSize() * (_isotopeTypeList.GetSize() + 1) / 2; i++) {
		CDF *df;
		try { df = new CDF(); } catch(...) { df = NULL; }
		if(df == NULL) NewException((double)sizeof(CDF), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		df->m_fMinVal = 0.0f;
		df->m_fMaxVal = _rdfMax;
		df->m_iResolution = _rdfRes;
		df->Create();
		_rdfList.Add(df);
	}
}

void CSFacObservation::process(CTimeStep *ts) {
	int i, j;
	for(i = 0; i < _atomIndex.GetSize(); i++) {
		for(j = i+1; j < _atomIndex.GetSize(); j++) {
			float dist = FoldedLength(ts->m_vaCoords[_atomIndex[i]] - ts->m_vaCoords[_atomIndex[j]]);
			int a = _isotopeList[i];
			int b = _isotopeList[j];
			if(a < b)
				((CDF *)_rdfList[(_isotopeTypeList.GetSize()-1)*a - a*(a-1)/2 + b])->AddToBin(dist);
			else
				((CDF *)_rdfList[(_isotopeTypeList.GetSize()-1)*b - b*(b-1)/2 + a])->AddToBin(dist);
		}
	}
}

void CSFacObservation::finalize() {
	int i, j, k, l;
	CDF *rdfTotal;
	try { rdfTotal = new CDF(); } catch(...) { rdfTotal = NULL; }
	if(rdfTotal == NULL) NewException((double)sizeof(CDF), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	rdfTotal->m_fMinVal = 0.0f;
	rdfTotal->m_fMaxVal = _rdfMax;
	rdfTotal->m_iResolution = _rdfRes;
	rdfTotal->Create();
	
	CDF *sfacTotal;
	try { sfacTotal = new CDF(); } catch(...) { sfacTotal = NULL; }
	if(sfacTotal == NULL) NewException((double)sizeof(CDF), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	sfacTotal->m_fMinVal = 0.0f;
	sfacTotal->m_fMaxVal = _sfacMax;
	sfacTotal->m_iResolution = _sfacRes;
	sfacTotal->Create();
	
	CDF *neutronSFac;
	try { neutronSFac = new CDF(); } catch(...) { neutronSFac = NULL; }
	if(neutronSFac == NULL) NewException((double)sizeof(CDF), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	neutronSFac->m_fMinVal = 0.0f;
	neutronSFac->m_fMaxVal = _sfacMax;
	neutronSFac->m_iResolution = _sfacRes;
	neutronSFac->Create();
	
	CDF *xraySFac;
	try { xraySFac = new CDF(); } catch(...) { xraySFac = NULL; }
	if(xraySFac == NULL) NewException((double)sizeof(CDF), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	xraySFac->m_fMinVal = 0.0f;
	xraySFac->m_fMaxVal = _sfacMax;
	xraySFac->m_iResolution = _sfacRes;
	xraySFac->Create();
	
	for(i = 0; i < _isotopeTypeList.GetSize(); i++) {
		for(j = i; j < _isotopeTypeList.GetSize(); j++) {
			mprintf("    Processing partial structure factor %s-%s\n", ((CIsotope *)_isotopeTypeList[i])->label(), ((CIsotope *)_isotopeTypeList[j])->label());
			CDF *df = (CDF *)_rdfList[(_isotopeTypeList.GetSize()-1)*i - i*(i-1)/2 + j];
			df->MultiplyBin(1.0 / g_iSteps);
			if(i == j)
				df->MultiplyBin(2.0);
			df->CorrectRadialDist();
			df->MultiplyBin(g_fBoxX * g_fBoxY * g_fBoxZ / (4.0f/3.0f * Pi) / _isotopeTypeCount[i] / _isotopeTypeCount[j]);
			if (g_bDoubleBox)
				df->MultiplyBin(g_iDoubleBoxFactor);
			
			char name[BUF_SIZE];
#ifdef TARGET_LINUX
			snprintf(name, BUF_SIZE, "sfac_%s_rdf_%s_%s.csv", _name, ((CIsotope *)_isotopeTypeList[i])->label(), ((CIsotope *)_isotopeTypeList[j])->label());
#else
			sprintf(name, "sfac_%s_rdf_%s_%s.csv", _name, ((CIsotope *)_isotopeTypeList[i])->label(), ((CIsotope *)_isotopeTypeList[j])->label());
#endif
			mprintf("    Writing RDF to %s...\n", name);
			df->Write("", name, "", false);
			
			for(k = 0; k < _rdfRes; k++)
				rdfTotal->m_pBin[k] += df->m_pBin[k] * ((i == j) ? 1.0f : 2.0f) * _isotopeTypeCount[i] * _isotopeTypeCount[j] / _atomIndex.GetSize() / _atomIndex.GetSize();
			
			CDF *sfac;
			try { sfac = new CDF(); } catch(...) { sfac = NULL; }
			if(sfac == NULL) NewException((double)sizeof(CDF), __FILE__, __LINE__, __PRETTY_FUNCTION__);
			sfac->m_fMinVal = 0.0f;
			sfac->m_fMaxVal = _sfacMax;
			sfac->m_iResolution = _sfacRes;
			sfac->Create();
			
			for(k = 0; k < _sfacRes; k++) {
				float f = 0;
				for(l = 0; l < _rdfRes; l++) {
					f += ((0.5f + l) / _rdfRes * _rdfMax) * (df->m_pBin[l] - 1.0f) / ((0.5f + k) / _sfacRes * _sfacMax) * sinf(((0.5f + l) / _rdfRes * _rdfMax) * ((0.5f + k) / _sfacRes * _sfacMax));
				}
				sfac->m_pBin[k] = f * 4.0f * Pi * g_iGesAtomCount / g_fBoxX / g_fBoxY / g_fBoxZ * _rdfMax / _rdfRes;
			}
			
#ifdef TARGET_LINUX
			snprintf(name, BUF_SIZE, "sfac_%s_sfac_%s_%s.csv", _name, ((CIsotope *)_isotopeTypeList[i])->label(), ((CIsotope *)_isotopeTypeList[j])->label());
#else
			sprintf(name, "sfac_%s_sfac_%s_%s.csv", _name, ((CIsotope *)_isotopeTypeList[i])->label(), ((CIsotope *)_isotopeTypeList[j])->label());
#endif
			mprintf("    Writing Structure Factor to %s...\n", name);
			sfac->Write("", name, "", false);
			
			for(k = 0; k < _sfacRes; k++) {
				sfacTotal->m_pBin[k] += sfac->m_pBin[k] * ((i == j) ? 1.0f : 2.0f) * _isotopeTypeCount[i] * _isotopeTypeCount[j] / _atomIndex.GetSize() / _atomIndex.GetSize();
				neutronSFac->m_pBin[k] += sfac->m_pBin[k] * ((i == j) ? 1.0f : 2.0f) * _isotopeTypeCount[i] * _isotopeTypeCount[j] / _atomIndex.GetSize() / _atomIndex.GetSize() * ((CIsotope *)_isotopeTypeList[i])->neutronFactor() * ((CIsotope *)_isotopeTypeList[j])->neutronFactor();
				xraySFac->m_pBin[k] += sfac->m_pBin[k] * ((i == j) ? 1.0f : 2.0f) * _isotopeTypeCount[i] * _isotopeTypeCount[j] / _atomIndex.GetSize() / _atomIndex.GetSize() * ((CIsotope *)_isotopeTypeList[i])->xrayFactor((0.5f + k) / _sfacRes * _sfacMax) * ((CIsotope *)_isotopeTypeList[j])->xrayFactor((0.5f + k) / _sfacRes * _sfacMax);
			}
			
			delete sfac;
		}
	}
	
	if(_normFFac) {
		float neutronFactor = 0.0f;
		for(i = 0; i < _isotopeTypeList.GetSize(); i++) {
			neutronFactor += (float)_isotopeTypeCount[i] / _atomIndex.GetSize() * ((CIsotope *)_isotopeTypeList[i])->neutronFactor();
		}
		neutronSFac->MultiplyBin(1.0f / neutronFactor / neutronFactor);
		for(i = 0; i < _sfacRes; i++) {
			float xrayFactor = 0.0f;
			for(j = 0; j < _isotopeTypeList.GetSize(); j++) {
				xrayFactor += (float)_isotopeTypeCount[j] / _atomIndex.GetSize() * ((CIsotope *)_isotopeTypeList[j])->xrayFactor((0.5f + i) / _sfacRes * _sfacMax);
			}
			xraySFac->m_pBin[i] /= xrayFactor * xrayFactor;
		}
	}
	
	char name[BUF_SIZE];
#ifdef TARGET_LINUX
	snprintf(name, BUF_SIZE, "sfac_%s_rdf_total.csv", _name);
#else
	sprintf(name, "sfac_%s_rdf_total.csv", _name);
#endif
	mprintf("    Writing total RDF to %s...\n", name);
	rdfTotal->Write("", name, "", false);
	
#ifdef TARGET_LINUX
	snprintf(name, BUF_SIZE, "sfac_%s_sfac_total.csv", _name);
#else
	sprintf(name, "sfac_%s_sfac_total.csv", _name);
#endif
	mprintf("    Writing total Structure Factor to %s...\n", name);
	sfacTotal->Write("", name, "", false);
	
#ifdef TARGET_LINUX
	snprintf(name, BUF_SIZE, "sfac_%s_neutron.csv", _name);
#else
	sprintf(name, "sfac_%s_neutron.csv", _name);
#endif
	mprintf("    Writing Neutron Scattering Function to %s...\n", name);
	neutronSFac->Write("", name, "", false);
	
#ifdef TARGET_LINUX
	snprintf(name, BUF_SIZE, "sfac_%s_xray.csv", _name);
#else
	sprintf(name, "sfac_%s_xray.csv", _name);
#endif
	mprintf("    Writing X-Ray Scattering Function to %s...\n", name);
	xraySFac->Write("", name, "", false);
	
	delete rdfTotal;
	delete sfacTotal;
}

bool gatherStructureFactor() {
	createIsotopeList();
	
	if(AskYesNo("    Compute structure factor of whole system (y/n)? [yes] ", true)) {
		mprintf(YELLOW, "\n>>> Structure Factor Observation 1 >>>\n\n");
		
		CSFacObservation *obs;
		try { obs = new CSFacObservation(true); } catch(...) { obs = NULL; }
		if(obs == NULL) NewException((double)sizeof(CSFacObservation), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		g_sFacObserv.Add(obs);
		
		mprintf(YELLOW, "<<< End of Structure Factor Observation 1 <<<\n\n");
	}
	
	if(AskYesNo("    Compute structure factors for specific atoms (y/n)? [no] ", false)) {
		while(true) {
			mprintf(YELLOW, "\n>>> Structure Factor Observation %d >>>\n\n", g_sFacObserv.GetSize() + 1);
			
			CSFacObservation *obs;
			try { obs = new CSFacObservation(); } catch(...) { obs = NULL; }
			if(obs == NULL) NewException((double)sizeof(CSFacObservation), __FILE__, __LINE__, __PRETTY_FUNCTION__);
			g_sFacObserv.Add(obs);
			
			mprintf(YELLOW, "<<< End of Structure Factor Observation %d <<<\n\n", g_sFacObserv.GetSize());
			
			if(!AskYesNo("    Add another observation (y/n)? [no] ", false))
				break;
			mprintf("\n");
		}
	}
	mprintf("\n");
	
	return true;
}

bool initializeStructureFactor() {
	int i;
	for(i = 0; i < g_sFacObserv.GetSize(); i++) {
		((CSFacObservation *)g_sFacObserv[i])->initialize();
	}
	return true;
}

void processStructureFactor(CTimeStep *ts) {
	int i;
	for(i = 0; i < g_sFacObserv.GetSize(); i++) {
		((CSFacObservation *)g_sFacObserv[i])->process(ts);
	}
}

void finalizeStructureFactor() {
	int i;
	for(i = 0; i < g_sFacObserv.GetSize(); i++) {
		mprintf(YELLOW, ">>> Structure Factor Observation %d >>>\n\n", i + 1);
		((CSFacObservation *)g_sFacObserv[i])->finalize();
		mprintf(YELLOW, "\n<<< End of Structure Factor Observation %d <<<\n\n", i + 1);
	}
	deleteIsotopeList();
}

// CStructureFactor::CStructureFactor()
// {
// }
// 
// 
// CStructureFactor::~CStructureFactor()
// {
// }
// 
// 
// void CStructureFactor::Create()
// {
// 	int z, z2;
// 	CDF *df;
// 
// 	for (z=0;z<g_oaAtoms.GetSize();z++)
// 	{
// 		if (z == g_iVirtAtomType)
// 			continue;
// 
// 		if (((CAtom*)g_oaAtoms[z])->m_pElement->m_fCoherentNCS == 0)
// 			eprintf("\nError: Coherent Neutron Cross Section for %s not defined. Edit elementdata.cpp.\n",((CAtom*)g_oaAtoms[z])->m_sName);
// 
// 		for (z2=0;z2<g_oaAtoms.GetSize();z2++)
// 		{
// 			if (z2 == g_iVirtAtomType)
// 				continue;
// 
// 			if (z2 < z)
// 			{
// 				m_oaRDFs.Add(NULL);
// 				m_oaSFacs.Add(NULL);
// 			} else
// 			{
// 				df = new CDF();
// 				df->m_fMinVal = 0;
// 				df->m_fMaxVal = m_fRDFRange;
// 				df->m_iResolution = m_iRDFRes;
// 				df->SetLabelX("Distance [pm]");
// 				df->SetLabelY("g(r)");
// 				df->Create();
// 				m_oaRDFs.Add(df);
// 
// 				df = new CDF();
// 				df->m_fMinVal = 0;
// 				df->m_fMaxVal = m_fSQRange;
// 				df->m_iResolution = m_iSQRes;
// 				df->SetLabelX("Wave Vector Modulus [Angstrom^-1]");
// 				df->SetLabelY("S(k)");
// 				df->Create();
// 				m_oaSFacs.Add(df);
// 			}
// 		}
// 	}
// }
// 
// 
// void CStructureFactor::Parse()
// {
// 	mprintf(WHITE,">>> Structure Factor Analysis >>>\n\n");
// 
// 	m_fRDFRange = AskFloat("    Enter maximum RDF distance to observe (in pm): [%d] ",(float)HalfBox(),HalfBox());
// 	m_iRDFRes = AskUnsignedInteger("    Enter RDF binning resolution: [300] ",300);
// 	m_fSQRange = AskFloat("    Enter maximum wave vector modulus (in Angstrom^-1): [50] ",50);
// 	m_iSQRes = AskUnsignedInteger("    Enter Structure Factor binning resolution: [2000] ",2000);
// 
// 	mprintf("\n");
// 
// 	m_bDumpElementRDFs = AskYesNo("    Write out all element RDFs (y/n)? [yes] ",true);
// 	m_bDumpTotalRDF = AskYesNo("    Write out overall RDF (y/n)? [yes] ",true);
// 	m_bDumpElementSFac = AskYesNo("    Write out all element Structure Factor contributions (y/n)? [yes] ",true);
// 
// 	mprintf(WHITE,"\n<<< End of Structure Factor Analysis <<<\n\n");
// }
// 
// 
// void CStructureFactor::ProcessStep(CTimeStep *ts)
// {
// 	int z, z2, t1, t2, i;
// 	float v;
// 
// 	i = g_oaAtoms.GetSize()-1;
// 
// 	for (z=0;z<g_iGesAtomCount;z++)
// 	{
// 		t1 = g_waAtomRealElement[z];
// 		for (z2=z+1;z2<g_iGesAtomCount;z2++)
// 		{
// 			t2 = g_waAtomRealElement[z2];
// 
// 			v = FoldedLength(ts->m_vaCoords[z] - ts->m_vaCoords[z2]);
// 
// 			if (t1 >= t2)
// 			{
// 				((CDF*)m_oaRDFs[t2*i+t1])->AddToBin(v);
// 			} else
// 			{
// 				((CDF*)m_oaRDFs[t1*i+t2])->AddToBin(v);
// 			}
// 		}
// 	}
// }
// 
// 
// void CStructureFactor::Finish()
// {
// 	int z, z2, z3;
// 	CDF *df, *dftotal, *dfstotal;
// 	double fac/*, tf*/;
// 	char buf[256];
// 
// 	dftotal = NULL;
// 
// 	mprintf(WHITE,"* Structure Factor\n");
// 
// 	if (m_bDumpTotalRDF)
// 	{
// 		dftotal = new CDF();
// 		dftotal->m_fMinVal = 0;
// 		dftotal->m_fMaxVal = m_fRDFRange;
// 		dftotal->m_iResolution = m_iRDFRes;
// 		dftotal->SetLabelX("Distance [pm]");
// 		dftotal->SetLabelY("g(r)");
// 		dftotal->Create();
// //		tf = 0;
// 	}
// 
// 	dfstotal = new CDF();
// 	dfstotal->m_fMinVal = 0;
// 	dfstotal->m_fMaxVal = m_fSQRange;
// 	dfstotal->m_iResolution = m_iSQRes;
// 	dfstotal->SetLabelX("Wave Vector Modulus [Angstrom^-1]");
// 	dfstotal->SetLabelY("S(k)");
// 	dfstotal->Create();
// 
// 	for (z=0;z<g_oaAtoms.GetSize();z++)
// 	{
// 		if (z == g_iVirtAtomType)
// 			continue;
// 
// 		for (z2=0;z2<g_oaAtoms.GetSize();z2++)
// 		{
// 			if (z2 == g_iVirtAtomType)
// 				continue;
// 
// 			if (z2 < z)
// 				continue;
// 
// 			df = (CDF*)m_oaRDFs[z*(g_oaAtoms.GetSize()-1)+z2];
// 
// 			mprintf(WHITE,"  Processing Contributions of type %s - %s\n",((CAtom*)g_oaAtoms[z])->m_sName,((CAtom*)g_oaAtoms[z2])->m_sName);
// 
// 			df->MultiplyBin(1.0 / g_iSteps);
// 			if (z == z2)
// 				df->MultiplyBin(2.0);
// 			df->CorrectRadialDist();
// 			df->MultiplyBin(g_fBoxX*g_fBoxY*g_fBoxZ / (4.0/3.0*Pi) / ((CAtom*)g_oaAtoms[z])->m_iCount / ((CAtom*)g_oaAtoms[z2])->m_iCount);
// 
// 			if (g_bDoubleBox)
// 				df->MultiplyBin(g_iDoubleBoxFactor);
// 
// 			if (m_bDumpElementRDFs)
// 			{
// 				sprintf(buf,"sfac_rdf_%s_%s.csv",((CAtom*)g_oaAtoms[z])->m_sName,((CAtom*)g_oaAtoms[z2])->m_sName);
// 				mprintf("    Writing RDF to %s ...\n",buf);
// 				df->Write("",buf,"",false);
// 			}
// 
// 			if (m_bDumpTotalRDF)
// 			{
// 				if (z == z2)
// 					fac = 1.0;
// 						else fac = 2.0;
// 				for (z3=0;z3<m_iRDFRes;z3++)
// 					dftotal->m_pBin[z3] += df->m_pBin[z3] * fac * (double)((CAtom*)g_oaAtoms[z])->m_iCount * ((CAtom*)g_oaAtoms[z2])->m_iCount / g_iGesAtomCount / g_iGesAtomCount;
// 	//			tf += fac * (double)((CAtom*)g_oaAtoms[z])->m_iCount * ((CAtom*)g_oaAtoms[z2])->m_iCount / g_iGesAtomCount / g_iGesAtomCount;
// 			}
// 
// 			TransformRDF(df,(CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2]);
// 
// 			if (z == z2)
// 				fac = 1.0;
// 					else fac = 2.0;
// 			for (z3=0;z3<m_iSQRes;z3++)
// 				dfstotal->m_pBin[z3] += ((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[z3] * fac * (double)((CAtom*)g_oaAtoms[z])->m_pElement->m_fCoherentNCS * ((CAtom*)g_oaAtoms[z2])->m_pElement->m_fCoherentNCS * ((CAtom*)g_oaAtoms[z])->m_iCount * ((CAtom*)g_oaAtoms[z2])->m_iCount / g_iGesAtomCount / g_iGesAtomCount;
// 
// 			if (m_bDumpElementSFac)
// 			{
// 				sprintf(buf,"sfac_%s_%s.csv",((CAtom*)g_oaAtoms[z])->m_sName,((CAtom*)g_oaAtoms[z2])->m_sName);
// 				mprintf("    Writing Structure Factor Contribution to %s ...\n",buf);
// 			//	mprintf("    Correction Factor: %G\n",((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[0]);
// 			/*	tf = 0;
// 				for (z3=0;z3<m_iSQRes;z3++)
// 				{
// 					if (((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[z3] < tf)
// 						tf = ((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[z3];
// 				}
// 				if (tf < 0)
// 				{
// 					for (z3=0;z3<m_iSQRes;z3++)
// 						((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[z3] /= fabs(tf);
// 				}
// 				for (z3=0;z3<m_iSQRes;z3++)
// 					((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[z3] += 1.0;*/
// 				((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->Write("",buf,"",false);
// 			}
// 		}
// 	}
// 
// 	if (m_bDumpTotalRDF)
// 	{
// 		mprintf(WHITE,"  * Overall RDF\n");
// 	//	mprintf("    tf = %f.\n",tf);
// 		sprintf(buf,"sfac_rdf_total.csv");
// 		mprintf("    Writing RDF to %s ...\n",buf);
// 		dftotal->Write("",buf,"",false);
// 	}
// 
// 	mprintf(WHITE,"  * Overall S(k)\n");
// 	sprintf(buf,"sfac_total.csv");
// 	mprintf("    Writing S(k) to %s ...\n",buf);
// /*	tf = 0;
// 	for (z3=0;z3<m_iSQRes;z3++)
// 	{
// 		if (dfstotal->m_pBin[z3] < tf)
// 			tf = dfstotal->m_pBin[z3];
// 	}
// 	if (tf < 0)
// 	{
// 		for (z3=0;z3<m_iSQRes;z3++)
// 			dfstotal->m_pBin[z3] /= fabs(tf);
// 	}
// 	for (z3=0;z3<m_iSQRes;z3++)
// 		dfstotal->m_pBin[z3] += 1.0;*/
// 	dfstotal->Write("",buf,"",false);
// }
// 
// 
// void CStructureFactor::TransformRDF(CDF *pin, CDF *pout)
// {
// 	int z, z2;
// 	double tf, tf2;
// 
// 	for (z=0;z<m_iSQRes;z++)
// 	{
// 		tf = 0;
// 
// 		for (z2=0;z2<m_iRDFRes;z2++)
// 		{
// 			tf2 = ((z2+0.5)/m_iRDFRes*m_fRDFRange/100.0) * (pin->m_pBin[z2]-1.0) / ((z+0.5)/m_iSQRes*m_fSQRange) * sin(((z2+0.5)/m_iRDFRes*m_fRDFRange/100.0) * ((z+0.5)/m_iSQRes*m_fSQRange));
// 			tf += tf2;
// 	//		mprintf("\n  %.4fA * %.4f / %.4fA^-1 * sin( %.4fA^-1 * %.4fA ) = %.4f",((z2+0.5)/m_iRDFRes*m_fRDFRange/100.0),(pin->m_pBin[z2]-1.0),((z+0.5)/m_iSQRes*m_fRDFRange/100.0),((z2+0.5)/m_iRDFRes*m_fRDFRange/100.0),((z+0.5)/m_iSQRes*m_fRDFRange/100.0),tf2);
// 		}
// 
// 		pout->m_pBin[z] = tf*4.0*Pi/g_fBoxX/g_fBoxY/g_fBoxZ*1000000.0;
// 	//	mprintf("\n\n## 1 + 4*Pi*%.4f*%.4f = %.4f\n",tf,(1.0/m_iRDFRes*m_fRDFRange/10000.0),pout->m_pBin[z]);
// 	}
// }
