/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2013 Martin Brehm
                  2012-2013 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/


#include "sdengine.h"
#include "lmwrapper.h"
#include "tools.h"


CSDEngine::CSDEngine()
{
	m_oaGroups.SetName("CSDEngine::m_oaGroups");
	m_iaBlockSizes.SetName("CSDEngine::m_iaBlockSizes");
	m_faTempBuffer.SetName("CSDEngine::m_faTempBuffer");
	m_faSum.SetName("CSDEngine::m_faSum");
	m_faSqSum.SetName("CSDEngine::m_faSqSum");
	m_faMean.SetName("CSDEngine::m_faMean");
	m_faVari.SetName("CSDEngine::m_faVari");
	m_faCorLen.SetName("CSDEngine::m_faCorLen");
}


CSDEngine::~CSDEngine()
{
}


CSDGroup::CSDGroup()
{
	m_oaBlockSets.SetName("CSDGroup::m_oaBlockSets");
}


CSDGroup::~CSDGroup()
{
}


CSDBlockSet::CSDBlockSet()
{
	m_iCounter = 0;
	m_faBlocks.SetName("CSDBlockSet::m_faBlocks");
}


CSDBlockSet::~CSDBlockSet()
{
}


void CSDEngine::Init(int bins)
{
	int z, z2;
	CSDGroup *gr;
	CSDBlockSet *bs;

	m_iSteps = 0;

	m_faTempBuffer.SetSize(bins);
	m_faSum.SetSize(bins);
	m_faSqSum.SetSize(bins);
	m_faMean.SetSize(bins);
	m_faVari.SetSize(bins);
	m_faCorLen.SetSize(bins);

	for (z=0;z<bins;z++)
	{
		m_faTempBuffer[z] = 0;
		m_faSum[z] = 0;
		m_faSqSum[z] = 0;
		m_faMean[z] = 0;
		m_faVari[z] = 0;
		m_faCorLen[z] = 0;

		gr = new CSDGroup();
		m_oaGroups.Add(gr);
		gr->m_oaBlockSets.SetSize(m_iaBlockSizes.GetSize());
		gr->m_faBlockVari.SetSize(m_iaBlockSizes.GetSize());
		for (z2=0;z2<m_iaBlockSizes.GetSize();z2++)
		{
			bs = new CSDBlockSet();
			gr->m_oaBlockSets[z2] = bs;
			gr->m_faBlockVari[z2] = 0;

			bs->m_faBlocks.Add(0);
		}
	}
}


void CSDEngine::FinishStep()
{
	int z, z2;
	CSDGroup *gr;
	CSDBlockSet *bs;

	m_iSteps++;

	for (z=0;z<m_oaGroups.GetSize();z++)
	{
		gr = (CSDGroup*)m_oaGroups[z];

		m_faSum[z] += m_faTempBuffer[z];
		m_faSqSum[z] += m_faTempBuffer[z]*m_faTempBuffer[z];

		for (z2=0;z2<m_iaBlockSizes.GetSize();z2++)
		{
			bs = (CSDBlockSet*)gr->m_oaBlockSets[z2];
			bs->m_faBlocks[bs->m_faBlocks.GetSize()-1] += m_faTempBuffer[z];

			bs->m_iCounter++;
			if (bs->m_iCounter >= m_iaBlockSizes[z2])
			{
				bs->m_iCounter = 0;
				bs->m_faBlocks[bs->m_faBlocks.GetSize()-1] /= m_iaBlockSizes[z2];
				bs->m_faBlocks.Add(0);
			}
		}

		m_faTempBuffer[z] = 0;
	}
}


void CSDEngine::DumpData(CDF *df, float timessigma, bool verbose)
{
	int z, z2, ti;
	CSDGroup *gr;
//	FILE *a;
	FILE *b;
	char buf[256];
	double *x;
	double *y, *y2, ya;
	double da, ds, dr, tf, dsa, dsmi, dsma;
	CLMWrapper *lm;
	double par[3], *curve;

//	a = fopen("block.csv","wt");
	b = NULL;

	mprintf("    Calculating correlation time of histogram...\n");

	try { df->m_pAdditionalSets = new double*[4]; } catch(...) { df->m_pAdditionalSets = NULL; }
	if (df->m_pAdditionalSets == NULL) NewException((double)4*sizeof(double*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { df->m_pAdditionalSetLabels = new char*[4]; } catch(...) { df->m_pAdditionalSetLabels = NULL; }
	if (df->m_pAdditionalSetLabels == NULL) NewException((double)4*sizeof(char*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	df->m_iAdditionalSets = 4;

	for (z=0;z<4;z++)
	{
		try { df->m_pAdditionalSets[z] = new double[df->m_iResolution]; } catch(...) { df->m_pAdditionalSets[z] = NULL; }
		if (df->m_pAdditionalSets[z] == NULL) NewException((double)df->m_iResolution*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	}

	sprintf(buf,"%.2f%c confidence range",NormalDistIntegral(timessigma)*2.0-1.0,'%');
	try { df->m_pAdditionalSetLabels[0] = new char[strlen(buf)+1]; } catch(...) { df->m_pAdditionalSetLabels[0] = NULL; }
	if (df->m_pAdditionalSetLabels[0] == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	strcpy(df->m_pAdditionalSetLabels[0],buf);

	sprintf(buf,"%.2f%c confidence range",NormalDistIntegral(timessigma)*2.0-1.0,'%');
	try { df->m_pAdditionalSetLabels[1] = new char[strlen(buf)+1]; } catch(...) { df->m_pAdditionalSetLabels[1] = NULL; }
	if (df->m_pAdditionalSetLabels[1] == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	strcpy(df->m_pAdditionalSetLabels[1],buf);

	sprintf(buf,"Standard deviation");
	try { df->m_pAdditionalSetLabels[2] = new char[strlen(buf)+1]; } catch(...) { df->m_pAdditionalSetLabels[2] = NULL; }
	if (df->m_pAdditionalSetLabels[2] == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	strcpy(df->m_pAdditionalSetLabels[2],buf);

	sprintf(buf,"Correlation Length");
	try { df->m_pAdditionalSetLabels[3] = new char[strlen(buf)+1]; } catch(...) { df->m_pAdditionalSetLabels[4] = NULL; }
	if (df->m_pAdditionalSetLabels[3] == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	strcpy(df->m_pAdditionalSetLabels[3],buf);

	try { lm = new CLMWrapper(); } catch(...) { lm = NULL; }
	if (lm == NULL) NewException((double)sizeof(CLMWrapper),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { x = new double[m_iaBlockSizes.GetSize()]; } catch(...) { x = NULL; }
	if (x == NULL) NewException((double)m_iaBlockSizes.GetSize()*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { y = new double[m_iaBlockSizes.GetSize()]; } catch(...) { y = NULL; }
	if (y== NULL) NewException((double)m_iaBlockSizes.GetSize()*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { y2 = new double[m_iaBlockSizes.GetSize()]; } catch(...) { y2 = NULL; }
	if (y2 == NULL) NewException((double)m_iaBlockSizes.GetSize()*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { curve = new double[m_iaBlockSizes.GetSize()]; } catch(...) { curve = NULL; }
	if (curve == NULL) NewException((double)m_iaBlockSizes.GetSize()*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	mprintf("    Fitting trajectory block decomposition...\n");
	mprintf(WHITE,"      [");

	tf = m_oaGroups.GetSize() / 60.0;

	dsa = 0;
	dsmi = 1e20;
	dsma = 0;
	ti = 0;
	for (z=0;z<m_oaGroups.GetSize();z++)
	{
		if (fmod(z,tf) < 1.0)
			mprintf(WHITE,"#");

		if (df->m_pBin[z] == 0)
			continue;

		gr = (CSDGroup*)m_oaGroups[z];

//		mprintf("  * Gruppe %d:\n",z);

		if (verbose)
		{
			sprintf(buf,"extra%03d.csv",z);
			b = fopen(buf,"wt");
			fprintf(b,"# Block;  Block size;  sqrt(size);  log(size);  1/size;  Vari;  Size*Vari;  S;  log(S);  log(S2);  Fit\n");
		}

		ya = 0;
		for (z2=0;z2<m_iaBlockSizes.GetSize();z2++)
		{
			x[z2] = log10(m_iaBlockSizes[z2]);
			y[z2] = log10(m_iaBlockSizes[z2]*gr->m_faBlockVari[z2]/m_faVari[z]);
			if (z2 == 0)
			{
				y2[z2] = y[z2];
				ya = y[z2];
			} else
			{
				if (y[z2] > ya)
				{
					y2[z2] = y[z2];
					ya = y[z2];
				} else y2[z2] = ya;
			}
		}

		par[0] = 50.0;
		par[1] = -0.01;
		par[2] = 1.0;

		lm->Fit_SD(m_iaBlockSizes.GetSize(),x,y2,par,curve,&dr,200);

		if (verbose)
			fprintf(b,"# A=%G;  B=%G;  C=%G;  R=%G\n",par[0],par[1],par[2],dr);

		ds = pow(10.0,par[0]*(1.0-exp(par[1]*pow(log10(m_iSteps),par[2]))));

		if (ds < dsmi)
			dsmi = ds;
		if (ds > dsma)
			dsma = ds;
		dsa += ds;
		ti++;

		m_faCorLen[z] = ds;

//		mprintf("    Fit done. y(x) = %10G * (1 - exp( %10G * x^%10G ) ).   R = %10G.  S = %10G\n",par[0],par[1],par[2],dr,ds);

		if (verbose)
		{
			for (z2=0;z2<m_iaBlockSizes.GetSize();z2++)
				fprintf(b,"%d;  %d;  %G;  %G;  %G;  %G;  %G;  %G;  %G;  %G;  %G\n",z2,(int)m_iaBlockSizes[z2],sqrt(m_iaBlockSizes[z2]),log10(m_iaBlockSizes[z2]),1.0/m_iaBlockSizes[z2],gr->m_faBlockVari[z2],m_iaBlockSizes[z2]*gr->m_faBlockVari[z2],m_iaBlockSizes[z2]*gr->m_faBlockVari[z2]/m_faVari[z],y[z2],y2[z2],curve[z2]);
		}

/*		x = new double[m_iaBlockSizes.GetSize()];
		y = new double[m_iaBlockSizes.GetSize()];

		xa = 0;
		ya = 0;
		for (z2=0;z2<m_iaBlockSizes.GetSize();z2++)
		{
			bs = (CSDBlockSet*)gr->m_oaBlockSets[z2];
//			mprintf("    - Blockgroesse %d (%d Schritte)\n",z2,m_iaBlockSizes[z2]);
//			mprintf("        %d Bloecke, Counter %d.\n",bs->m_faBlocks.GetSize(),bs->m_iCounter);

			if (z == 100)
			{
				for (z3=0;z3<bs->m_faBlocks.GetSize()-1;z3++)
				{
					fprintf(a,"%d;  %d;  %G\n",z2,z3,bs->m_faBlocks[z3]);
				}
			}


			x[z2] = log10(m_iaBlockSizes[z2]);
			y[z2] = log10(m_iaBlockSizes[z2]*gr->m_faBlockVari[z2]/m_faVari[z]);
			xa += x[z2];
			ya += y[z2];
		}
	
		xa /= m_iaBlockSizes.GetSize();
		ya /= m_iaBlockSizes.GetSize();

		dxy = 0;
		dxx = 0;
		dyy = 0;

		for (z2=0;z2<m_iaBlockSizes.GetSize();z2++)
		{
			dxy += (x[z2] - xa) * (y[z2] - ya);
			dxx += (x[z2] - xa) * (x[z2] - xa);
			dyy += (y[z2] - ya) * (y[z2] - ya);
		}

		db = dxy / dxx;
		da = ya - db * xa;

		ds = pow(10.0, da + db * log10(m_iSteps));

		dr = sqrt( dxy*dxy / dxx / dyy );

		mprintf("    y(x) = %10G + %10G * x;    S = %10G;  R = %10G\n",da,db,ds,dr);
		
		for (z2=0;z2<m_iaBlockSizes.GetSize();z2++)
			fprintf(b,"%d;  %d;  %G;  %G;  %G;  %G;  %G;  %G;  %G;  %G\n",z2,(int)m_iaBlockSizes[z2],sqrt(m_iaBlockSizes[z2]),log10(m_iaBlockSizes[z2]),1.0/m_iaBlockSizes[z2],gr->m_faBlockVari[z2],m_iaBlockSizes[z2]*gr->m_faBlockVari[z2],m_iaBlockSizes[z2]*gr->m_faBlockVari[z2]/m_faVari[z],log10(m_iaBlockSizes[z2]*gr->m_faBlockVari[z2]/m_faVari[z]),da+db*x[z2]);


		delete[] x;
		delete[] y;*/

		if (verbose)
			fclose(b);

	}
	delete[] x;
	delete[] y;
	delete[] y2;
	delete[] curve;
	delete lm;

	mprintf(WHITE,"]\n");

	dsa /= ti;

	mprintf("    Correlation length: Min %.3f, Max %.3f, Avg %.3f time steps.\n",dsmi,dsma,dsa);

	for (z=0;z<m_oaGroups.GetSize();z++)
	{
		ds = 0;
		da = 0;
		for (z2=z-10;z2<=z+10;z2++)
		{
			if (z2 < 0)
				continue;
			if (z2 >= m_oaGroups.GetSize())
				continue;
			if (df->m_pBin[z2] == 0)
				continue;
			ds += m_faCorLen[z2];
			da += 1.0;
		}
		ds /= da;

		if (df->m_pBin[z] != 0)
		{
//			df->m_pAdditionalSets[2][z] = sqrt(ds / m_iSteps) * df->m_pBin[z];
			df->m_pAdditionalSets[2][z] = sqrt(ds / m_iSteps * m_faVari[z]);
			df->m_pAdditionalSets[3][z] = ds;

			df->m_pAdditionalSets[0][z] = df->m_pBin[z] - timessigma*df->m_pAdditionalSets[2][z];
			df->m_pAdditionalSets[1][z] = df->m_pBin[z] + timessigma*df->m_pAdditionalSets[2][z];
		} else
		{
			df->m_pAdditionalSets[0][z] = 0;
			df->m_pAdditionalSets[1][z] = 0;
			df->m_pAdditionalSets[2][z] = 0;
			df->m_pAdditionalSets[3][z] = 0;
		}
	}

//	fclose(a);
}


void CSDEngine::FinishAnalysis()
{
	int z, z2, z3;
	CSDGroup *gr;
	CSDBlockSet *bl;

	mprintf("    Calculating block averages...\n");

	for (z=0;z<m_oaGroups.GetSize();z++)
	{
		m_faMean[z] = m_faSum[z] / m_iSteps;
		m_faVari[z] = (m_faSqSum[z] / m_iSteps) - (m_faMean[z]*m_faMean[z]);

//		mprintf("    # Bin %3d:   Sum %10G;   SqSum %10G;  Mean %10G;  Vari %10G\n",z,m_faSum[z],m_faSqSum[z],m_faMean[z],m_faVari[z]);

		gr = (CSDGroup*)m_oaGroups[z];

		for (z2=0;z2<gr->m_oaBlockSets.GetSize();z2++)
		{
			bl = (CSDBlockSet*)gr->m_oaBlockSets[z2];

			for (z3=0;z3<bl->m_faBlocks.GetSize()-1;z3++)
				gr->m_faBlockVari[z2] += pow(bl->m_faBlocks[z3] - m_faMean[z], 2);

			gr->m_faBlockVari[z2] /= ((double)bl->m_faBlocks.GetSize()-1);
		}
	}
}


