/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2013 Martin Brehm
                  2012-2013 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/


#include "fesa.h"
#include "globalvar.h"
#include "maintools.h"


CFESAnalysis::CFESAnalysis()
{
}


CFESAnalysis::~CFESAnalysis()
{
}


void CFESAnalysis::Parse()
{
	char buf[256];
	unsigned char ty, rty, at;

	mprintf(WHITE,"\n>>> Free Energy Surface Analysis >>>\n\n");

	m_bIntra = (AskYesNo("    Perform this analysis intermolecular (y) or intramolecular (n)? [yes] ",true)==false);

	if (m_bIntra)
	{
		m_iMol1 = AskMolecule("    From which molecule should the atoms be taken");

_again1:
		AskString("    Which first atom to use from %s? [#2] ",buf,"#2",((CMolecule*)g_oaMolecules[m_iMol1])->m_sName);
		if (!ParseAtom(buf,m_iMol1,ty,rty,at))
			goto _again1;
		m_iMolAtomType1 = ty;
		m_iMolAtomRealType1 = rty;
		m_iMolAtom1 = at;

_again2:
		AskString("    Which second atom to use from %s? [#2] ",buf,"#2",((CMolecule*)g_oaMolecules[m_iMol1])->m_sName);
		if (!ParseAtom(buf,m_iMol1,ty,rty,at))
			goto _again2;
		m_iMolAtomType2 = ty;
		m_iMolAtomRealType2 = rty;
		m_iMolAtom2 = at;
	} else
	{
		m_iMol1 = AskMolecule("    From which molecule should the first atom be taken");
_again1b:
		AskString("    Which first atom to use from %s? [#2] ",buf,"#2",((CMolecule*)g_oaMolecules[m_iMol1])->m_sName);
		if (!ParseAtom(buf,m_iMol1,ty,rty,at))
			goto _again1b;
		m_iMolAtomType1 = ty;
		m_iMolAtomRealType1 = rty;
		m_iMolAtom1 = at;

		m_iMol2 = AskMolecule("    From which molecule should the second atom be taken");
_again2b:
		AskString("    Which second atom to use from %s? [#2] ",buf,"#2",((CMolecule*)g_oaMolecules[m_iMol2])->m_sName);
		if (!ParseAtom(buf,m_iMol2,ty,rty,at))
			goto _again2b;
		m_iMolAtomType2 = ty;
		m_iMolAtomRealType2 = rty;
		m_iMolAtom2 = at;
	}

	mprintf("\n");

	m_fMinDist = AskFloat("    Please enter minimal distance (in pm): [0] ",0);
	m_fMaxDist = AskFloat("    Please enter maximal distance (in pm): [%d] ",(float)HalfBox(),HalfBox());
	m_iResolution = AskUnsignedInteger("    Please enter distance resolution: [300] ",300);


	mprintf(WHITE,"\n<<< End of Free Energy Surface Analysis <<<\n\n");
	BTOUT;
}


void CFESAnalysis::BuildName()
{

}


void CFESAnalysis::Create()
{
	m_pCrossToLower = new CDF();

	m_pCrossToLower->m_fMinVal = m_fMinDist;
	m_pCrossToLower->m_fMaxVal = m_fMaxDist;
	m_pCrossToLower->m_iResolution = m_iResolution;
	m_pCrossToLower->Create();

	m_pCrossToUpper = new CDF();

	m_pCrossToUpper->m_fMinVal = m_fMinDist;
	m_pCrossToUpper->m_fMaxVal = m_fMaxDist;
	m_pCrossToUpper->m_iResolution = m_iResolution;
	m_pCrossToUpper->Create();

	if (m_bIntra)
		m_pLastBuffer = new float[((CMolecule*)g_oaMolecules[m_iMol1])->m_laSingleMolIndex.GetSize()];
			else m_pLastBuffer = new float[((CMolecule*)g_oaMolecules[m_iMol1])->m_laSingleMolIndex.GetSize()*((CMolecule*)g_oaMolecules[m_iMol2])->m_laSingleMolIndex.GetSize()];
}


void CFESAnalysis::ProcessStep(CTimeStep *ts)
{
	int z, z2, z3, ti1, ti2, tia, tib;
	CMolecule *m1, *m2;
	CSingleMolecule *sm1, *sm2;
	float tf;

	if (m_bIntra)
	{
		m1 = (CMolecule*)g_oaMolecules[m_iMol1];
		for (z=0;z<m1->m_laSingleMolIndex.GetSize();z++)
		{
			sm1 = (CSingleMolecule*)g_oaSingleMolecules[m1->m_laSingleMolIndex[z]];

			ti1 = ((CxIntArray*)sm1->m_oaAtomOffset[m_iMolAtomType1])->GetAt(m_iMolAtom1);
			ti2 = ((CxIntArray*)sm1->m_oaAtomOffset[m_iMolAtomType2])->GetAt(m_iMolAtom2);

			tf = FoldedLength(ts->m_vaCoords[ti1] - ts->m_vaCoords[ti2]);

			if (g_iSteps != 1)
			{
				tia = (m_pLastBuffer[z]-m_fMinDist)/(m_fMaxDist-m_fMinDist)*m_iResolution;
				tib = (tf-m_fMinDist)/(m_fMaxDist-m_fMinDist)*m_iResolution;

				if (((tia >= 0) && (tia < m_iResolution)) || ((tib >= 0) && (tib < m_iResolution)))
				{
					if (tia < 0)
						tia = 0;
					if (tib < 0)
						tib = 0;
					if (tia >= m_iResolution)
						tia = m_iResolution-1;
					if (tib >= m_iResolution)
						tib = m_iResolution-1;
					
					if (tf > m_pLastBuffer[z])
					{
						for (z3=tia;z3<=tib;z3++)
							m_pCrossToUpper->m_pBin[z3]++;
					} else
					{
						for (z3=tib;z3<=tia;z3++)
							m_pCrossToLower->m_pBin[z3]++;
					}
				}
			}

			m_pLastBuffer[z] = tf;
		}
	} else
	{
		m1 = (CMolecule*)g_oaMolecules[m_iMol1];
		m2 = (CMolecule*)g_oaMolecules[m_iMol2];
		for (z=0;z<m1->m_laSingleMolIndex.GetSize();z++)
		{
			sm1 = (CSingleMolecule*)g_oaSingleMolecules[m1->m_laSingleMolIndex[z]];

			for (z2=0;z2<m2->m_laSingleMolIndex.GetSize();z2++)
			{
				sm2 = (CSingleMolecule*)g_oaSingleMolecules[m2->m_laSingleMolIndex[z2]];

				ti1 = ((CxIntArray*)sm1->m_oaAtomOffset[m_iMolAtomType1])->GetAt(m_iMolAtom1);
				ti2 = ((CxIntArray*)sm2->m_oaAtomOffset[m_iMolAtomType2])->GetAt(m_iMolAtom2);

				tf = FoldedLength(ts->m_vaCoords[ti1] - ts->m_vaCoords[ti2]);

				if (g_iSteps != 1)
				{
					tia = (m_pLastBuffer[z*m2->m_laSingleMolIndex.GetSize()+z2]-m_fMinDist)/(m_fMaxDist-m_fMinDist)*m_iResolution;
					tib = (tf-m_fMinDist)/(m_fMaxDist-m_fMinDist)*m_iResolution;

					if (((tia >= 0) && (tia < m_iResolution)) || ((tib >= 0) && (tib < m_iResolution)))
					{
						if (tia < 0)
							tia = 0;
						if (tib < 0)
							tib = 0;
						if (tia >= m_iResolution)
							tia = m_iResolution-1;
						if (tib >= m_iResolution)
							tib = m_iResolution-1;
						
						if (tf > m_pLastBuffer[z*m2->m_laSingleMolIndex.GetSize()+z2])
						{
							for (z3=tia;z3<tib;z3++)
								m_pCrossToUpper->m_pBin[z3]++;
						} else
						{
							for (z3=tib;z3<tia;z3++)
								m_pCrossToLower->m_pBin[z3]++;
						}
					}
				}

				m_pLastBuffer[z*m2->m_laSingleMolIndex.GetSize()+z2] = tf;
			}
		}
	}
}


void CFESAnalysis::Finish()
{
	m_pCrossToLower->Write("","tolower.csv","",false);
	m_pCrossToUpper->Write("","toupper.csv","",false);

	m_pCrossToLower->CorrectRadialDist();
	m_pCrossToUpper->CorrectRadialDist();

	m_pCrossToLower->Write("","tolower_rad.csv","",false);
	m_pCrossToUpper->Write("","toupper_rad.csv","",false);

}
