/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2013 Martin Brehm
                  2012-2013 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "structurefactor.h"
#include "moltools.h"
#include "globalvar.h"
#include "maintools.h"


CStructureFactor::CStructureFactor()
{
}


CStructureFactor::~CStructureFactor()
{
}


void CStructureFactor::Create()
{
	int z, z2;
	CDF *df;

	for (z=0;z<g_oaAtoms.GetSize();z++)
	{
		if (z == g_iVirtAtomType)
			continue;

		if (((CAtom*)g_oaAtoms[z])->m_pElement->m_fCoherentNCS == 0)
			eprintf("\nError: Coherent Neutron Cross Section for %s not defined. Edit elementdata.cpp.\n",((CAtom*)g_oaAtoms[z])->m_sName);

		for (z2=0;z2<g_oaAtoms.GetSize();z2++)
		{
			if (z2 == g_iVirtAtomType)
				continue;

			if (z2 < z)
			{
				m_oaRDFs.Add(NULL);
				m_oaSFacs.Add(NULL);
			} else
			{
				df = new CDF();
				df->m_fMinVal = 0;
				df->m_fMaxVal = m_fRDFRange;
				df->m_iResolution = m_iRDFRes;
				df->SetLabelX("Distance [pm]");
				df->SetLabelY("g(r)");
				df->Create();
				m_oaRDFs.Add(df);

				df = new CDF();
				df->m_fMinVal = 0;
				df->m_fMaxVal = m_fSQRange;
				df->m_iResolution = m_iSQRes;
				df->SetLabelX("Wave Vector Modulus [Angstrom^-1]");
				df->SetLabelY("S(k)");
				df->Create();
				m_oaSFacs.Add(df);
			}
		}
	}
}


void CStructureFactor::Parse()
{
	mprintf(WHITE,">>> Structure Factor Analysis >>>\n\n");

	m_fRDFRange = AskFloat("    Enter maximum RDF distance to observe (in pm): [%d] ",(float)HalfBox(),HalfBox());
	m_iRDFRes = AskUnsignedInteger("    Enter RDF binning resolution: [300] ",300);
	m_fSQRange = AskFloat("    Enter maximum wave vector modulus (in Angstrom^-1): [50] ",50);
	m_iSQRes = AskUnsignedInteger("    Enter Structure Factor binning resolution: [2000] ",2000);

	mprintf("\n");

	m_bDumpElementRDFs = AskYesNo("    Write out all element RDFs (y/n)? [yes] ",true);
	m_bDumpTotalRDF = AskYesNo("    Write out overall RDF (y/n)? [yes] ",true);
	m_bDumpElementSFac = AskYesNo("    Write out all element Structure Factor contributions (y/n)? [yes] ",true);

	mprintf(WHITE,"\n<<< End of Structure Factor Analysis <<<\n\n");
}


void CStructureFactor::ProcessStep(CTimeStep *ts)
{
	int z, z2, t1, t2, i;
	float v;

	i = g_oaAtoms.GetSize()-1;

	for (z=0;z<g_iGesAtomCount;z++)
	{
		t1 = g_waAtomRealElement[z];
		for (z2=z+1;z2<g_iGesAtomCount;z2++)
		{
			t2 = g_waAtomRealElement[z2];

			v = FoldedLength(ts->m_vaCoords[z] - ts->m_vaCoords[z2]);

			if (t1 >= t2)
			{
				((CDF*)m_oaRDFs[t2*i+t1])->AddToBin(v);
			} else
			{
				((CDF*)m_oaRDFs[t1*i+t2])->AddToBin(v);
			}
		}
	}
}


void CStructureFactor::Finish()
{
	int z, z2, z3;
	CDF *df, *dftotal, *dfstotal;
	double fac/*, tf*/;
	char buf[256];

	dftotal = NULL;

	mprintf(WHITE,"* Structure Factor\n");

	if (m_bDumpTotalRDF)
	{
		dftotal = new CDF();
		dftotal->m_fMinVal = 0;
		dftotal->m_fMaxVal = m_fRDFRange;
		dftotal->m_iResolution = m_iRDFRes;
		dftotal->SetLabelX("Distance [pm]");
		dftotal->SetLabelY("g(r)");
		dftotal->Create();
//		tf = 0;
	}

	dfstotal = new CDF();
	dfstotal->m_fMinVal = 0;
	dfstotal->m_fMaxVal = m_fSQRange;
	dfstotal->m_iResolution = m_iSQRes;
	dfstotal->SetLabelX("Wave Vector Modulus [Angstrom^-1]");
	dfstotal->SetLabelY("S(k)");
	dfstotal->Create();

	for (z=0;z<g_oaAtoms.GetSize();z++)
	{
		if (z == g_iVirtAtomType)
			continue;

		for (z2=0;z2<g_oaAtoms.GetSize();z2++)
		{
			if (z2 == g_iVirtAtomType)
				continue;

			if (z2 < z)
				continue;

			df = (CDF*)m_oaRDFs[z*(g_oaAtoms.GetSize()-1)+z2];

			mprintf(WHITE,"  Processing Contributions of type %s - %s\n",((CAtom*)g_oaAtoms[z])->m_sName,((CAtom*)g_oaAtoms[z2])->m_sName);

			df->MultiplyBin(1.0 / g_iSteps);
			if (z == z2)
				df->MultiplyBin(2.0);
			df->CorrectRadialDist();
			df->MultiplyBin(g_fBoxX*g_fBoxY*g_fBoxZ / (4.0/3.0*Pi) / ((CAtom*)g_oaAtoms[z])->m_iCount / ((CAtom*)g_oaAtoms[z2])->m_iCount);

			if (g_bDoubleBox)
				df->MultiplyBin(g_iDoubleBoxFactor);

			if (m_bDumpElementRDFs)
			{
				sprintf(buf,"sfac_rdf_%s_%s.csv",((CAtom*)g_oaAtoms[z])->m_sName,((CAtom*)g_oaAtoms[z2])->m_sName);
				mprintf("    Writing RDF to %s ...\n",buf);
				df->Write("",buf,"",false);
			}

			if (m_bDumpTotalRDF)
			{
				if (z == z2)
					fac = 1.0;
						else fac = 2.0;
				for (z3=0;z3<m_iRDFRes;z3++)
					dftotal->m_pBin[z3] += df->m_pBin[z3] * fac * (double)((CAtom*)g_oaAtoms[z])->m_iCount * ((CAtom*)g_oaAtoms[z2])->m_iCount / g_iGesAtomCount / g_iGesAtomCount;
	//			tf += fac * (double)((CAtom*)g_oaAtoms[z])->m_iCount * ((CAtom*)g_oaAtoms[z2])->m_iCount / g_iGesAtomCount / g_iGesAtomCount;
			}

			TransformRDF(df,(CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2]);

			if (z == z2)
				fac = 1.0;
					else fac = 2.0;
			for (z3=0;z3<m_iSQRes;z3++)
				dfstotal->m_pBin[z3] += ((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[z3] * fac * (double)((CAtom*)g_oaAtoms[z])->m_pElement->m_fCoherentNCS * ((CAtom*)g_oaAtoms[z2])->m_pElement->m_fCoherentNCS * ((CAtom*)g_oaAtoms[z])->m_iCount * ((CAtom*)g_oaAtoms[z2])->m_iCount / g_iGesAtomCount / g_iGesAtomCount;

			if (m_bDumpElementSFac)
			{
				sprintf(buf,"sfac_%s_%s.csv",((CAtom*)g_oaAtoms[z])->m_sName,((CAtom*)g_oaAtoms[z2])->m_sName);
				mprintf("    Writing Structure Factor Contribution to %s ...\n",buf);
			//	mprintf("    Correction Factor: %G\n",((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[0]);
			/*	tf = 0;
				for (z3=0;z3<m_iSQRes;z3++)
				{
					if (((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[z3] < tf)
						tf = ((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[z3];
				}
				if (tf < 0)
				{
					for (z3=0;z3<m_iSQRes;z3++)
						((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[z3] /= fabs(tf);
				}
				for (z3=0;z3<m_iSQRes;z3++)
					((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->m_pBin[z3] += 1.0;*/
				((CDF*)m_oaSFacs[z*(g_oaAtoms.GetSize()-1)+z2])->Write("",buf,"",false);
			}
		}
	}

	if (m_bDumpTotalRDF)
	{
		mprintf(WHITE,"  * Overall RDF\n");
	//	mprintf("    tf = %f.\n",tf);
		sprintf(buf,"sfac_rdf_total.csv");
		mprintf("    Writing RDF to %s ...\n",buf);
		dftotal->Write("",buf,"",false);
	}

	mprintf(WHITE,"  * Overall S(k)\n");
	sprintf(buf,"sfac_total.csv");
	mprintf("    Writing S(k) to %s ...\n",buf);
/*	tf = 0;
	for (z3=0;z3<m_iSQRes;z3++)
	{
		if (dfstotal->m_pBin[z3] < tf)
			tf = dfstotal->m_pBin[z3];
	}
	if (tf < 0)
	{
		for (z3=0;z3<m_iSQRes;z3++)
			dfstotal->m_pBin[z3] /= fabs(tf);
	}
	for (z3=0;z3<m_iSQRes;z3++)
		dfstotal->m_pBin[z3] += 1.0;*/
	dfstotal->Write("",buf,"",false);
}


void CStructureFactor::TransformRDF(CDF *pin, CDF *pout)
{
	int z, z2;
	double tf, tf2;

	for (z=0;z<m_iSQRes;z++)
	{
		tf = 0;

		for (z2=0;z2<m_iRDFRes;z2++)
		{
			tf2 = ((z2+0.5)/m_iRDFRes*m_fRDFRange/100.0) * (pin->m_pBin[z2]-1.0) / ((z+0.5)/m_iSQRes*m_fSQRange) * sin(((z2+0.5)/m_iRDFRes*m_fRDFRange/100.0) * ((z+0.5)/m_iSQRes*m_fSQRange));
			tf += tf2;
	//		mprintf("\n  %.4fA * %.4f / %.4fA^-1 * sin( %.4fA^-1 * %.4fA ) = %.4f",((z2+0.5)/m_iRDFRes*m_fRDFRange/100.0),(pin->m_pBin[z2]-1.0),((z+0.5)/m_iSQRes*m_fRDFRange/100.0),((z2+0.5)/m_iRDFRes*m_fRDFRange/100.0),((z+0.5)/m_iSQRes*m_fRDFRange/100.0),tf2);
		}

		pout->m_pBin[z] = tf*4.0*Pi/g_fBoxX/g_fBoxY/g_fBoxZ*1000000.0;
	//	mprintf("\n\n## 1 + 4*Pi*%.4f*%.4f = %.4f\n",tf,(1.0/m_iRDFRes*m_fRDFRange/10000.0),pout->m_pBin[z]);
	}
}
