/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2013 Martin Brehm
                  2012-2013 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "microhet.h"
#include "travis.h"
#include "maintools.h"


CMicroHet::CMicroHet()
{
	m_oaObservations.SetName("CMicroHet::m_oaObservations");
	m_iaCuboidLength.SetName("CMicroHet::m_iaCuboidLength");
	m_faCuboidVolume.SetName("CMicroHet::m_faCuboidVolume");
}


CMicroHet::~CMicroHet()
{
}


CMicroHetObservation::CMicroHetObservation()
{
	m_oaAtomGroups.SetName("CMicroHetObservation::m_oaAtomGroups");
	m_faEmptyBinHalf.SetName("CMicroHetObservation::m_faEmptyBinHalf");
	m_faEmptyBinZero.SetName("CMicroHetObservation::m_faEmptyBinZero");
}


CMicroHetObservation::~CMicroHetObservation()
{
}


void CMicroHet::Parse()
{
	CMicroHetObservation *o;
	int z, i1, i2, i3;

	mprintf(WHITE,">>> Microheterogeneity Analysis >>>\n\n");

	m_bSphere = AskYesNo("    Use cuboid (n) or sphere (y)? [no] ",false);

	m_iRes = AskUnsignedInteger("    Please enter the spatial resolution per dimension: [100] ",100);
	m_iResSqr = m_iRes * m_iRes;
	m_iResTri = m_iResSqr * m_iRes;
	m_fBinVolume = g_fBoxX * g_fBoxY * g_fBoxZ / 1e9 / m_iRes / m_iRes / m_iRes;

	mprintf("\n    One bin has the spatial extent of %.3f x %.3f x %.3f pm.\n",g_fBoxX/m_iRes,g_fBoxY/m_iRes,g_fBoxZ/m_iRes);
	mprintf("    This equals a volume of %.3f pm^3.\n",m_fBinVolume*1E9);

	mprintf("\n    This will require %s of RAM.\n\n",FormatBytes((double)m_iResTri*sizeof(MH_BIN_TYPE)));

	if (AskYesNo("    Use multiple analysis boxes with different sizes (y/n)? [yes] ",true))
	{
		if (AskYesNo("    Use equidistant box sizes (y) or enter sizes manually (n)? [yes] ",true))
		{
			i1 = AskUnsignedInteger("    Enter smallest box size (in bins): [1] ",1);
			i2 = AskUnsignedInteger("    Enter box size increment (in bins): [1] ",1);
			i3 = AskUnsignedInteger("    Enter increment count: [%d] ",(m_iRes-i1)/i2+1,(m_iRes-i1)/i2+1);
			for (z=0;z<i3;z++)
				m_iaCuboidLength.Add(i1+z*i2);
		} else
		{
			do {
				i1 = AskUnsignedInteger("    Enter box size (in bins): [finished] ",0);
				if (i1 != 0)
					m_iaCuboidLength.Add(i1);
			} while (i1 != 0);
		}
	} else
	{
		i1 = AskUnsignedInteger("    Please enter the edge length of the analysis box (in bins): [%d] ",m_iRes/2,m_iRes/2);
		m_iaCuboidLength.Add(i1);
	}

	mprintf("\n    Using %d different analysis boxes:\n\n",m_iaCuboidLength.GetSize());
	for (z=0;z<m_iaCuboidLength.GetSize();z++)
	{
		mprintf("    - %3d bins edge length (%8.3f x %8.3f x %8.3f pm)\n",m_iaCuboidLength[z],g_fBoxX/m_iRes*m_iaCuboidLength[z],g_fBoxY/m_iRes*m_iaCuboidLength[z],g_fBoxZ/m_iRes*m_iaCuboidLength[z]);
	}

	do {
		mprintf(WHITE,"\n*** Analysis %d ***\n\n",m_oaObservations.GetSize()+1);

		try { o = new CMicroHetObservation(); } catch(...) { o = NULL; }
		if (o == NULL) NewException((double)sizeof(CMicroHetObservation),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		o->m_pParent = this;
		o->Parse();
		m_oaObservations.Add(o);
		mprintf("\n");
	} while (AskYesNo("    Add another microheterogeneity analysis (y/n)? [no] ",false));

	mprintf(WHITE,"\n<<< End of Microheterogeneity Analysis <<<\n\n");
}


void CMicroHetObservation::Parse()
{
	int z, z2, z3, z4, c;
	CMolecule *m;
	CAtomGroup *ag;
	char buf[256];

	if (g_fTimestepLength == 0)
		g_fTimestepLength = AskFloat("    Enter the length of one trajectory time step in fs: [0.5] ",0.5f);

	m_bMass = AskYesNo("    Use mass (y) or atom count (n)? [yes] ",true);

	if (m_bMass)
		mprintf("\n    When observing any center (\"#\") in a molecule, the center bears the total molecule mass.\n");

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		mprintf("\n");
		if (AskYesNo("    Use atoms from molecule %s (y/n)? [yes] ",true,((CMolecule*)g_oaMolecules[z])->m_sName))
		{
			try { ag = new CAtomGroup(); } catch(...) { ag = NULL; }
			if (ag == NULL) NewException((double)sizeof(CAtomGroup),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
_a:
			AskString("      Which atoms to use from this molecule? [all] ",buf,"*");
			if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[z],buf))
				goto _a;
			m_oaAtomGroups.Add(ag);
		}
	}

	BuildName();

	mprintf("\n    Observing %s.\n",m_sName);

	c = 0;
	for (z2=0;z2<m_oaAtomGroups.GetSize();z2++)
	{
		ag = (CAtomGroup*)m_oaAtomGroups[z2];
		for (z3=0;z3<ag->m_oaAtoms.GetSize();z3++)
		{
			if (m_bMass)
			{
				if (ag->m_baRealAtomType[z3] == g_iVirtAtomType)
				{
					m = ag->m_pMolecule;
					for (z4=0;z4<m->m_baAtomIndex.GetSize();z4++)
						c += (int)((m->m_waAtomCount[z4] * ((CAtom*)g_oaAtoms[m->m_baAtomIndex[z4]])->m_pElement->m_fMass) + 0.5) * ag->m_pMolecule->m_laSingleMolIndex.GetSize();
				} else c += (int)(((CAtom*)g_oaAtoms[ag->m_baRealAtomType[z3]])->m_pElement->m_fMass+0.5) * ((CxIntArray*)ag->m_oaAtoms[z3])->GetSize() * ag->m_pMolecule->m_laSingleMolIndex.GetSize();
			} else c += ((CxIntArray*)ag->m_oaAtoms[z3])->GetSize() * ag->m_pMolecule->m_laSingleMolIndex.GetSize();
		}
	}
	mprintf("\n    Largest expected bin value is %d.\n",c);
	mprintf("    Maximum allowed bin value is %.0f.\n\n",pow(256.0,sizeof(MH_BIN_TYPE)));
	m_iHistoRes = c+1;
	m_fMaxVal = c;

	if (c >= pow(256.0,sizeof(MH_BIN_TYPE)))
	{
		AskYesNo("    WARNING: Bin overflow may occur (adjust MH_BIN_TYPE in microhet.h)! [Ok] ",true);
		mprintf("\n");
	}

	if (m_pParent->m_iaCuboidLength.GetSize() > 1)
		m_bCDF = AskYesNo("    Create two-dimensional distribution function (y/n)? [yes] ",true);
			else m_bCDF = false;

	if (m_bCDF)
	{
		mprintf("\n");
		m_bCDF_Dens = AskYesNo("    Create density 2D plot (y/n)? [yes] ",true);
		if (m_bCDF_Dens)
		{
			m_fRangeFac = AskFloat("      Please enter maximum vertical CDF range (in units of uniform density): [5.0] ",5.0);
			m_bCDF_Dens_Uniform = AskYesNo("      Take density relative to uniform density (y) or absolute values (n)? [no] ",false);
		}
		m_bCDF_Tot = AskYesNo("    Create absolute value 2D plot (y/n)? [yes] ",true);
		if (m_bCDF_Tot)
			m_bCubicRoot = AskYesNo("      Plot absolute values (n) or cubic root of values (y)? [yes] ",true);
				else m_bCubicRoot = false;
		m_bCDF_Mode3 = AskYesNo("    Create \"mode 3\" 2D plot (y/n)? [no] ",false);

		m_bEdgeUniform = AskYesNo("    Take cube edge length relative to simulation box (y) or absolute (n)? [yes] ",true);
		m_iCDFRes = AskUnsignedInteger("    Please enter vertical CDF resolution: [150] ",150);
	} else
	{
		m_bCDF_Dens = false;
		m_bCDF_Tot = false;
		m_bCDF_Mode3 = false;
	}

	mprintf("\n");
	m_bEmpty2D = AskYesNo("    Create empty bin 2D plot (y/n)? [no] ",false);
	if (m_bEmpty2D)
	{
		m_iEmpty2DRes = AskUnsignedInteger("      Enter the resolution of the temporal axis: [150] ",150);
		m_iEmpty2DStride = AskUnsignedInteger("      How many time steps to use per temporal bin? [1] ",1);
	}

	mprintf("\n");
	m_bCSV = AskYesNo("    Write out additional CSV data files (y/n)? [no] ",false);

	if (m_bCSV)
	{
		m_bDensity = AskYesNo("    Use density (y) or absolute value (n)? [yes] ",true);
		if (m_bDensity)
			m_bUniform = AskYesNo("    Take density relative to uniform density (y) or absolute values (n)? [no] ",false);
	} else m_bDensity = true;
}


void CMicroHet::Create()
{
	double f;
	int z, zx, zy, zz, zyq, zzq, chq, cl;

	try { m_pIBin = new MH_BIN_TYPE[m_iResTri]; } catch(...) {m_pIBin = NULL; }
	if (m_pIBin == NULL) NewException((double)m_iResTri*sizeof(MH_BIN_TYPE),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<m_iaCuboidLength.GetSize();z++)
	{
		f = 0;
		if (m_bSphere)
		{
			cl = m_iaCuboidLength[z];
			chq = (cl*cl)/4;
			for (zz=0;zz<cl;zz++)
			{
				zzq = zz*zz - zz*cl + 2*chq;
				for (zy=0;zy<cl;zy++)
				{
					zyq = zzq + zy*zy - zy*cl;
					for (zx=0;zx<cl;zx++)
					{
						if (zyq + zx*zx - zx*cl <= 0)
							f += m_fBinVolume;
					}
				}
			}
		} else
		{
			f = m_iaCuboidLength[z] * m_iaCuboidLength[z] * m_iaCuboidLength[z] * m_fBinVolume;
		}
		m_faCuboidVolume.Add(f);
	}

	for (z=0;z<m_oaObservations.GetSize();z++)
	{
		((CMicroHetObservation*)m_oaObservations[z])->Create();
	}

	m_iCounter = 0;
}


void CMicroHet::Zero()
{
	unsigned long z;

	for (z=0;z<m_iResTri;z++)
		m_pIBin[z] = 0;
}


void CMicroHetObservation::Create()
{
	int z;

	try { m_pHistogram = new CDF*[m_pParent->m_iaCuboidLength.GetSize()]; } catch(...) { m_pHistogram = NULL; }
	if (m_pHistogram == NULL) NewException((double)m_pParent->m_iaCuboidLength.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<m_pParent->m_iaCuboidLength.GetSize();z++)
	{
		try { m_pHistogram[z] = new CDF(); } catch(...) { m_pHistogram[z] = NULL; }
		if (m_pHistogram[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		m_pHistogram[z]->m_iResolution = m_iHistoRes;
		m_pHistogram[z]->m_fMinVal = 0;

		if (m_bCSV)
		{
			m_pHistogram[z]->SetLabelY("Occurrence");

			if (m_bDensity)
			{
				if (m_bUniform)
				{
					m_pHistogram[z]->m_fMaxVal = g_fBoxX*g_fBoxY*g_fBoxZ/1e9 / m_pParent->m_faCuboidVolume[z];
					if (m_bMass)
					{
						m_pHistogram[z]->SetLabelX("Relative mass density");
					} else
					{
						m_pHistogram[z]->SetLabelX("Relative particle density");
					}
				} else
				{
					if (m_bMass)
					{
						m_pHistogram[z]->SetLabelX("Mass density [g/cm^3]");
						m_pHistogram[z]->m_fMaxVal = m_fMaxVal / m_pParent->m_faCuboidVolume[z] * 0.0016605;
					} else
					{
						m_pHistogram[z]->SetLabelX("Particle density [1/nm^3]");
						m_pHistogram[z]->m_fMaxVal = m_fMaxVal / m_pParent->m_faCuboidVolume[z];
					}
				}
			} else
			{
				m_pHistogram[z]->m_fMaxVal = m_fMaxVal;
				if (m_bCubicRoot)
				{
					if (m_bMass)
						m_pHistogram[z]->SetLabelX("Cubic Root of Mass [g/mol]");
							else m_pHistogram[z]->SetLabelX("Cubic Root of Particle count");
				} else
				{
					if (m_bMass)
						m_pHistogram[z]->SetLabelX("Mass [g/mol]");
							else m_pHistogram[z]->SetLabelX("Particle count");
				}
			}
		}

		m_pHistogram[z]->m_bLeft = true;

		m_pHistogram[z]->Create();
	}

	if (m_bEmpty2D)
	{
		m_pEmptyBinTD = new C2DF();
		m_pEmptyBinTD->m_iRes[0] = m_iEmpty2DRes;
		m_pEmptyBinTD->m_iRes[1] = m_pParent->m_iaCuboidLength.GetSize();
		m_pEmptyBinTD->m_fMinVal[0] = 0;
		m_pEmptyBinTD->m_fMaxVal[0] = m_iEmpty2DRes * m_iEmpty2DStride * g_iStride * g_fTimestepLength / 1000.0;
		m_pEmptyBinTD->m_fMinVal[1] = 0;
		m_pEmptyBinTD->m_fMaxVal[1] = m_pParent->m_iaCuboidLength.GetSize() * g_fBoxX / m_pParent->m_iRes;
		m_pEmptyBinTD->SetLabelX("Simulation Time [ps]");
		m_pEmptyBinTD->SetLabelY("Bin Size [pm]");
		m_pEmptyBinTD->Create();
	}
}


void CMicroHet::Bin(CMicroHetObservation *o, int zi, unsigned long *zero)
{
	unsigned long z;

	*zero = 0;
	for (z=0;z<m_iResTri;z++)
	{
		o->m_pHistogram[zi]->AddToBinInt_Fast(m_pIBin[z]);

		if (m_pIBin[z] == 0)
			(*zero)++;
	}
}


void CMicroHet::Process(CTimeStep *ts)
{
	unsigned long zero;
	double dz, dzl, dt;
	int z, z2, z3, z4, z5, cl, zi;
	int px, py, pz;
	int ival;
	CMicroHetObservation *o;
	CAtomGroup *ag;
	CMolecule *m;
	CSingleMolecule *sm;
	CxVector3 *v;

	ival = 0;
	dz = 0;
	for (zi=0;zi<m_iaCuboidLength.GetSize();zi++)
	{
		cl = m_iaCuboidLength[zi];

		for (z=0;z<m_oaObservations.GetSize();z++)
		{
			Zero();

			o = (CMicroHetObservation*)m_oaObservations[z];

			for (z2=0;z2<o->m_oaAtomGroups.GetSize();z2++)
			{
				ag = (CAtomGroup*)o->m_oaAtomGroups[z2];

				for (z3=0;z3<ag->m_oaAtoms.GetSize();z3++)
				{
					if (o->m_bMass)
					{
						if (ag->m_baRealAtomType[z3] == g_iVirtAtomType)
						{
							ival = 0;
							m = ag->m_pMolecule;
							for (z4=0;z4<m->m_baAtomIndex.GetSize();z4++)
								ival += (int)((m->m_waAtomCount[z4] * ((CAtom*)g_oaAtoms[m->m_baAtomIndex[z4]])->m_pElement->m_fMass) + 0.5);
						} else ival = (int)(((CAtom*)g_oaAtoms[ag->m_baRealAtomType[z3]])->m_pElement->m_fMass+0.5);
					}

					for (z4=0;z4<((CxIntArray*)ag->m_oaAtoms[z3])->GetSize();z4++)
					{
						for (z5=0;z5<ag->m_pMolecule->m_laSingleMolIndex.GetSize();z5++)
						{
							sm = (CSingleMolecule*)g_oaSingleMolecules[ag->m_pMolecule->m_laSingleMolIndex[z5]];
							v = &ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[ag->m_baAtomType[z3]])->GetAt(z4)];

							px = (int)(v->GetAt(0) / g_fBoxX * m_iRes);
							py = (int)(v->GetAt(1) / g_fBoxY * m_iRes);
							pz = (int)(v->GetAt(2) / g_fBoxZ * m_iRes);

							if (o->m_bMass)
								ProcessAtom(px,py,pz,ival,cl);
									else ProcessAtom(px,py,pz,0,cl);

						} // END FOR Z5
					} // END FOR Z4
				} // END FOR Z3
			} // END FOR Z2

			Bin(o,zi,&zero);

			dzl = dz; // Last bin size
			dz = (double)zero / m_iResTri;

	//		mprintf("\nzi: %d, zero=%d, dz=%G, dzl=%G.",zi,zero,dz,dzl);

			if (o->m_bEmpty2D)
				o->m_pEmptyBinTD->AddToBin_IntY(((double)m_iCounter)*g_iStride*g_fTimestepLength/1000.0,zi,dz);

	//		mprintf("\n%d; %G; %d; %G",m_iCounter,((double)m_iCounter)*g_iStride*g_fTimestepLength/1000.0,zi,dz);

			if (dz < 0.5)
			{
				if (zi == 0)
				{
					eprintf("\nUnimplemented (A).");
				} else
				{
					if (dzl >= 0.5)
					{
						dt = zi-(0.5-dz)/(dzl-dz);
						o->m_faEmptyBinHalf.Add(dt);
//						o->m_faEmptyBinHalf2.Add(zi);
				//		mprintf("\n### %d %f ###",zi,dt);
					}
				}
			}

			if (dz == 0)
			{
				if (zi == 0)
				{
					eprintf("\nUnimplemented (A).");
				} else
				{
					if (dzl != 0)
						o->m_faEmptyBinZero.Add(zi-1);
				}
			}
		} // END FOR Z
	} // END FOR ZI

	m_iCounter++;
}


void CMicroHet::ProcessAtom(int px, int py, int pz, int mass, int cl)
{
	int /*ch,*/ chq, /*ix,*/ iy, iz, zzq, zyq, zx, zy, zz, ty, tz;
	MH_BIN_TYPE *ib2;

//	ch = cl/2;
	chq = (cl*cl)/4;

	if (mass == 0)
	{
		if (m_bSphere)
		{
			iz = pz;
			for (zz=0;zz<cl;zz++)
			{
				zzq = zz*zz - zz*cl + 2*chq;
				iz++;
				if (iz >= m_iRes)
					iz = 0;
				tz = iz*m_iResSqr;
				iy = py;
				for (zy=0;zy<cl;zy++)
				{
					zyq = zzq + zy*zy - zy*cl;
					iy++;
					if (iy >= m_iRes)
						iy = 0;
					ty = iy*m_iRes;

					ib2 = &m_pIBin[tz+ty+px];
					if (px+cl >= m_iRes)
					{
						for (zx=0;zx<m_iRes-px;zx++)
						{
							if (zyq + zx*zx - zx*cl <= 0)
								(*ib2)++;
							ib2++;
						}
						ib2 -= m_iRes;
						for (;zx<cl;zx++)
						{
							if (zyq + zx*zx - zx*cl <= 0)
								(*ib2)++;
							ib2++;
						}
					} else
					{
						for (zx=0;zx<cl;zx++)
						{
							if (zyq + zx*zx - zx*cl <= 0)
								(*ib2)++;
							ib2++;
						}
					}
				}
			}
/*			iz = pz;
			for (zz=-ch;zz<ch;zz++)
			{
				zzq = zz*zz;
				iz++;
				if (iz >= m_iRes)
					iz = 0;
				tz = iz*m_iResSqr;
				iy = py;
				for (zy=-ch;zy<ch;zy++)
				{
					zyq = zy*zy;
					iy++;
					if (iy >= m_iRes)
						iy = 0;
					ty = iy*m_iRes;
					ix = px;
					for (zx=-ch;zx<ch;zx++)
					{
						ix++;
						if (ix >= m_iRes)
							ix = 0;
						if (zzq + zyq + zx*zx <= chq)
							m_pIBin[tz+ty+ix]++;
					}
				}
			}*/
		} else // ELSE IF NOT SPHERE
		{
			iz = pz;
			for (zz=0;zz<cl;zz++)
			{
				iz++;
				if (iz >= m_iRes)
					iz = 0;
				tz = iz*m_iResSqr;
				iy = py;
				for (zy=0;zy<cl;zy++)
				{
					iy++;
					if (iy >= m_iRes)
						iy = 0;
					ty = iy*m_iRes;

					ib2 = &m_pIBin[tz+ty+px];
					if (px+cl >= m_iRes)
					{
						for (zx=0;zx<m_iRes-px;zx++)
						{
							(*ib2)++;
							ib2++;
						}
						ib2 -= m_iRes;
						for (;zx<cl;zx++)
						{
							(*ib2)++;
							ib2++;
						}
					} else
					{
						for (zx=0;zx<cl;zx++)
						{
							(*ib2)++;
							ib2++;
						}
					}

				}
			}
		} // END IF NOT SPHERE*/
	} else // ELSE IF MASS
	{
		if (m_bSphere)
		{
/*			iz = pz;
			for (zz=0;zz<cl;zz++)
			{
				zzq = zz*zz - zz*cl + cl*cl;
				iz++;
				if (iz >= m_iRes)
					iz = 0;
				tz = iz*m_iResSqr;
				iy = py;
				for (zy=0;zy<cl;zy++)
				{
					zyq = zy*zy - zy*cl + cl*cl;
					iy++;
					if (iy >= m_iRes)
						iy = 0;
					ty = iy*m_iRes;

					ix = px;
					for (zx=0;zx<cl;zx++)
					{
						ix++;
						if (ix >= m_iRes)
							ix = 0;
						if (zzq + zyq + zx*zx - zx*cl <= chq)
							m_pIBin[tz+ty+ix] += mass;
					}

				}
			}*/

			iz = pz;
			for (zz=0;zz<cl;zz++)
			{
				zzq = zz*zz - zz*cl + 2*chq;
				iz++;
				if (iz >= m_iRes)
					iz = 0;
				tz = iz*m_iResSqr;
				iy = py;
				for (zy=0;zy<cl;zy++)
				{
					zyq = zzq + zy*zy - zy*cl;
					iy++;
					if (iy >= m_iRes)
						iy = 0;
					ty = iy*m_iRes;

					ib2 = &m_pIBin[tz+ty+px];
					if (px+cl >= m_iRes)
					{
						for (zx=0;zx<m_iRes-px;zx++)
						{
							if (zyq + zx*zx - zx*cl <= 0)
								(*ib2) += mass;
							ib2++;
						}
						ib2 -= m_iRes;
						for (;zx<cl;zx++)
						{
							if (zyq + zx*zx - zx*cl <= 0)
								(*ib2) += mass;
							ib2++;
						}
					} else
					{
						for (zx=0;zx<cl;zx++)
						{
							if (zyq + zx*zx - zx*cl <= 0)
								(*ib2) += mass;
							ib2++;
						}
					}
				}
			}

/*			iz = pz;
			for (zz=-ch;zz<ch;zz++)
			{
				zzq = zz*zz;
				iz++;
				if (iz >= m_iRes)
					iz = 0;
				tz = iz*m_iResSqr;
				iy = py;
				for (zy=-ch;zy<ch;zy++)
				{
					zyq = zy*zy;
					iy++;
					if (iy >= m_iRes)
						iy = 0;
					ty = iy*m_iRes;

					ix = px;
					for (zx=-ch;zx<ch;zx++)
					{
						ix++;
						if (ix >= m_iRes)
							ix = 0;
						if (zzq + zyq + zx*zx <= chq)
							m_pIBin[tz+ty+ix] += mass;
					}

				}
			}*/

		} else // ELSE IF NOT SPHERE
		{
			iz = pz;
			for (zz=0;zz<cl;zz++)
			{
				iz++;
				if (iz >= m_iRes)
					iz = 0;
				tz = iz*m_iResSqr;
				iy = py;
				for (zy=0;zy<cl;zy++)
				{
					iy++;
					if (iy >= m_iRes)
						iy = 0;
					ty = iy*m_iRes;

				/*	ix = px;
					for (zx=0;zx<cl;zx++)
					{
						ix++;
						if (ix >= m_iRes)
							ix = 0;
						ib[tz+ty+ix] += mass;
					}*/
					
					ib2 = &m_pIBin[tz+ty+px];
					if (px+cl >= m_iRes)
					{
						for (zx=0;zx<m_iRes-px;zx++)
						{
							(*ib2) += mass;
							ib2++;
						}
						ib2 -= m_iRes;
						for (;zx<cl;zx++)
						{
							(*ib2) += mass;
							ib2++;
						}
					} else
					{
						for (zx=0;zx<cl;zx++)
						{
							(*ib2) += mass;
							ib2++;
						}
					}

			/*		ib2 = &ib[tz+ty+px];
					for (zx=0;zx<cl;zx++)
					{
						*ib2 += mass;
						ib2++;
					}*/
				}
			}
		} // END IF NOT SPHERE
	} // END IF MASS
}


void CMicroHetObservation::BuildCDF(int mode, int obs, const char *multibuf)
{
	double d;
	int zi, z2, z;
	char buf[256];
	FILE *a;

	mprintf("    Creating Microheterogeneity 2D plot (%s)...\n",(mode==0)?"density":((mode==1)?"abs. value":"mode 3"));

	try { m_pCDF = new C2DF(); } catch(...) { m_pCDF = NULL; }
	if (m_pCDF == NULL) NewException((double)sizeof(C2DF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	m_pCDF->m_fMinVal[0] = 0;

	if (m_bEdgeUniform)
		m_pCDF->m_fMaxVal[0] = 1.0;
			else m_pCDF->m_fMaxVal[0] = m_pParent->m_iaCuboidLength[m_pParent->m_iaCuboidLength.GetSize()-1]*g_fBoxX/m_pParent->m_iRes;

	m_pCDF->m_iRes[0] = m_pParent->m_iaCuboidLength[m_pParent->m_iaCuboidLength.GetSize()-1] / (m_pParent->m_iaCuboidLength[1] - m_pParent->m_iaCuboidLength[0]) + 1;
	m_iCDFOffset = m_pParent->m_iaCuboidLength[0] / (m_pParent->m_iaCuboidLength[1] - m_pParent->m_iaCuboidLength[0]);

	m_pCDF->m_fMinVal[1] = 0;
	if (mode == 0)
	{
		if (m_bCDF_Dens_Uniform)
		{
			m_pCDF->m_fMaxVal[1] = m_fRangeFac;
			if (m_bMass)
			{
				m_pCDF->SetLabelY("Relative mass density");
			} else
			{
				m_pCDF->SetLabelY("Relative particle density");
			}
		} else
		{
			if (m_bMass)
			{
				m_pCDF->m_fMaxVal[1] = m_fMaxVal / g_fBoxX / g_fBoxY / g_fBoxZ * 1e9 * m_fRangeFac * 0.0016605;
				m_pCDF->m_fMaxVal[1] = (((int)(m_pCDF->m_fMaxVal[1]*10.0))+1)/10.0; // Aufrunden
				m_pCDF->SetLabelY("Mass density [g/cm^3]");
			} else
			{
				m_pCDF->m_fMaxVal[1] = m_fMaxVal / g_fBoxX / g_fBoxY / g_fBoxZ * 1e9 * m_fRangeFac;
				m_pCDF->SetLabelY("Particle density [1/nm^3]");
			}
		}
	} else if (mode == 1)
	{
		m_pCDF->m_fMaxVal[1] = m_fMaxVal;
		if (m_bCubicRoot)
		{
			if (m_bMass)
				m_pCDF->SetLabelY("Cubic Root of Mass [g/mol]");
					else m_pCDF->SetLabelY("Cubic Root of Particle count");
		} else
		{
			if (m_bMass)
				m_pCDF->SetLabelY("Mass [g/mol]");
					else m_pCDF->SetLabelY("Particle count");
		}
	} else
	{
		if (m_bMass)
		{
			m_pCDF->m_fMaxVal[1] = m_fMaxVal / g_fBoxX / g_fBoxY / g_fBoxZ * 1e9 * 0.0016605;
			m_pCDF->m_fMaxVal[1] = (((int)(m_pCDF->m_fMaxVal[1]*10.0))+1)/10.0; // Aufrunden
			m_pCDF->SetLabelY("Mass bla");
		} else
		{
			m_pCDF->m_fMaxVal[1] = m_fMaxVal / g_fBoxX / g_fBoxY / g_fBoxZ * 1e9;
			m_pCDF->SetLabelY("Particle count bla");
		}
	}
	m_pCDF->m_iRes[1] = m_iCDFRes;

	if (m_pParent->m_bSphere)
	{
		if (m_bEdgeUniform)
			m_pCDF->SetLabelX("Relative sphere diameter");
				else m_pCDF->SetLabelX("Sphere diameter [pm]");
	} else
	{
		if (m_bEdgeUniform)
			m_pCDF->SetLabelX("Relative cube edge length");
				else m_pCDF->SetLabelX("Cube edge length [pm]");
	}

	m_pCDF->SetLabelZ("Occurrence");

	m_pCDF->m_iPlotType = 2;
	m_pCDF->m_iSmoothGrade = 0;
	m_pCDF->m_fPlotExp = 0.15;
	m_pCDF->m_iInterpolationOrder = 1;
	m_pCDF->m_iPlotPixel = 768;
	m_pCDF->m_bContourLines = false;
	m_pCDF->m_iGPInterpolation = 1;

	m_pCDF->Create();

	for (zi=0;zi<m_pParent->m_iaCuboidLength.GetSize();zi++)
	{
		if (mode == 2) // "Mode 3"
		{
			if (m_bMass)
				m_pHistogram[zi]->m_fMaxVal = m_fMaxVal / pow(m_pParent->m_faCuboidVolume[zi],2.0/3.0) * 0.0016605 / pow(m_pParent->m_faCuboidVolume[m_pParent->m_iaCuboidLength.GetSize()-1],1.0/3.0);
					else m_pHistogram[zi]->m_fMaxVal = m_fMaxVal / pow(m_pParent->m_faCuboidVolume[zi],2.0/3.0) / pow(m_pParent->m_faCuboidVolume[m_pParent->m_iaCuboidLength.GetSize()-1],1.0/3.0);

			for (z2=0;z2<m_pHistogram[zi]->m_iResolution;z2++)
				m_pCDF->AddToBin_IntX(zi+m_iCDFOffset,((double)z2)*m_pHistogram[zi]->m_fMaxVal/m_pHistogram[zi]->m_iResolution,m_pHistogram[zi]->m_pBin[z2]);
		} else if (mode == 1) // Absolute Value
		{
			m_pHistogram[zi]->m_fMaxVal = m_fMaxVal;

			if (m_bCubicRoot)
			{
				d = pow(m_fMaxVal,2.0/3.0);
				for (z2=0;z2<m_pHistogram[zi]->m_iResolution;z2++)
					m_pCDF->AddToBin_IntX(zi+m_iCDFOffset,pow(((double)z2)*m_pHistogram[zi]->m_fMaxVal/m_pHistogram[zi]->m_iResolution,1.0/3.0)*d,m_pHistogram[zi]->m_pBin[z2]);
			} else
			{
				for (z2=0;z2<m_pHistogram[zi]->m_iResolution;z2++)
					m_pCDF->AddToBin_IntX(zi+m_iCDFOffset,((double)z2)*m_pHistogram[zi]->m_fMaxVal/m_pHistogram[zi]->m_iResolution,m_pHistogram[zi]->m_pBin[z2]);
			}
		} else // Density
		{
			if (m_bCDF_Dens_Uniform)
			{
				m_pHistogram[zi]->m_fMaxVal = g_fBoxX*g_fBoxY*g_fBoxZ/1e9 / m_pParent->m_faCuboidVolume[zi];
			} else
			{
				if (m_bMass)
					m_pHistogram[zi]->m_fMaxVal = m_fMaxVal / m_pParent->m_faCuboidVolume[zi] * 0.0016605;
						else m_pHistogram[zi]->m_fMaxVal = m_fMaxVal / m_pParent->m_faCuboidVolume[zi];
			}

			for (z2=0;z2<m_pHistogram[zi]->m_iResolution;z2++)
				m_pCDF->AddToBin_IntX(zi+m_iCDFOffset,((double)z2)*m_pHistogram[zi]->m_fMaxVal/m_pHistogram[zi]->m_iResolution,m_pHistogram[zi]->m_pBin[z2]);
		}
	}

	m_pCDF->NormalizeBinIntegral(1000000.0);

	if (g_pDatabase->GetBool("/PLOT2D/FORMATS/WRITE_TRIPLES"))
	{
		sprintf(buf,"microhet_cdf_obs%d%s_%s_triples.csv",obs+1,multibuf,(mode==0)?"dens":((mode==1)?"abs":"mode3"));
		mprintf("      Saving 2D Microheterogeneity triples as %s ...\n",buf);
		m_pCDF->Write("",buf,"");
	}

	if (g_pDatabase->GetBool("/PLOT2D/FORMATS/WRITE_MATRIX"))
	{
		sprintf(buf,"microhet_cdf_obs%d%s_%s_matrix.csv",obs+1,multibuf,(mode==0)?"dens":((mode==1)?"abs":"mode3"));
		mprintf("      Saving 2D Microheterogeneity matrix as %s ...\n",buf);
		m_pCDF->WriteCSV("",buf,"");
	}

	if (g_pDatabase->GetBool("/PLOT2D/FORMATS/WRITE_MATHEMATICA"))
	{
		sprintf(buf,"microhet_cdf_obs%d%s_%s.nb",obs+1,multibuf,(mode==0)?"dens":((mode==1)?"abs":"mode3"));
		mprintf("      Saving Microheterogeneity Mathematica notebook as %s ...\n",buf);
		m_pCDF->WriteMathematicaNb("",buf,"",false);
	}

	if (g_pDatabase->GetBool("/PLOT2D/FORMATS/WRITE_GNUPLOT"))
	{
		sprintf(buf,"microhet_cdf_obs%d%s_%s",obs+1,multibuf,(mode==0)?"dens":((mode==1)?"abs":"mode3"));
		mprintf("      Saving Microheterogeneity Gnuplot Input as %s.gp ...\n",buf);
		m_pCDF->WriteGnuplotInput("",buf,"",false);
	}

	if (mode == 1)
	{
		sprintf(buf,"microhet_empty_td%s.csv",multibuf);
		a = OpenFileWrite(buf,true);
		mfprintf(a,"# Time [ps];  empty bin 50%c size;  empty bin 0%c size\n",'%','%','%');
		for (z=0;z<m_faEmptyBinHalf.GetSize();z++)
			mfprintf(a,"%G;  %G;  %G\n",z*g_iStride*g_fTimestepLength/1000.0,m_faEmptyBinHalf[z]*g_fBoxX/m_pParent->m_iRes,m_faEmptyBinZero[z]*g_fBoxX/m_pParent->m_iRes);
		fclose(a);

		if (m_bEmpty2D)
		{
			m_pEmptyBinTD->MultiplyBin(1.0/m_pParent->m_iCounter);

			m_pEmptyBinTD->NormalizeBinIntegral(1000000.0);

			if (g_pDatabase->GetBool("/PLOT2D/FORMATS/WRITE_TRIPLES"))
			{
				sprintf(buf,"microhet_empty2d_td_obs%d%s_triples.csv",obs+1,multibuf);
				mprintf("      Saving 2D Empty Bin triples as %s ...\n",buf);
				m_pEmptyBinTD->Write("",buf,"");
			}

			if (g_pDatabase->GetBool("/PLOT2D/FORMATS/WRITE_MATRIX"))
			{
				sprintf(buf,"microhet_empty2d_td_obs%d%s_matrix.csv",obs+1,multibuf);
				mprintf("      Saving 2D Empty Bin matrix as %s ...\n",buf);
				m_pEmptyBinTD->WriteCSV("",buf,"");
			}

			if (g_pDatabase->GetBool("/PLOT2D/FORMATS/WRITE_MATHEMATICA"))
			{
				sprintf(buf,"microhet_empty2d_td_obs%d%s.nb",obs+1,multibuf);
				mprintf("      Saving Empty Bin Mathematica notebook as %s ...\n",buf);
				m_pEmptyBinTD->WriteMathematicaNb("",buf,"",false);
			}

			if (g_pDatabase->GetBool("/PLOT2D/FORMATS/WRITE_GNUPLOT"))
			{
				sprintf(buf,"microhet_empty2d_td_obs%d%s",obs+1,multibuf);
				mprintf("      Saving Empty Bin Gnuplot Input as %s.gp ...\n",buf);
				m_pEmptyBinTD->WriteGnuplotInput("",buf,"",false);
			}
		}
	}
}


void CMicroHetObservation::BuildName()
{
	char buf[1024];
	int z;

	buf[0] = 0;

	for (z=0;z<m_oaAtomGroups.GetSize();z++)
	{
		strcat(buf,((CAtomGroup*)m_oaAtomGroups[z])->m_pMolecule->m_sName);
		strcat(buf,": ");
		strcat(buf,((CAtomGroup*)m_oaAtomGroups[z])->m_sName);
		if (z < m_oaAtomGroups.GetSize()-1)
			strcat(buf,"; ");
	}

	try { m_sName = new char[strlen(buf)+1]; } catch(...) { m_sName = NULL; }
	if (m_sName == NULL) NewException((double)sizeof(char)*(strlen(buf)+1),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	strcpy(m_sName,buf);
}


void CMicroHetObservation::BuildZeroPlot(const char *s, bool absolute)
{
	int zi, z, z2, ti;
	double d;
	CDF *df;
	char buf[256];

	try { df = new CDF(); } catch(...) { df = NULL; }
	if (df == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	if (absolute)
	{
		df->m_fMinVal = (double)m_pParent->m_iaCuboidLength[0] / (double)m_pParent->m_iRes * g_fBoxX;
		df->m_fMaxVal = ((double)(m_pParent->m_iaCuboidLength.GetSize()+1.0) / m_pParent->m_iaCuboidLength.GetSize()) * (double)m_pParent->m_iaCuboidLength[m_pParent->m_iaCuboidLength.GetSize()-1] / (double)m_pParent->m_iRes * g_fBoxX;
	} else
	{
		df->m_fMinVal = (double)m_pParent->m_iaCuboidLength[0] / (double)m_pParent->m_iRes;
		df->m_fMaxVal = (double)m_pParent->m_iaCuboidLength[m_pParent->m_iaCuboidLength.GetSize()-1] / (double)m_pParent->m_iRes;
	}

	df->m_iResolution = m_pParent->m_iaCuboidLength.GetSize();
	df->m_bLeft = true;

	ti = 0;
	for (z2=0;z2<m_iHistoRes;z2++)
	{
		for (zi=0;zi<m_pParent->m_iaCuboidLength.GetSize();zi++)
		{
			if (m_pHistogram[zi]->m_pBin[z2] != 0)
			{
				ti++;
				goto _next;
			}
		}
_next:;
	}

	df->CreateMulti(ti);

	if (absolute)
		df->SetLabelX("Cube edge length [pm]");
			else df->SetLabelX("Relative cube edge length");

	df->SetLabelY("Total percentage");

	ti = 0;
	for (z2=0;z2<m_iHistoRes;z2++)
	{
		for (zi=0;zi<m_pParent->m_iaCuboidLength.GetSize();zi++)
			if (m_pHistogram[zi]->m_pBin[z2] != 0)
				goto _found;
		continue;
_found:
		if (m_bMass)
			sprintf(buf,"Mass %d percentage",z2);
				else sprintf(buf,"Count %d percentage",z2);
		df->SetLabelMulti(ti,buf);
		for (zi=0;zi<m_pParent->m_iaCuboidLength.GetSize();zi++)
		{
			d = 0;
			for (z=0;z<m_iHistoRes;z++)
				d += m_pHistogram[zi]->m_pBin[z];
			df->AddToBin_Multi_Int(ti,zi,m_pHistogram[zi]->m_pBin[z2] / d * 100.0);
		}
		ti++;
	}

	mprintf("      Saving entry plot as %s.csv ...\n",s);
	df->WriteMulti("",s,".csv");
/*	mprintf("      Saving entry plot AGR file as %s.agr ...\n",s);
	df->WriteMultiAgr("",s,".agr","Bin Value Percentage",false);*/

	delete df;
}


void CMicroHetObservation::BuildSlices(char *s)
{
	int zi, z2;
	CDF *df;
	double d;
	char buf[256];

	try { df = new CDF(); } catch(...) { df = NULL; }
	if (df == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	df->m_fMinVal = 0;
	df->m_fMaxVal = /*m_fMaxVal*/m_iHistoRes;
	df->m_iResolution = m_iHistoRes;
	df->m_bLeft = true;

	df->CreateMulti(m_pParent->m_iaCuboidLength.GetSize());

	df->SetLabelY("Total percentage");

	if (m_bMass)
		df->SetLabelX("Mass [g/mol]");
			else df->SetLabelX("Particle count");

	d = 0;
	for (zi=0;zi<m_pHistogram[0]->m_iResolution;zi++)
		d += m_pHistogram[0]->m_pBin[zi];

	for (z2=0;z2<m_pParent->m_iaCuboidLength.GetSize();z2++)
	{
		sprintf(buf,"%.1f pm",(double)m_pParent->m_iaCuboidLength[z2]/m_pParent->m_iRes*g_fBoxX);
		df->SetLabelMulti(z2,buf);
		for (zi=0;zi<m_pHistogram[z2]->m_iResolution;zi++)
			df->AddToBin_Multi_Int(z2,zi,m_pHistogram[z2]->m_pBin[zi]/d*100.0);
	}

	for (zi=0;zi<df->m_iResolution;zi++)
		df->m_pBin[zi] /= m_pParent->m_iaCuboidLength.GetSize();

	mprintf("      Saving Microheterogeneity plot as %s.csv ...\n",s);
	df->WriteMulti("",s,".csv");
	mprintf("      Saving Microheterogeneity plot AGR file as %s.agr ...\n",s);
	df->WriteMultiAgr("",s,".agr","Microheterogeneity plot",false);

	delete df;
}
