/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2013 Martin Brehm
                  2012-2013 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "vorowrapper.h"
#include "voro++.h"
#include "tools.h"
#include "globalvar.h"
#include "maintools.h"


CVoroWrapper::CVoroWrapper()
{
	m_pContainer = NULL;
}


CVoroWrapper::~CVoroWrapper()
{
}


void CVoroWrapper::Build(CTimeStep *ts)
{
	int ijk, q, z, z2, z3, id, i, faces, co;
	CVoroAtom *va;
	CVoroMolecule *vm;
	double /**pp,*/ ta, surf, volume, vg;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	int *nbtemp;

	try { nbtemp = new int[m_oaVoroAtoms.GetSize()]; } catch(...) { nbtemp = NULL; }
	if (nbtemp == NULL) NewException((double)m_oaVoroAtoms.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		vm->tempfaces = 0;
		vm->tempsurf = 0;
		vm->tempvol = 0;
		for (z2=0;z2<m_oaVoroMolecules.GetSize();z2++)
			vm->m_pNbhTempMol[z2] = 0;
		for (z2=0;z2<g_iGesAtomCount;z2++)
			vm->m_pNbhTempAtoms[z2] = 0;
	}

	for (z=0;z<g_iGesAtomCount;z++)
		m_pAtomTouched[z] = false;

	co = 0;
	vg = 0;
	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				co++;

				ijk=vl.ijk;
				q=vl.q;
	//			pp=m_pContainer->p[ijk]+m_pContainer->ps*q;

				id = m_pContainer->id[ijk][q];
				c.face_areas(fa);
				c.face_orders(fo);
				c.neighbors(nb);

				m_pAtomTouched[id] = true;

				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				surf = c.surface_area();
				faces = c.number_of_faces();
				volume = c.volume();

				vg += volume;

				va->m_pVolume->AddToBin(volume*1000.0);
				va->m_pSurfaceArea->AddToBin(surf*100.0);
				va->m_pFaces->AddToBin_Int(faces);
				va->m_pMaxRadius->AddToBin(sqrt(c.max_radius_squared())*1000.0);
				va->m_pAVRatio->AddToBin(10.6347231054330961*volume/pow(surf,1.5));

				vm->tempvol += volume*1000.0;

				for (z2=0;z2<m_oaVoroAtoms.GetSize();z2++)
				{
					nbtemp[z2] = 0;
				}

				for (z2=0;z2<m_oaVoroMolecules.GetSize();z2++)
				{
					va->m_pNbhTempMol[z2] = 0;
				}

				ta = 0;
				for (z2=0;z2<faces;z2++)
				{
					va->m_pFaceOrders->AddToBin_Int(fo[z2]);

					if (m_pAssignMolecules[id] != m_pAssignMolecules[nb[z2]])
					{
						nbtemp[m_pAssignAtoms[nb[z2]]]++;
						va->m_pNbhDistAtoms[m_pAssignAtoms[nb[z2]]]->AddToBin(ts->FoldedDistance(id,nb[z2]));

						((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]])->m_pNbhDistAtoms[m_pAssignAtoms[id]]->AddToBin(ts->FoldedDistance(id,((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]])->m_iCenterOffset));

						if (vm->m_pNbhTempAtoms[nb[z2]] == 0)
						{
							vm->m_pNbhTempAtoms[nb[z2]]++;
							vm->m_pNbhDistAtoms[m_pAssignAtoms[nb[z2]]]->AddToBin(ts->FoldedDistance(id,((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]])->m_iCenterOffset));
						}

						if (vm->m_pNbhTempMol[m_pAssignMolecules[nb[z2]]] == 0)
						{
							vm->m_pNbhTempMol[m_pAssignMolecules[nb[z2]]]++;
							vm->m_pNbhDistMolecules[m_pAssignMoleculeTypes[nb[z2]]]->AddToBin(ts->FoldedDistance(((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]])->m_iCenterOffset,((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]])->m_iCenterOffset));
						}

						if (va->m_pNbhTempMol[m_pAssignMolecules[nb[z2]]] == 0)
							va->m_pNbhDistMolecules[m_pAssignMoleculeTypes[nb[z2]]]->AddToBin(ts->FoldedDistance(id,((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]])->m_iCenterOffset));

						va->m_pNbhTempMol[m_pAssignMolecules[nb[z2]]]++;
						vm->tempsurf += fa[z2]*100.0;
						vm->tempfaces++;
						ta += fa[z2]*100.0;
					}
				}

				va->m_pExposedSurface->AddToBin(ta);
				va->m_pExposedSurfacePerc->AddToBin(ta/surf);

				for (z2=0;z2<m_oaVoroAtoms.GetSize();z2++)
				{
					va->m_pNbhAtoms[z2]->AddToBin_Int(nbtemp[z2]);
				}

				for (z2=0;z2<g_oaMolecules.GetSize();z2++)
				{
					i = 0;
					for (z3=0;z3<m_oaVoroMolecules.GetSize();z3++)
					{
						if (((CVoroMolecule*)m_oaVoroMolecules[z3])->m_iMolecule != z2)
							continue;
						if (va->m_pNbhTempMol[z3] != 0)
							i++;
					}
					va->m_pNbhMolecules[z2]->AddToBin_Int(i);
				}
			} else
			{
				mprintf("\nWarning: Compute_Cell failed.");
			}
		} while (vl.inc());
	}

	for (z=0;z<g_iGesAtomCount;z++)
	{
		if (!m_pAtomTouched[z])
			mprintf("\nWarning: Atom %s(%d) %s%d not touched by voronoi!",((CMolecule*)g_oaMolecules[g_waAtomMolIndex[z]])->m_sName,g_laAtomSMIndex[z]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,g_waAtomMolNumber[z]+1);
	}

	if ((co != g_iGesAtomCount) || (fabs(vg*1000.0-g_fBoxX*g_fBoxY*g_fBoxZ/1000000.0)/(g_fBoxX*g_fBoxY*g_fBoxZ/1000000.0) > 0.001))
	{
		mprintf("\nVoro++: Problem. %d/%d atoms touched. V=%f/%f A^3.",co,g_iGesAtomCount,vg*1000.0,g_fBoxX*g_fBoxY*g_fBoxZ/1000000.0);
	}

	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		vm->m_pVolume->AddToBin(vm->tempvol);
		vm->m_pSurfaceArea->AddToBin(vm->tempsurf);
		vm->m_pFaces->AddToBin_Int(vm->tempfaces);
		vm->m_pAVRatio->AddToBin(10.6347231054330961*vm->tempvol/pow(vm->tempsurf,1.5));

		for (z2=0;z2<m_oaVoroAtoms.GetSize();z2++)
		{
			i = 0;
			for (z3=0;z3<g_iGesAtomCount;z3++)
			{
				if (m_pAssignAtoms[z3] != z2)
					continue;
				if (vm->m_pNbhTempAtoms[z3] != 0)
					i++;
			}
			vm->m_pNbhAtoms[z2]->AddToBin_Int(i);
		}

		for (z2=0;z2<g_oaMolecules.GetSize();z2++)
		{
			i = 0;
			for (z3=0;z3<m_oaVoroMolecules.GetSize();z3++)
			{
				if (((CVoroMolecule*)m_oaVoroMolecules[z3])->m_iMolecule != z2)
					continue;
				if (vm->m_pNbhTempMol[z3] != 0)
					i++;
			}
			vm->m_pNbhMolecules[z2]->AddToBin_Int(i);
		}
	}

	delete[] nbtemp;
	delete m_pContainer;
}


void CVoroWrapper::Init()
{
	int z, z2, z3, z4, i;
	CVoroAtom *va;
	CVoroMolecule *vm;
	CMolecule *m;
	CSingleMolecule *sm;

	m_iBlocksX = (int)(pow(g_iGesAtomCount/optimal_particles/g_fBoxX/g_fBoxY/g_fBoxZ,1/3.0)*g_fBoxX)+1;
	m_iBlocksY = (int)(pow(g_iGesAtomCount/optimal_particles/g_fBoxX/g_fBoxY/g_fBoxZ,1/3.0)*g_fBoxY)+1;
	m_iBlocksZ = (int)(pow(g_iGesAtomCount/optimal_particles/g_fBoxX/g_fBoxY/g_fBoxZ,1/3.0)*g_fBoxZ)+1;

	m_fBoxDens = g_iGesAtomCount / g_fBoxX / g_fBoxY / g_fBoxZ * 1000000.0; // Particles / Angstrom^3

	if (m_bVoroStat || m_bSurfCover || m_bWritePOV || g_bVoid)
	{
		try { m_pAssignAtoms = new long[g_iGesAtomCount]; } catch(...) { m_pAssignAtoms = NULL; }
		if (m_pAssignAtoms == NULL) NewException((double)g_iGesAtomCount*sizeof(long),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		try { m_pAssignMolecules = new long[g_iGesAtomCount]; } catch(...) { m_pAssignMolecules = NULL; }
		if (m_pAssignMolecules == NULL) NewException((double)g_iGesAtomCount*sizeof(long),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		try { m_pAssignMoleculeTypes = new long[g_iGesAtomCount]; } catch(...) { m_pAssignMoleculeTypes = NULL; }
		if (m_pAssignMoleculeTypes == NULL) NewException((double)g_iGesAtomCount*sizeof(long),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		try { m_pAtomTouched = new bool[g_iGesAtomCount]; } catch(...) { m_pAtomTouched = NULL; }
		if (m_pAtomTouched == NULL) NewException((double)g_iGesAtomCount*sizeof(bool),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		i = 0;
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[z];
			for (z2=0;z2<m->m_waAtomCount.GetSize()-1;z2++)
			{
				for (z3=0;z3<m->m_waAtomCount[z2];z3++)
				{
					try { va = new CVoroAtom(); } catch(...) { va = NULL; }
					if (va == NULL) NewException((double)sizeof(CVoroAtom),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					va->m_pParent = this;
					va->m_iAtomType = z2;
					va->m_iAtom = z3;
					va->m_iRealAtomType = m->m_baAtomIndex[z2];
					va->m_iMolecule = z;

					m_oaVoroAtoms.Add(va);

					for (z4=0;z4<m->m_laSingleMolIndex.GetSize();z4++)
					{
						sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z4]];
	//					mprintf("Mol %d, AT %d, A %d, Sm %d, Offset %f, i=%d.\n",z+1,z2+1,z3+1,z4+1,((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3),i);
						m_pAssignAtoms[((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3)] = i;
					}

					i++;
				}
			}
		}


		i = 0;
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[z];
			for (z4=0;z4<m->m_laSingleMolIndex.GetSize();z4++)
			{
				sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z4]];

				try { vm = new CVoroMolecule(); } catch(...) { vm = NULL; }
				if (vm == NULL) NewException((double)sizeof(CVoroMolecule),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				vm->m_pParent = this;
				vm->m_iMolecule = z;
				vm->m_iSingleMol = z4;
				vm->m_iCenterOffset = ((CxIntArray*)sm->m_oaAtomOffset[sm->m_oaAtomOffset.GetSize()-1])->GetAt(1); // 1 is #2, which is Center of Mass

				m_oaVoroMolecules.Add(vm);

				for (z2=0;z2<m->m_waAtomCount.GetSize()-1;z2++)
				{
					for (z3=0;z3<m->m_waAtomCount[z2];z3++)
					{
						m_pAssignMolecules[((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3)] = i;
						m_pAssignMoleculeTypes[((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3)] = z;
					}
				}

				i++;
			}
		}
	}

	g_iVoroMemory = 16;
}


void CVoroWrapper::Init2()
{
	int z;

	if (m_bVoroStat)
	{
		for (z=0;z<m_oaVoroAtoms.GetSize();z++)
		{
			((CVoroAtom*)m_oaVoroAtoms[z])->InitAnalyses();
			try { ((CVoroAtom*)m_oaVoroAtoms[z])->m_pNbhTempMol = new int[m_oaVoroMolecules.GetSize()]; } catch(...) { ((CVoroAtom*)m_oaVoroAtoms[z])->m_pNbhTempMol = NULL; }
			if (((CVoroAtom*)m_oaVoroAtoms[z])->m_pNbhTempMol == NULL) NewException((double)m_oaVoroMolecules.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		}

		for (z=0;z<m_oaVoroMolecules.GetSize();z++)
		{
			((CVoroMolecule*)m_oaVoroMolecules[z])->InitAnalyses();

			try { ((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempMol = new int[m_oaVoroMolecules.GetSize()]; } catch(...) { ((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempMol = NULL; }
			if (((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempMol == NULL) NewException((double)m_oaVoroMolecules.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			try { ((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempAtoms = new int[g_iGesAtomCount]; } catch(...) { ((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempAtoms = NULL; }
			if (((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempAtoms == NULL) NewException((double)g_iGesAtomCount*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		}
	}
}


void CVoroWrapper::Parse()
{
	CTimeStep *t;
	CMolecule *m;
	CSingleMolecule *sm;
	char buf[256];
	int z, z2, z3;
	CxDoubleArray *fa;
//	CAtom *at;

	mprintf(WHITE,">>> Voronoi Analysis >>>\n\n");

	m_bVoroStat = AskYesNo("    Compute Voronoi statistics (y/n)? [yes] ",true);

	m_bSurfCover = AskYesNo("    Calculate surface coverage of one molecule (y/n)? [no] ",false);

	if (m_bSurfCover)
	{
		if (g_oaMolecules.GetSize() > 1)
		{
			m_iSurfCoverMol = AskRangeInteger_ND("    Use Voronoi cell of which molecule type (1-%d)? ",1,g_oaMolecules.GetSize(),g_oaMolecules.GetSize())-1;
		} else m_iSurfCoverMol = 0;

		m = (CMolecule*)g_oaMolecules[m_iSurfCoverMol];

		if (m->m_laSingleMolIndex.GetSize() > 1)
/*			m_iSurfCoverSM = AskInteger_ND("    Which representants of %s to use (1-%d)? ",m->m_sName,m->m_laSingleMolIndex.GetSize())-1;
				else m_iSurfCoverSM = 0;*/
		{
			AskString("    Which representants of %s to use (allowed 1-%d, eg. 5-7,9)? [all] ",buf,"",m->m_sName,m->m_laSingleMolIndex.GetSize());
			if (strlen(buf) == 0)
			{
				for (z=0;z<m->m_laSingleMolIndex.GetSize();z++)
					m_waSurfCoverSM.Add(z);
			} else ParseIntList(buf,&m_waSurfCoverSM);
		} else m_waSurfCoverSM.Add(0);

		mprintf("\n    Observing voronoi cells of %d molecules.\n\n",m_waSurfCoverSM.GetSize());

		g_iFixMol = m_iSurfCoverMol;

		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[z];
			for (z2=0;z2<m->m_waAtomCount.GetSize()-1;z2++)
			{
//				at = (CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]];
				for (z3=0;z3<m->m_waAtomCount[z2];z3++)
				{
					try { fa = new CxDoubleArray(); } catch(...) { fa = NULL; }
					if (fa == NULL) NewException((double)sizeof(CxDoubleArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					fa->SetGrow(1000);
					m_oaSurfCoverData.Add(fa);
					m_waSurfCoverMol.Add(z);
					m_waSurfCoverElem.Add(z2);
					m_waSurfCoverAtom.Add(z3);
				}
			}
		}
	}

	m_bWritePOV = AskYesNo("    Create POV-Ray render input of one Voronoi cell (y/n)? [no] ",false);

	if (m_bWritePOV)
	{
		mprintf(YELLOW,"\n    This requires POV-Ray v3.7 or later to render!\n\n");
		m_fPOVScript = OpenFileWrite("pov_render.bat",true);

		AskString("    Enter name of POV-Ray executable: [pvengine64] ",m_sPOVExe,"pvengine64");

		m_iPOVResX = AskUnsignedInteger("    Enter image width (in pixels): [800] ",800);

		m_iPOVResY = AskUnsignedInteger("    Enter image height (in pixels): [600] ",600);

		m_fPOVCameraDist = AskFloat("    Enter camera distance from molecule: [20.0] ",20.0f);

		if (g_oaMolecules.GetSize() > 1)
		{
			mprintf("    Molecule types: ");
			for (z=0;z<g_oaMolecules.GetSize();z++)
			{
				mprintf("%s (%d)",((CMolecule*)g_oaMolecules[z])->m_sName,z+1);
				if (z < g_oaMolecules.GetSize()-1)
					mprintf(", ");
			}
			mprintf("\n\n");

			m_iPOVMol = AskRangeInteger_ND("    Write Voronoi cell of which molecule type (1-%d)? ",1,g_oaMolecules.GetSize(),g_oaMolecules.GetSize())-1;
		} else m_iPOVMol = 0;

		m = (CMolecule*)g_oaMolecules[m_iPOVMol];
		if (m->m_laSingleMolIndex.GetSize() > 1)
		{
			m_iPOVSM = AskRangeInteger_ND("    Which representant of %s to use (1-%d)? ",1,m->m_laSingleMolIndex.GetSize(),m->m_sName,m->m_laSingleMolIndex.GetSize())-1;
		} else m_iPOVSM = 0;

		g_iFixMol = m_iPOVMol;

		m_bPOVRot = AskYesNo("    Fix rotational freedom of the reference molecule (y/n)? [yes] ",true);

		if (m_bPOVRot)
		{
//			g_iRefSystemDim = 3;
_ref3again:
			mprintf("\n    The first atom will be put into the center.\n");
			mprintf("    The second atom will point straight to the right.\n");
			mprintf("    The third atom will point around upwards.\n");
			mprintf("    2nd and 3rd atom may be equal, the up direction is not fixed then.\n\n");
			mprintf("    Please enter three reference atoms (e.g. C1,H2,O1): ");
			inpprintf("! Please enter three reference atoms (e.g. C1,H2,O1):\n");
			myget(buf);
			if (!ParseRefSystem(g_iFixMol,buf,3))
				goto _ref3again;

			if ((g_iFixAtomType[0] == g_iFixAtomType[1]) && (g_iFixAtom[0] == g_iFixAtom[1]))
			{
				eprintf("At least atoms 1 and 2 need to be different.\n\n");
				inpprintf("! At least atoms 1 and 2 need to be different.\n");
				goto _ref3again;
			}

 			mprintf("\n");
			mprintf(WHITE,"Reference plane: Fixing in %s the %d. %s-, the %d. %s- and the %d. %s atom.\n",((CMolecule*)g_oaMolecules[g_iFixMol])->m_sName,g_iFixAtom[0]+1,((CAtom*)g_oaAtoms[g_iFixRealAtomType[0]])->m_sName,g_iFixAtom[1]+1,((CAtom*)g_oaAtoms[g_iFixRealAtomType[1]])->m_sName,g_iFixAtom[2]+1,((CAtom*)g_oaAtoms[g_iFixRealAtomType[2]])->m_sName);
			mprintf("\n");
		} else
		{
//			g_iRefSystemDim = 1;
_ref1again:
			mprintf("    Please enter the atom to put into the box center (e.g. C3): [#2] ");
			inpprintf("! Please enter the atom to put into the box center (e.g. C3): [#2]\n");
			myget(buf);
			if (strlen(buf)==0)
			{
				if (!ParseRefSystem(m_iPOVMol,"#2",1))
				{
					eprintf("Weird error.\n");
					inpprintf("! Weird error.\n");
					abort();
				}
			} else if (!ParseRefSystem(m_iPOVMol,buf,1))
				goto _ref1again;
 			mprintf("\n");
			mprintf(WHITE,"Reference atom: Fixing in %s the %d. %s atom.\n",((CMolecule*)g_oaMolecules[m_iPOVMol])->m_sName,g_iFixAtom[0]+1,((CAtom*)g_oaAtoms[g_iFixRealAtomType[0]])->m_sName);
			mprintf("\n");
		}

		m_bPOVAtoms = AskYesNo("    Draw atoms of reference molecule (y/n)? [yes] ",true);

		m_bPOVDrawNeighbors = AskYesNo("    Draw adjacent molecules (y/n)? [no] ",false);

		if (m_bPOVAtoms)
			m_bPOVAtomGrey = AskYesNo("    Draw the atoms in shades of grey (y) or color (n)? [no] ",false);

		m_bPOVVertices = AskYesNo("    Draw voronoi cell vertices (y/n)? [no] ",false);

		m_bPOVEdges = AskYesNo("    Draw voronoi cell edges (y/n)? [yes] ",true);

		m_bPOVFaces = AskYesNo("    Draw voronoi cell faces (y/n)? [yes] ",true);

		if (m_bPOVFaces)
		{
			m_fPOVFaceOpac = AskRangeFloat("    Enter face transparency (0.0-1.0): [0.25] ",0.0,1.0f,0.25f);
			m_bPOVFaceColor = AskYesNo("    Colorize faces depending on atom types (y/n)? [no] ",false);
		} else
		{
			m_bPOVFaceColor = false;
		}

		if (m_bPOVFaceColor)
		{
			m_bPOVFaceColorRef = AskYesNo("    Take colors from ref. mol atom types (y) or from neighbors (n)? [no] ",false);
			ParsePOVFaceColor();
		}

		if (m_bPOVDrawNeighbors && m_bPOVFaceColor && !m_bPOVFaceColorRef)
		{
			m_bPOVNbColorFace = AskYesNo("    Colorize adjacent molecules according to face color (y/n)? [no] ",false);
			m_bPOVHighlightNbAtoms = AskYesNo("    Highlight neighboring atoms from molecules (y/n)? [no] ",false);
		} else
		{
			m_bPOVNbColorFace = false;
			m_bPOVHighlightNbAtoms = false;
		}

		m_vPOVRotPos[0] = 0;
		m_vPOVRotPos[1] = 0;
		m_vPOVRotPos[2] = 0;
		if (AskYesNo("    Rotate voronoi cell during course of trajectory (y/n)? [no] ",false))
		{
			m_vPOVRotInc[0] = AskFloat("    Enter X angle per step in degree: [0.0] ",5.0);
			m_vPOVRotInc[1] = AskFloat("    Enter Y angle per step in degree: [0.0] ",0);
			m_vPOVRotInc[2] = AskFloat("    Enter Z angle per step in degree: [0.0] ",0);
		} else
		{
			m_vPOVRotInc[0] = 0;
			m_vPOVRotInc[1] = 0;
			m_vPOVRotInc[2] = 0;
		}
	}
	
	mprintf("\n");
	mprintf("    Initializing Voronoi data structures...\n");
	g_pVoroWrapper->Init();
	mprintf("\n");

	mprintf("*** Voro: Box density is %f particles / Angstrom^3.\n",m_fBoxDens);
	mprintf("*** Voro: Using %d x %d x %d blocks.\n",m_iBlocksX,m_iBlocksY,m_iBlocksZ);
	mprintf("*** Voro: Using cell memory for %d particles.\n",g_iVoroMemory);

	if (m_bVoroStat || m_bSurfCover || m_bWritePOV)
	{
		mprintf(WHITE,"*** Dumping Voronoi Infos to \"voro.txt\".\n");

		try { t = new CTimeStep(); } catch(...) { t = NULL; }
		if (t == NULL) NewException((double)sizeof(CTimeStep),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		t->CopyFrom(&g_TimeStep);

		m = (CMolecule*)g_oaMolecules[0];
		sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[0]];

		t->CenterPos(t->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[sm->m_baAtomIndex.GetSize()-1])->GetAt(1)]);
		t->CenterPos(CxVector3(-g_fBoxX/2.0,-g_fBoxY/2.0,-g_fBoxZ/2.0));
		t->FoldAtomsPositive();
		g_pVoroWrapper->Dump("voro.txt",t);
	//	g_pVoroWrapper->WritePOV(t,"voro.pov",0,0);
		delete t;
		g_pVoroWrapper->Init2();
	}

//	AskYesNo("    This is hard-coded for Ulrikes Bmim-Br ^^ [accept] ",false);

	mprintf(WHITE,"\n<<< End of Voronoi Analysis <<<\n\n");
}


void CVoroWrapper::Dump(const char *s, CTimeStep *ts)
{
	int ijk, q, z, z2, ci, id;
	double *pp, cv;
	FILE *a;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	CVoroAtom *va, *va2;
	CVoroMolecule *vm, *vm2;

	a = OpenFileWrite(s,true);

	mfprintf(a,"*** Voro: Box density is %f particles / Angstrom^3.\n",m_fBoxDens);
	mfprintf(a,"*** Voro: Using %d x %d x %d blocks.\n",m_iBlocksX,m_iBlocksY,m_iBlocksZ);

//	d0 = g_iGesAtomCount / g_fBoxX / g_fBoxY / g_fBoxZ * 1000000.0;
//	d = 2.35 / pow(d0,1/3.0);

	mfprintf(a,"*** Voro: Creating Container...\n");
	fflush(a);

//	m_pContainer = new container_periodic_poly(1.0,0,1.0,0,0,1.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory);

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	mfprintf(a,"*** Voro: Adding %d particles...\n",g_iGesAtomCount);
	fflush(a);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	m_fMaxVol = 0;
	m_fMaxSurf = 0;

	c_loop_all_periodic vl(*m_pContainer);

	ci = 0;
	cv = 0;
	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				ci++;
				ijk=vl.ijk;
				q=vl.q;
				pp=m_pContainer->p[ijk]+m_pContainer->ps*q;
				id = m_pContainer->id[ijk][q];

				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				mfprintf(a,"Cell %5d (%s[%d] %s%d); p=(%.2f | %.2f | %.2f); r=%.2f pm; %d Faces; Vol=%.4f A^3, Surface=%.4f A^2; Max.Radius=%.3f pm\n",id,((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,vm->m_iSingleMol+1,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1,pp[0]*1000.0,pp[1]*1000.0,pp[2]*1000.0,pp[3]*1000.0,c.number_of_faces(),c.volume()*1000.0,c.surface_area()*100.0,sqrt(c.max_radius_squared())*1000.0);

				if (c.volume()*1000.0 > m_fMaxVol)
					m_fMaxVol = c.volume()*1000.0;

				cv += c.volume()*1000.0;

				if (c.surface_area()*100.0 > m_fMaxSurf)
					m_fMaxSurf = c.surface_area()*100.0;

				c.face_areas(fa);
				c.face_orders(fo);
				c.neighbors(nb);

				for (z2=0;z2<c.number_of_faces();z2++)
				{
					va2 = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[nb[z2]]];
					vm2 = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]];
					mfprintf(a,"  - Face %2d: %2d Vertices, Area=%7.4f A^2, Neighbor is %5d (%s[%d] %s%d)\n",z2+1,fo[z2],fa[z2]*100.0,nb[z2],((CMolecule*)g_oaMolecules[va2->m_iMolecule])->m_sName,vm2->m_iSingleMol+1,((CAtom*)g_oaAtoms[va2->m_iRealAtomType])->m_sName,va2->m_iAtom+1);
				}
			}
		} while (vl.inc());
	}
	mprintf("    %d/%d Particles touched.\n",ci,g_iGesAtomCount);
	mprintf("    Volume %f/%f Angstrom^3.\n",cv,g_fBoxX*g_fBoxY*g_fBoxZ/1000000.0);
	m_fMaxVol *= 3.0;
	m_fMaxSurf *= 3.0;
	delete m_pContainer;
	mfprintf(a,"*** The End ***\n");
	fclose(a);
}


void CVoroWrapper::Finish()
{
	int z, z2;
	char buf[256];
	CVoroAtom *va;
	CVoroMolecule *vm, *vmbase;

	if (m_bWritePOV)
		fclose(m_fPOVScript);

	if (m_bSurfCover)
	{
		mprintf(WHITE,"*** Voronoi Surface Coverage\n\n");
		mprintf("    Writing Surface data to voro_surfcover.csv\n");
		FinishSurfCover("voro_surfcover.csv");
		mprintf("\n");
	}

	if (m_bVoroStat)
	{
		mprintf(WHITE,"*** Voronoi Functions\n\n");

		mprintf("    Creating the sub-directory \"voronoi\"...\n");
		system("mkdir voronoi");
		mprintf("\n");

		mprintf("    Atom Information:\n");
		for (z=0;z<m_oaVoroAtoms.GetSize();z++)
		{
			va = (CVoroAtom*)m_oaVoroAtoms[z];
			sprintf(buf,"voronoi/voro_atom_%s_%s%d",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
			mprintf("      - Writing %s ...\n",buf);
			va->Dump(buf);
		}

		mprintf("\n    Molecule Information:\n");
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			vmbase = NULL;
			for (z2=0;z2<m_oaVoroMolecules.GetSize();z2++)
			{
				vm = (CVoroMolecule*)m_oaVoroMolecules[z2];
				if (vm == NULL)
					continue;
				if (vm->m_iMolecule != z)
					continue;
				if (vmbase == NULL)
				{
					vmbase = vm;
				} else
				{
					vmbase->Add(vm);
					delete vm;
					m_oaVoroMolecules[z2] = NULL;
				}
			}
			sprintf(buf,"voronoi/voro_molecule_%s",((CMolecule*)g_oaMolecules[vmbase->m_iMolecule])->m_sName);
			mprintf("      - Writing %s ...\n",buf);
			vmbase->Dump(buf);
		}
		mprintf("\n    Mean Atom Neighbor Count Matrix:\n");
		mprintf("      Writing voro_atom_nbcount.csv ...\n");
		WriteNbAtomCountMatrix("voro_atom_nbcount.csv");

		mprintf("\n    Mean Atom Neighbor Distance Matrix:\n");
		mprintf("      Writing voro_atom_nbdist.csv ...\n");
		WriteNbAtomDistMatrix("voro_atom_nbdist.csv");

		mprintf("\n    Mean Molecule Neighbor Count Matrix:\n");
		mprintf("      Writing voro_molecule_nbcount.csv ...\n");
		WriteNbMoleculeCountMatrix("voro_molecule_nbcount.csv");

		mprintf("\n    Mean Molecule Neighbor Distance Matrix:\n");
		mprintf("      Writing voro_molecule_nbdist.csv ...\n");
		WriteNbMoleculeDistMatrix("voro_molecule_nbdist.csv");

		mprintf("\n    Atom information table:\n");
		mprintf("      Writing voro_atoms.csv ...\n");
		WriteAtomInfo("voro_atoms.csv");

		mprintf("\n    Molecule information table:\n");
		mprintf("      Writing voro_molecules.csv ...\n");
		WriteMoleculeInfo("voro_molecules.csv");
		mprintf("\n");
	}
}


CVoroAtom::CVoroAtom()
{
}


CVoroAtom::~CVoroAtom()
{
}


void CVoroAtom::InitAnalyses()
{
	int z;

	try { m_pSurfaceArea = new CDF(); } catch(...) { m_pSurfaceArea = NULL; }
	if (m_pSurfaceArea == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pSurfaceArea->m_fMinVal = 0;
	m_pSurfaceArea->m_fMaxVal = m_pParent->m_fMaxSurf;
	m_pSurfaceArea->m_iResolution = 1000;
	m_pSurfaceArea->SetLabelX("Surface Area [A^2]");
	m_pSurfaceArea->SetLabelY("Occurrence");
	m_pSurfaceArea->Create();

	try { m_pExposedSurface = new CDF(); } catch(...) { m_pExposedSurface = NULL; }
	if (m_pExposedSurface == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pExposedSurface->m_fMinVal = 0;
	m_pExposedSurface->m_fMaxVal = m_pParent->m_fMaxSurf;
	m_pExposedSurface->m_iResolution = 1000;
	m_pExposedSurface->SetLabelX("Exposed Surface Area [A^2]");
	m_pExposedSurface->SetLabelY("Occurrence");
	m_pExposedSurface->Create();

	try { m_pExposedSurfacePerc = new CDF(); } catch(...) { m_pExposedSurfacePerc = NULL; }
	if (m_pExposedSurfacePerc == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pExposedSurfacePerc->m_fMinVal = 0;
	m_pExposedSurfacePerc->m_fMaxVal = 100.0;
	m_pExposedSurfacePerc->m_iResolution = 1000;
	m_pExposedSurfacePerc->SetLabelX("Exposed Surface Percentage");
	m_pExposedSurfacePerc->SetLabelY("Occurrence");
	m_pExposedSurfacePerc->Create();

	try { m_pVolume = new CDF(); } catch(...) { m_pVolume = NULL; }
	if (m_pVolume == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pVolume->m_fMinVal = 0;
	m_pVolume->m_fMaxVal = m_pParent->m_fMaxVol;
	m_pVolume->m_iResolution = 1000;
	m_pVolume->SetLabelX("Volume [A^3]");
	m_pVolume->SetLabelY("Occurrence");
	m_pVolume->Create();

	try { m_pMaxRadius = new CDF(); } catch(...) { m_pMaxRadius = NULL; }
	if (m_pMaxRadius == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pMaxRadius->m_fMinVal = 0;
	m_pMaxRadius->m_fMaxVal = g_fBoxX/4.0;
	m_pMaxRadius->m_iResolution = int(g_fBoxX*0.2);
	m_pMaxRadius->SetLabelX("Max. Radius [pm]");
	m_pMaxRadius->SetLabelY("Occurrence");
	m_pMaxRadius->Create();

	try { m_pAVRatio = new CDF(); } catch(...) { m_pAVRatio = NULL; }
	if (m_pAVRatio == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pAVRatio->m_fMinVal = 0;
	m_pAVRatio->m_fMaxVal = 1.0;
	m_pAVRatio->m_iResolution = 1000;
	m_pAVRatio->SetLabelX("A/V Ratio");
	m_pAVRatio->SetLabelY("Occurrence");
	m_pAVRatio->Create();

	try { m_pFaces = new CDF(); } catch(...) { m_pFaces = NULL; }
	if (m_pFaces == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pFaces->m_fMinVal = 0;
	m_pFaces->m_fMaxVal = 100;
	m_pFaces->m_iResolution = 101;
	m_pFaces->SetLabelX("Face Count");
	m_pFaces->SetLabelY("Occurrence");
	m_pFaces->Create();

	try { m_pFaceOrders = new CDF(); } catch(...) { m_pFaceOrders = NULL; }
	if (m_pFaceOrders == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pFaceOrders->m_fMinVal = 0;
	m_pFaceOrders->m_fMaxVal = 50;
	m_pFaceOrders->m_iResolution = 51;
	m_pFaceOrders->SetLabelX("Face Order");
	m_pFaceOrders->SetLabelY("Occurrence");
	m_pFaceOrders->Create();

	try { m_pNbhAtoms = new CDF*[m_pParent->m_oaVoroAtoms.GetSize()]; } catch(...) { m_pNbhAtoms = NULL; }
	if (m_pNbhAtoms == NULL) NewException((double)m_pParent->m_oaVoroAtoms.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		try { m_pNbhAtoms[z] = new CDF(); } catch(...) { m_pNbhAtoms[z] = NULL; }
		if (m_pNbhAtoms[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhAtoms[z]->m_fMinVal = 0;
		m_pNbhAtoms[z]->m_fMaxVal = 50;
		m_pNbhAtoms[z]->m_iResolution = 51;
		m_pNbhAtoms[z]->SetLabelX("Neighbor Count");
		m_pNbhAtoms[z]->SetLabelY("Occurrence");
		m_pNbhAtoms[z]->Create();
	}

	try { m_pNbhDistAtoms = new CDF*[m_pParent->m_oaVoroAtoms.GetSize()]; } catch(...) { m_pNbhDistAtoms = NULL; }
	if (m_pNbhDistAtoms == NULL) NewException((double)m_pParent->m_oaVoroAtoms.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		try { m_pNbhDistAtoms[z] = new CDF(); } catch(...) { m_pNbhDistAtoms[z] = NULL; }
		if (m_pNbhDistAtoms[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhDistAtoms[z]->m_fMinVal = 0;
		m_pNbhDistAtoms[z]->m_fMaxVal = g_fBoxX*2.0;
		m_pNbhDistAtoms[z]->m_iResolution = int(g_fBoxX*0.5);
		m_pNbhDistAtoms[z]->SetLabelX("Neighbor Distance [pm]");
		m_pNbhDistAtoms[z]->SetLabelY("Occurrence");
		m_pNbhDistAtoms[z]->Create();
	}

	try { m_pNbhDistMolecules = new CDF*[g_oaMolecules.GetSize()]; } catch(...) { m_pNbhDistMolecules = NULL; }
	if (m_pNbhDistMolecules == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		try { m_pNbhDistMolecules[z] = new CDF(); } catch(...) { m_pNbhDistMolecules[z] = NULL; }
		if (m_pNbhDistMolecules[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhDistMolecules[z]->m_fMinVal = 0;
		m_pNbhDistMolecules[z]->m_fMaxVal = g_fBoxX*2.0;
		m_pNbhDistMolecules[z]->m_iResolution = int(g_fBoxX*0.5);
		m_pNbhDistMolecules[z]->SetLabelX("Neighbor Distance [pm]");
		m_pNbhDistMolecules[z]->SetLabelY("Occurrence");
		m_pNbhDistMolecules[z]->Create();
	}

	try { m_pNbhMolecules = new CDF*[g_oaMolecules.GetSize()]; } catch(...) { m_pNbhMolecules = NULL; }
	if (m_pNbhMolecules == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		try { m_pNbhMolecules[z] = new CDF(); } catch(...) { m_pNbhMolecules[z] = NULL; }
		if (m_pNbhMolecules[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhMolecules[z]->m_fMinVal = 0;
		m_pNbhMolecules[z]->m_fMaxVal = 50;
		m_pNbhMolecules[z]->m_iResolution = 51;
		m_pNbhMolecules[z]->SetLabelX("Neighbor Count");
		m_pNbhMolecules[z]->SetLabelY("Occurrence");
		m_pNbhMolecules[z]->Create();
	}
}


CVoroMolecule::CVoroMolecule()
{
}


CVoroMolecule::~CVoroMolecule()
{
}


void CVoroMolecule::InitAnalyses()
{
	int z;

	try { m_pSurfaceArea = new CDF(); } catch(...) { m_pSurfaceArea = NULL; }
	if (m_pSurfaceArea == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pSurfaceArea->m_fMinVal = 0;
	m_pSurfaceArea->m_fMaxVal = m_pParent->m_fMaxSurf*((CMolecule*)g_oaMolecules[m_iMolecule])->m_iAtomGes;
	m_pSurfaceArea->m_iResolution = 1000;
	m_pSurfaceArea->SetLabelX("Surface Area [A^2]");
	m_pSurfaceArea->SetLabelY("Occurrence");
	m_pSurfaceArea->Create();

	try { m_pVolume = new CDF(); } catch(...) { m_pVolume = NULL; }
	if (m_pVolume == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pVolume->m_fMinVal = 0;
	m_pVolume->m_fMaxVal = m_pParent->m_fMaxVol*((CMolecule*)g_oaMolecules[m_iMolecule])->m_iAtomGes;
	m_pVolume->m_iResolution = 1000;
	m_pVolume->SetLabelX("Volume [A^3]");
	m_pVolume->SetLabelY("Occurrence");
	m_pVolume->Create();

	try { m_pAVRatio = new CDF(); } catch(...) { m_pAVRatio = NULL; }
	if (m_pAVRatio == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pAVRatio->m_fMinVal = 0;
	m_pAVRatio->m_fMaxVal = 1.0;
	m_pAVRatio->m_iResolution = 1000;
	m_pAVRatio->SetLabelX("A/V Ratio");
	m_pAVRatio->SetLabelY("Occurrence");
	m_pAVRatio->Create();

	try { m_pFaces = new CDF(); } catch(...) { m_pFaces = NULL; }
	if (m_pFaces == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pFaces->m_fMinVal = 0;
	m_pFaces->m_fMaxVal = 100*((CMolecule*)g_oaMolecules[m_iMolecule])->m_iAtomGes;
	m_pFaces->m_iResolution = int(100*((CMolecule*)g_oaMolecules[m_iMolecule])->m_iAtomGes)+1;
	m_pFaces->SetLabelX("Face count");
	m_pFaces->SetLabelY("Occurrence");
	m_pFaces->Create();

	try { m_pNbhAtoms = new CDF*[m_pParent->m_oaVoroAtoms.GetSize()]; } catch(...) { m_pNbhAtoms = NULL; }
	if (m_pNbhAtoms == NULL) NewException((double)m_pParent->m_oaVoroAtoms.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		try { m_pNbhAtoms[z] = new CDF(); } catch(...) { m_pNbhAtoms[z] = NULL; }
		if (m_pNbhAtoms[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhAtoms[z]->m_fMinVal = 0;
		m_pNbhAtoms[z]->m_fMaxVal = 50;
		m_pNbhAtoms[z]->m_iResolution = 51;
		m_pNbhAtoms[z]->SetLabelX("Neighbor count");
		m_pNbhAtoms[z]->SetLabelY("Occurrence");
		m_pNbhAtoms[z]->Create();
	}

	try { m_pNbhDistAtoms = new CDF*[m_pParent->m_oaVoroAtoms.GetSize()]; } catch(...) { m_pNbhDistAtoms = NULL; }
	if (m_pNbhDistAtoms == NULL) NewException((double)m_pParent->m_oaVoroAtoms.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		try { m_pNbhDistAtoms[z] = new CDF(); } catch(...) { m_pNbhDistAtoms[z] = NULL; }
		if (m_pNbhDistAtoms[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhDistAtoms[z]->m_fMinVal = 0;
		m_pNbhDistAtoms[z]->m_fMaxVal = g_fBoxX*2.0;
		m_pNbhDistAtoms[z]->m_iResolution = int(g_fBoxX*0.5);
		m_pNbhDistAtoms[z]->SetLabelX("Neighbor distance [pm]");
		m_pNbhDistAtoms[z]->SetLabelY("Occurrence");
		m_pNbhDistAtoms[z]->Create();
	}

	try { m_pNbhDistMolecules = new CDF*[g_oaMolecules.GetSize()]; } catch(...) { m_pNbhDistMolecules = NULL; }
	if (m_pNbhDistMolecules == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		try { m_pNbhDistMolecules[z] = new CDF(); } catch(...) { m_pNbhDistMolecules[z] = NULL; }
		if (m_pNbhDistMolecules[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhDistMolecules[z]->m_fMinVal = 0;
		m_pNbhDistMolecules[z]->m_fMaxVal = g_fBoxX*2.0;
		m_pNbhDistMolecules[z]->m_iResolution = int(g_fBoxX*0.5);
		m_pNbhDistMolecules[z]->SetLabelX("Neighbor distance [pm]");
		m_pNbhDistMolecules[z]->SetLabelY("Occurrence");
		m_pNbhDistMolecules[z]->Create();
	}

	try { m_pNbhMolecules = new CDF*[g_oaMolecules.GetSize()]; } catch(...) { m_pNbhMolecules = NULL; }
	if (m_pNbhMolecules == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		try { m_pNbhMolecules[z] = new CDF(); } catch(...) { m_pNbhMolecules[z] = NULL; }
		if (m_pNbhMolecules[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhMolecules[z]->m_fMinVal = 0;
		m_pNbhMolecules[z]->m_fMaxVal = 50;
		m_pNbhMolecules[z]->m_iResolution = 51;
		m_pNbhMolecules[z]->SetLabelX("Neighbor count");
		m_pNbhMolecules[z]->SetLabelY("Occurrence");
		m_pNbhMolecules[z]->Create();
	}
}


void CVoroAtom::Dump(const char *s)
{
	FILE *a;
	char buf[256];
	int z, z2;

	sprintf(buf,"%s.txt",s);
	a = OpenFileWrite(buf,true);

	mfprintf(a,"*** Voronoi Analysis for %s %s%d ***\n\n",((CMolecule*)g_oaMolecules[m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[m_iRealAtomType])->m_sName,m_iAtom+1);

	m_pVolume->CalcMeanSD();
	mfprintf(a,"  * Volume:\n");
	mfprintf(a,"      Avg. %10G Angstrom^3\n",m_pVolume->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^3\n",m_pVolume->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^3\n",m_pVolume->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^3\n",m_pVolume->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_volume.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pVolume->m_fBinEntries,m_pVolume->m_fSkipEntries);
	m_pVolume->NormBinSum(100.0);
	m_pVolume->Write("",s,"_volume.csv",true);

	m_pSurfaceArea->CalcMeanSD();
	mfprintf(a,"  * Surface Area:\n");
	mfprintf(a,"      Avg. %10G Angstrom^2\n",m_pSurfaceArea->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^2\n",m_pSurfaceArea->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^2\n",m_pSurfaceArea->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^2\n",m_pSurfaceArea->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_surface.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pSurfaceArea->m_fBinEntries,m_pSurfaceArea->m_fSkipEntries);
	m_pSurfaceArea->NormBinSum(100.0);
	m_pSurfaceArea->Write("",s,"_surface.csv",true);

	m_pExposedSurface->CalcMeanSD();
	mfprintf(a,"  * Exposed Surface Area:\n");
	mfprintf(a,"      Avg. %10G Angstrom^2\n",m_pExposedSurface->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^2\n",m_pExposedSurface->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^2\n",m_pExposedSurface->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^2\n",m_pExposedSurface->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_exposed_surface.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pExposedSurface->m_fBinEntries,m_pExposedSurface->m_fSkipEntries);
	m_pExposedSurface->NormBinSum(100.0);
	m_pExposedSurface->Write("",s,"_exposed_surface.csv",true);

	m_pExposedSurfacePerc->CalcMeanSD();
	mfprintf(a,"  * Exposed Surface Area Percentage:\n");
	mfprintf(a,"      Avg. %10G percent\n",m_pExposedSurfacePerc->m_fMean);
	mfprintf(a,"      Min. %10G percent\n",m_pExposedSurfacePerc->m_fMinInput);
	mfprintf(a,"      Max. %10G percent\n",m_pExposedSurfacePerc->m_fMaxInput);
	mfprintf(a,"      SD   %10G percent\n",m_pExposedSurfacePerc->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_exposed_surface_perc.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pExposedSurfacePerc->m_fBinEntries,m_pExposedSurfacePerc->m_fSkipEntries);
	m_pExposedSurfacePerc->NormBinSum(100.0);
	m_pExposedSurfacePerc->Write("",s,"_exposed_surface_perc.csv",true);

	m_pMaxRadius->CalcMeanSD();
	mfprintf(a,"  * Maximum Radius:\n");
	mfprintf(a,"      Avg. %.4f pm\n",m_pMaxRadius->m_fMean);
	mfprintf(a,"      Min. %.4f pm\n",m_pMaxRadius->m_fMinInput);
	mfprintf(a,"      Max. %.4f pm\n",m_pMaxRadius->m_fMaxInput);
	mfprintf(a,"      SD   %.4f pm\n\n",m_pMaxRadius->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_maxradius.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pMaxRadius->m_fBinEntries,m_pMaxRadius->m_fSkipEntries);
	m_pMaxRadius->NormBinSum(100.0);
	m_pMaxRadius->Write("",s,"_maxradius.csv",true);

	m_pFaceOrders->CalcMeanSD();
	mfprintf(a,"  * Face Orders:\n");
	mfprintf(a,"      Avg. %.6f Edges\n",m_pFaceOrders->m_fMean);
	mfprintf(a,"      Min. %.0f Edges\n",m_pFaceOrders->m_fMinInput);
	mfprintf(a,"      Max. %.0f Edges\n",m_pFaceOrders->m_fMaxInput);
	mfprintf(a,"      SD   %.6f Edges\n\n",m_pFaceOrders->m_fSD);
	m_pFaceOrders->NormBinSum(100.0);
	for (z=0;z<m_pFaceOrders->m_iResolution;z++)
		if (m_pFaceOrders->m_pBin[z] != 0)
			mfprintf(a,"      - %2d Edges: %10.6f percent\n",z,m_pFaceOrders->m_pBin[z]);
	mfprintf(a,"      Saving Histogram as %s_faceorders.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pFaceOrders->m_fBinEntries,m_pFaceOrders->m_fSkipEntries);
	m_pFaceOrders->Write_Int("",s,"_faceorders.csv");

	m_pFaces->CalcMeanSD();
	mfprintf(a,"  * Faces:\n");
	mfprintf(a,"      Avg. %.6f\n",m_pFaces->m_fMean);
	mfprintf(a,"      Min. %.0f\n",m_pFaces->m_fMinInput);
	mfprintf(a,"      Max. %.0f\n",m_pFaces->m_fMaxInput);
	mfprintf(a,"      SD   %.6f\n\n",m_pFaces->m_fSD);
	m_pFaces->NormBinSum(100.0);
	for (z=0;z<m_pFaces->m_iResolution;z++)
		if (m_pFaces->m_pBin[z] != 0)
			mfprintf(a,"      - %2d Faces: %10.6f percent\n",z,m_pFaces->m_pBin[z]);
	mfprintf(a,"      Saving Histogram as %s_faces.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pFaces->m_fBinEntries,m_pFaces->m_fSkipEntries);
	m_pFaces->Write_Int("",s,"_faces.csv");

	m_pAVRatio->CalcMeanSD();
	mfprintf(a,"  * A/V Ratio:\n");
	mfprintf(a,"      Avg. %.6f\n",m_pAVRatio->m_fMean);
	mfprintf(a,"      Min. %.6f\n",m_pAVRatio->m_fMinInput);
	mfprintf(a,"      Max. %.6f\n",m_pAVRatio->m_fMaxInput);
	mfprintf(a,"      SD   %.6f\n\n",m_pAVRatio->m_fSD);
	m_pAVRatio->NormBinSum(100.0);
	mfprintf(a,"      Saving Histogram as %s_avratio.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pAVRatio->m_fBinEntries,m_pAVRatio->m_fSkipEntries);
	m_pAVRatio->Write("",s,"_avratio.csv",false);

	mfprintf(a,"###### Neighboring Molecules ######\n\n");

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m_pNbhMolecules[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s molecules:\n",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Avg. %.6f\n",m_pNbhMolecules[z]->m_fMean);
		mfprintf(a,"      Min. %.0f\n",m_pNbhMolecules[z]->m_fMinInput);
		mfprintf(a,"      Max. %.0f\n",m_pNbhMolecules[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.6f\n",m_pNbhMolecules[z]->m_fSD);
		m_pNbhMolecules[z]->NormBinSum(100.0);
		for (z2=0;z2<m_pNbhMolecules[z]->m_iResolution;z2++)
			if (m_pNbhMolecules[z]->m_pBin[z2] != 0)
				mfprintf(a,"      - %2d Molecules: %10.6f percent\n",z2,m_pNbhMolecules[z]->m_pBin[z2]);
		sprintf(buf,"_nbcount_mol_%s.csv",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhMolecules[z]->m_fBinEntries,m_pNbhMolecules[z]->m_fSkipEntries);
		m_pNbhMolecules[z]->Write_Int("",s,buf);
	}

	mfprintf(a,"###### Neighboring Molecule Distances ######\n\n");

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m_pNbhDistMolecules[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s molecule distances:\n",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Avg. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMean);
		mfprintf(a,"      Min. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMinInput);
		mfprintf(a,"      Max. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.4f pm\n",m_pNbhDistMolecules[z]->m_fSD);
		m_pNbhDistMolecules[z]->NormBinSum(100.0);
		sprintf(buf,"_nbdist_mol_%s.csv",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhDistMolecules[z]->m_fBinEntries,m_pNbhDistMolecules[z]->m_fSkipEntries);
		m_pNbhDistMolecules[z]->Write("",s,buf,true);
	}

	mfprintf(a,"###### Neighboring Atoms ######\n\n");

	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		sprintf(buf,"%s %s%d",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		m_pNbhAtoms[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s atoms:\n",buf);
		mfprintf(a,"      Avg. %.6f\n",m_pNbhAtoms[z]->m_fMean);
		mfprintf(a,"      Min. %.0f\n",m_pNbhAtoms[z]->m_fMinInput);
		mfprintf(a,"      Max. %.0f\n",m_pNbhAtoms[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.6f\n",m_pNbhAtoms[z]->m_fSD);
		m_pNbhAtoms[z]->NormBinSum(100.0);
		for (z2=0;z2<m_pNbhAtoms[z]->m_iResolution;z2++)
			if (m_pNbhAtoms[z]->m_pBin[z2] != 0)
				mfprintf(a,"      - %2d Atoms: %10.6f percent\n",z2,m_pNbhAtoms[z]->m_pBin[z2]);
		sprintf(buf,"_nbcount_atom_%s_%s%d.csv",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhAtoms[z]->m_fBinEntries,m_pNbhAtoms[z]->m_fSkipEntries);
		m_pNbhAtoms[z]->Write_Int("",s,buf);
	}

	mfprintf(a,"###### Neighboring Atom Distances ######\n\n");

	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		sprintf(buf,"%s %s%d",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		m_pNbhDistAtoms[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s atom distances:\n",buf);
		mfprintf(a,"      Avg. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMean);
		mfprintf(a,"      Min. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMinInput);
		mfprintf(a,"      Max. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.4f pm\n",m_pNbhDistAtoms[z]->m_fSD);
		m_pNbhDistAtoms[z]->NormBinSum(100.0);
		sprintf(buf,"_nbdist_atom_%s_%s%d.csv",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhDistAtoms[z]->m_fBinEntries,m_pNbhDistAtoms[z]->m_fSkipEntries);
		m_pNbhDistAtoms[z]->Write("",s,buf,true);
	}

	fclose(a);
}

void CVoroMolecule::Dump(const char *s)
{
	FILE *a;
	char buf[256];
	int z, z2;

	sprintf(buf,"%s.txt",s);
	a = OpenFileWrite(buf,true);

	mfprintf(a,"*** Voronoi Analysis for Molecule %s ***\n\n",((CMolecule*)g_oaMolecules[m_iMolecule])->m_sName);

	m_pVolume->CalcMeanSD();
	mfprintf(a,"  * Volume:\n");
	mfprintf(a,"      Avg. %10G Angstrom^3\n",m_pVolume->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^3\n",m_pVolume->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^3\n",m_pVolume->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^3\n",m_pVolume->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_volume.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pVolume->m_fBinEntries,m_pVolume->m_fSkipEntries);
	m_pVolume->NormBinSum(100.0);
	m_pVolume->Write("",s,"_volume.csv",true);

	m_pSurfaceArea->CalcMeanSD();
	mfprintf(a,"  * Surface Area:\n");
	mfprintf(a,"      Avg. %10G Angstrom^2\n",m_pSurfaceArea->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^2\n",m_pSurfaceArea->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^2\n",m_pSurfaceArea->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^2\n",m_pSurfaceArea->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_surface.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pSurfaceArea->m_fBinEntries,m_pSurfaceArea->m_fSkipEntries);
	m_pSurfaceArea->NormBinSum(100.0);
	m_pSurfaceArea->Write("",s,"_surface.csv",true);

	m_pAVRatio->CalcMeanSD();
	mfprintf(a,"  * A/V Ratio:\n");
	mfprintf(a,"      Avg. %10G Angstrom^2\n",m_pAVRatio->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^2\n",m_pAVRatio->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^2\n",m_pAVRatio->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^2\n",m_pAVRatio->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_avratio.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pAVRatio->m_fBinEntries,m_pAVRatio->m_fSkipEntries);
	m_pAVRatio->NormBinSum(100.0);
	m_pAVRatio->Write("",s,"_avratio.csv",true);

	m_pFaces->CalcMeanSD();
	mfprintf(a,"  * Faces:\n");
	mfprintf(a,"      Avg. %.6f\n",m_pFaces->m_fMean);
	mfprintf(a,"      Min. %.0f\n",m_pFaces->m_fMinInput);
	mfprintf(a,"      Max. %.0f\n",m_pFaces->m_fMaxInput);
	mfprintf(a,"      SD   %.6f\n\n",m_pFaces->m_fSD);
	m_pFaces->NormBinSum(100.0);
	for (z=0;z<m_pFaces->m_iResolution;z++)
		if (m_pFaces->m_pBin[z] != 0)
			mfprintf(a,"      - %2d Faces: %10.6f percent\n",z,m_pFaces->m_pBin[z]);
	mfprintf(a,"      Saving Histogram as %s_faces.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pFaces->m_fBinEntries,m_pFaces->m_fSkipEntries);
	m_pFaces->Write_Int("",s,"_faces.csv");

	mfprintf(a,"###### Neighboring Molecules ######\n\n");

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m_pNbhMolecules[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s molecules:\n",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Avg. %.6f\n",m_pNbhMolecules[z]->m_fMean);
		mfprintf(a,"      Min. %.0f\n",m_pNbhMolecules[z]->m_fMinInput);
		mfprintf(a,"      Max. %.0f\n",m_pNbhMolecules[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.6f\n",m_pNbhMolecules[z]->m_fSD);
		m_pNbhMolecules[z]->NormBinSum(100.0);
		for (z2=0;z2<m_pNbhMolecules[z]->m_iResolution;z2++)
			if (m_pNbhMolecules[z]->m_pBin[z2] != 0)
				mfprintf(a,"      - %2d Molecules: %10.6f percent\n",z2,m_pNbhMolecules[z]->m_pBin[z2]);
		sprintf(buf,"_nbcount_mol_%s.csv",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhMolecules[z]->m_fBinEntries,m_pNbhMolecules[z]->m_fSkipEntries);
		m_pNbhMolecules[z]->Write_Int("",s,buf);
	}

	mfprintf(a,"###### Neighboring Molecule Distances ######\n\n");

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m_pNbhDistMolecules[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s molecule distances:\n",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Avg. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMean);
		mfprintf(a,"      Min. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMinInput);
		mfprintf(a,"      Max. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.4f pm\n",m_pNbhDistMolecules[z]->m_fSD);
		m_pNbhDistMolecules[z]->NormBinSum(100.0);
		sprintf(buf,"_nbdist_mol_%s.csv",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhDistMolecules[z]->m_fBinEntries,m_pNbhDistMolecules[z]->m_fSkipEntries);
		m_pNbhDistMolecules[z]->Write("",s,buf,true);
	}

	mfprintf(a,"###### Neighboring Atoms ######\n\n");

	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		sprintf(buf,"%s %s%d",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		m_pNbhAtoms[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s atoms:\n",buf);
		mfprintf(a,"      Avg. %.6f\n",m_pNbhAtoms[z]->m_fMean);
		mfprintf(a,"      Min. %.0f\n",m_pNbhAtoms[z]->m_fMinInput);
		mfprintf(a,"      Max. %.0f\n",m_pNbhAtoms[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.6f\n",m_pNbhAtoms[z]->m_fSD);
		m_pNbhAtoms[z]->NormBinSum(100.0);
		for (z2=0;z2<m_pNbhAtoms[z]->m_iResolution;z2++)
			if (m_pNbhAtoms[z]->m_pBin[z2] != 0)
				mfprintf(a,"      - %2d Atoms: %10.6f percent\n",z2,m_pNbhAtoms[z]->m_pBin[z2]);
		sprintf(buf,"_nbcount_atom_%s_%s%d.csv",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhAtoms[z]->m_fBinEntries,m_pNbhAtoms[z]->m_fSkipEntries);
		m_pNbhAtoms[z]->Write_Int("",s,buf);
	}

	mfprintf(a,"###### Neighboring Atom Distances ######\n\n");

	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		sprintf(buf,"%s %s%d",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		m_pNbhDistAtoms[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s atom distances:\n",buf);
		mfprintf(a,"      Avg. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMean);
		mfprintf(a,"      Min. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMinInput);
		mfprintf(a,"      Max. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.4f pm\n",m_pNbhDistAtoms[z]->m_fSD);
		m_pNbhDistAtoms[z]->NormBinSum(100.0);
		sprintf(buf,"_nbdist_atom_%s_%s%d.csv",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhDistAtoms[z]->m_fBinEntries,m_pNbhDistAtoms[z]->m_fSkipEntries);
		m_pNbhDistAtoms[z]->Write("",s,buf,true);
	}

	fclose(a);
}

void CVoroMolecule::Add(CVoroMolecule *m)
{
	int z;

	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		m_pNbhAtoms[z]->AddFrom(m->m_pNbhAtoms[z]);
		m_pNbhDistAtoms[z]->AddFrom(m->m_pNbhDistAtoms[z]);
	}
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m_pNbhMolecules[z]->AddFrom(m->m_pNbhMolecules[z]);
		m_pNbhDistMolecules[z]->AddFrom(m->m_pNbhDistMolecules[z]);
	}
	m_pVolume->AddFrom(m->m_pVolume);
	m_pSurfaceArea->AddFrom(m->m_pSurfaceArea);
	m_pAVRatio->AddFrom(m->m_pAVRatio);
}

void CVoroWrapper::WriteNbAtomCountMatrix(const char *s)
{
	FILE *a;
	int z, z2;
	CVoroAtom *va;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Row=From, Column=To\n");
	mfprintf(a,"*");
	for (z=0;z<m_oaVoroAtoms.GetSize();z++)
	{
		va = (CVoroAtom*)m_oaVoroAtoms[z];
		mfprintf(a,";  %s %s%d",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
	}
	mfprintf(a,"\n");
	
	for (z=0;z<m_oaVoroAtoms.GetSize();z++)
	{
		va = (CVoroAtom*)m_oaVoroAtoms[z];
		mfprintf(a,"%s %s%d",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
		for (z2=0;z2<m_oaVoroAtoms.GetSize();z2++)
			mfprintf(a,";  %f",va->m_pNbhAtoms[z2]->m_fMean);
		mfprintf(a,"\n");
	}
	
	fclose(a);
}

void CVoroWrapper::WriteNbAtomDistMatrix(const char *s)
{
	FILE *a;
	int z, z2;
	CVoroAtom *va;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Row=From, Column=To, all numbers in pm\n");
	mfprintf(a,"*");
	for (z=0;z<m_oaVoroAtoms.GetSize();z++)
	{
		va = (CVoroAtom*)m_oaVoroAtoms[z];
		mfprintf(a,";  %s %s%d",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
	}
	mfprintf(a,"\n");
	
	for (z=0;z<m_oaVoroAtoms.GetSize();z++)
	{
		va = (CVoroAtom*)m_oaVoroAtoms[z];
		mfprintf(a,"%s %s%d",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
		for (z2=0;z2<m_oaVoroAtoms.GetSize();z2++)
			mfprintf(a,";  %f",va->m_pNbhDistAtoms[z2]->m_fMean);
		mfprintf(a,"\n");
	}
	
	fclose(a);
}

void CVoroWrapper::WriteNbMoleculeCountMatrix(const char *s)
{
	FILE *a;
	int z, z2;
	CVoroMolecule *vm;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Row=From, Column=To\n");
	mfprintf(a,"*");
	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		if (vm == NULL)
			continue;
		mfprintf(a,";  %s",((CMolecule*)g_oaMolecules[vm->m_iMolecule])->m_sName);
	}
	mfprintf(a,"\n");
	
	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		if (vm == NULL)
			continue;
		mfprintf(a,"%s",((CMolecule*)g_oaMolecules[vm->m_iMolecule])->m_sName);
		for (z2=0;z2<m_oaVoroMolecules.GetSize();z2++)
		{
			if (m_oaVoroMolecules[z2] == NULL)
				continue;
			mfprintf(a,";  %f",vm->m_pNbhMolecules[((CVoroMolecule*)m_oaVoroMolecules[z2])->m_iMolecule]->m_fMean);
		}
		mfprintf(a,"\n");
	}
	
	fclose(a);
}

void CVoroWrapper::WriteNbMoleculeDistMatrix(const char *s)
{
	FILE *a;
	int z, z2;
	CVoroMolecule *vm;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Row=From, Column=To, all numbers in pm\n");
	mfprintf(a,"*");
	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		if (vm == NULL)
			continue;
		mfprintf(a,";  %s",((CMolecule*)g_oaMolecules[vm->m_iMolecule])->m_sName);
	}
	mfprintf(a,"\n");
	
	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		if (vm == NULL)
			continue;
		mfprintf(a,"%s",((CMolecule*)g_oaMolecules[vm->m_iMolecule])->m_sName);
		for (z2=0;z2<m_oaVoroMolecules.GetSize();z2++)
		{
			if (m_oaVoroMolecules[z2] == NULL)
				continue;
			mfprintf(a,";  %f",vm->m_pNbhDistMolecules[((CVoroMolecule*)m_oaVoroMolecules[z2])->m_iMolecule]->m_fMean);
		}
		mfprintf(a,"\n");
	}
	
	fclose(a);
}

void CVoroWrapper::WriteAtomInfo(const char *s)
{
	FILE *a;
	int z;
	CVoroAtom *va;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Molecule;  Atom;  Mean Volume [A^3];  Mean Surface [A^2];  Mean Exposed Surface [A^2];  Mean Exposed Surface Percentage;  Mean Faces;  Mean Face Orders;  Mean Max Radius [pm];  Mean A/V Ratio\n");

	for (z=0;z<m_oaVoroAtoms.GetSize();z++)
	{
		va = (CVoroAtom*)m_oaVoroAtoms[z];
		mfprintf(a,"%s;  %s%d;  ",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
		mfprintf(a,"%f;  ",va->m_pVolume->m_fMean);
		mfprintf(a,"%f;  ",va->m_pSurfaceArea->m_fMean);
		mfprintf(a,"%f;  ",va->m_pExposedSurface->m_fMean);
		mfprintf(a,"%f;  ",va->m_pExposedSurfacePerc->m_fMean);
		mfprintf(a,"%f;  ",va->m_pFaces->m_fMean);
		mfprintf(a,"%f;  ",va->m_pFaceOrders->m_fMean);
		mfprintf(a,"%f;  ",va->m_pMaxRadius->m_fMean);
		mfprintf(a,"%f\n",va->m_pAVRatio->m_fMean);
	}
	
	fclose(a);
}

void CVoroWrapper::WriteMoleculeInfo(const char *s)
{
	FILE *a;
	int z;
	CVoroMolecule *vm;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Molecule;  Mean Volume [A^3];  Mean Surface [A^2];  Mean Faces;  Mean A/V Ratio\n");

	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		if (vm == NULL)
			continue;
		mfprintf(a,"%s;  ",((CMolecule*)g_oaMolecules[vm->m_iMolecule])->m_sName);
		mfprintf(a,"%f;  ",vm->m_pVolume->m_fMean);
		mfprintf(a,"%f;  ",vm->m_pSurfaceArea->m_fMean);
		mfprintf(a,"%f;  ",vm->m_pFaces->m_fMean);
		mfprintf(a,"%f\n",vm->m_pAVRatio->m_fMean);
	}
	
	fclose(a);
}


void CVoroWrapper::WritePOV(CTimeStep *ts)
{
	char buf[256];
	CxVector3 vec, vec2, vec3;
	CMolecule *m;
	CSingleMolecule *sm;

	m = (CMolecule*)g_oaMolecules[m_iPOVMol];
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[m_iPOVSM]];

	sprintf(buf,"voro_%s_%d_%05d.pov",m->m_sName,m_iPOVSM+1,(int)(g_iSteps/g_iStride));

	if (m_bPOVRot)
	{
		vec = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[0]])->GetAt(g_iFixAtom[0])];
		vec2 = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[1]])->GetAt(g_iFixAtom[1])];
		vec3 = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[2]])->GetAt(g_iFixAtom[2])];
		m_mPOVMat.MatUltra(vec2-vec, vec3-vec); 
//		g_pTempTimestep->Transform(mat);
	}

	vec = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[0]])->GetAt(g_iFixAtom[0])] - CxVector3(g_fBoxX/2.0,g_fBoxY/2.0,g_fBoxZ/2.0);

	ts->CenterPos(vec);

	ts->FoldAtomsPositive();

	WritePOV(ts,buf,m_iPOVMol,m_iPOVSM);

	m_vPOVRotPos += m_vPOVRotInc;
}


void CVoroWrapper::WritePOV(CTimeStep *ts, const char *s, int mol, int smol)
{
	FILE *b;
	int ijk, q, z, z2, z3, z4, id, faces, co, fc, ti, o, o2;
	CVoroAtom *va;
	CVoroMolecule *vm;
	CMolecule *m;
	CElement *el, *el2;
	CSingleMolecule *sm;
	bool n=false;
	double *pp, tx, ty, tz;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	vector<int> fv;
	CxVector3 vec1, vec2, vec3, vec1b, vec2b, vecA, vecB, vecC, vecD, cam;
	CxIntArray nbmol, nbsm;
	CxIntArray nbatoms;
	CMolBond *mb;
	float cr, cg, cb;
	double r_edge = 0.005;

	m = (CMolecule*)g_oaMolecules[mol];
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[smol]];

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

	mfprintf(m_fPOVScript,"\"%s\" /EXIT /RENDER \"%s\" +w%d +h%d +fn +a0.3\n",m_sPOVExe,s,m_iPOVResX,m_iPOVResY);
	fflush(m_fPOVScript);

	b = OpenFileWrite(s,true);

	mfprintf(b,"// Written by TRAVIS\n");
	mfprintf(b,"// See http://www.travis-analyzer.de\n\n");
	mfprintf(b,"#version 3.6;\n");
	mfprintf(b,"\n");

	if (m_bPOVFaces)
	{
		mfprintf(b,"/**** Faces ****/\n");

		mfprintf(b,"#declare face_draw          = true;\n");
		if (m_bPOVFaceColor)
			mfprintf(b,"//#declare face_color         = < 0.3, 0.3, 1.0 >;\n");
				else mfprintf(b,"#declare face_color         = < 0.3, 0.3, 1.0 >;\n");
		mfprintf(b,"#declare face_trans         = %.3f;\n",m_fPOVFaceOpac);
		mfprintf(b,"#declare face_specular      = 0.6;\n");
		mfprintf(b,"#declare face_reflect       = 0;\n");
		mfprintf(b,"#declare face_ambient       = 0.3;\n");
		mfprintf(b,"#declare face_diffuse       = 0.8;\n");

		mfprintf(b,"\n");
	}

	if (m_bPOVEdges)
	{
		mfprintf(b,"/**** Edges ****/\n");

		mfprintf(b,"#declare edge_draw           = true;\n");
		if (m_bPOVFaces)
		{
			r_edge = 0.005;
			mfprintf(b,"#declare edge_color          = < 0.3, 0.3, 0.3, 0, 0 >;\n");
		} else
		{
			r_edge = 0.005;
			mfprintf(b,"#declare edge_color          = < 0.4, 0.4, 0.6, 0, 0 >;\n");
		}
		mfprintf(b,"#declare edge_r              = %f;\n",r_edge);
		mfprintf(b,"#declare edge_specular       = 0.8;\n");
		mfprintf(b,"#declare edge_reflect        = 0;\n");
		mfprintf(b,"#declare edge_ambient        = 0.1;\n");
		mfprintf(b,"#declare edge_diffuse        = 0.9;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare edge_draw_halo1     = false;\n");
		mfprintf(b,"#declare edge_r_halo1        = 0.008;\n");
		mfprintf(b,"#declare edge_l_halo1        = 0.006;\n");
		mfprintf(b,"#declare edge_d_halo1        = 0.2;\n");
		mfprintf(b,"#declare edge_color_halo1    = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare edge_specular_halo1 = 0;\n");
		mfprintf(b,"#declare edge_reflect_halo1  = 0;\n");
		mfprintf(b,"#declare edge_ambient_halo1  = 1.0;\n");
		mfprintf(b,"#declare edge_diffuse_halo1  = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare edge_draw_halo2     = false;\n");
		mfprintf(b,"#declare edge_r_halo2        = 0.006;\n");
		mfprintf(b,"#declare edge_l_halo2        = 0.008;\n");
		mfprintf(b,"#declare edge_d_halo2        = 0.4;\n");
		mfprintf(b,"#declare edge_color_halo2    = < 1, 1, 1, 0, 0 >;\n");
		mfprintf(b,"#declare edge_specular_halo2 = 0;\n");
		mfprintf(b,"#declare edge_reflect_halo2  = 0;\n");
		mfprintf(b,"#declare edge_ambient_halo2  = 1.0;\n");
		mfprintf(b,"#declare edge_diffuse_halo2  = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare edge_draw_halo3     = false;\n");
		mfprintf(b,"#declare edge_r_halo3        = 0.004;\n");
		mfprintf(b,"#declare edge_l_halo3        = 0.012;\n");
		mfprintf(b,"#declare edge_d_halo3        = 0.401;\n");
		mfprintf(b,"#declare edge_color_halo3    = < 1, 1, 1, 0, 0.5 >;\n");
		mfprintf(b,"#declare edge_specular_halo3 = 0;\n");
		mfprintf(b,"#declare edge_reflect_halo3  = 0;\n");
		mfprintf(b,"#declare edge_ambient_halo3  = 1.0;\n");
		mfprintf(b,"#declare edge_diffuse_halo3  = 0;\n");

		mfprintf(b,"\n");
	}

	if (m_bPOVVertices)
	{
		mfprintf(b,"/**** Vertices ****/\n");
		mfprintf(b,"#declare vertex_draw         = true;\n");
		mfprintf(b,"#declare vertex_r            = %.4f;\n",r_edge+0.0015);
		mfprintf(b,"#declare vertex_color        = < 0.2, 0.2, 0.2, 0, 0 >;\n");
		mfprintf(b,"#declare vertex_specular     = 0.6;\n");
		mfprintf(b,"#declare vertex_reflect      = 0;\n");
		mfprintf(b,"#declare vertex_ambient      = 0.2;\n");
		mfprintf(b,"#declare vertex_diffuse      = 0.8;\n");
		mfprintf(b,"\n");
	}

	if (m_bPOVAtoms)
	{
		mfprintf(b,"/**** Atoms ****/\n");

		mfprintf(b,"#declare atom_draw           = true;\n");
		mfprintf(b,"#declare atom_r              = 0.65;\n");
		if (m_bPOVFaces)
			mfprintf(b,"#declare atom_specular      = 0.2;\n");
				else mfprintf(b,"#declare atom_specular      = 0.7;\n");
		mfprintf(b,"#declare atom_reflect        = 0;\n");
		mfprintf(b,"#declare atom_ambient        = 0.2;\n");
		mfprintf(b,"#declare atom_diffuse        = 0.7;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare atom_draw_halo1      = true;\n");
		mfprintf(b,"#declare atom_r_halo1         = 0.008;\n");
		mfprintf(b,"#declare atom_d_halo1         = 0.0125;\n");
		mfprintf(b,"#declare atom_color_halo1     = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare atom_specular_halo1  = 0;\n");
		mfprintf(b,"#declare atom_reflect_halo1   = 0;\n");
		mfprintf(b,"#declare atom_ambient_halo1   = 1.0;\n");
		mfprintf(b,"#declare atom_diffuse_halo1   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare atom_draw_halo2      = true;\n");
		mfprintf(b,"#declare atom_r_halo2         = 0.003;\n");
		mfprintf(b,"#declare atom_d_halo2         = 0.01875;\n");
		if (m_bPOVFaces)
			mfprintf(b,"#declare atom_color_halo2     = < 1.7, 1.7, 1.7, 0, 0 >;\n");
				else mfprintf(b,"#declare atom_color_halo2     = < 1, 1, 1, 0, 0 >;\n");
		mfprintf(b,"#declare atom_specular_halo2  = 0;\n");
		mfprintf(b,"#declare atom_reflect_halo2   = 0;\n");
		mfprintf(b,"#declare atom_ambient_halo2   = 1.0;\n");
		mfprintf(b,"#declare atom_diffuse_halo2   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare atom_draw_halo3      = true;\n");
		mfprintf(b,"#declare atom_r_halo3         = 0.003;\n");
		mfprintf(b,"#declare atom_d_halo3         = 0.025;\n");
		if (m_bPOVFaces)
			mfprintf(b,"#declare atom_color_halo3     = < 1.7, 1.7, 1.7, 0, 0.5 >;\n");
				else mfprintf(b,"#declare atom_color_halo3     = < 1, 1, 1, 0, 0.5 >;\n");
		mfprintf(b,"#declare atom_specular_halo3  = 0;\n");
		mfprintf(b,"#declare atom_reflect_halo3   = 0;\n");
		mfprintf(b,"#declare atom_ambient_halo3   = 1.0;\n");
		mfprintf(b,"#declare atom_diffuse_halo3   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"/**** Bonds ****/\n");

		mfprintf(b,"#declare bond_draw        = true;\n");
		mfprintf(b,"#declare bond_r           = 0.015;\n");
		mfprintf(b,"#declare bond_specular    = 0.7;\n");
		mfprintf(b,"#declare bond_reflect     = 0;\n");
		mfprintf(b,"#declare bond_ambient     = 0.2;\n");
		mfprintf(b,"#declare bond_diffuse     = 0.7;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare bond_draw_halo1      = true;\n");
		mfprintf(b,"#declare bond_r_halo1         = 0.008;\n");
		mfprintf(b,"#declare bond_d_halo1         = 0;\n");
		mfprintf(b,"#declare bond_color_halo1     = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare bond_specular_halo1  = 0;\n");
		mfprintf(b,"#declare bond_reflect_halo1   = 0;\n");
		mfprintf(b,"#declare bond_ambient_halo1   = 1.0;\n");
		mfprintf(b,"#declare bond_diffuse_halo1   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare bond_draw_halo2      = true;\n");
		mfprintf(b,"#declare bond_r_halo2         = 0.003;\n");
		mfprintf(b,"#declare bond_d_halo2         = 0.0286;\n");
		if (m_bPOVFaces)
			mfprintf(b,"#declare bond_color_halo2     = < 1.7, 1.7, 1.7, 0, 0 >;\n");
				else mfprintf(b,"#declare bond_color_halo2     = < 1, 1, 1, 0, 0 >;\n");
		mfprintf(b,"#declare bond_specular_halo2  = 0;\n");
		mfprintf(b,"#declare bond_reflect_halo2   = 0;\n");
		mfprintf(b,"#declare bond_ambient_halo2   = 1.0;\n");
		mfprintf(b,"#declare bond_diffuse_halo2   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare bond_draw_halo3      = true;\n");
		mfprintf(b,"#declare bond_r_halo3         = 0.003;\n");
		mfprintf(b,"#declare bond_d_halo3         = 0.0333;\n");
		if (m_bPOVFaces)
			mfprintf(b,"#declare bond_color_halo3     = < 1.7, 1.7, 1.7, 0, 0.5 >;\n");
				else mfprintf(b,"#declare bond_color_halo3     = < 1, 1, 1, 0, 0.5 >;\n");
		mfprintf(b,"#declare bond_specular_halo3  = 0;\n");
		mfprintf(b,"#declare bond_reflect_halo3   = 0;\n");
		mfprintf(b,"#declare bond_ambient_halo3   = 1.0;\n");
		mfprintf(b,"#declare bond_diffuse_halo3   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare bond_draw_stub      = true;\n");
		mfprintf(b,"#declare bond_r_stub         = 0.004;\n");
		mfprintf(b,"#declare bond_l_stub         = 0.004;\n");
		mfprintf(b,"#declare bond_color_stub     = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare bond_specular_stub  = 0;\n");
		mfprintf(b,"#declare bond_reflect_stub   = 0;\n");
		mfprintf(b,"#declare bond_ambient_stub   = 1.0;\n");
		mfprintf(b,"#declare bond_diffuse_stub   = 0;\n");

		mfprintf(b,"\n");
	}

	if (m_bPOVDrawNeighbors)
	{
		mfprintf(b,"/**** Neighboring Atoms ****/\n");

		mfprintf(b,"#declare nbatom_draw        = true;\n");
		mfprintf(b,"#declare nbatom_r           = 0.65;\n");
		mfprintf(b,"#declare nbatom_specular    = 0.7;\n");
		mfprintf(b,"#declare nbatom_reflect     = 0;\n");
		mfprintf(b,"#declare nbatom_ambient     = 0.2;\n");
		mfprintf(b,"#declare nbatom_diffuse     = 0.7;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbatom_draw_halo1      = true;\n");
		mfprintf(b,"#declare nbatom_r_halo1         = 0.008;\n");
		mfprintf(b,"#declare nbatom_d_halo1         = 0.0125;\n");
		mfprintf(b,"#declare nbatom_color_halo1     = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare nbatom_specular_halo1  = 0;\n");
		mfprintf(b,"#declare nbatom_reflect_halo1   = 0;\n");
		mfprintf(b,"#declare nbatom_ambient_halo1   = 1.0;\n");
		mfprintf(b,"#declare nbatom_diffuse_halo1   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbatom_draw_halo2      = true;\n");
		mfprintf(b,"#declare nbatom_r_halo2         = 0.003;\n");
		mfprintf(b,"#declare nbatom_d_halo2         = 0.01875;\n");
		mfprintf(b,"#declare nbatom_color_halo2     = < 1, 1, 1, 0, 0 >;\n");
		mfprintf(b,"#declare nbatom_specular_halo2  = 0;\n");
		mfprintf(b,"#declare nbatom_reflect_halo2   = 0;\n");
		mfprintf(b,"#declare nbatom_ambient_halo2   = 1.0;\n");
		mfprintf(b,"#declare nbatom_diffuse_halo2   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbatom_draw_halo3      = true;\n");
		mfprintf(b,"#declare nbatom_r_halo3         = 0.003;\n");
		mfprintf(b,"#declare nbatom_d_halo3         = 0.025;\n");
		mfprintf(b,"#declare nbatom_color_halo3     = < 1, 1, 1, 0, 0.5 >;\n");
		mfprintf(b,"#declare nbatom_specular_halo3  = 0;\n");
		mfprintf(b,"#declare nbatom_reflect_halo3   = 0;\n");
		mfprintf(b,"#declare nbatom_ambient_halo3   = 1.0;\n");
		mfprintf(b,"#declare nbatom_diffuse_halo3   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"/**** Neighboring Bonds ****/\n");

		mfprintf(b,"#declare nbbond_draw        = true;\n");
		mfprintf(b,"#declare nbbond_r           = 0.015;\n");
		mfprintf(b,"#declare nbbond_specular    = 0.7;\n");
		mfprintf(b,"#declare nbbond_reflect     = 0;\n");
		mfprintf(b,"#declare nbbond_ambient     = 0.2;\n");
		mfprintf(b,"#declare nbbond_diffuse     = 0.7;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbbond_draw_halo1      = true;\n");
		mfprintf(b,"#declare nbbond_r_halo1         = 0.008;\n");
		mfprintf(b,"#declare nbbond_d_halo1         = 0;\n");
		mfprintf(b,"#declare nbbond_color_halo1     = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare nbbond_specular_halo1  = 0;\n");
		mfprintf(b,"#declare nbbond_reflect_halo1   = 0;\n");
		mfprintf(b,"#declare nbbond_ambient_halo1   = 1.0;\n");
		mfprintf(b,"#declare nbbond_diffuse_halo1   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbbond_draw_halo2      = true;\n");
		mfprintf(b,"#declare nbbond_r_halo2         = 0.003;\n");
		mfprintf(b,"#declare nbbond_d_halo2         = 0.0286;\n");
		mfprintf(b,"#declare nbbond_color_halo2     = < 1, 1, 1, 0, 0 >;\n");
		mfprintf(b,"#declare nbbond_specular_halo2  = 0;\n");
		mfprintf(b,"#declare nbbond_reflect_halo2   = 0;\n");
		mfprintf(b,"#declare nbbond_ambient_halo2   = 1.0;\n");
		mfprintf(b,"#declare nbbond_diffuse_halo2   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbbond_draw_halo3      = true;\n");
		mfprintf(b,"#declare nbbond_r_halo3         = 0.003;\n");
		mfprintf(b,"#declare nbbond_d_halo3         = 0.0333;\n");
		mfprintf(b,"#declare nbbond_color_halo3     = < 1, 1, 1, 0, 0.5 >;\n");
		mfprintf(b,"#declare nbbond_specular_halo3  = 0;\n");
		mfprintf(b,"#declare nbbond_reflect_halo3   = 0;\n");
		mfprintf(b,"#declare nbbond_ambient_halo3   = 1.0;\n");
		mfprintf(b,"#declare nbbond_diffuse_halo3   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbbond_draw_stub      = true;\n");
		mfprintf(b,"#declare nbbond_r_stub         = 0.004;\n");
		mfprintf(b,"#declare nbbond_l_stub         = 0.004;\n");
		mfprintf(b,"#declare nbbond_color_stub     = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare nbbond_specular_stub  = 0;\n");
		mfprintf(b,"#declare nbbond_reflect_stub   = 0;\n");
		mfprintf(b,"#declare nbbond_ambient_stub   = 1.0;\n");
		mfprintf(b,"#declare nbbond_diffuse_stub   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"/**** Rays ****/\n");

		mfprintf(b,"#declare ray_draw            = true;\n");
		mfprintf(b,"#declare ray_color           = <1.0, 1.0, 0.2, 0, 0>;\n");
		mfprintf(b,"#declare ray_r               = 0.005;\n");
		mfprintf(b,"#declare ray_specular        = 0.4;\n");
		mfprintf(b,"#declare ray_reflect         = 0;\n");
		mfprintf(b,"#declare ray_ambient         = 0.2;\n");
		mfprintf(b,"#declare ray_diffuse         = 0.9;\n");

		mfprintf(b,"\n");
		
		mfprintf(b,"#declare ray_draw_halo1      = true;\n");
		mfprintf(b,"#declare ray_r_halo1         = 0.005;\n");
		mfprintf(b,"#declare ray_d_halo1         = 0;\n");
		mfprintf(b,"#declare ray_color_halo1     = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare ray_specular_halo1  = 0;\n");
		mfprintf(b,"#declare ray_reflect_halo1   = 0;\n");
		mfprintf(b,"#declare ray_ambient_halo1   = 1.0;\n");
		mfprintf(b,"#declare ray_diffuse_halo1   = 0;\n");

		mfprintf(b,"\n");
		
		mfprintf(b,"#declare ray_draw_halo2      = true;\n");
		mfprintf(b,"#declare ray_r_halo2         = 0.002;\n");
		mfprintf(b,"#declare ray_d_halo2         = 0.001;\n");
		mfprintf(b,"#declare ray_color_halo2     = < 1, 1, 1, 0, 0 >;\n");
		mfprintf(b,"#declare ray_specular_halo2  = 0;\n");
		mfprintf(b,"#declare ray_reflect_halo2   = 0;\n");
		mfprintf(b,"#declare ray_ambient_halo2   = 1.0;\n");
		mfprintf(b,"#declare ray_diffuse_halo2   = 0;\n");

		mfprintf(b,"\n");
		
		mfprintf(b,"#declare ray_draw_halo3      = true;\n");
		mfprintf(b,"#declare ray_r_halo3         = 0.002;\n");
		mfprintf(b,"#declare ray_d_halo3         = 0.0011;\n");
		mfprintf(b,"#declare ray_color_halo3     = < 1, 1, 1, 0, 0.5 >;\n");
		mfprintf(b,"#declare ray_specular_halo3  = 0;\n");
		mfprintf(b,"#declare ray_reflect_halo3   = 0;\n");
		mfprintf(b,"#declare ray_ambient_halo3   = 1.0;\n");
		mfprintf(b,"#declare ray_diffuse_halo3   = 0;\n");

		mfprintf(b,"\n");
		
		mfprintf(b,"#declare ray_draw_stub       = true;\n");
		mfprintf(b,"#declare ray_r_stub          = 0.003;\n");
		mfprintf(b,"#declare ray_l_stub          = 0.004;\n");
		mfprintf(b,"#declare ray_color_stub      = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare ray_specular_stub   = 0;\n");
		mfprintf(b,"#declare ray_reflect_stub    = 0;\n");
		mfprintf(b,"#declare ray_ambient_stub    = 1.0;\n");
		mfprintf(b,"#declare ray_diffuse_stub    = 0;\n");

		mfprintf(b,"\n");
	}

	mfprintf(b,"/**** Element Colors ****/\n");
	for (z=0;z<g_oaAtoms.GetSize();z++)
	{
		if (z == g_iVirtAtomType)
			continue;
		el = ((CAtom*)g_oaAtoms[z])->m_pElement;
		mfprintf(b,"#declare element_%s_color   = < %f, %f, %f, 0, 0 >;\n",el->m_sLabel,el->m_iColorR/255.0,el->m_iColorG/255.0,el->m_iColorB/255.0);
	}

	mfprintf(b,"\n/**** Element Radii ****/\n");
	for (z=0;z<g_oaAtoms.GetSize();z++)
	{
		if (z == g_iVirtAtomType)
			continue;
		el = ((CAtom*)g_oaAtoms[z])->m_pElement;
		mfprintf(b,"#declare element_%s_radius  = %f;\n",el->m_sLabel,el->m_fRadius/1000.0);
	}
	mfprintf(b,"\n");

	cam[0] = 0;
	cam[1] = 0;
	cam[2] = m_fPOVCameraDist;

	mfprintf(b,"\ncamera {\n");
	mfprintf(b,"	location < %f, %f, %f >\n",cam[0],cam[1],cam[2]);
	mfprintf(b,"	sky y\n");
	mfprintf(b,"	right -0.06*x*image_width/image_height\n");
	mfprintf(b,"	up 0.06*y\n");
	mfprintf(b,"	look_at < 0, 0, 0 >\n");
	mfprintf(b,"}\n");
	mfprintf(b,"\n");
	mfprintf(b,"// Solid colored Background\n");
	mfprintf(b,"background { rgb < 1.0, 1.0, 1.0 > }\n");
	mfprintf(b,"\n");

	mfprintf(b,"/***** Color Gradient Background ****/");
	mfprintf(b,"/*sky_sphere {\n");
	mfprintf(b,"  pigment {\n");
	mfprintf(b,"    gradient y\n");
	mfprintf(b,"    color_map {\n");
	mfprintf(b,"      [ 0 color rgb < 0.05, 0.05, 0.05 > ]\n");
	mfprintf(b,"      [ 1 color rgb < 0.20, 0.16, 0.50 > ]\n");
	mfprintf(b,"    }\n");
	mfprintf(b,"    scale 0.1\n");
	mfprintf(b,"    translate -0.05\n");
	mfprintf(b,"  }\n");
	mfprintf(b,"}*/\n\n");

	mfprintf(b,"// Two light sources\n");
	mfprintf(b,"light_source { < -8, 20, 20 > color rgb 0.8 }\n");
	mfprintf(b,"light_source { < 25, 12, 20 > color rgb 0.5 }\n\n");

	mfprintf(b,"\n/****** Below this line, no changes should be necessary **********/\n\n\n");

	mfprintf(b,"#macro m_face_color(col)\n");
	mfprintf(b,"  #ifdef(face_color)\n");
	mfprintf(b,"    < face_color.x, face_color.y, face_color.z, 0, face_trans >\n");
	mfprintf(b,"  #else\n");
	mfprintf(b,"    < col.x, col.y, col.z, 0, face_trans >\n");
	mfprintf(b,"  #end\n");
	mfprintf(b,"#end\n");

	mfprintf(b,"\nunion {\n");

	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				co++;

				ijk=vl.ijk;
				q=vl.q;
				pp=m_pContainer->p[ijk]+m_pContainer->ps*q;

				id = m_pContainer->id[ijk][q];

				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				ti = 0;

				if ((vm->m_iMolecule == mol) && (vm->m_iSingleMol == smol))
				{
					c.neighbors(nb);
					faces = c.number_of_faces();
					c.face_vertices(fv);

					mfprintf(b,"\n// Voronoi cell %d\n",m_pContainer->id[vl.ijk][vl.q]);
					mfprintf(b,"union {\n");

					for (z=0;z<faces;z++)
					{
				//		mprintf("Face %d/%d:\n",z+1,faces);
						fc = fv[ti];
				//		mprintf("  %d Vertices\n",fc);

						if (m_pAssignMolecules[id] != m_pAssignMolecules[nb[z]])
						{
							tx = 0;
							ty = 0;
							tz = 0;
							for (z2=0;z2<fc;z2++)
							{
					//			mprintf("  - Vertex %d/%d; Id=%d, ti=%d (%.6f, %.6f, %.6f) -> (%.6f, %.6f, %.6f)\n",z2+1,fc,fv[ti+z2+1]+1,ti,c.pts[fv[ti+z2+1]*3]*0.5,c.pts[fv[ti+z2+1]*3+1]*0.5,c.pts[fv[ti+z2+1]*3+2]*0.5,*pp+c.pts[fv[ti+z2+1]*3]*0.5,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5);

								vec1 = CxVector3(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0);

								if (m_bPOVRot)
									vec1 = m_mPOVMat * vec1;

								if (m_bPOVVertices)
								{
									mfprintf(b,"#if (vertex_draw)\n");
									mfprintf(b,"  sphere { < %.6f, %.6f, %.6f >, vertex_r\n",vec1[0],vec1[1],vec1[2]);
									mfprintf(b,"    pigment { rgbft vertex_color } finish { reflection vertex_reflect specular vertex_specular ambient vertex_ambient diffuse vertex_diffuse } }\n");
									mfprintf(b,"#end\n");
								}

								if (m_bPOVEdges)
								{
									n = false;

									if (z2 < fc-1)
									{
										if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+z2+2]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+z2+2]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+z2+2]*3+2],2)) > 0.0000000001)
										{
											vec2 = CxVector3(*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
											n = true;
										}
									} else
									{
										if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+1]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+1]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+1]*3+2],2)) > 0.0000000001)
										{
											vec2 = CxVector3(*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);
											n = true;
										}
									}

									if (n)
									{
										if (m_bPOVRot)
											vec2 = m_mPOVMat * vec2;

										mfprintf(b,"#if (edge_draw)\n");
										mfprintf(b,"  cylinder { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f >, edge_r\n",vec1[0],vec1[1],vec1[2],vec2[0],vec2[1],vec2[2]);
										mfprintf(b,"    pigment { rgbft edge_color } finish { reflection edge_reflect specular edge_specular ambient edge_ambient diffuse edge_diffuse } }\n");
										mfprintf(b,"#end\n");

										vec3 = cam - (vec1 + vec2) / 2.0;
										vec3.Normalize();

										vec2b = vec2-vec1;
										vec2b.Normalize();
										vec1b = CrossP(vec3,vec2b);

										vec1b.Normalize();

						//				if (halo)
										{
											mfprintf(b,"#if (edge_draw_halo1)\n");
											mfprintf(b,"  triangle {\n    < ");
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >,\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    pigment { rgbft edge_color_halo1 } finish { reflection edge_reflect_halo1 specular edge_specular_halo1 ambient edge_ambient_halo1 diffuse edge_diffuse_halo1 } no_reflection no_radiosity }\n");

											mfprintf(b,"  triangle {\n    < ");
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    pigment { rgbft edge_color_halo1 } finish { reflection edge_reflect_halo1 specular edge_specular_halo1 ambient edge_ambient_halo1 diffuse edge_diffuse_halo1 } no_reflection no_radiosity }\n");
											mfprintf(b,"#end\n");


											mfprintf(b,"#if (edge_draw_halo2)\n");
											mfprintf(b,"  triangle {\n    < ");
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >,\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    pigment { rgbft edge_color_halo2 } finish { reflection edge_reflect_halo2 specular edge_specular_halo2 ambient edge_ambient_halo2 diffuse edge_diffuse_halo2 } no_reflection no_radiosity }\n");

											mfprintf(b,"  triangle {\n    < ");
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    pigment { rgbft edge_color_halo2 } finish { reflection edge_reflect_halo2 specular edge_specular_halo2 ambient edge_ambient_halo2 diffuse edge_diffuse_halo2 } no_reflection no_radiosity }\n");
											mfprintf(b,"#end\n");

											
											mfprintf(b,"#if (edge_draw_halo3)\n");
											mfprintf(b,"  triangle {\n    < ");
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >,\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    pigment { rgbft edge_color_halo3 } finish { reflection edge_reflect_halo3 specular edge_specular_halo3 ambient edge_ambient_halo3 diffuse edge_diffuse_halo3 } no_reflection no_radiosity }\n");

											mfprintf(b,"  triangle {\n    < ");
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    < ");
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
											mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
											mfprintf(b,"    pigment { rgbft edge_color_halo3 } finish { reflection edge_reflect_halo3 specular edge_specular_halo3 ambient edge_ambient_halo3 diffuse edge_diffuse_halo3 } no_reflection no_radiosity }\n");
											mfprintf(b,"#end\n");
										}
									}
								}

								tx += vec1[0];
								ty += vec1[1];
								tz += vec1[2];
							}

							if (m_bPOVFaces)
							{
								tx /= fc;
								ty /= fc;
								tz /= fc;

								if (m_bPOVDrawNeighbors)
								{
									vec1 = ts->m_vaCoords[nb[z]];
									vec1 /= 1000.0;
									vec1 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

									el = ((CAtom*)g_oaAtoms[g_waAtomRealElement[nb[z]]])->m_pElement;

									if (m_bPOVRot)
										vec1 = m_mPOVMat * vec1;

									vec2[0] = tx;
									vec2[1] = ty;
									vec2[2] = tz;

									mfprintf(b,"#if (ray_draw)\n");
									mfprintf(b,"  cylinder { <%.6f, %.6f, %.6f>, <%.6f, %.6f, %.6f>, ray_r\n",vec1[0],vec1[1],vec1[2],vec2[0],vec2[1],vec2[2]);
									mfprintf(b,"    pigment { rgbft ray_color } finish { reflection ray_reflect specular ray_specular ambient ray_ambient diffuse ray_diffuse } }\n");
									mfprintf(b,"#end\n");

									vec3 = cam - (vec1 + vec2) / 2.0;
									vec3.Normalize();

									vec2b = vec2-vec1;
									vec1b = CrossP(vec3,vec2b);

									vec1b.Normalize();

					//				if (shadow)
									{
										mfprintf(b,"#if (ray_draw_halo1)\n");
										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >,\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo1 } finish { reflection ray_reflect_halo1 specular ray_specular_halo1 ambient ray_ambient_halo1 diffuse ray_diffuse_halo1 } no_reflection no_radiosity }\n");

										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo1 } finish { reflection ray_reflect_halo1 specular ray_specular_halo1 ambient ray_ambient_halo1 diffuse ray_diffuse_halo1 } no_reflection no_radiosity }\n");
										mfprintf(b,"#end\n");
									}

					//				if (halo)
									{
										mfprintf(b,"#if (ray_draw_halo2)\n");
										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >,\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo2 } finish { reflection ray_reflect_halo2 specular ray_specular_halo2 ambient ray_ambient_halo2 diffuse ray_diffuse_halo2 } no_reflection no_radiosity }\n");

										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo2 } finish { reflection ray_reflect_halo2 specular ray_specular_halo2 ambient ray_ambient_halo2 diffuse ray_diffuse_halo2 } no_reflection no_radiosity }\n");
										mfprintf(b,"#end\n");

										
										mfprintf(b,"#if (ray_draw_halo3)\n");
										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >,\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo3 } finish { reflection ray_reflect_halo3 specular ray_specular_halo3 ambient ray_ambient_halo3 diffuse ray_diffuse_halo3 } no_reflection no_radiosity }\n");

										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo3 } finish { reflection ray_reflect_halo3 specular ray_specular_halo3 ambient ray_ambient_halo3 diffuse ray_diffuse_halo3 } no_reflection no_radiosity }\n");
										mfprintf(b,"#end\n");
									}

					//				if (shadow)
									{
										vec3 = vec2 - vec1;
										vec3.Normalize();

										mfprintf(b,"#if (ray_draw_stub)\n");
										mfprintf(b,"  cylinder { <%.6f, %.6f, %.6f>,\n    < ",vec1[0],vec1[1],vec1[2]);
										mfprintf(b,"%.6f + (%.6f) * ( ((%.6f) * nbatom_r) - ((%.6f) * nbatom_r - 0.5 * sqrt(4*(%.6f)*(%.6f)*nbatom_r*nbatom_r - 4*(ray_r+ray_r_stub)*(ray_r+ray_r_stub) )) + ray_l_stub ), ",vec1[0],vec3[0],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
										mfprintf(b,"%.6f + (%.6f) * ( ((%.6f) * nbatom_r) - ((%.6f) * nbatom_r - 0.5 * sqrt(4*(%.6f)*(%.6f)*nbatom_r*nbatom_r - 4*(ray_r+ray_r_stub)*(ray_r+ray_r_stub) )) + ray_l_stub ), ",vec1[1],vec3[1],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
										mfprintf(b,"%.6f + (%.6f) * ( ((%.6f) * nbatom_r) - ((%.6f) * nbatom_r - 0.5 * sqrt(4*(%.6f)*(%.6f)*nbatom_r*nbatom_r - 4*(ray_r+ray_r_stub)*(ray_r+ray_r_stub) )) + ray_l_stub ) >,\n",vec1[2],vec3[2],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
										mfprintf(b,"    ray_r+ray_r_stub\n");
										mfprintf(b,"    pigment { rgbft ray_color_stub } finish { reflection ray_reflect_stub specular ray_specular_stub ambient ray_ambient_stub diffuse ray_diffuse_stub } no_radiosity }\n");

										mfprintf(b,"  cylinder { <%.6f, %.6f, %.6f>,\n    < ",vec2[0],vec2[1],vec2[2]);
										mfprintf(b,"%.6f - (%.6f) * ray_l_stub, ",vec2[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f) * ray_l_stub, ",vec2[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f) * ray_l_stub >,",vec2[2],vec3[2]);
										mfprintf(b,"    ray_r+ray_r_stub\n");
										mfprintf(b,"    pigment { rgbft ray_color_stub } finish { reflection ray_reflect_stub specular ray_specular_stub ambient ray_ambient_stub diffuse ray_diffuse_stub } no_radiosity }\n");
										mfprintf(b,"#end\n");
									}
								}

								mfprintf(b,"#if (face_draw)\n");
								mfprintf(b,"\n  mesh { \n");

								for (z2=0;z2<fc;z2++)
								{
									vec1 = CxVector3(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0);

									if (z2 < fc-1)
										vec2 = CxVector3(*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
											else vec2 = CxVector3(*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);

									if (m_bPOVRot)
									{
										vec1 = m_mPOVMat * vec1;
										vec2 = m_mPOVMat * vec2;
									}

									mfprintf(b,"    triangle { <%.6f, %.6f, %.6f>, <%.6f, %.6f, %.6f>, <%.6f, %.6f, %.6f> }\n",tx,ty,tz,vec1[0],vec1[1],vec1[2],vec2[0],vec2[1],vec2[2]);
								}

								if (m_bPOVDrawNeighbors)
								{
									nbatoms.Add(nb[z]);
									for (z2=0;z2<nbmol.GetSize();z2++)
									{
										if ((nbmol[z2] == ((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z]]])->m_iMolecule) && (nbsm[z2] == ((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z]]])->m_iSingleMol))
											goto _nbdone;
									}
									nbmol.Add(((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z]]])->m_iMolecule);
									nbsm.Add(((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z]]])->m_iSingleMol);
_nbdone:;
								}

								if (m_bPOVFaceColor)
								{
									if (m_bPOVFaceColorRef)
										POVFindColor(id,&cr,&cg,&cb,0);
											else POVFindColor(nb[z],&cr,&cg,&cb,0);

									mfprintf(b,"    texture { pigment { rgbft m_face_color(<%.3f, %.3f, %.3f>) } finish { reflection face_reflect specular face_specular ambient face_ambient diffuse face_diffuse }  }\n",cr,cg,cb);
								} else mfprintf(b,"    texture { pigment { rgbft m_face_color(<1,1,1>) } finish { reflection face_reflect specular face_specular ambient face_ambient diffuse face_diffuse }  }\n");

								mfprintf(b,"  }\n\n");
								mfprintf(b,"#end\n");
							}
						}

						ti += fv[ti]+1;
					}

					mfprintf(b,"\n");
					mfprintf(b,"}\n\n");

					if (m_bPOVAtoms)
					{
						vec1 = CxVector3(*pp-g_fBoxX/2000.0,pp[1]-g_fBoxY/2000.0,pp[2]-g_fBoxZ/2000.0);
						el = ((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement;

						if (m_bPOVRot)
							vec1 = m_mPOVMat * vec1;

						mfprintf(b,"#if (atom_draw)\n");
						mfprintf(b,"// Particle %d\nsphere { < %.6f, %.6f, %.6f >, element_%s_radius*atom_r\n",m_pContainer->id[vl.ijk][vl.q],vec1[0],vec1[1],vec1[2],el->m_sLabel);

						if (m_bPOVAtomGrey)
						{
							z2 = (el->m_iColorR + el->m_iColorG + el->m_iColorB) / 4.0;
							mfprintf(b,"  pigment { rgb < %f, %f, %f > } finish { reflection atom_reflect specular atom_specular ambient atom_ambient diffuse atom_diffuse } }\n",z2/255.0,z2/255.0,z2/255.0);
						} else mfprintf(b,"  pigment { rgbft element_%s_color } finish { reflection atom_reflect specular atom_specular ambient atom_ambient diffuse atom_diffuse } }\n",el->m_sLabel);
						mfprintf(b,"#end\n");

						vec2 = cam - vec1;
						vec2.Normalize();

		//				if (shadow)
						{
							mfprintf(b,"#if (atom_draw_halo1)\n");
							mfprintf(b,"  disc { < %.6f - (%.6f) * atom_d_halo1, %.6f - (%.6f) * atom_d_halo1, %.6f - (%.6f) * atom_d_halo1 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
							mfprintf(b,"    < %.6f, %.6f, %.6f >, (%f * atom_r) + atom_r_halo1\n",vec2[0],vec2[1],vec2[2],el->m_fRadius/1000.0);
							mfprintf(b,"    pigment { rgbft atom_color_halo1 } finish { reflection atom_reflect_halo1 specular atom_specular_halo1 ambient atom_ambient_halo1 diffuse atom_diffuse_halo1 } no_reflection no_radiosity }\n");
							mfprintf(b,"#end\n");
						}

		//				if (halo)
						{
							mfprintf(b,"#if (atom_draw_halo2)\n");
							mfprintf(b,"  disc { < %.6f - (%.6f) * atom_d_halo2, %.6f - (%.6f) * atom_d_halo2, %.6f - (%.6f) * atom_d_halo2 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
							mfprintf(b,"    < %.6f, %.6f, %.6f >, (%f * atom_r) + atom_r_halo1 + atom_r_halo2\n",vec2[0],vec2[1],vec2[2],el->m_fRadius/1000.0);
							mfprintf(b,"    pigment { rgbft atom_color_halo2 } finish { reflection atom_reflect_halo2 specular atom_specular_halo2 ambient atom_ambient_halo2 diffuse atom_diffuse_halo2 } no_reflection no_radiosity }\n");
							mfprintf(b,"#end\n");

							mfprintf(b,"#if (atom_draw_halo3)\n");
							mfprintf(b,"  disc { < %.6f - (%.6f) * atom_d_halo3, %.6f - (%.6f) * atom_d_halo3, %.6f - (%.6f) * atom_d_halo3 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
							mfprintf(b,"    < %.6f, %.6f, %.6f >, (%f * atom_r) + atom_r_halo1 + atom_r_halo2 + atom_r_halo3\n",vec2[0],vec2[1],vec2[2],el->m_fRadius/1000.0);
							mfprintf(b,"    pigment { rgbft atom_color_halo3 } finish { reflection atom_reflect_halo3 specular atom_specular_halo3 ambient atom_ambient_halo3 diffuse atom_diffuse_halo3 } no_reflection no_radiosity }\n");
							mfprintf(b,"#end\n");
						}
					}
				}
			}
		} while (vl.inc());
	}

	for (z2=0;z2<sm->m_oaBonds.GetSize();z2++)
	{
		mb = (CMolBond*)sm->m_oaBonds[z2];
		o = mb->m_iAtomOffset[0];
		o2 = mb->m_iAtomOffset[1];

		el = ((CAtom*)g_oaAtoms[g_waAtomRealElement[o]])->m_pElement;
		el2 = ((CAtom*)g_oaAtoms[g_waAtomRealElement[o2]])->m_pElement;

		vec1 = ts->m_vaCoords[o];
		vec1 /= 1000.0;
		vec1 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

		vec2 = ts->m_vaCoords[o2];
		vec2 /= 1000.0;
		vec2 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

		if ((vec1-vec2).GetLength() > 0.3)
			continue;

		if (m_bPOVRot)
		{
			vec1 = m_mPOVMat * vec1;
			vec2 = m_mPOVMat * vec2;
		}

		vec3 = (vec1/el->m_fRadius + vec2/el2->m_fRadius) / (1.0/el->m_fRadius+1.0/el2->m_fRadius);

		mfprintf(b,"#if (bond_draw)\n");
		mfprintf(b,"  cylinder { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f >, bond_r\n",vec1[0],vec1[1],vec1[2],vec3[0],vec3[1],vec3[2]);
		mfprintf(b,"    pigment { rgbft element_%s_color } finish { reflection bond_reflect specular bond_specular ambient bond_ambient diffuse bond_diffuse } }\n",el->m_sLabel);

		mfprintf(b,"  cylinder { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f >, bond_r\n",vec3[0],vec3[1],vec3[2],vec2[0],vec2[1],vec2[2]);
		mfprintf(b,"    pigment { rgbft element_%s_color } finish { reflection bond_reflect specular bond_specular ambient bond_ambient diffuse bond_diffuse } }\n",el2->m_sLabel);
		mfprintf(b,"#end\n");

		vec3 = cam - (vec1 + vec2) / 2.0;
		vec3.Normalize();

		vec2b = vec2-vec1;
		vec1b = CrossP(vec3,vec2b);

		vec1b.Normalize();

//		if (shadow)
		{
			mfprintf(b,"#if (bond_draw_halo1)\n");
			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >,\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo1 } finish { reflection bond_reflect_halo1 specular bond_specular_halo1 ambient bond_ambient_halo1 diffuse bond_diffuse_halo1 } no_reflection no_radiosity }\n");

			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo1 } finish { reflection bond_reflect_halo1 specular bond_specular_halo1 ambient bond_ambient_halo1 diffuse bond_diffuse_halo1 } no_reflection no_radiosity }\n");
			mfprintf(b,"#end\n");
		}

//		if (halo)
		{
			mfprintf(b,"#if (bond_draw_halo2)\n");
			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >,\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo2 } finish { reflection bond_reflect_halo2 specular bond_specular_halo2 ambient bond_ambient_halo2 diffuse bond_diffuse_halo2 } no_reflection no_radiosity }\n");

			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo2 } finish { reflection bond_reflect_halo2 specular bond_specular_halo2 ambient bond_ambient_halo2 diffuse bond_diffuse_halo2 } no_reflection no_radiosity }\n");
			mfprintf(b,"#end\n");

			
			mfprintf(b,"#if (bond_draw_halo3)\n");
			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >,\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo3 } finish { reflection bond_reflect_halo3 specular bond_specular_halo3 ambient bond_ambient_halo3 diffuse bond_diffuse_halo3 } no_reflection no_radiosity }\n");

			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo3 } finish { reflection bond_reflect_halo3 specular bond_specular_halo3 ambient bond_ambient_halo3 diffuse bond_diffuse_halo3 } no_reflection no_radiosity }\n");
			mfprintf(b,"#end\n");
		}

//		if (shadow)
		{
			vec3 = vec2 - vec1;
			vec3.Normalize();

			mfprintf(b,"#if (bond_draw_stub)\n");
			mfprintf(b,"  cylinder {\n    < %.6f, %.6f, %.6f >,\n",vec1[0],vec1[1],vec1[2]);
			mfprintf(b,"    < %.6f + (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ),\n",vec1[0],vec3[0],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
			mfprintf(b,"      %.6f + (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ),\n",vec1[1],vec3[1],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
			mfprintf(b,"      %.6f + (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ) >\n,",vec1[2],vec3[2],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
			mfprintf(b,"    bond_r + bond_r_stub\n");
			mfprintf(b,"    pigment { rgbft bond_color_stub } finish { reflection bond_reflect_stub specular bond_specular_stub ambient bond_ambient_stub diffuse bond_diffuse_stub } no_radiosity }\n");

			mfprintf(b,"  cylinder {\n    < %.6f, %.6f, %.6f >,\n",vec2[0],vec2[1],vec2[2]);
			mfprintf(b,"    < %.6f - (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ),\n",vec2[0],vec3[0],el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0);
			mfprintf(b,"      %.6f - (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ),\n",vec2[1],vec3[1],el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0);
			mfprintf(b,"      %.6f - (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ) >\n,",vec2[2],vec3[2],el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0);
			mfprintf(b,"    bond_r + bond_r_stub\n");
			mfprintf(b,"    pigment { rgbft bond_color_stub } finish { reflection bond_reflect_stub specular bond_specular_stub ambient bond_ambient_stub diffuse bond_diffuse_stub } no_radiosity }\n");
			mfprintf(b,"#end\n");
		}
	}


	if (m_bPOVDrawNeighbors)
	{
		for (z=0;z<nbmol.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[nbmol[z]];
			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[nbsm[z]]];

			for (z2=0;z2<m->m_baAtomIndex.GetSize();z2++)
			{
				if (m->m_baAtomIndex[z2] == g_iVirtAtomType)
					continue;

				el = ((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_pElement;
				for (z3=0;z3<((CxIntArray*)sm->m_oaAtomOffset[z2])->GetSize();z3++)
				{
					o = ((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3);
					vec1 = ts->m_vaCoords[o];
					vec1 /= 1000.0;

					vec1 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

					if (m_bPOVRot)
						vec1 = m_mPOVMat * vec1;

					if (m_bPOVHighlightNbAtoms)
					{
						n = false;
						for (z4=0;z4<nbatoms.GetSize();z4++)
						{
							if (nbatoms[z4] == o)
							{
								n = true;
								break;
							}
						}
					}

					if (m_bPOVNbColorFace)
					{
						if (m_bPOVHighlightNbAtoms)
							POVFindColor(o,&cr,&cg,&cb,n?0:0.3);
								else POVFindColor(o,&cr,&cg,&cb,0);
					} else
					{
						cr = el->m_iColorR / 255.0;
						cg = el->m_iColorG / 255.0;
						cb = el->m_iColorB / 255.0;
					}

					mfprintf(b,"#if (nbatom_draw)\n");
					if (m_bPOVHighlightNbAtoms)
						mfprintf(b,"  sphere { <%.6f, %.6f, %.6f>, %f\n",vec1[0],vec1[1],vec1[2],n?0.04:0.02);
							mfprintf(b,"  sphere { <%.6f, %.6f, %.6f>, element_%s_radius*nbatom_r\n",vec1[0],vec1[1],vec1[2],el->m_sLabel);

					if (m_bPOVNbColorFace && m_bPOVHighlightNbAtoms)
						mfprintf(b,"    pigment { rgb <%f, %f, %f> } finish { reflection nbatom_reflect specular nbatom_specular ambient nbatom_ambient diffuse nbatom_diffuse } }\n",cr,cg,cb);
							else mfprintf(b,"    pigment { rgbft element_%s_color } finish { reflection nbatom_reflect specular nbatom_specular ambient nbatom_ambient diffuse nbatom_diffuse } }\n",el->m_sLabel);
					mfprintf(b,"#end\n");

					vec2 = cam - vec1;
					vec2.Normalize();

	//				if (shadow)
					{
						mfprintf(b,"#if (nbatom_draw_halo1)\n");
						mfprintf(b,"  disc { < %.6f - (%.6f) * nbatom_d_halo1, %.6f - (%.6f) * nbatom_d_halo1, %.6f - (%.6f) * nbatom_d_halo1 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
						mfprintf(b,"    < %.6f, %.6f, %.6f >, (%f * nbatom_r) + nbatom_r_halo1\n",vec2[0],vec2[1],vec2[2],el->m_fRadius/1000.0);
						mfprintf(b,"    pigment { rgbft nbatom_color_halo1 } finish { reflection nbatom_reflect_halo1 specular nbatom_specular_halo1 ambient nbatom_ambient_halo1 diffuse nbatom_diffuse_halo1 } no_reflection no_radiosity }\n");
					}

	//				if (halo)
					{
						mfprintf(b,"#if (nbatom_draw_halo2)\n");
						mfprintf(b,"  disc { < %.6f - (%.6f) * nbatom_d_halo2, %.6f - (%.6f) * nbatom_d_halo2, %.6f - (%.6f) * nbatom_d_halo2 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
						mfprintf(b,"    < %.6f, %.6f, %.6f >, (%f * nbatom_r) + nbatom_r_halo1 + nbatom_r_halo2\n",vec2[0],vec2[1],vec2[2],el->m_fRadius/1000.0);
						mfprintf(b,"    pigment { rgbft nbatom_color_halo2 } finish { reflection nbatom_reflect_halo2 specular nbatom_specular_halo2 ambient nbatom_ambient_halo2 diffuse nbatom_diffuse_halo2 } no_reflection no_radiosity }\n");
						mfprintf(b,"#end\n");

						mfprintf(b,"#if (nbatom_draw_halo3)\n");
						mfprintf(b,"  disc { < %.6f - (%.6f) * nbatom_d_halo3, %.6f - (%.6f) * nbatom_d_halo3, %.6f - (%.6f) * nbatom_d_halo3 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
						mfprintf(b,"    < %.6f, %.6f, %.6f >, (%f * nbatom_r) + nbatom_r_halo1 + nbatom_r_halo2 + nbatom_r_halo3\n",vec2[0],vec2[1],vec2[2],el->m_fRadius/1000.0);
						mfprintf(b,"    pigment { rgbft nbatom_color_halo3 } finish { reflection nbatom_reflect_halo3 specular nbatom_specular_halo3 ambient nbatom_ambient_halo3 diffuse nbatom_diffuse_halo3 } no_reflection no_radiosity }\n");
						mfprintf(b,"#end\n");
					}
				}
			}

			for (z2=0;z2<sm->m_oaBonds.GetSize();z2++)
			{
				mb = (CMolBond*)sm->m_oaBonds[z2];
				o = mb->m_iAtomOffset[0];
				o2 = mb->m_iAtomOffset[1];

				el = ((CAtom*)g_oaAtoms[g_waAtomRealElement[o]])->m_pElement;
				el2 = ((CAtom*)g_oaAtoms[g_waAtomRealElement[o2]])->m_pElement;

				vec1 = ts->m_vaCoords[o];
				vec1 /= 1000.0;
				vec1 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

				vec2 = ts->m_vaCoords[o2];
				vec2 /= 1000.0;
				vec2 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

				if ((vec1-vec2).GetLength() > 0.3)
					continue;

				if (m_bPOVRot)
				{
					vec1 = m_mPOVMat * vec1;
					vec2 = m_mPOVMat * vec2;
				}

				vec3 = (vec1/el->m_fRadius + vec2/el2->m_fRadius) / (1.0/el->m_fRadius+1.0/el2->m_fRadius);

				if (m_bPOVNbColorFace)
				{
					if (m_bPOVHighlightNbAtoms)
					{
						n = false;
						for (z4=0;z4<nbatoms.GetSize();z4++)
						{
							if (nbatoms[z4] == o)
							{
								n = true;
								break;
							}
						}
						POVFindColor(o,&cr,&cg,&cb,n?0:0.3);
					} else POVFindColor(o,&cr,&cg,&cb,0);
				} else
				{
					cr = el->m_iColorR / 255.0;
					cg = el->m_iColorG / 255.0;
					cb = el->m_iColorB / 255.0;
				}

				mfprintf(b,"#if (nbbond_draw)\n");
				mfprintf(b,"  cylinder { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f >, nbbond_r\n",vec1[0],vec1[1],vec1[2],vec3[0],vec3[1],vec3[2]);

				if (m_bPOVNbColorFace && m_bPOVHighlightNbAtoms)
					mfprintf(b,"    pigment { rgb < %f, %f, %f > } finish { reflection nbbond_reflect specular nbbond_specular ambient nbbond_ambient diffuse nbbond_diffuse } }\n",cr,cg,cb);
						else mfprintf(b,"    pigment { rgbft element_%s_color } finish { reflection nbbond_reflect specular nbbond_specular ambient nbbond_ambient diffuse nbbond_diffuse } }\n",el->m_sLabel);

				if (m_bPOVNbColorFace)
				{
					if (m_bPOVHighlightNbAtoms)
					{
						n = false;
						for (z4=0;z4<nbatoms.GetSize();z4++)
						{
							if (nbatoms[z4] == o2)
							{
								n = true;
								break;
							}
						}
						POVFindColor(o2,&cr,&cg,&cb,n?0:0.3);
					} else POVFindColor(o2,&cr,&cg,&cb,0);
				} else
				{
					cr = el2->m_iColorR / 255.0;
					cg = el2->m_iColorG / 255.0;
					cb = el2->m_iColorB / 255.0;
				}

				mfprintf(b,"  cylinder { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f >, nbbond_r\n",vec3[0],vec3[1],vec3[2],vec2[0],vec2[1],vec2[2]);
				if (m_bPOVNbColorFace && m_bPOVHighlightNbAtoms)
					mfprintf(b,"    pigment { rgb < %f, %f, %f > } finish { reflection nbbond_reflect specular nbbond_specular ambient nbbond_ambient diffuse nbbond_diffuse } }\n",cr,cg,cb);
						else mfprintf(b,"    pigment { rgbft element_%s_color } finish { reflection nbbond_reflect specular nbbond_specular ambient nbbond_ambient diffuse nbbond_diffuse } }\n",el->m_sLabel);
				mfprintf(b,"#end\n");

				vec3 = cam - (vec1 + vec2) / 2.0;
				vec3.Normalize();

				vec2b = vec2-vec1;
				vec1b = CrossP(vec3,vec2b);

				vec1b.Normalize();

//				if (shadow)
				{
					mfprintf(b,"#if (nbbond_draw_halo1)\n");
					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >,\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo1 } finish { reflection nbbond_reflect_halo1 specular nbbond_specular_halo1 ambient nbbond_ambient_halo1 diffuse nbbond_diffuse_halo1 } no_reflection no_radiosity }\n");

					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo1 } finish { reflection nbbond_reflect_halo1 specular nbbond_specular_halo1 ambient nbbond_ambient_halo1 diffuse nbbond_diffuse_halo1 } no_reflection no_radiosity }\n");
					mfprintf(b,"#end\n");
				}

//				if (halo)
				{
					mfprintf(b,"#if (nbbond_draw_halo2)\n");
					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >,\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo2 } finish { reflection nbbond_reflect_halo2 specular nbbond_specular_halo2 ambient nbbond_ambient_halo2 diffuse nbbond_diffuse_halo2 } no_reflection no_radiosity }\n");

					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo2 } finish { reflection nbbond_reflect_halo2 specular nbbond_specular_halo2 ambient nbbond_ambient_halo2 diffuse nbbond_diffuse_halo2 } no_reflection no_radiosity }\n");
					mfprintf(b,"#end\n");

					
					mfprintf(b,"#if (nbbond_draw_halo3)\n");
					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >,\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo3 } finish { reflection nbbond_reflect_halo3 specular nbbond_specular_halo3 ambient nbbond_ambient_halo3 diffuse nbbond_diffuse_halo3 } no_reflection no_radiosity }\n");

					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo3 } finish { reflection nbbond_reflect_halo3 specular nbbond_specular_halo3 ambient nbbond_ambient_halo3 diffuse nbbond_diffuse_halo3 } no_reflection no_radiosity }\n");
					mfprintf(b,"#end\n");
				}

//				if (shadow)
				{
					vec3 = vec2 - vec1;
					vec3.Normalize();

					mfprintf(b,"#if (nbbond_draw_stub)\n");
					mfprintf(b,"  cylinder {\n    < %.6f, %.6f, %.6f >,\n",vec1[0],vec1[1],vec1[2]);
					mfprintf(b,"    < %.6f + (%.6f) * ( (%.6f)*nbatom_r - ( (%.6f)*nbatom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec1[0],vec3[0],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
					mfprintf(b,"      %.6f + (%.6f) * ( (%.6f)*nbatom_r - ( (%.6f)*nbatom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec1[1],vec3[1],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
					mfprintf(b,"      %.6f + (%.6f) * ( (%.6f)*nbatom_r - ( (%.6f)*nbatom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ) >\n,",vec1[2],vec3[2],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
					mfprintf(b,"    nbbond_r + nbbond_r_stub\n");
					mfprintf(b,"    pigment { rgbft nbbond_color_stub } finish { reflection nbbond_reflect_stub specular nbbond_specular_stub ambient nbbond_ambient_stub diffuse nbbond_diffuse_stub } no_radiosity }\n");

					mfprintf(b,"  cylinder {\n    < %.6f, %.6f, %.6f >,\n",vec2[0],vec2[1],vec2[2]);
					mfprintf(b,"    < %.6f - (%.6f) * ( (%.6f)*nbatom_r - ( (%.6f)*nbatom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec2[0],vec3[0],el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0);
					mfprintf(b,"      %.6f - (%.6f) * ( (%.6f)*nbatom_r - ( (%.6f)*nbatom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec2[1],vec3[1],el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0);
					mfprintf(b,"      %.6f - (%.6f) * ( (%.6f)*nbatom_r - ( (%.6f)*nbatom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ) >\n,",vec2[2],vec3[2],el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0);
					mfprintf(b,"    nbbond_r + nbbond_r_stub\n");
					mfprintf(b,"    pigment { rgbft nbbond_color_stub } finish { reflection nbbond_reflect_stub specular nbbond_specular_stub ambient nbbond_ambient_stub diffuse nbbond_diffuse_stub } no_radiosity }\n");
					mfprintf(b,"#end\n");
				}
			}
		}
	}

	mfprintf(b,"\n/*** Manual change of cell rotation only works if drawing of all halos is disabled! ****/\n");
	mfprintf(b,"    rotate <%f, %f, %f>\n\n    no_shadow\n}\n",m_vPOVRotPos[0],m_vPOVRotPos[1],m_vPOVRotPos[2]);

	fclose(b);

	delete m_pContainer;
}


/*void CVoroWrapper::WriteXYZCell(CTimeStep *ts, int mol, int smol)
{
	FILE *a, *b;
	int ijk, q, z, z2, z3, id, faces, co, fc, ti, ec;
	CVoroAtom *va;
	CVoroMolecule *vm;
	CMolecule *m;
	CSingleMolecule *sm;
	double *pp, tx, ty, tz;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	vector<int> fv;
	int *nbtemp;

	m = (CMolecule*)g_oaMolecules[mol];
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[smol]];

	try { nbtemp = new int[m_oaVoroAtoms.GetSize()]; } catch(...) { nbtemp = NULL; }
	if (nbtemp == NULL) NewException((double)m_oaVoroAtoms.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

	a = OpenFileWrite(s,true);

	ec = 0;
	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		ec += ((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();

	mfprintf(a,"%d\n\n",ec+2000);

	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		for (z2=0;z2<((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();z2++)
			mfprintf(a," %s  %g  %g  %g\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z]])->m_sName,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0]/100.0-g_fBoxX/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1]/100.0-g_fBoxY/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2]/100.0-g_fBoxZ/200.0);

	for (z=0;z<10;z++)
	{
		for (z2=0;z2<10;z2++)
		{
			for (z3=0;z3<10;z3++)
			{
				mfprintf(a,"B  %g  %g  %g\n",30.0+z*3.0,z2*3.0,z3*3.0);
				mfprintf(a,"B  %g  %g  %g\n",30.2+z*3.0,z2*3.0,z3*3.0);
			}
		}
	}

	mfprintf(a,"%d\n\n",m->m_iAtomGes+2000);
	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		for (z2=0;z2<((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();z2++)
			mfprintf(a," %s  %g  %g  %g\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z]])->m_sName,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0]/100.0-g_fBoxX/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1]/100.0-g_fBoxY/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2]/100.0-g_fBoxZ/200.0);

	b = OpenFileWrite("voro.pov",true);


	mfprintf(b,"#version 3.6;\n");
	mfprintf(b,"\n");

	mfprintf(b,"#declare r_atom=1.0;\n");
	mfprintf(b,"#declare r_vertex=0;\n");
	mfprintf(b,"#declare r_edge=0.004;\n");
	mfprintf(b,"//#declare r_vertex=0.01;\n");
	mfprintf(b,"//#declare r_edge=0.01;\n");
	mfprintf(b,"#declare color_edge=<0,1,0,0,0>;\n");
	mfprintf(b,"#declare color_vertex=<0,1,0,0,0>;\n");
	mfprintf(b,"#declare color_face=<0.5,0.5,1,0,0.4>;\n");
	mfprintf(b,"//#declare color_face=<0,0,0,1,1>;\n");
	mfprintf(b,"#declare spec_atom=0.7;\n");
	mfprintf(b,"#declare spec_edge=0.3;\n");
	mfprintf(b,"#declare spec_vertex=0.3;\n");
	mfprintf(b,"#declare spec_face=0.3;\n");
	mfprintf(b,"#declare refl_atom=0;\n");
	mfprintf(b,"#declare ambient_vertex=0.2;\n");
	mfprintf(b,"#declare ambient_edge=0.2;\n");
	mfprintf(b,"#declare ambient_atom=0.4;\n");
	mfprintf(b,"#declare ambient_face=0.3;\n");
	mfprintf(b,"#declare diffuse_vertex=0.8;\n");
	mfprintf(b,"#declare diffuse_edge=0.8;\n");
	mfprintf(b,"#declare diffuse_atom=0.7;\n");
	mfprintf(b,"#declare diffuse_face=0.9;\n");


	mfprintf(b,"// Right-handed coordinate system in which the z-axis points upwards\n");
	mfprintf(b,"camera {\n");
	mfprintf(b,"	location <0,-20,0>\n");
	mfprintf(b,"	sky z\n");
	mfprintf(b,"	right -0.06*x*image_width/image_height\n");
	mfprintf(b,"	up 0.06*z\n");
	mfprintf(b,"	look_at <0, 0, 0>\n");
	mfprintf(b,"}\n");
	mfprintf(b,"\n");
	mfprintf(b,"// White background\n");
	mfprintf(b,"background{rgb 1}\n");
	mfprintf(b,"\n");
	mfprintf(b,"// Two lights with slightly different colors\n");
	mfprintf(b,"light_source{<-8,-20,30> color rgb <0.77,0.75,0.75>}\n");
	mfprintf(b,"light_source{<25,-12,12> color rgb <0.38,0.40,0.40>}\n\n");

	mfprintf(b,"\nunion {\n");

	ec = 0;
	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				co++;

				ijk=vl.ijk;
				q=vl.q;
				pp=m_pContainer->p[ijk]+m_pContainer->ps*q;

				id = m_pContainer->id[ijk][q];

				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				ti = 0;

				if ((vm->m_iMolecule == mol) && (vm->m_iSingleMol == smol))
				{
					c.neighbors(nb);
					faces = c.number_of_faces();
					c.face_vertices(fv);

					mfprintf(b,"// Voronoi cell %d\n",m_pContainer->id[vl.ijk][vl.q]);
					mfprintf(b,"union {\n");

					for (z=0;z<faces;z++)
					{
				//		mprintf("Face %d/%d:\n",z+1,faces);
						fc = fv[ti];
				//		mprintf("  %d Vertices\n",fc);

						if (m_pAssignMolecules[id] != m_pAssignMolecules[nb[z]])
						{
							tx = 0;
							ty = 0;
							tz = 0;
							for (z2=0;z2<fc;z2++)
							{
					//			mprintf("  - Vertex %d/%d; Id=%d, ti=%d (%g, %g, %g) -> (%g, %g, %g)\n",z2+1,fc,fv[ti+z2+1]+1,ti,c.pts[fv[ti+z2+1]*3]*0.5,c.pts[fv[ti+z2+1]*3+1]*0.5,c.pts[fv[ti+z2+1]*3+2]*0.5,*pp+c.pts[fv[ti+z2+1]*3]*0.5,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5);
								mfprintf(b,"sphere{<%g,%g,%g>, r_vertex\n pigment{rgbft color_vertex } finish{specular spec_vertex ambient ambient_vertex diffuse diffuse_vertex } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0);
								tx += *pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0;
								ty += pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0;
								tz += pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0;
								if (z2 < fc-1)
								{
									if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+z2+2]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+z2+2]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+z2+2]*3+2],2)) > 0.0000000001)
									{
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										ec++;
										if (ec == 1000)
										{
											eprintf("Out of edges.\n");
											return;
										}
										mfprintf(b,"cylinder{<%g,%g,%g>,<%g,%g,%g>, r_edge\n pigment{rgbft color_edge } finish{specular spec_edge ambient ambient_edge diffuse diffuse_edge } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
									}
								} else
								{
									if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+1]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+1]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+1]*3+2],2)) > 0.0000000001)
									{
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										ec++;
										if (ec == 1000)
										{
											eprintf("Out of edges.\n");
											return;
										}
										mfprintf(b,"cylinder{<%g,%g,%g>,<%g,%g,%g>, r_edge\n pigment{rgbft color_edge } finish{specular spec_edge ambient ambient_edge diffuse diffuse_edge } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);
									}
								}
								fflush(b);
							}
							tx /= fc;
							ty /= fc;
							tz /= fc;

							mfprintf(b,"mesh { \n");
							for (z2=0;z2<fc-1;z2++)
							{
								mfprintf(b," triangle { <%g, %g, %g>, <%g, %g, %g>, <%g, %g, %g> }\n",tx,ty,tz,*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
							}
							mfprintf(b," triangle { <%g, %g, %g>, <%g, %g, %g>, <%g, %g, %g> }\n",tx,ty,tz,*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);
							mfprintf(b,"\ntexture {\n");
							mfprintf(b,"pigment{rgbft color_face } finish{specular spec_face ambient ambient_face diffuse diffuse_face }\n");
							mfprintf(b,"  }\n}\n");
						}

						ti += fv[ti]+1;
					}

					mfprintf(b,"\n");
					mfprintf(b,"}\n\n");

					mfprintf(b,"// Particle %d\nsphere{<%g,%g,%g>, %g*r_atom\n",m_pContainer->id[vl.ijk][vl.q],*pp-g_fBoxX/2000.0,pp[1]-g_fBoxY/2000.0,pp[2]-g_fBoxZ/2000.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_fRadius/1000.0);
					mfprintf(b,"pigment{rgb <%f,%f,%f>} finish{ reflection refl_atom specular spec_atom ambient ambient_atom diffuse diffuse_atom } }\n",((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorR/255.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorG/255.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorB/255.0);
				}
			}
		} while (vl.inc());
	}

	mfprintf(b,"\n    rotate <0, 0, 50>\n  rotate <-30, 0, 0>\n}\n");

	fclose(b);

	mprintf("%d/1000 Edges used.\n",ec);

	for (z=ec;z<1000;z++)
		mfprintf(a," B  30 0 0\n He  30 0 0\n");

	fclose(a);

}*/

/*void CVoroWrapper::WriteXYZCell_Start(CTimeStep *ts, const char *s, int mol, int smol)
{
	FILE *a, *b;
	int ijk, q, z, z2, z3, id,  faces, co, fc, ti, ec;
	CVoroAtom *va;
	CVoroMolecule *vm;
	CMolecule *m;
	CSingleMolecule *sm;
	double *pp, tx, ty, tz;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	vector<int> fv;
	int *nbtemp;

	m = (CMolecule*)g_oaMolecules[mol];
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[smol]];

	try { nbtemp = new int[m_oaVoroAtoms.GetSize()]; } catch(...) { nbtemp = NULL; }
	if (nbtemp == NULL) NewException((double)m_oaVoroAtoms.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

	a = OpenFileWrite(s,true);

	ec = 0;
	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		ec += ((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();

	mfprintf(a,"%d\n\n",ec+m_iMaxEdgesR3);

	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		for (z2=0;z2<((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();z2++)
			mfprintf(a," %s  %g  %g  %g\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z]])->m_sName,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0]/100.0-g_fBoxX/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1]/100.0-g_fBoxY/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2]/100.0-g_fBoxZ/200.0);

	for (z=0;z<10;z++)
	{
		for (z2=0;z2<10;z2++)
		{
			for (z3=0;z3<10;z3++)
			{
				mfprintf(a,"B  %g  %g  %g\n",30.0+z*3.0,z2*3.0,z3*3.0);
				mfprintf(a,"B  %g  %g  %g\n",30.2+z*3.0,z2*3.0,z3*3.0);
			}
		}
	}

	mfprintf(a,"%d\n\n",m->m_iAtomGes+2000);
	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		for (z2=0;z2<((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();z2++)
			mfprintf(a," %s  %g  %g  %g\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z]])->m_sName,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0]/100.0-g_fBoxX/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1]/100.0-g_fBoxY/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2]/100.0-g_fBoxZ/200.0);

	b = OpenFileWrite("voro.pov",true);


	mfprintf(b,"#version 3.6;\n");
	mfprintf(b,"\n");

	mfprintf(b,"#declare r_atom=1.0;\n");
	mfprintf(b,"#declare r_vertex=0;\n");
	mfprintf(b,"#declare r_edge=0.004;\n");
	mfprintf(b,"//#declare r_vertex=0.01;\n");
	mfprintf(b,"//#declare r_edge=0.01;\n");
	mfprintf(b,"#declare color_edge=<0,1,0,0,0>;\n");
	mfprintf(b,"#declare color_vertex=<0,1,0,0,0>;\n");
	mfprintf(b,"#declare color_face=<0.5,0.5,1,0,0.4>;\n");
	mfprintf(b,"//#declare color_face=<0,0,0,1,1>;\n");
	mfprintf(b,"#declare spec_atom=0.7;\n");
	mfprintf(b,"#declare spec_edge=0.3;\n");
	mfprintf(b,"#declare spec_vertex=0.3;\n");
	mfprintf(b,"#declare spec_face=0.3;\n");
	mfprintf(b,"#declare refl_atom=0;\n");
	mfprintf(b,"#declare ambient_vertex=0.2;\n");
	mfprintf(b,"#declare ambient_edge=0.2;\n");
	mfprintf(b,"#declare ambient_atom=0.4;\n");
	mfprintf(b,"#declare ambient_face=0.3;\n");
	mfprintf(b,"#declare diffuse_vertex=0.8;\n");
	mfprintf(b,"#declare diffuse_edge=0.8;\n");
	mfprintf(b,"#declare diffuse_atom=0.7;\n");
	mfprintf(b,"#declare diffuse_face=0.9;\n");


	mfprintf(b,"// Right-handed coordinate system in which the z-axis points upwards\n");
	mfprintf(b,"camera {\n");
	mfprintf(b,"	location <0,-20,0>\n");
	mfprintf(b,"	sky z\n");
	mfprintf(b,"	right -0.06*x*image_width/image_height\n");
	mfprintf(b,"	up 0.06*z\n");
	mfprintf(b,"	look_at <0, 0, 0>\n");
	mfprintf(b,"}\n");
	mfprintf(b,"\n");
	mfprintf(b,"// White background\n");
	mfprintf(b,"background{rgb 1}\n");
	mfprintf(b,"\n");
	mfprintf(b,"// Two lights with slightly different colors\n");
	mfprintf(b,"light_source{<-8,-20,30> color rgb <0.77,0.75,0.75>}\n");
	mfprintf(b,"light_source{<25,-12,12> color rgb <0.38,0.40,0.40>}\n\n");

	mfprintf(b,"\nunion {\n");

	ec = 0;
	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				co++;

				ijk=vl.ijk;
				q=vl.q;
				pp=m_pContainer->p[ijk]+m_pContainer->ps*q;

				id = m_pContainer->id[ijk][q];

				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				ti = 0;

				if ((vm->m_iMolecule == mol) && (vm->m_iSingleMol == smol))
				{
					c.neighbors(nb);
					faces = c.number_of_faces();
					c.face_vertices(fv);

					mfprintf(b,"// Voronoi cell %d\n",m_pContainer->id[vl.ijk][vl.q]);
					mfprintf(b,"union {\n");

					for (z=0;z<faces;z++)
					{
				//		mprintf("Face %d/%d:\n",z+1,faces);
						fc = fv[ti];
				//		mprintf("  %d Vertices\n",fc);

						if (m_pAssignMolecules[id] != m_pAssignMolecules[nb[z]])
						{
							tx = 0;
							ty = 0;
							tz = 0;
							for (z2=0;z2<fc;z2++)
							{
					//			mprintf("  - Vertex %d/%d; Id=%d, ti=%d (%g, %g, %g) -> (%g, %g, %g)\n",z2+1,fc,fv[ti+z2+1]+1,ti,c.pts[fv[ti+z2+1]*3]*0.5,c.pts[fv[ti+z2+1]*3+1]*0.5,c.pts[fv[ti+z2+1]*3+2]*0.5,*pp+c.pts[fv[ti+z2+1]*3]*0.5,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5);
								mfprintf(b,"sphere{<%g,%g,%g>, r_vertex\n pigment{rgbft color_vertex } finish{specular spec_vertex ambient ambient_vertex diffuse diffuse_vertex } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0);
								tx += *pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0;
								ty += pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0;
								tz += pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0;
								if (z2 < fc-1)
								{
									if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+z2+2]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+z2+2]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+z2+2]*3+2],2)) > 0.0000000001)
									{
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										ec++;
										if (ec == 1000)
										{
											eprintf("Out of edges.\n");
											return;
										}
										mfprintf(b,"cylinder{<%g,%g,%g>,<%g,%g,%g>, r_edge\n pigment{rgbft color_edge } finish{specular spec_edge ambient ambient_edge diffuse diffuse_edge } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
									}
								} else
								{
									if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+1]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+1]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+1]*3+2],2)) > 0.0000000001)
									{
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										ec++;
										if (ec == 1000)
										{
											eprintf("Out of edges.\n");
											return;
										}
										mfprintf(b,"cylinder{<%g,%g,%g>,<%g,%g,%g>, r_edge\n pigment{rgbft color_edge } finish{specular spec_edge ambient ambient_edge diffuse diffuse_edge } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);
									}
								}
								fflush(b);
							}
							tx /= fc;
							ty /= fc;
							tz /= fc;

							mfprintf(b,"mesh { \n");
							for (z2=0;z2<fc-1;z2++)
							{
								mfprintf(b," triangle { <%g, %g, %g>, <%g, %g, %g>, <%g, %g, %g> }\n",tx,ty,tz,*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
							}
							mfprintf(b," triangle { <%g, %g, %g>, <%g, %g, %g>, <%g, %g, %g> }\n",tx,ty,tz,*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);
							mfprintf(b,"\ntexture {\n");
							mfprintf(b,"pigment{rgbft color_face } finish{specular spec_face ambient ambient_face diffuse diffuse_face }\n");
							mfprintf(b,"  }\n}\n");
						}

						ti += fv[ti]+1;
					}

					mfprintf(b,"\n");
					mfprintf(b,"}\n\n");

					mfprintf(b,"// Particle %d\nsphere{<%g,%g,%g>, %g*r_atom\n",m_pContainer->id[vl.ijk][vl.q],*pp-g_fBoxX/2000.0,pp[1]-g_fBoxY/2000.0,pp[2]-g_fBoxZ/2000.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_fRadius/1000.0);
					mfprintf(b,"pigment{rgb <%f,%f,%f>} finish{ reflection refl_atom specular spec_atom ambient ambient_atom diffuse diffuse_atom } }\n",((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorR/255.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorG/255.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorB/255.0);
				}
			}
		} while (vl.inc());
	}

	mfprintf(b,"\n    rotate <0, 0, 50>\n  rotate <-30, 0, 0>\n}\n");

	fclose(b);

	mprintf("%d/1000 Edges used.\n",ec);

	for (z=ec;z<1000;z++)
		mfprintf(a," B  30 0 0\n He  30 0 0\n");

	fclose(a);

}*/

void CVoroWrapper::ParsePOVFaceColor()
{
	int z, z2, z3;
	CMolecule *m;
	CAtom *at;
	char buf[256];
	CxWordArray wa;
	float fr, fg, fb;

	if (m_faPOVFaceColor.GetSize() == 0)
	{
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			if (m_bPOVFaceColorRef && (z != m_iPOVMol))
				continue;

			m = (CMolecule*)g_oaMolecules[z];
			for (z2=0;z2<m->m_waAtomCount.GetSize()-1;z2++)
			{
				at = (CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]];
				for (z3=0;z3<m->m_waAtomCount[z2];z3++)
				{
					m_faPOVFaceColor.Add(at->m_pElement->m_iColorR/255.0f);
					m_faPOVFaceColor.Add(at->m_pElement->m_iColorG/255.0f);
					m_faPOVFaceColor.Add(at->m_pElement->m_iColorB/255.0f);
					m_waPOVFaceColorMol.Add(z);
					m_waPOVFaceColorElem.Add(z2);
					m_waPOVFaceColorAtom.Add(z3);
				}
			}
		}
	}

_again:
	mprintf(WHITE,"\n *** Color assignment:\n\n");
	for (z=0;z<m_waPOVFaceColorMol.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[m_waPOVFaceColorMol[z]];
		mprintf(YELLOW,"  #");
		mprintf(WHITE,"%3d",z+1);
		mprintf("%15s %2s%-2d  ( %.3f, %.3f, %.3f )\n",m->m_sName,((CAtom*)g_oaAtoms[m->m_baAtomIndex[m_waPOVFaceColorElem[z]]])->m_sName,m_waPOVFaceColorAtom[z]+1,m_faPOVFaceColor[z*3],m_faPOVFaceColor[z*3+1],m_faPOVFaceColor[z*3+2]);
		if ((z+1)%10 == 0)
			mprintf("\n");
	}
	mprintf("\n");

_iagain:
	AskString("    Which colors to change (e.g. 1-5,8): [done] ",buf,"");

	if (strlen(buf) == 0)
		return;

	if (!ParseIntList(buf,&wa))
	{
		eprintf("Wrong input.\n");
		goto _iagain;
	}

	fr = AskFloat("    Enter red component (0.0 - 1.0): [0.7] ",0.7f);
	fg = AskFloat("    Enter green component (0.0 - 1.0): [0.7] ",0.7f);
	fb = AskFloat("    Enter blue component (0.0 - 1.0): [0.7] ",0.7f);

	for (z=0;z<wa.GetSize();z++)
	{
		m_faPOVFaceColor[(wa[z]-1)*3] = fr;
		m_faPOVFaceColor[(wa[z]-1)*3+1] = fg;
		m_faPOVFaceColor[(wa[z]-1)*3+2] = fb;
	}

	mprintf("\n    Changed %d color values.\n",wa.GetSize());
	wa.RemoveAll();

	goto _again;
}


void CVoroWrapper::POVFindColor(int offset, float *r, float *g, float *b, float bleach)
{
	int z;
	float gr;

	for (z=0;z<m_waPOVFaceColorMol.GetSize();z++)
	{
		if ((g_waAtomMolIndex[offset] == m_waPOVFaceColorMol[z]) &&
			(g_waAtomElement[offset] == m_waPOVFaceColorElem[z]) &&
			(g_waAtomMolNumber[offset] == m_waPOVFaceColorAtom[z]))
		{
			*r = m_faPOVFaceColor[z*3];
			*g = m_faPOVFaceColor[z*3+1];
			*b = m_faPOVFaceColor[z*3+2];

//			gr = (*r + *g + *b) / 3.0;
			gr = 1.0f;

			*r = *r * (1.0-bleach) + gr*bleach;
			*g = *g * (1.0-bleach) + gr*bleach;
			*b = *b * (1.0-bleach) + gr*bleach;

			return;
		}
	}

	*r = 0;
	*g = 0;
	*b = 0;
}


void CVoroWrapper::ProcessSurfCover(CTimeStep *ts)
{
	int ijk, q, z, z3, id, faces;
//	CVoroAtom *va;
	CVoroMolecule *vm;
//	CMolecule *m;
//	CSingleMolecule *sm;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	CxVector3 vec;
	double atot;
	bool *tb;
	CxDoubleArray *tfa;

//	m = (CMolecule*)g_oaMolecules[m_iSurfCoverMol];
/*	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[m_iSurfCoverSM]];

	vec = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[0]])->GetAt(g_iFixAtom[0])] - CxVector3(g_fBoxX/2.0,g_fBoxY/2.0,g_fBoxZ/2.0);

	ts->CenterPos(vec);*/

	ts->FoldAtomsPositive();

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

	atot = 0;

	try { tb = new bool[m_waSurfCoverAtom.GetSize()]; } catch(...) { tb = NULL; }
	if (tb == NULL) NewException((double)m_waSurfCoverAtom.GetSize()*sizeof(bool),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<m_waSurfCoverAtom.GetSize();z++)
		tb[z] = false;

	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				ijk=vl.ijk;
				q=vl.q;

				id = m_pContainer->id[ijk][q];

//				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				if ((vm->m_iMolecule == m_iSurfCoverMol) && (/*vm->m_iSingleMol == m_iSurfCoverSM*/m_waSurfCoverSM.Contains(vm->m_iSingleMol)))
				{
					c.neighbors(nb);
					faces = c.number_of_faces();
					c.face_areas(fa);

					for (z=0;z<faces;z++)
					{
						if (m_pAssignMolecules[id] != m_pAssignMolecules[nb[z]])
						{
							for (z3=0;z3<m_waSurfCoverMol.GetSize();z3++)
							{
								if ((g_waAtomMolIndex[nb[z]] == m_waSurfCoverMol[z3]) &&
									(g_waAtomElement[nb[z]] == m_waSurfCoverElem[z3]) &&
									(g_waAtomMolNumber[nb[z]] == m_waSurfCoverAtom[z3]))
								{
									tfa = (CxDoubleArray*)m_oaSurfCoverData[z3];

									if (!tb[z3])
									{
										tb[z3] = true;
										tfa->Add(fa[z]);
									} else (*tfa)[tfa->GetSize()-1] += fa[z];

									atot += fa[z];

									goto _done;
								}
							}
							eprintf("Error.\n");
_done:;
						}
					}
				}
			}
		} while (vl.inc());
	}

	for (z=0;z<m_waSurfCoverAtom.GetSize();z++)
	{
		if (!tb[z])
			((CxDoubleArray*)m_oaSurfCoverData[z])->Add(0);

		if (atot > 0)
			((CxDoubleArray*)m_oaSurfCoverData[z])->GetAt(((CxDoubleArray*)m_oaSurfCoverData[z])->GetSize()-1) /= atot;
	}

	delete m_pContainer;
	delete[] tb;
}


void CVoroWrapper::FinishSurfCover(const char *s)
{
	FILE *a;
	int z, z2;
	CMolecule *m;
	double t;

	a = OpenFileWrite(s,true);

	mfprintf(a,"# Step;  ");

	for (z=0;z<m_waSurfCoverMol.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[m_waSurfCoverMol[z]];
		mfprintf(a,"%s %s%d",m->m_sName,((CAtom*)g_oaAtoms[m->m_baAtomIndex[m_waSurfCoverElem[z]]])->m_sName,m_waSurfCoverAtom[z]+1);
		if (z < m_waSurfCoverMol.GetSize()-1)
			mfprintf(a,";  ");
	}
	mfprintf(a,"\n");

	mfprintf(a,"# Averages;  ");
	for (z2=0;z2<m_waSurfCoverMol.GetSize();z2++)
	{
		t = 0;
		for (z=0;z<((CxDoubleArray*)m_oaSurfCoverData[z2])->GetSize();z++)
			t += ((CxDoubleArray*)m_oaSurfCoverData[z2])->GetAt(z);

		if (t > 0)
			t /= ((CxDoubleArray*)m_oaSurfCoverData[z2])->GetSize();

		mfprintf(a,"%f",t*100.0f);
		if (z2 < m_waSurfCoverMol.GetSize()-1)
			mfprintf(a,";  ");
	}
	mfprintf(a,"\n");

	for (z=0;z<((CxDoubleArray*)m_oaSurfCoverData[0])->GetSize();z++)
	{
		mfprintf(a,"%d;  ",z);
		for (z2=0;z2<m_waSurfCoverMol.GetSize();z2++)
		{
			mfprintf(a,"%f",((CxDoubleArray*)m_oaSurfCoverData[z2])->GetAt(z)*100.0f);
			if (z2 < m_waSurfCoverMol.GetSize()-1)
				mfprintf(a,";  ");
		}
		mfprintf(a,"\n");
	}

	fclose(a);
}
