/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2013 Martin Brehm
                  2012-2013 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "resp.h"
#include "tools.h"
#include "travis.h"


CResp::CResp()
{
}


CResp::~CResp()
{
}


void CResp::PrepareResp_Stage1(int mol)
{
	FILE *a;
	int z, z2, h, ti, ti2;
	double ac;
	bool *tb;
	char buf[256];

	mprintf("Preparing Stage 1 of RESP calculation...\n");

	if (g_oaMolecules.GetSize() > 1)
		sprintf(buf,"Resp1.m%d.inp",mol+1);
			else sprintf(buf,"Resp1.inp");

	a = OpenFileWrite(buf,true);

	mfprintf(a,"&FORCE_EVAL\n");
	mfprintf(a,"  METHOD Quickstep\n");
	mfprintf(a,"  &DFT\n");
	mfprintf(a,"    BASIS_SET_FILE_NAME %s\n",m_sBasisFile);
	mfprintf(a,"    POTENTIAL_FILE_NAME %s\n\n",m_sPotFile);
	mfprintf(a,"    CHARGE %d\n\n",m_iCharge[mol]);
	mfprintf(a,"    &MGRID\n");
	mfprintf(a,"      CUTOFF 400\n");
	mfprintf(a,"      NGRIDS 5\n");
	mfprintf(a,"      REL_CUTOFF 30\n");
	mfprintf(a,"    &END MGRID\n\n");
	mfprintf(a,"    &QS\n");
	mfprintf(a,"      EPS_DEFAULT 1.0E-12\n");
	mfprintf(a,"      EPS_PGF_ORB 1.0E-32\n");
	mfprintf(a,"      METHOD GAPW\n");
	mfprintf(a,"    &END QS\n");
	mfprintf(a,"    &SCF\n");
	mfprintf(a,"#      SCF_GUESS RESTART\n");
	mfprintf(a,"      SCF_GUESS ATOMIC\n");
	mfprintf(a,"      MAX_SCF 15\n");
	mfprintf(a,"      &OT\n");
	mfprintf(a,"        ENERGY_GAP 0.001\n");
	mfprintf(a,"        PRECONDITIONER FULL_ALL\n");
	mfprintf(a,"        MINIMIZER DIIS\n");
	mfprintf(a,"      &END\n");
	mfprintf(a,"      EPS_SCF 1.0E-6\n");
	mfprintf(a,"      &OUTER_SCF\n");
	mfprintf(a,"        MAX_SCF 20\n");
	mfprintf(a,"        EPS_SCF 1.0E-6\n");
	mfprintf(a,"      &END\n");
	mfprintf(a,"      &PRINT\n");
	mfprintf(a,"        &RESTART\n");
	mfprintf(a,"          BACKUP_COPIES 0\n");
	mfprintf(a,"        &END\n");
	mfprintf(a,"      &END\n");
	mfprintf(a,"    &END SCF\n\n");
	mfprintf(a,"    &POISSON\n");
	mfprintf(a,"      POISSON_SOLVER WAVELET\n");
	mfprintf(a,"      PERIODIC NONE\n");
	mfprintf(a,"    &END POISSON\n\n");
	mfprintf(a,"    &XC\n");
	mfprintf(a,"      &XC_FUNCTIONAL NONE\n");
	mfprintf(a,"      &END XC_FUNCTIONAL\n");
 	mfprintf(a,"      &XC_GRID\n");
 	mfprintf(a,"        XC_DERIV NN10_SMOOTH\n");
 	mfprintf(a,"        XC_SMOOTH_RHO NN10\n");
 	mfprintf(a,"      &END XC_GRID\n");
	mfprintf(a,"      &HF\n");
	mfprintf(a,"        &SCREENING\n");
	mfprintf(a,"          EPS_SCHWARZ 1.0E-10\n");
	mfprintf(a,"        &END\n");
	mfprintf(a,"        &MEMORY\n");
	mfprintf(a,"          MAX_MEMORY %d\n",m_iRAM);
	mfprintf(a,"          EPS_STORAGE_SCALING 1.0E-1\n");
	mfprintf(a,"        &END MEMORY\n");
	mfprintf(a,"        &INTERACTION_POTENTIAL\n");
	mfprintf(a,"          CUTOFF_RADIUS 10.0\n");
	mfprintf(a,"          POTENTIAL_TYPE TRUNCATED\n");
	mfprintf(a,"          T_C_G_DATA %s\n",m_sTCG);
	mfprintf(a,"        &END INTERACTION_POTENTIAL\n");
	mfprintf(a,"      &END HF\n");
	mfprintf(a,"    &END XC\n");
	mfprintf(a,"  &END DFT\n\n");
	mfprintf(a,"  &PROPERTIES\n");
	mfprintf(a,"    &RESP\n");
	mfprintf(a,"      INTEGER_TOTAL_CHARGE .FALSE.\n");         
	mfprintf(a,"      RESTRAIN_HEAVIES_STRENGTH 1.0E-06\n");    
	mfprintf(a,"      RESTRAIN_HEAVIES_TO_ZERO .TRUE.\n");  
	
	if (m_bNewCP2k)
	{
		mfprintf(a,"      &NONPERIODIC_SYS\n");                       
		mfprintf(a,"        RMAX 2.5\n");                       
		mfprintf(a,"        RMIN 2.1\n");                        
		mfprintf(a,"      &END NONPERIODIC_SYS\n");                       
	} else
	{
		mfprintf(a,"      RMAX 2.5\n");                       
		mfprintf(a,"      RMIN 2.1\n");                        
	}

	mfprintf(a,"      STRIDE 2 2 2    \n");     
	
	if (m_bNewCP2k)
	{
		mfprintf(a,"      &PRINT\n");                       
		mfprintf(a,"        &PROGRAM_RUN_INFO ON\n");
		mfprintf(a,"          FILENAME RESP\n");
		mfprintf(a,"          ADD_LAST NUMERIC\n");
		mfprintf(a,"          COMMON_ITERATION_LEVELS 100000\n");
		mfprintf(a,"        &END PROGRAM_RUN_INFO\n");
		mfprintf(a,"      &END PRINT\n");                       
	} else
	{
		mfprintf(a,"      &PROGRAM_RUN_INFO ON\n");
		mfprintf(a,"        FILENAME RESP\n");
		mfprintf(a,"        ADD_LAST NUMERIC\n");
		mfprintf(a,"        COMMON_ITERATION_LEVELS 100000\n");
		mfprintf(a,"      &END PROGRAM_RUN_INFO\n");
	}

	h = -1;
	for (z=0;z<g_oaAtoms.GetSize()-1;z++)
	{
		if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"H")==0)
		{
			h = z;
			break;
		}
	}

	try { tb = new bool[g_iGesAtomCount]; } catch(...) { tb = NULL; }
	if (tb == NULL) NewException((double)g_iGesAtomCount*sizeof(bool),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<g_iGesAtomCount;z++)
		tb[z] = false;

	ti = 0;
	for (z=0;z<g_iGesAtomCount;z++)
	{
		if ((g_waAtomMolIndex[z] != mol) || (g_laAtomSMLocalIndex[z] != 0))
			continue;

		ti++;

		if (g_waAtomRealElement[z] == h)
			continue;
		
		if (tb[z])
			continue;

		tb[z] = true;

		ac = g_faAtomCode[z];

		ti2 = ti;

		for (z2=z+1;z2<g_iGesAtomCount;z2++)
		{
			if ((g_waAtomMolIndex[z2] != mol) || (g_laAtomSMLocalIndex[z2] != 0))
				continue;

			ti2++;

			if (ac == g_faAtomCode[z2])
			{
				mfprintf(a,"\n#     %s%d <--> %s%d\n",((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,g_waAtomMolNumber[z]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[z2]])->m_sName,g_waAtomMolNumber[z2]+1);
				mfprintf(a,"      &CONSTRAINT\n");
				mfprintf(a,"        ATOM_COEF 1.0 -1.0\n");
				mfprintf(a,"        ATOM_LIST %d %d\n",ti,ti2);
				mfprintf(a,"        TARGET 0.0\n");
				mfprintf(a,"      &END CONSTRAINT\n");
				tb[z2] = true;
			}
		}
	}

	delete[] tb;

	mfprintf(a,"    &END RESP\n");
	mfprintf(a,"  &END PROPERTIES\n\n");
	mfprintf(a,"  &SUBSYS\n");
	mfprintf(a,"    &CELL\n");
	mfprintf(a,"      ABC %.2f %.2f %.2f\n",m_fNeedBoxX[mol]/100.0f,m_fNeedBoxY[mol]/100.0f,m_fNeedBoxZ[mol]/100.0f);
	mfprintf(a,"      PERIODIC NONE\n");
	mfprintf(a,"    &END CELL\n\n");
	mfprintf(a,"    &COORD\n");

	for (z=0;z<g_iGesAtomCount;z++)
	{
		if ((g_waAtomMolIndex[z] != mol) || (g_laAtomSMLocalIndex[z] != 0))
			continue;

		mfprintf(a,"      %2s  %10.6f  %10.6f  %10.6f  # %s%d\n",((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,m_pTimeStep->m_vaCoords[z][0]/100.0f,m_pTimeStep->m_vaCoords[z][1]/100.0f,m_pTimeStep->m_vaCoords[z][2]/100.0f,((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,g_waAtomMolNumber[z]+1);
	}

	mfprintf(a,"    &END COORD\n");

	for (z=0;z<g_oaAtoms.GetSize()-1;z++)
	{
		mfprintf(a,"\n    &KIND %s\n",((CAtom*)g_oaAtoms[z])->m_sName);
		mfprintf(a,"      BASIS_SET %s\n",m_sBasis);
		mfprintf(a,"      POTENTIAL ALL\n");
		mfprintf(a,"    &END KIND\n");
	}

	mfprintf(a,"\n    &TOPOLOGY\n");
	mfprintf(a,"      &CENTER_COORDINATES\n");
	mfprintf(a,"      &END CENTER_COORDINATES\n");
	mfprintf(a,"    &END TOPOLOGY\n");

	mfprintf(a,"  &END SUBSYS\n");
	mfprintf(a,"&END FORCE_EVAL\n\n");
	mfprintf(a,"&GLOBAL\n");
	if (g_oaMolecules.GetSize() > 1)
		mfprintf(a,"  PROJECT Resp1.m%d\n",mol+1);
			else mfprintf(a,"  PROJECT Resp1\n");
	mfprintf(a,"  RUN_TYPE ENERGY\n");
	mfprintf(a,"  PRINT_LEVEL LOW\n");
	mfprintf(a,"&END GLOBAL\n");

	fclose(a);
}


void CResp::PrepareResp_Stage2(int mol)
{
	FILE *a;
	int z, z2, h, ti, ti2;
	double ac;
	bool *tb;
	char buf[256];

	mprintf("Preparing Stage 2 of RESP calculation...\n");

	if (g_oaMolecules.GetSize() > 1)
		sprintf(buf,"Resp2.m%d.inp",mol+1);
			else sprintf(buf,"Resp2.inp");

	a = OpenFileWrite(buf,true);

	mfprintf(a,"&FORCE_EVAL\n");
	mfprintf(a,"  METHOD Quickstep\n");
	mfprintf(a,"  &DFT\n");
	mfprintf(a,"    BASIS_SET_FILE_NAME %s\n",m_sBasisFile);
	mfprintf(a,"    POTENTIAL_FILE_NAME %s\n\n",m_sPotFile);
	mfprintf(a,"    CHARGE %d\n\n",m_iCharge[mol]);
	mfprintf(a,"    &MGRID\n");
	mfprintf(a,"      CUTOFF 400\n");
	mfprintf(a,"      NGRIDS 5\n");
	mfprintf(a,"      REL_CUTOFF 30\n");
	mfprintf(a,"    &END MGRID\n\n");
	mfprintf(a,"    &QS\n");
	mfprintf(a,"      EPS_DEFAULT 1.0E-12\n");
	mfprintf(a,"      EPS_PGF_ORB 1.0E-32\n");
	mfprintf(a,"      METHOD GAPW\n");
	mfprintf(a,"    &END QS\n");
	mfprintf(a,"    &SCF\n");
	mfprintf(a,"      SCF_GUESS RESTART\n");
	mfprintf(a,"      MAX_SCF 200\n");
	mfprintf(a,"      &OT\n");
	mfprintf(a,"        PRECONDITIONER NONE\n");
	mfprintf(a,"        MINIMIZER DIIS\n");
	mfprintf(a,"      &END\n");
	mfprintf(a,"      EPS_SCF 1.0E-6\n");
	mfprintf(a,"      &PRINT\n");
	mfprintf(a,"        &RESTART\n");
	mfprintf(a,"          BACKUP_COPIES 0\n");
	mfprintf(a,"        &END\n");
	mfprintf(a,"      &END\n");
	mfprintf(a,"    &END SCF\n\n");
	mfprintf(a,"    &POISSON\n");
	mfprintf(a,"      POISSON_SOLVER WAVELET\n");
	mfprintf(a,"      PERIODIC NONE\n");
	mfprintf(a,"    &END POISSON\n\n");

	if (g_oaMolecules.GetSize() > 1)
		mfprintf(a,"    WFN_RESTART_FILE_NAME Resp1.m%d-RESTART.wfn\n\n",mol+1);
			else mfprintf(a,"    WFN_RESTART_FILE_NAME Resp1-RESTART.wfn\n\n");

	mfprintf(a,"    &XC\n");
	mfprintf(a,"      &XC_FUNCTIONAL NONE\n");
	mfprintf(a,"      &END XC_FUNCTIONAL\n");
 	mfprintf(a,"      &XC_GRID\n");
 	mfprintf(a,"        XC_DERIV NN10_SMOOTH\n");
 	mfprintf(a,"        XC_SMOOTH_RHO NN10\n");
 	mfprintf(a,"      &END XC_GRID\n");
	mfprintf(a,"      &HF\n");
	mfprintf(a,"        &SCREENING\n");
	mfprintf(a,"          EPS_SCHWARZ 1.0E-10\n");
	mfprintf(a,"        &END\n");
	mfprintf(a,"        &MEMORY\n");
	mfprintf(a,"          MAX_MEMORY %d\n",m_iRAM);
	mfprintf(a,"          EPS_STORAGE_SCALING 1.0E-1\n");
	mfprintf(a,"        &END MEMORY\n");
	mfprintf(a,"        &INTERACTION_POTENTIAL\n");
	mfprintf(a,"          CUTOFF_RADIUS 10.0\n");
	mfprintf(a,"          POTENTIAL_TYPE TRUNCATED\n");
	mfprintf(a,"          T_C_G_DATA %s\n",m_sTCG);
	mfprintf(a,"        &END INTERACTION_POTENTIAL\n");
	mfprintf(a,"      &END HF\n");
	mfprintf(a,"    &END XC\n");
	mfprintf(a,"  &END DFT\n\n");
	mfprintf(a,"  &PROPERTIES\n");
	mfprintf(a,"    &RESP\n");
	mfprintf(a,"      INTEGER_TOTAL_CHARGE .FALSE.\n");         
	mfprintf(a,"      RESTRAIN_HEAVIES_STRENGTH 1.0E-06\n");    
	mfprintf(a,"      RESTRAIN_HEAVIES_TO_ZERO .TRUE.\n");     
	
	if (m_bNewCP2k)
	{
		mfprintf(a,"      &NONPERIODIC_SYS\n");                       
		mfprintf(a,"        RMAX 2.5\n");                       
		mfprintf(a,"        RMIN 2.1\n");                        
		mfprintf(a,"      &END NONPERIODIC_SYS\n");                       
	} else
	{
		mfprintf(a,"      RMAX 2.5\n");                       
		mfprintf(a,"      RMIN 2.1\n");                        
	}

	mfprintf(a,"      STRIDE 2 2 2    \n");     
	
	if (m_bNewCP2k)
	{
		mfprintf(a,"      &PRINT\n");                       
		mfprintf(a,"        &PROGRAM_RUN_INFO ON\n");
		mfprintf(a,"          FILENAME RESP\n");
		mfprintf(a,"          ADD_LAST NUMERIC\n");
		mfprintf(a,"          COMMON_ITERATION_LEVELS 100000\n");
		mfprintf(a,"        &END PROGRAM_RUN_INFO\n");
		mfprintf(a,"      &END PRINT\n");                       
	} else
	{
		mfprintf(a,"      &PROGRAM_RUN_INFO ON\n");
		mfprintf(a,"        FILENAME RESP\n");
		mfprintf(a,"        ADD_LAST NUMERIC\n");
		mfprintf(a,"        COMMON_ITERATION_LEVELS 100000\n");
		mfprintf(a,"      &END PROGRAM_RUN_INFO\n");
	}

	h = -1;
	for (z=0;z<g_oaAtoms.GetSize()-1;z++)
	{
		if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"H")==0)
		{
			h = z;
			break;
		}
	}

	ti = 0;
	for (z=0;z<g_iGesAtomCount;z++)
	{
		if ((g_waAtomMolIndex[z] != mol) || (g_laAtomSMLocalIndex[z] != 0))
			continue;

		ti++;

		if (g_waAtomRealElement[z] == h)
			continue;

		mfprintf(a,"\n#     %s%d\n",((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,g_waAtomMolNumber[z]+1);
		mfprintf(a,"      &CONSTRAINT\n");
		mfprintf(a,"        ATOM_COEF 1.0\n");
		mfprintf(a,"        ATOM_LIST %d\n",ti);
		mfprintf(a,"        TARGET %f\n",m_faCharges[z]);
		mfprintf(a,"      &END CONSTRAINT\n");
	}

	try { tb = new bool[g_iGesAtomCount]; } catch(...) { tb = NULL; }
	if (tb == NULL) NewException((double)g_iGesAtomCount*sizeof(bool),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<g_iGesAtomCount;z++)
		tb[z] = false;

	ti = 0;
	for (z=0;z<g_iGesAtomCount;z++)
	{
		if ((g_waAtomMolIndex[z] != mol) || (g_laAtomSMLocalIndex[z] != 0))
			continue;

		ti++;

		if (g_waAtomRealElement[z] != h)
			continue;

		if (tb[z])
			continue;

		tb[z] = true;

		ti2 = ti;

		ac = g_faAtomCode[z];

		for (z2=z+1;z2<g_iGesAtomCount;z2++)
		{
			if ((g_waAtomMolIndex[z2] != mol) || (g_laAtomSMLocalIndex[z2] != 0))
				continue;

			ti2++;

			if (ac == g_faAtomCode[z2])
			{
				mfprintf(a,"\n#     %s%d <--> %s%d\n",((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,g_waAtomMolNumber[z]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[z2]])->m_sName,g_waAtomMolNumber[z2]+1);
				mfprintf(a,"      &CONSTRAINT\n");
				mfprintf(a,"        ATOM_COEF 1.0 -1.0\n");
				mfprintf(a,"        ATOM_LIST %d %d\n",ti,ti2);
				mfprintf(a,"        TARGET 0.0\n");
				mfprintf(a,"      &END CONSTRAINT\n");
				tb[z2] = true;
			}
		}
	}

	delete[] tb;

	mfprintf(a,"    &END RESP\n");
	mfprintf(a,"  &END PROPERTIES\n\n");
	mfprintf(a,"  &SUBSYS\n");
	mfprintf(a,"    &CELL\n");
	mfprintf(a,"      ABC %.2f %.2f %.2f\n",m_fNeedBoxX[mol]/100.0f,m_fNeedBoxY[mol]/100.0f,m_fNeedBoxZ[mol]/100.0f);
	mfprintf(a,"      PERIODIC NONE\n");
	mfprintf(a,"    &END CELL\n\n");
	mfprintf(a,"    &COORD\n");

	for (z=0;z<g_iGesAtomCount;z++)
	{
		if ((g_waAtomMolIndex[z] != mol) || (g_laAtomSMLocalIndex[z] != 0))
			continue;

		mfprintf(a,"      %2s  %10.6f  %10.6f  %10.6f  # %s%d\n",((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,m_pTimeStep->m_vaCoords[z][0]/100.0f,m_pTimeStep->m_vaCoords[z][1]/100.0f,m_pTimeStep->m_vaCoords[z][2]/100.0f,((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,g_waAtomMolNumber[z]+1);
	}

	mfprintf(a,"    &END COORD\n");

	for (z=0;z<g_oaAtoms.GetSize()-1;z++)
	{
		mfprintf(a,"\n    &KIND %s\n",((CAtom*)g_oaAtoms[z])->m_sName);
		mfprintf(a,"      BASIS_SET %s\n",m_sBasis);
		mfprintf(a,"      POTENTIAL ALL\n");
		mfprintf(a,"    &END KIND\n");
	}

	mfprintf(a,"\n    &TOPOLOGY\n");
	mfprintf(a,"      &CENTER_COORDINATES\n");
	mfprintf(a,"      &END CENTER_COORDINATES\n");
	mfprintf(a,"    &END TOPOLOGY\n");

	mfprintf(a,"  &END SUBSYS\n");
	mfprintf(a,"&END FORCE_EVAL\n\n");
	mfprintf(a,"&GLOBAL\n");
	if (g_oaMolecules.GetSize() > 1)
		mfprintf(a,"  PROJECT Resp2.m%d\n",mol+1);
			else mfprintf(a,"  PROJECT Resp2\n");
	mfprintf(a,"  RUN_TYPE ENERGY\n");
	mfprintf(a,"  PRINT_LEVEL LOW\n");
	mfprintf(a,"&END GLOBAL\n");

	fclose(a);
}

void CResp::ParseResp()
{
	char buf[256];
	int z0, z, z2, ti;
	bool b;
	CMolecule *m;
	CSingleMolecule *sm;
	float xmi, xma, ymi, yma, zmi, zma;

//	ScanNeedBox();
	m_faCharges.SetSize(g_iGesAtomCount);
	for (z=0;z<g_iGesAtomCount;z++)
		m_faCharges[z] = 0;

	mprintf(WHITE,"\n>>> CP2k RESP Calculation >>>\n\n");

	m_bNewCP2k = AskYesNo("    Use new CP2k RESP implementation (Sep 2012) (y) or old one (n)? [yes] ",true);

	if (g_oaMolecules.GetSize() > 1)
		mprintf("    A RESP calculation is performed for each molecule type independently.\n\n");

	b = false;
	for (z=0;z<g_oaMolecules.GetSize();z++)
		if (((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex.GetSize() > 1)
			b = true;

	if (b)
	{
		m_bEachMol = AskYesNo("    Perform RESP calculations for all molecules (y) or only once per type (n)? [no] ",false);
	} else m_bEachMol = false;

	if (!m_bEachMol)
		m_bGeoOpt = AskYesNo("    Perform a BLYP-D geometry optimization first (y/n)? [yes] ",true);
			else m_bGeoOpt = false;

	if (m_bGeoOpt)
	{
		AskString("    Please enter GeoOpt basis set name: [DZVP-MOLOPT-SR-GTH] ",m_sGeoBasis,"DZVP-MOLOPT-SR-GTH");
_gbagain:
		AskString("    Please enter location of GeoOpt basis set file: [BASIS_MOLOPT] ",m_sGeoBasisFile,"BASIS_MOLOPT");
		if (!FileExist(m_sGeoBasisFile))
		{
			eprintf("Cannot open \"%s\" for reading.\n",m_sGeoBasisFile);
			goto _gbagain;
		}
	}

	m_b1stPass = AskYesNo("    Perform 1st RESP pass (y/n)? [yes] ",true);
	m_b2ndPass = AskYesNo("    Perform 2nd RESP pass (y/n)? [yes] ",true);
	m_iRAM = AskInteger("    How much RAM per core to use for the HF integrals (in MB): [1000] ",1000);

	AskString("    Please enter RESP basis set name: [6-31++G**] ",m_sBasis,"6-31++G**");

_bagain:
	AskString("    Please enter location of basis set file: [EMSL_BASIS_SETS] ",m_sBasisFile,"EMSL_BASIS_SETS");
	if (!FileExist(m_sBasisFile))
	{
		eprintf("Cannot open \"%s\" for reading.\n",m_sBasisFile);
		goto _bagain;
	}

_pagain:
	AskString("    Please enter location of pseudopotential file: [POTENTIAL] ",m_sPotFile,"POTENTIAL");
	if (!FileExist(m_sPotFile))
	{
		eprintf("Cannot open \"%s\" for reading.\n",m_sPotFile);
		goto _pagain;
	}

_tagain:
	AskString("    Please enter location of t_c_g.dat: [t_c_g.dat] ",m_sTCG,"t_c_g.dat");
	if (!FileExist(m_sTCG))
	{
		eprintf("Cannot open \"%s\" for reading.\n",m_sTCG);
		goto _tagain;
	}

	AskString_ND("    Enter CP2k run command (without input file and output redirection): ",buf);

	m_bReviewInput = AskYesNo("    Do you want to check/modify the CP2k input files before executing (y/n)? [no] ",false);

	try { m_sCP2kCommand = new char[strlen(buf)+1]; } catch(...) { m_sCP2kCommand = NULL; }
	if (m_sCP2kCommand == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	strcpy(m_sCP2kCommand,buf);

	try { m_fNeedBoxX = new float[g_oaMolecules.GetSize()]; } catch(...) { m_fNeedBoxX = NULL; }
	if (m_fNeedBoxX == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(float),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_fNeedBoxY = new float[g_oaMolecules.GetSize()]; } catch(...) { m_fNeedBoxY = NULL; }
	if (m_fNeedBoxY == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(float),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_fNeedBoxZ = new float[g_oaMolecules.GetSize()]; } catch(...) { m_fNeedBoxZ = NULL; }
	if (m_fNeedBoxZ == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(float),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_iCharge = new int[g_oaMolecules.GetSize()]; } catch(...) { m_iCharge = NULL; }
	if (m_iCharge == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_fScaleCharge = new float[g_oaMolecules.GetSize()]; } catch(...) { m_fScaleCharge = NULL; }
	if (m_fScaleCharge == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(float),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_fRespSum = new float[g_oaMolecules.GetSize()]; } catch(...) { m_fRespSum = NULL; }
	if (m_fRespSum == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(float),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z0=0;z0<g_oaMolecules.GetSize();z0++)
	{
		mprintf("\n");
		m = (CMolecule*)g_oaMolecules[z0];
		sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[0]];
		if (g_oaMolecules.GetSize() > 1)
			mprintf(YELLOW,"*** Molecule %d: %s\n\n",z0+1,m->m_sName);

		if (m->m_iAtomGesNoVirt == 1)
		{
			mprintf("    Molecule contains only 1 atom. Will not perform RESP calculation.\n\n");
			m_fScaleCharge[z0] = AskFloat("    Set molecule charge to which value? [0.0] ",0.0);
			continue;
		}

		xmi = 1e20f;
		xma = -1e20f;
		ymi = 1e20f;
		yma = -1e20f;
		zmi = 1e20f;
		zma = -1e20f;

		for (z=0;z<m->m_baAtomIndex.GetSize();z++)
		{
			for (z2=0;z2<m->m_waAtomCount[z];z2++)
			{
				if (m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0] < xmi)
					xmi = m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0];
				if (m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0] > xma)
					xma = m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0];

				if (m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1] < ymi)
					ymi = m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1];
				if (m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1] > yma)
					yma = m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1];

				if (m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2] < zmi)
					zmi = m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2];
				if (m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2] > zma)
					zma = m_pTimeStep->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2];
			}
		}

		m_fNeedBoxX[z0] = xma-xmi;
		m_fNeedBoxY[z0] = yma-ymi;
		m_fNeedBoxZ[z0] = zma-zmi;

		mprintf("    The molecule has the following spatial extent: X=%.0f pm, Y=%.0f pm, Z=%.0f pm.\n\n",m_fNeedBoxX[z0],m_fNeedBoxY[z0],m_fNeedBoxZ[z0]);

		xma = m_fNeedBoxX[z0];
		if (m_fNeedBoxY[z0] > xma)
			xma = m_fNeedBoxY[z0];
		if (m_fNeedBoxZ[z0] > xma)
			xma = m_fNeedBoxZ[z0];

		m_fNeedBoxX[z0] = AskRangeFloat("    Enter the cubic box edge length: [%.1f] ",xma,1E20f,xma+1000.0f,xma+1000.0f);
		m_fNeedBoxY[z0] = m_fNeedBoxX[z0];
		m_fNeedBoxZ[z0] = m_fNeedBoxX[z0];

		m_iCharge[z0] = AskInteger("    Enter the integer (\"quantum chemical\") charge of the molecule: [0] ",0);

		if (m_iCharge[z0] != 0)
			m_fScaleCharge[z0] = AskFloat("    Scale molecule charge to which value? [%.1f] ",m_iCharge[z0]*0.8f,m_iCharge[z0]*0.8f);
				else m_fScaleCharge[z0] = 0;
	}

	mprintf(WHITE,"\n<<< End of CP2k RESP Calculation <<<\n\n");

	mprintf(WHITE,"### Starting Operation ###\n");

	for (z0=0;z0<g_oaMolecules.GetSize();z0++)
	{
		mprintf("\n");
		m = (CMolecule*)g_oaMolecules[z0];
		sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[0]];
		if (g_oaMolecules.GetSize() > 1)
			mprintf(YELLOW,"*** Molecule %d: %s\n\n",z0+1,m->m_sName);

		if (m->m_iAtomGesNoVirt == 1)
		{
			mprintf("    Molecule contains only 1 atom. Will not perform RESP calculation.\n\n");
			for (z=0;z<g_iGesAtomCount;z++)
			{
				if (g_waAtomMolIndex[z] != z0)
					continue;
				m_faCharges[z] = m_fScaleCharge[z0];
			}
			continue;
		}

		if (m_bGeoOpt)
		{
			PrepareGeoOpt(z0);
			if (m_bReviewInput)
			{
				if (g_oaMolecules.GetSize() > 1)
					mprintf("\n    Input file \"GeoOpt.m%d.inp\" written. You may modify the input file now.\n\n",z0+1);
						else mprintf("\n    Input file \"GeoOpt.inp\" written. You may modify the input file now.\n\n");
				AskString("    Press return to start geometry optimization: ",buf,"");
			}
			mprintf("\n");
			RunGeoOpt(z0);
		}
		if (m_b1stPass)
		{
			PrepareResp_Stage1(z0);
			if (m_bReviewInput)
			{
				if (g_oaMolecules.GetSize() > 1)
					mprintf("\n    Input file \"Resp1.m%d.inp\" written. You may modify the input file now.\n\n",z0+1);
						else mprintf("\n    Input file \"Resp1.inp\" written. You may modify the input file now.\n\n");
				AskString("    Press return to start first RESP pass: ",buf,"");
			}
			mprintf("\n");
			RunResp1(z0);
		}
		if (g_oaMolecules.GetSize() > 1)
			sprintf(buf,"Resp1.m%d-RESP.resp",z0+1);
				else sprintf(buf,"Resp1-RESP.resp");
		if (!ReadRespCharges(buf,z0))
		{
			eprintf("\nAborting RESP calculation for this molecule.\n");
			continue;
		}
		CalcRespSum(z0);
		mprintf("  (total charge is %f)\n\n",m_fRespSum[z0]);
		if (m_b2ndPass)
		{
			PrepareResp_Stage2(z0);
			if (m_bReviewInput)
			{
				if (g_oaMolecules.GetSize() > 1)
					mprintf("\n    Input file \"Resp2.m%d.inp\" written. You may modify the input file now.\n\n",z0+1);
						else mprintf("\n    Input file \"Resp2.inp\" written. You may modify the input file now.\n\n");
				AskString("    Press return to start second RESP pass: ",buf,"");
			}
			mprintf("\n");
			RunResp2(z0);
		}
		if (g_oaMolecules.GetSize() > 1)
			sprintf(buf,"Resp2.m%d-RESP.resp",z0+1);
				else sprintf(buf,"Resp2-RESP.resp");
		if (!ReadRespCharges(buf,z0))
		{
			eprintf("\nAborting RESP calculation for this molecule.\n");
			continue;
		}
		CalcRespSum(z0);
		mprintf("  (total charge is %f)\n\n",m_fRespSum[z0]);
		if ((m_iCharge[z0] != 0) && (m_fRespSum[z0] != 0))
		{
			mprintf("Scaling total charge to %f...\n",m_fScaleCharge[z0]);
			for (z=0;z<g_iGesAtomCount;z++)
			{
				if (g_waAtomMolIndex[z] != z0)
					continue;
				m_faCharges[z] *= m_fScaleCharge[z0] / m_fRespSum[z0];
			}
		} else if (m_iCharge[z0] == 0)
		{
			mprintf("Shifting total charge to 0...\n");
			ti = 0;
			for (z=0;z<m->m_baAtomIndex.GetSize();z++)
			{
				if (m->m_baAtomIndex[z] == g_iVirtAtomType)
					continue;
				ti += m->m_waAtomCount[z];
			}
			for (z=0;z<g_iGesAtomCount;z++)
			{
				if (g_waAtomMolIndex[z] != z0)
					continue;
				m_faCharges[z] -= m_fRespSum[z0] / ti;
			}
		}
	}
	mprintf("\n");
	WriteRespCharges("resp.out");
	WriteMol2("result.mol2");

	mprintf(WHITE,"\n### Finished Operation ###\n\n");
}


void CResp::PrepareGeoOpt(int mol)
{
	FILE *a;
	int z, ti;
	char buf[256];

	mprintf("Preparing energy minimisation input...\n");

	if (g_oaMolecules.GetSize() > 1)
		sprintf(buf,"GeoOpt.m%d.inp",mol+1);
			else sprintf(buf,"GeoOpt.inp");

	a = OpenFileWrite(buf,true);

	mfprintf(a,"&FORCE_EVAL\n");
	mfprintf(a,"  METHOD Quickstep\n");
	mfprintf(a,"  &DFT\n");
	mfprintf(a,"    BASIS_SET_FILE_NAME %s\n",m_sGeoBasisFile);
	mfprintf(a,"    POTENTIAL_FILE_NAME %s\n\n",m_sPotFile);
	mfprintf(a,"    CHARGE %d\n\n",m_iCharge[mol]);
	mfprintf(a,"    &MGRID\n");
	mfprintf(a,"      CUTOFF 280\n");
	mfprintf(a,"      NGRIDS 5\n");
	mfprintf(a,"      REL_CUTOFF 30\n");
	mfprintf(a,"    &END MGRID\n\n");
	mfprintf(a,"    &QS\n");
	mfprintf(a,"      EPS_DEFAULT 1.0E-10\n");
	mfprintf(a,"    &END QS\n");
	mfprintf(a,"    &SCF\n");
	mfprintf(a,"      SCF_GUESS ATOMIC\n");
	mfprintf(a,"      MAX_SCF 200\n");
	mfprintf(a,"      &OT\n");
	mfprintf(a,"        ENERGY_GAP 0.001\n");
	mfprintf(a,"        PRECONDITIONER FULL_ALL\n");
	mfprintf(a,"        MINIMIZER DIIS\n");
	mfprintf(a,"      &END\n");
	mfprintf(a,"      EPS_SCF 1.0E-5\n");
	mfprintf(a,"      &PRINT\n");
	mfprintf(a,"        &RESTART\n");
	mfprintf(a,"          BACKUP_COPIES 0\n");
	mfprintf(a,"        &END\n");
	mfprintf(a,"      &END\n");
	mfprintf(a,"    &END SCF\n\n");
	mfprintf(a,"    &XC\n");
	mfprintf(a,"      &XC_FUNCTIONAL BLYP\n");
	mfprintf(a,"      &END XC_FUNCTIONAL\n");
	mfprintf(a,"      &XC_GRID\n");
	mfprintf(a,"        XC_DERIV NN10_SMOOTH\n");
	mfprintf(a,"        XC_SMOOTH_RHO NN10\n");
	mfprintf(a,"      &END XC_GRID\n");
	mfprintf(a,"      &vdW_POTENTIAL\n");
	mfprintf(a,"         DISPERSION_FUNCTIONAL PAIR_POTENTIAL\n");
	mfprintf(a,"         &PAIR_POTENTIAL\n");
	mfprintf(a,"            TYPE DFTD2\n");
	mfprintf(a,"            REFERENCE_FUNCTIONAL BLYP\n");
	mfprintf(a,"         &END PAIR_POTENTIAL\n");
	mfprintf(a,"      &END vdW_POTENTIAL\n");
	mfprintf(a,"    &END XC\n");
	mfprintf(a,"  &END DFT\n\n");
	mfprintf(a,"  &SUBSYS\n");
	mfprintf(a,"    &CELL\n");
	mfprintf(a,"      ABC %.2f %.2f %.2f\n",m_fNeedBoxX[mol]/100.0f,m_fNeedBoxY[mol]/100.0f,m_fNeedBoxZ[mol]/100.0f);
	mfprintf(a,"    &END CELL\n\n");
	mfprintf(a,"    &COORD\n");
	
	for (z=0;z<g_iGesAtomCount;z++)
	{
		if ((g_waAtomMolIndex[z] != mol) || (g_laAtomSMLocalIndex[z] != 0))
			continue;

		mfprintf(a,"      %2s  %10.6f  %10.6f  %10.6f  # %s%d\n",((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,m_pTimeStep->m_vaCoords[z][0]/100.0f,m_pTimeStep->m_vaCoords[z][1]/100.0f,m_pTimeStep->m_vaCoords[z][2]/100.0f,((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,g_waAtomMolNumber[z]+1);
	}

	mfprintf(a,"    &END COORD\n");

	for (z=0;z<g_oaAtoms.GetSize()-1;z++)
	{
		if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"H")==0)
			ti = 1;
		else if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"B")==0)
			ti = 3;
		else if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"C")==0)
			ti = 4;
		else if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"N")==0)
			ti = 5;
		else if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"O")==0)
			ti = 6;
		else if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"F")==0)
			ti = 7;
		else if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"P")==0)
			ti = 5;
		else if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"S")==0)
			ti = 6;
		else if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"Cl")==0)
			ti = 7;
		else if (mystricmp(((CAtom*)g_oaAtoms[z])->m_sName,"Br")==0)
			ti = 7;
		else
		{
			eprintf("Atom \"%s\" not known to RESP module.\n",((CAtom*)g_oaAtoms[z])->m_sName);
			ti = 0;
		}

		mfprintf(a,"\n    &KIND %s\n",((CAtom*)g_oaAtoms[z])->m_sName);
		mfprintf(a,"      BASIS_SET %s\n",m_sGeoBasis);
		mfprintf(a,"      POTENTIAL GTH-BLYP-q%d\n",ti);
		mfprintf(a,"    &END KIND\n");
	}

	mfprintf(a,"  &END SUBSYS\n");
	mfprintf(a,"&END FORCE_EVAL\n\n");
	mfprintf(a,"&GLOBAL\n");

	if (g_oaMolecules.GetSize() > 1)
		mfprintf(a,"  PROJECT GeoOpt.m%d\n",mol+1);
			else mfprintf(a,"  PROJECT GeoOpt\n");

	mfprintf(a,"  RUN_TYPE GEO_OPT\n");
	mfprintf(a,"  PRINT_LEVEL LOW\n");
	mfprintf(a,"&END GLOBAL\n\n");
	mfprintf(a,"&MOTION\n");
	mfprintf(a,"  &GEO_OPT\n");
	mfprintf(a,"    OPTIMIZER BFGS\n");
	mfprintf(a,"    MAX_ITER 200\n");
	mfprintf(a,"  &END GEO_OPT\n");
	mfprintf(a,"  &PRINT\n");
	mfprintf(a,"    &RESTART\n");
	mfprintf(a,"      BACKUP_COPIES 0\n");
	mfprintf(a,"    &END\n");
	mfprintf(a,"  &END\n");
	mfprintf(a,"&END MOTION\n");

	fclose(a);
}


void CResp::RunGeoOpt(int mol)
{
#ifdef TARGET_LINUX
	CMolecule *m;
	int z, ti;
	char buf[256];
	FILE *a;

	mprintf("Running energy minimization...\n");
	if (g_oaMolecules.GetSize() > 1)
		sprintf(buf,"rm GeoOpt.m%d-pos-1.xyz",mol+1);
			else sprintf(buf,"rm GeoOpt-pos-1.xyz");
	mprintf("Executing \"%s\"...\n",buf);
	system(buf);

	if (g_oaMolecules.GetSize() > 1)
		sprintf(buf,"%s GeoOpt.m%d.inp | tee GeoOpt.m%d.log | grep -i 'ot diis\\|optimization step\\|\\ \\ =\\ \\ \\|ot sd\\|\\ change'",m_sCP2kCommand,mol+1,mol+1);
			else sprintf(buf,"%s GeoOpt.inp | tee GeoOpt.log | grep -i 'ot diis\\|optimization step\\|\\ \\ =\\ \\ \\|ot sd\\|\\ change'",m_sCP2kCommand);

	mprintf(YELLOW,"\nExecuting CP2k...\n");
	mprintf("  (\"%s\")\n\n",buf);
	system(buf);
	mprintf(YELLOW,"\nFinished energy minimization.\n\n");

	mprintf("Extracting final geometry...\n");
	m = (CMolecule*)g_oaMolecules[mol];
	ti = 0;
	for (z=0;z<m->m_baAtomIndex.GetSize();z++)
	{
		if (m->m_baAtomIndex[z] == g_iVirtAtomType)
			continue;
		ti += m->m_waAtomCount[z];
	}
	if (g_oaMolecules.GetSize() > 1)
		sprintf(buf,"tail -n %d GeoOpt.m%d-pos-1.xyz > GeoOpt.m%d.xyz",ti+2,mol+1,mol+1);
			else sprintf(buf,"tail -n %d GeoOpt-pos-1.xyz > GeoOpt.xyz",ti+2);
	mprintf("  (Command is \"%s\")\n",buf);
	system(buf);

	if (g_oaMolecules.GetSize() > 1)
		sprintf(buf,"GeoOpt.m%d.xyz",mol+1);
			else sprintf(buf,"GeoOpt.xyz");
	a = fopen(buf,"rb");
	if (a == NULL)
	{
		eprintf("Could not open \"%s\" for reading.\n",buf);
		return;
	}
	ReadCoord(a,mol);
	fclose(a);
#else
	eprintf("Only possible on Linux Operating Systems.\n");
#endif
}


bool CResp::ReadRespCharges(const char *s, int mol)
{
	FILE *a;
	int z, z2;
	char buf[256];
	float tf;
	CMolecule *m;

	m = (CMolecule*)g_oaMolecules[mol];

	a = fopen(s,"rt");
	if (a == NULL)
	{
		eprintf("CResp::ReadRespCharges(): Could not open \"%s\".\n",s);
		return false;
	}

	do {
		fgets(buf,256,a);
		if (feof(a))
		{
			eprintf("CResp::ReadRespCharges(): Unexpected end of file in \"%s\".\n",s);
			return false;
		}
	} while (strstr(buf,"RESP charges") == NULL);

	fgets(buf,256,a);
	fgets(buf,256,a);

	mprintf(WHITE,"\nReading RESP charges for %s from %s ...\n",m->m_sName,s);
	for (z=0;z<g_iGesAtomCount;z++)
	{
		if ((g_waAtomMolIndex[z] != mol) || (g_laAtomSMLocalIndex[z] != 0))
			continue;

		fgets(buf,256,a);
		if (feof(a))
		{
			eprintf("CResp::ReadRespCharges(char*): Unexpected end of file in %s while reading charge for atom %d.\n",s,z+1);
			fclose(a);
			return false;
		}
		buf[strlen(buf)-1] = 0;

		if (m_bNewCP2k)
			tf = (float)atof(&buf[20]);
				else tf = (float)atof(&buf[12]);

		mprintf("  - %2s%-2d: %9.6f\n",((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,g_waAtomMolNumber[z]+1,tf);

		for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
			m_faCharges[((CxIntArray*)((CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]])->m_oaAtomOffset[g_waAtomElement[z]])->GetAt(g_waAtomMolNumber[z])] = tf;
	}
	mprintf("Done.\n\n");
	fclose(a);
	return true;
}


void CResp::CalcRespSum(int mol)
{
	int z;

	m_fRespSum[mol] = 0;
	for (z=0;z<g_iGesAtomCount;z++)
	{
		if ((g_waAtomMolIndex[z] != mol) || (g_laAtomSMLocalIndex[z] != 0))
			continue;

		m_fRespSum[mol] += m_faCharges[z];
	}
}


void CResp::WriteRespCharges(const char *s)
{
	FILE *a;
	int z, z2, z3, z4;
	CMolecule *m;
	CSingleMolecule *sm;

	mprintf("Writing resulting charges to \"%s\"...\n",s);
	a = OpenFileWrite(s,true);
	mfprintf(a,"#  Element  Number  Line  Molecule  Mol.Number  Charge\n");
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
		{
			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];
			for (z3=0;z3<sm->m_baAtomIndex.GetSize();z3++)
			{
				if (sm->m_baAtomIndex[z3] == g_iVirtAtomType)
					continue;
				for (z4=0;z4<((CxIntArray*)sm->m_oaAtomOffset[z3])->GetSize();z4++)
					mfprintf(a,"%2s  %3d  %5d  %15s  %4d  %f\n",((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z3]])->m_sName,z4+1,((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)+1,m->m_sName,z2+1,m_faCharges[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)]);
			}
		}
	}
	fclose(a);
}


void CResp::RunResp1(int mol)
{
#ifdef TARGET_LINUX
	char buf[256];

	mprintf("Running RESP Stage 1...\n");
	if (g_oaMolecules.GetSize() > 1)
		sprintf(buf,"rm Resp1.m%d-RESP.resp",mol+1);
			else sprintf(buf,"rm Resp1-RESP.resp");
	mprintf("Executing \"%s\"...\n",buf);
	system(buf);

	if (g_oaMolecules.GetSize() > 1)
		sprintf(buf,"%s Resp1.m%d.inp | tee Resp1.m%d.log | grep -i 'ot diis\\|ot sd\\|\\ change'",m_sCP2kCommand,mol+1,mol+1);
			else sprintf(buf,"%s Resp1.inp | tee Resp1.log | grep -i 'ot diis\\|ot sd\\|\\ change'",m_sCP2kCommand);

	mprintf(YELLOW,"\nExecuting CP2k...\n");
	mprintf("  (Command is \"%s\")\n\n",buf);
	system(buf);
	mprintf(YELLOW,"\nFinished RESP Stage 1.\n\n");
#else
	eprintf("Only possible on Linux Operating Systems.\n");
#endif
}


void CResp::RunResp2(int mol)
{
#ifdef TARGET_LINUX
	char buf[256];

	mprintf("Running RESP Stage 2...\n");
	if (g_oaMolecules.GetSize() > 1)
		sprintf(buf,"rm Resp2.m%d-RESP.resp",mol+1);
			else sprintf(buf,"rm Resp2-RESP.resp");
	mprintf("Executing \"%s\"...\n",buf);
	system(buf);

	if (g_oaMolecules.GetSize() > 1)
		sprintf(buf,"%s Resp2.m%d.inp | tee Resp2.m%d.log | grep -i 'ot diis\\|ot sd\\|\\ change'",m_sCP2kCommand,mol+1,mol+1);
			else sprintf(buf,"%s Resp2.inp | tee Resp2.log | grep -i 'ot diis\\|ot sd\\|\\ change'",m_sCP2kCommand);
	
	mprintf(YELLOW,"\nExecuting CP2k...\n");
	mprintf("  (Command is \"%s\")\n\n",buf);
	system(buf);
	mprintf(YELLOW,"\nFinished RESP Stage 2.\n\n");
#else
	eprintf("Only possible on Linux Operating Systems.\n");
#endif
}


void CResp::WriteMol2(const char *s)
{
	FILE *a;
	mprintf("Saving Mol2 file as \"%s\"...\n",s);
	m_pTimeStep->m_faCharge.CopyFrom(&m_faCharges);
	a = OpenFileWrite(s,true);
	if (a == NULL)
		return;
	m_pTimeStep->WriteMol2(a);
	fclose(a);
}


void CResp::ReadCoord(FILE *a, int mol)
{
	char buf[256], obuf[256], *p, *q;
	const char *separators = " ,;\"'\t";
	int z;

	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);

	for (z=0;z<(long)g_iGesAtomCount;z++) // den ersten Zeitschritt einlesen
	{
		if ((g_waAtomMolIndex[z] != mol) || (g_laAtomSMLocalIndex[z] != 0))
			continue;

		buf[0] = 0;
		fgets_bin(buf,256,a);
		if (feof(a))
		{
			eprintf("\nCResp::ReadCoord(): Unexpected end of file (2). \"%s\"\n",buf);
			return;
		}
		buf[strlen(buf)-1] = 0;
		strcpy(obuf,buf);

		p = &buf[0];
		while (strchr(separators,*p) != NULL)
			p++;
		q = p+1;
		while ((strchr(separators,*q) == NULL) && (*q != 0))
			q++;
		if (*q == 0)
		{
			eprintf("\nCResp::ReadCoord(): %d: Incomplete line (1): \"%s\"\n",z+1,obuf);
			return;
		}
		*q = 0;

		q++;
		while (strchr(separators,*q) != NULL)
			q++;
		p = q;
		while ((strchr(separators,*p) == NULL) && (*p != 0))
			p++;
		if (*p == 0)
		{
			eprintf("\nCResp::ReadCoord(): %d: Incomplete line (2): \"%s\"",z+1,obuf);
			return;
		}
		*p = 0;
		m_pTimeStep->m_vaCoords[z][0] = (float)atof(q) * 100.0f;
		q = p+1;
		while (strchr(separators,*q) != NULL)
			q++;
		p = q;
		while ((strchr(separators,*p) == NULL) && (*p != 0))
			p++;
		if (*p == 0)
		{
			eprintf("\nCResp::ReadCoord(): %d: Incomplete line (3) \"%s\"",z+1,obuf);
			return;
		}
		*p = 0;
		m_pTimeStep->m_vaCoords[z][1] = (float)atof(q) * 100.0f;
		q = p+1;
		while (strchr(separators,*q) != NULL)
			q++;
		p = q;
		while ((strchr(separators,*p) == NULL) && (*p != 0))
			p++;
		if (*p != 0)
			*p = 0;
		m_pTimeStep->m_vaCoords[z][2] = (float)atof(q) * 100.0f;
	}
}
