/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2013 Martin Brehm
                  2012-2013 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "reordyn.h"
#include "travis.h"
#include "maintools.h"

void CReorDyn::Parse()
{
	BTIN;
	char buf[256];
	int z2, ti;
	CAtomGroup *ag;
	float tf;
	CFFT tfft;

	try { m_pRDyn = new CDF(); } catch(...) { m_pRDyn = NULL; }
	if (m_pRDyn == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pRDyn->m_bLeft = true;

	if (!g_bIRSpec)
		mprintf(YELLOW,"\n>>> Vector Reorientation Dynamics >>>\n\n");
			else mprintf(YELLOW,"\n>>> Infrared Spectrum >>>\n\n");

	mprintf("    All atoms will be taken from the OM %s.\n\n",((CMolecule*)g_oaMolecules[m_iShowMol])->m_sName);

//	mprintf("    If you want to calculate an infrared spectrum, choose \"dipole (2)\" here.\n\n");

	if (!g_bIRSpec)
	{
		m_iVecType = AskRangeInteger("    Should the vector depict position (1), dipole (2), velocity (3) or force (4)? [1] ",1,4,1) - 1;
		mprintf("\n");
	} else m_iVecType = 1;

	z2 = 0;
	m_iCombinations = 0;
	ti = 1;

	if (m_iVecType == 0) // Position
	{
		m_bOrtho = (AskRangeInteger("\n    Should the vector connect 2 points (0) or stand perpendicular to 3 points (1)? [0] ",0,1,0) != 0);

		do {
			ti = 1;
			if (z2 != 0)
				mprintf("\n    %d. vector\n\n",z2+1);

			if (m_bOrtho)
			{
_ax1:			mprintf("      Please enter the atom(s) at the base point (e.g. C7): ");
				inpprintf("! Please enter the atom(s) at the base point (e.g. C7):\n");
				myget(buf);

				try { ag = new CAtomGroup(); } catch(...) { ag = NULL; }
				if (ag == NULL) NewException((double)sizeof(CAtomGroup),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[m_iShowMol],buf))
				{
					eprintf("Wrong input.\n");
					inpprintf("! Wrong input.\n");
					goto _ax1;
				}
				m_oaVectors.Add(ag);
				ti *= ag->m_iAtomGes;
_ax2:			mprintf("      Please enter the 2nd atom(s) of the normal plane (e.g. C7): ");
				inpprintf("! Please enter the 2nd atom(s) of the normal plane (e.g. C7):\n");
				myget(buf);

				try { ag = new CAtomGroup(); } catch(...) { ag = NULL; }
				if (ag == NULL) NewException((double)sizeof(CAtomGroup),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[m_iShowMol],buf))
				{
					eprintf("Wrong input.\n");
					inpprintf("! Wrong input.\n");
					goto _ax2;
				}
				m_oaVectors.Add(ag);
				ti *= ag->m_iAtomGes;
_ax3:			mprintf("      Please enter the 3rd atom(s) of the normal plane (e.g. C7): ");
				inpprintf("! Please enter the 3rd atom(s) of the normal plane (e.g. C7):\n");
				myget(buf);

				try { ag = new CAtomGroup(); } catch(...) { ag = NULL; }
				if (ag == NULL) NewException((double)sizeof(CAtomGroup),__FILE__,__LINE__,__PRETTY_FUNCTION__);

				if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[m_iShowMol],buf))
				{
					eprintf("Wrong input.\n");
					inpprintf("! Wrong input.\n");
					goto _ax3;
				}
				m_oaVectors.Add(ag);
				ti *= ag->m_iAtomGes;
			} else // IF ORTHO
			{
_ax4:			mprintf("      Please enter the atom(s) at the base point (e.g. C7): ");
				inpprintf("! Please enter the atom(s) at the base point (e.g. C7):\n");
				myget(buf);

				try { ag = new CAtomGroup(); } catch(...) { ag = NULL; }
				if (ag == NULL) NewException((double)sizeof(CAtomGroup),__FILE__,__LINE__,__PRETTY_FUNCTION__);

				if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[m_iShowMol],buf))
				{
					eprintf("Wrong input.\n");
					inpprintf("! Wrong input.\n");
					goto _ax4;
				}
				m_oaVectors.Add(ag);
				ti *= ag->m_iAtomGes;
_ax5:			mprintf("      Please enter the atom(s) at the tip point (e.g. C7): ");
				inpprintf("! Please enter the atom(s) at the tip point (e.g. C7):\n");
				myget(buf);

				try { ag = new CAtomGroup(); } catch(...) { ag = NULL; }
				if (ag == NULL) NewException((double)sizeof(CAtomGroup),__FILE__,__LINE__,__PRETTY_FUNCTION__);

				if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[m_iShowMol],buf))
				{
					eprintf("Wrong input.\n");
					inpprintf("! Wrong input.\n");
					goto _ax5;
				}
				m_oaVectors.Add(ag);
				ti *= ag->m_iAtomGes;
				m_oaVectors.Add(NULL);
			} // END IF NOT ORTHO
			z2++;
			m_iCombinations += ti;
		} while (AskYesNo("\n    Enter another set of vectors (y/n)? [no] ",false));
	} else if (m_iVecType == 1) // Dipol
	{
		mprintf("    Taking dipole vector of OM %s.\n\n",((CMolecule*)g_oaMolecules[m_iShowMol])->m_sName);
		m_iCombinations = 1;
		g_bDipole = true;
		ParseDipole();
	} else if (m_iVecType == 2) // Geschwindigkeit
	{
_ax6:		mprintf("      Velocity vector of which atoms to use (e.g. C7)? [#2] ");
		inpprintf("! Velocity vector of which atoms to use (e.g. C7)? [#2]\n");
		myget(buf);

		try { ag = new CAtomGroup(); } catch(...) { ag = NULL; }
		if (ag == NULL) NewException((double)sizeof(CAtomGroup),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		if (strlen(buf)==0)
		{
			if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[m_iShowMol],"#2"))
			{
				eprintf("Weird error.\n");
				inpprintf("! Weird error.\n");
				abort();
			}
		} else if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[m_iShowMol],buf))
		{
			eprintf("Wrong input.\n");
			inpprintf("! Wrong input.\n");
			goto _ax6;
		}
		m_oaVectors.Add(ag);
		ti *= ag->m_iAtomGes;
	} else if (m_iVecType == 3) // Kraft
	{
_ax7:		mprintf("      Force vector of which atoms to use (e.g. C7)? [#2] ");
		inpprintf("! Force vector of which atoms to use (e.g. C7)? [#2]\n");
		myget(buf);

		try { ag = new CAtomGroup(); } catch(...) { ag = NULL; }
		if (ag == NULL) NewException((double)sizeof(CAtomGroup),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		if (strlen(buf)==0)
		{
			if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[m_iShowMol],"#2"))
			{
				eprintf("Weird Error.\n");
				inpprintf("! Weird Error.\n");
				abort();
			}
		} else if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[m_iShowMol],buf))
		{
			eprintf("Wrong input.\n");
			inpprintf("! Wrong input.\n");
			goto _ax7;
		}
		m_oaVectors.Add(ag);
		ti *= ag->m_iAtomGes;
	}

	mprintf("\n    Observing %d vectors per OM.\n\n",m_iCombinations);

//_depth:

	if (g_bIRSpec)
	{
		mprintf(WHITE,"    Hint: ");
		mprintf("The resolution of the ACF may never be higher than the number of processed steps.\n");
		mprintf("          Suggested is 75 percent of the processed steps, but not more than approx. 16384.\n\n");
		if (g_iTrajSteps != -1)
			ti = (int(g_iTrajSteps*0.75)<5120)?int(g_iTrajSteps*0.75):4096;
				else ti = 4096;

		m_iDepth = AskUnsignedInteger("    Enter the resolution (=depth) of the vector ACF (in time steps): [%d] ",ti,ti);
	} else
	{
		mprintf(WHITE,"    Hint: ");
		mprintf("The resolution of the ACF may never be higher than the number of processed steps.\n");
		mprintf("          Suggested is up to 75 percent of the processed steps.\n\n");
		if (g_iTrajSteps != -1)
			m_iDepth = AskUnsignedInteger("    Enter the resolution (=depth) of the vector ACF (in time steps): [%d] ",int(g_iTrajSteps*0.75),int(g_iTrajSteps*0.75));
				else m_iDepth = AskUnsignedInteger("    Enter the resolution (=depth) of the vector ACF (in time steps): [10000] ",10000);
	}

/*	if (g_bRDynCacheMode)
	{
		tf = g_iTrajSteps*m_iCombinations*((CMolecule*)g_oaMolecules[m_iShowMol])->m_waSingleMolIndex.GetSize()*3.0f*sizeof(float)/1024.0f/1024.0f;
//		if (tf >= 10.0f)
			if (!AskYesNo("    This will occupy around %.0f MB RAM (for each R.Dyn.). Continue (y/n)? [yes] ",true,tf))
				goto _depth;
	} else
	{
		tf = m_iDepth*g_iGesVirtAtomCount*3.0f*sizeof(double)/1024.0f/1024.0f;
//		if (tf >= 10.0f)
			if (!AskYesNo("    This will occupy around %.0f MB RAM (once, not for each R.Dyn.). Continue (y/n)? [yes] ",true,tf))
				goto _depth;
	}*/
	if (g_bACFFFT)
	{
		ti = CalcFFTSize(m_iDepth,false);
		if (m_iDepth != ti)
		{
			mprintf(WHITE,"\n    The next \"fast\" size for FFT is %d. Using this instead of %d as depth.\n",tfft.NextFastSize(m_iDepth),m_iDepth);
			m_iDepth = tfft.NextFastSize(m_iDepth);
		}
	} else mprintf("\n");

	m_iStride = AskUnsignedInteger("    Take each n-th time step for temporal axis: [1] ",1);

	mprintf("\n");

	if (!g_bIRSpec)
		m_bSpectrum = AskYesNo("    Calculate reorientation spectrum (FFT of vector ACF) (y/n)? [no] ",false);
			else m_bSpectrum = true;		

	if (m_bSpectrum)
	{
		m_pACF = new CACF();
		m_pACF->m_iSize = m_iDepth;
		m_pACF->m_bSpectrum = true;

		m_pACF->m_bDerivative = AskYesNo("    Derive the vectors before autocorrelation (y/n)? [yes] ",true);

		if (m_pACF->m_bDerivative)
			m_pACF->m_iDerivative = AskRangeInteger("    Please enter degree of vector derivation (1-6): [1] ",1,6,1);
				else m_pACF->m_iDerivative = 0;

		m_pACF->m_bWindowFunction = AskYesNo("    Apply window function (Cos^2) to autocorrelation function (y/n)? [yes] ",true);

		tf = 33356.41 / g_fTimestepLength / 2.0;
		mprintf("\n    A time step length of %.1f fs allows a spectral range up to %.1f cm^-1.\n\n",g_fTimestepLength,tf);
		m_pACF->m_fSpecWaveNumber = AskRangeFloat("    Calculate spectrum up to which wave number (cm^-1)? [%.1f cm^-1] ",0,tf,(tf<5000.0)?tf:5000.0,(tf<5000.0)?tf:5000.0);
		m_pACF->m_iMirror = AskUnsignedInteger("    No mirroring (0), short-time enhancing (1) or long-time enhancing (2)? [1] ",1);
		m_pACF->m_iZeroPadding = AskUnsignedInteger("    Zero Padding: How many zeros to insert? [%d] ",m_iDepth*3,m_iDepth*3);

		ti = CalcFFTSize(m_iDepth+m_pACF->m_iZeroPadding,false);
		if (m_iDepth+m_pACF->m_iZeroPadding != ti)
		{
			mprintf(WHITE,"\n    The next \"fast\" size for FFT is %d. Using %d zeros for zero padding.\n",ti,ti-m_iDepth);
			m_pACF->m_iZeroPadding = ti-m_iDepth;
		}

		m_pACF->m_bACF_DB = AskYesNo("    Convert intensity axis of spectrum to decibel (y/n)? [no] ",false);
		m_pACF->Create();
	}

	BuildName();

	if (!g_bIRSpec)
		mprintf(YELLOW,"\n<<< End of Vector Reorientation Dynamics <<<\n\n");
			else mprintf(YELLOW,"\n<<< End of Infrared Spectrum <<<\n\n");
	BTOUT;
}


void CReorDyn::BuildName()
{
	BTIN;
	int z2;
	char tmp[256];
	CAtomGroup *ag;

	tmp[0] = 0;
	strcat(tmp,"[");
	if (m_iVecType == 0) // Position
	{
		for (z2=0;z2<m_oaVectors.GetSize()/3;z2++)
		{
			if (m_bOrtho)
			{
				ag = (CAtomGroup*)m_oaVectors[z2*3];
				strcat(tmp,ag->m_sName);
				strcat(tmp,"_");
				strcat(tmp,((CAtomGroup*)m_oaVectors[z2*3+1])->m_sName);
				strcat(tmp,"_");
				strcat(tmp,((CAtomGroup*)m_oaVectors[z2*3+2])->m_sName);
			} else
			{
				ag = (CAtomGroup*)m_oaVectors[z2*3];
				strcat(tmp,ag->m_sName);
				strcat(tmp,"_");
				strcat(tmp,((CAtomGroup*)m_oaVectors[z2*3+1])->m_sName);
			}
			if (z2<(m_oaVectors.GetSize()/3)-1)
				strcat(tmp,"]_[");
		}
	} else if (m_iVecType == 1) // Dipol
	{
		strcat(tmp,"dip_");
		strcat(tmp,((CMolecule*)g_oaMolecules[m_iShowMol])->m_sName);
	} else if (m_iVecType == 2) // Geschwindigkeit
	{
		strcat(tmp,"vel_");
		strcat(tmp,((CAtomGroup*)m_oaVectors[0])->m_sName);
	} else if (m_iVecType == 3) // Kraft
	{
		strcat(tmp,"frc_");
		strcat(tmp,((CAtomGroup*)m_oaVectors[0])->m_sName);
	}
	strcat(tmp,"]");

	try { m_sShortName = new char[strlen(tmp)+1]; } catch(...) { m_sShortName = NULL; }
	if (m_sShortName == NULL) NewException((double)(strlen(tmp)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	strcpy(m_sShortName,tmp);
	sprintf(tmp,"%s_",((CMolecule*)g_oaMolecules[m_iShowMol])->m_sName);
	strcat(tmp,m_sShortName);

	try { m_sName = new char[strlen(tmp)+1]; } catch(...) { m_sName = NULL; }
	if (m_sName == NULL) NewException((double)(strlen(tmp)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	strcpy(m_sName,tmp);
	BTOUT;
}


void CReorDyn::BuildAtomList(CSingleMolecule *obs, CxIntArray *vec)
{
	BXIN;
	int z, z1t, z1a, z2t, z2a, z3t, z3a;
	CAtomGroup *g1, *g2, *g3;
	CxIntArray *a1, *a2, *a3;

	vec->RemoveAll_KeepSize();
	for (z=0;z<m_oaVectors.GetSize()/3;z++)
	{
		if (m_bOrtho)
		{
			g1 = (CAtomGroup*)m_oaVectors[z*3];
			g2 = (CAtomGroup*)m_oaVectors[z*3+1];
			g3 = (CAtomGroup*)m_oaVectors[z*3+2];
			for (z1t=0;z1t<g1->m_baAtomType.GetSize();z1t++)
			{
				a1 = (CxIntArray*)g1->m_oaAtoms[z1t];
				for (z1a=0;z1a<a1->GetSize();z1a++)
				{
					for (z2t=0;z2t<g2->m_baAtomType.GetSize();z2t++)
					{
						a2 = (CxIntArray*)g2->m_oaAtoms[z2t];
						for (z2a=0;z2a<a2->GetSize();z2a++)
						{
							for (z3t=0;z3t<g3->m_baAtomType.GetSize();z3t++)
							{
								a3 = (CxIntArray*)g3->m_oaAtoms[z3t];
								for (z3a=0;z3a<a3->GetSize();z3a++)
								{
									vec->Add(((CxIntArray*)obs->m_oaAtomOffset[g1->m_baAtomType[z1t]])->GetAt(a1->GetAt(z1a)));
									vec->Add(((CxIntArray*)obs->m_oaAtomOffset[g2->m_baAtomType[z2t]])->GetAt(a2->GetAt(z2a)));
									vec->Add(((CxIntArray*)obs->m_oaAtomOffset[g3->m_baAtomType[z3t]])->GetAt(a3->GetAt(z3a)));
								}
							}
						}
					}
				}
			}
		} else
		{
			g1 = (CAtomGroup*)m_oaVectors[z*3];
			g2 = (CAtomGroup*)m_oaVectors[z*3+1];
			for (z1t=0;z1t<g1->m_baAtomType.GetSize();z1t++)
			{
				a1 = (CxIntArray*)g1->m_oaAtoms[z1t];
				for (z1a=0;z1a<a1->GetSize();z1a++)
				{
					for (z2t=0;z2t<g2->m_baAtomType.GetSize();z2t++)
					{
						a2 = (CxIntArray*)g2->m_oaAtoms[z2t];
						for (z2a=0;z2a<a2->GetSize();z2a++)
						{
							vec->Add(((CxIntArray*)obs->m_oaAtomOffset[g1->m_baAtomType[z1t]])->GetAt(a1->GetAt(z1a)));
							vec->Add(((CxIntArray*)obs->m_oaAtomOffset[g2->m_baAtomType[z2t]])->GetAt(a2->GetAt(z2a)));
							vec->Add(0);
						}
					}
				}
			}
		}
	}
	BXOUT;
}
