/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2013 Martin Brehm
                  2012-2013 Martin Thomas

    This file written by Martin Thomas.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "raman.h"

#include "acf.h"
#include "df.h"
#include "globalvar.h"
#include "maintools.h"
#include "moltools.h"
#include "timestep.h"
#include "tools.h"
#include "xfloatarray.h"
#include "xobarray.h"

#include <errno.h>
#include <stdio.h>
#include <string.h>

#ifdef TARGET_LINUX
#include <sys/stat.h>
#include <sys/types.h>
#endif

#define BUF_SIZE 4096

static bool g_newRaman;
static bool g_orientAvg;
static float g_fieldStrength;
static int g_step = 0;
static int g_stride;
static float g_laser;
static float g_temp;
static char *g_inputTemplate;
static char *g_templateFieldPos;
static char *g_templatePolPos;
static char *g_templateCoordPos;

static CxObArray g_ramObserv;

static FILE *g_polFile[3];
static CTimeStep *g_timestep[3];

CRamanDyn::CRamanDyn(int showMol) {
	m_iShowMol = showMol;
	
	mprintf(YELLOW, ">>> Raman Spectrum >>>\n\n");
	
	mprintf("    All atoms will be taken from the OM %s.\n", ((CMolecule *)g_oaMolecules[m_iShowMol])->m_sName);
	
	m_iVecType = 1;
	m_iCombinations = 1;
	g_bDipole = true;
	ParseDipole();
	
	if(g_iTrajSteps != -1) {
		int depth = g_iTrajSteps / g_stride * 0.75;
		if(depth > 4096)
			depth = 4096;
		m_iDepth = AskUnsignedInteger("    Enter the resolution (=depth) of the ACF (in time steps): [%d] ", depth, depth);
	} else {
		m_iDepth = AskUnsignedInteger("    Enter the resolution (=depth) of the ACF (in time steps): [2000] ", 2000);
	}
	
	int size = CalcFFTSize(m_iDepth, false);
	if(m_iDepth != size) {
		mprintf(WHITE,"    The next \"fast\" size for FFT is %d. Using this instead of %d as depth.\n", size, m_iDepth);
		m_iDepth = size;
	}
	
	m_iStride = 1;
	m_bSpectrum = true;
	
	bool derive = AskYesNo("    Derive the vectors before autocorrelation (y/n)? [yes] ", true);
	if(derive)
		_derivativeOrder = AskRangeInteger("    Please enter degree of vector derivation (1-2): [1] ", 1, 2, 1);
	else
		_derivativeOrder = 0;
	bool window = AskYesNo("    Apply window function (Cos^2) to autocorrelation function (y/n)? [yes] ", true);
	float possibleRange = 33356.41f / g_fTimestepLength / 2.0f;
	mprintf("\n    A time step length of %.2f fs allows a spectral range up to %.2f cm^-1.\n", g_fTimestepLength, possibleRange);
	float specWaveNumber = AskRangeFloat("\n    Calculate spectrum up to which wave number (cm^-1)? [%.2f cm^-1] ", 0, possibleRange, (possibleRange < 5000.0f) ? possibleRange : 5000.0f, (possibleRange < 5000.0f) ? possibleRange : 5000.0f);
	int mirror = AskRangeInteger("    No mirroring (0) or short-time enhancing (1)? [1] ", 0, 1, 1);
	int zeroPadding = AskUnsignedInteger("    Zero Padding: How many zeros to insert? [%d] ", m_iDepth * 3, m_iDepth * 3);
	
	size = CalcFFTSize(m_iDepth + zeroPadding, false);
	if(m_iDepth + zeroPadding != size) {
		mprintf(WHITE, "    The next \"fast\" size for FFT is %d. Using %d zeros for zero padding.\n", size, size-m_iDepth);
		zeroPadding = size-m_iDepth;
	}
	
	try { _isoACF = new CACF(); } catch(...) { _isoACF = NULL; }
	if(_isoACF == NULL) NewException((double)sizeof(CACF), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	_isoACF->m_iSize = m_iDepth;
	_isoACF->m_bSpectrum = true;
	_isoACF->m_bDerivative = derive;
	_isoACF->m_iDerivative = _derivativeOrder;
	_isoACF->m_bWindowFunction = window;
	_isoACF->m_fSpecWaveNumber = specWaveNumber;
	_isoACF->m_iMirror = mirror;
	_isoACF->m_iZeroPadding = zeroPadding;
	_isoACF->m_bACF_DB = false;
	
	try { _anisoACF = new CACF(); } catch(...) { _anisoACF = NULL; }
	if(_anisoACF == NULL) NewException((double)sizeof(CACF), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	_anisoACF->m_iSize = m_iDepth;
	_anisoACF->m_bSpectrum = true;
	_anisoACF->m_bDerivative = derive;
	_anisoACF->m_iDerivative = _derivativeOrder;
	_anisoACF->m_bWindowFunction = window;
	_anisoACF->m_fSpecWaveNumber = specWaveNumber;
	_anisoACF->m_iMirror = mirror;
	_anisoACF->m_iZeroPadding = zeroPadding;
	_anisoACF->m_bACF_DB = false;
	
	BuildName();
	
	mprintf(YELLOW, "<<< End of Raman Spectrum <<<\n\n");
}

CRamanDyn::~CRamanDyn() {
	int i, j, k;

	for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize() * m_iCombinations; i++)
		delete (CxFloatArray *)_dipole0[i];
	for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
		for(j = 0; j < 3; j++) {
			for(k = 0; k < (g_orientAvg ? 3 : 1); k++) {
				delete (CxFloatArray *)_polarizability[j][k][i];
			}
		}
	}
	delete _isoACF;
	delete _anisoACF;
}

void CRamanDyn::initialize() {
	int i, j, k;

	_isoACF->Create();
	_anisoACF->Create();
	
	if (g_iTrajSteps != -1)
		mprintf("    Raman Cache: Trying to allocate %s of memory...\n", FormatBytes((double)((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize() * g_iTrajSteps / g_iStride / g_stride * (g_orientAvg ? 9.9 : 3.3) * sizeof(float)));
	else
		mprintf("    Raman Cache: Trying to allocate %s of memory...\n", FormatBytes((double)((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize() * 2000 / g_iStride /g_stride * (g_orientAvg ? 9.9 : 3.3) * sizeof(float)));
	for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
		CxVector3 *dipoleVector;
		try { dipoleVector = new CxVector3(); } catch(...) { dipoleVector = NULL; }
		if(dipoleVector == NULL) NewException((double)sizeof(CxVector3), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		_dipole0.Add(dipoleVector);
	}
	for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
		for(j = 0; j < 3; j++) {
			for(k = 0; k < (g_orientAvg ? 3 : 1); k++) {
				CxFloatArray *floatArray;
				try { floatArray = new CxFloatArray(); } catch(...) { floatArray = NULL; }
				if(floatArray == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
				if(g_iTrajSteps != -1) {
					floatArray->SetMaxSize((long)(g_iTrajSteps / g_iStride / g_stride * 1.1));
					floatArray->SetGrow((long)(g_iTrajSteps / g_iStride / g_stride * 0.1));
				} else {
					floatArray->SetGrow(10000);
				}
				_polarizability[j][k].Add(floatArray);
			}
		}
	}
}

void CRamanDyn::getDipole0() {
	int i;

	for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
		CSingleMolecule *sm = (CSingleMolecule *)g_oaSingleMolecules[((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex[i]];
		*((CxVector3 *)_dipole0[i]) = sm->m_vDipole;
	}
}

void CRamanDyn::calcPolarizability(int fieldDirection) {
	int i;

	for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
		CSingleMolecule *sm = (CSingleMolecule *)g_oaSingleMolecules[((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex[i]];
		for(int j = 0; j < 3; j++) {
			((CxFloatArray *)_polarizability[j][fieldDirection][i])->Add(((*((CxVector3 *)_dipole0[i]))[j] - sm->m_vDipole[j]) / g_fieldStrength * 0.393430f); // 0.393430 - Umrechnung von Debye in a.u.
		}
	}
}

void CRamanDyn::finalize() {
	int i, j, k, l;
	char filename[BUF_SIZE];
#ifdef TARGET_WINDOWS
	_snprintf(filename, BUF_SIZE, "raman/pol_%s.dat", m_sName);
#else
	snprintf(filename, BUF_SIZE, "raman/pol_%s.dat", m_sName);
#endif
	mprintf("    Writing polarizabilities for first molecule to %s...\n", filename);
	FILE *pol_file;
	pol_file = OpenFileWrite(filename, false);
	for(i = 0; i < ((CxFloatArray *)_polarizability[0][0][0])->GetSize(); i++) {
		fprintf(pol_file, "%10.2f", i * g_fTimestepLength * g_iStride * g_stride);
		for(j = 0; j < (g_orientAvg ? 3 : 1); j++) {
			for(k = 0; k < 3; k++) {
				fprintf(pol_file, " %14.8f", (*((CxFloatArray *)_polarizability[k][j][0]))[i]);
			}
		}
		fprintf(pol_file, "\n");
	}
	fclose(pol_file);
	
	float step;
	switch(_derivativeOrder) {
		case 0:
			mprintf("    Not deriving polarizabilities.\n");
			break;
		case 1:
			mprintf("    Deriving polarizabilities (1st derivative)...\n");
			mprintf(WHITE, "      [");
			step = ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize() / 20.0f;
			for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
				if(fmodf((float)i, step) < 1.0f)
					mprintf(WHITE, "#");
				for(j = 0; j < ((CxFloatArray *)_polarizability[0][0][0])->GetSize() - 1; j++) {
					for(k = 0; k < 3; k++) {
						for(l = 0; l < (g_orientAvg ? 3 : 1); l++) {
							(*((CxFloatArray *)_polarizability[k][l][i]))[j] = 0.5f * ((*((CxFloatArray *)_polarizability[k][l][i]))[j+1] - (*((CxFloatArray *)_polarizability[k][l][i]))[j]);
						}
					}
				}
			}
			mprintf(WHITE, "]\n");
			break;
		case 2:
			mprintf("    Deriving polarizabilities (2nd derivative)...\n");
			mprintf(WHITE, "      [");
			step = ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize() / 20.0f;
			for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
				if(fmodf((float)i, step) < 1.0f)
					mprintf(WHITE, "#");
				for(j = 0; j < ((CxFloatArray *)_polarizability[0][0][0])->GetSize() - 2; j++) {
					for(k = 0; k < 3; k++) {
						for(l = 0; l < (g_orientAvg ? 3 : 1); l++) {
							(*((CxFloatArray *)_polarizability[k][l][i]))[j] = (*((CxFloatArray *)_polarizability[k][l][i]))[j+2] - 2.0f * (*((CxFloatArray *)_polarizability[k][l][i]))[j+1] + (*((CxFloatArray *)_polarizability[k][l][i]))[j];
						}
					}
				}
			}
			mprintf(WHITE, "]\n");
			break;
		default:
			mprintf(RED, "Higher derivatives not implemented.\n");
			abort();
			break;
	}
	
	mprintf("    Processing polarizability components...\n");
	step = ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize() / 20.0f;
	
	CAutoCorrelation *autoCorrelation;
	try { autoCorrelation = new CAutoCorrelation(); } catch(...) { autoCorrelation = NULL; }
	if(autoCorrelation == NULL) NewException((double)sizeof(CAutoCorrelation), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	autoCorrelation->Init(((CxFloatArray *)_polarizability[0][0][0])->GetSize() - _derivativeOrder, m_iDepth, g_bACFFFT);
	
	if(g_orientAvg) {
		CxFloatArray *isotropyPol, *isotropyACF;
		try { isotropyACF = new CxFloatArray(); } catch(...) { isotropyACF = NULL; }
		if(isotropyACF == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		isotropyACF->SetSize(m_iDepth);
		try { isotropyPol = new CxFloatArray(); } catch(...) { isotropyPol = NULL; }
		if(isotropyPol == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		isotropyPol->SetSize(((CxFloatArray *)_polarizability[0][0][0])->GetSize() - _derivativeOrder);
		
		mprintf("    Isotropic part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < m_iDepth; i++) {
			_isoACF->m_pData[i] = 0.0f;
		}
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			for(j = 0; j < ((CxFloatArray *)_polarizability[0][0][0])->GetSize() - _derivativeOrder; j++) {
				(*isotropyPol)[j] = 0.0f;
				for(k = 0; k < 3; k++) {
					(*isotropyPol)[j] += (*((CxFloatArray *)_polarizability[k][k][i]))[j];
				}
				(*isotropyPol)[j] /= 3.0f;
			}
			autoCorrelation->AutoCorrelate(isotropyPol, isotropyACF);
			for(j = 0; j < m_iDepth; j++) {
				_isoACF->m_pData[j] += (*isotropyACF)[j];
			}
		}
		_isoACF->NormalizeCached();
		mprintf(WHITE, "]\n");

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/acf_iso_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/acf_iso_%s.dat", m_sName);
#endif
		mprintf("    Saving ACF as %s...\n", filename);
		FILE *acf_file = OpenFileWrite(filename, false);
		for(i = 0; i < m_iDepth; i++)
			fprintf(acf_file, "%10.2f %12.8f\n", i * g_fTimestepLength * g_iStride * g_stride, _isoACF->m_pData[i]);
		fclose(acf_file);
		
		if(_isoACF->m_iMirror != 0) {
			mprintf("    Mirroring ACF...\n");
			_isoACF->Mirror(_isoACF->m_iMirror);
		}
		if(_isoACF->m_bWindowFunction != 0) {
			mprintf("    Applying window function to ACF...\n");
			_isoACF->Window();
		}

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/acf_iso_%s.mw.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/acf_iso_%s.mw.dat", m_sName);
#endif

		mprintf("    Saving ACF as %s...\n", filename);
		acf_file = OpenFileWrite(filename, false);
		for(i = 0; i < _isoACF->m_iSize; i++)
			fprintf(acf_file, "%10.2f %12.8f\n", i * g_fTimestepLength * g_iStride * g_stride, _isoACF->m_pData[i]);
		fclose(acf_file);
		
		mprintf("    Performing Fourier transformation...\n");
		CFFT *fft;
		try { fft = new CFFT(); } catch(...) { fft = NULL; }
		if(fft == NULL) NewException((double)sizeof(CFFT), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		fft->PrepareFFT_C2C(_isoACF->m_iSize + _isoACF->m_iZeroPadding);
		_isoACF->Transform(fft);
		delete fft;
		_isoACF->m_pSpectrum->SetMaxRWL(1e15f/299792458.0f/100.0f/g_fTimestepLength/g_iStride/g_stride);
		
#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/spectrum_iso_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/spectrum_iso_%s.dat", m_sName);
#endif

		mprintf("    Saving spectrum as %s...\n", filename);
		_isoACF->m_pSpectrum->Write("", filename, "");
		
		delete isotropyACF;
		delete isotropyPol;
		
		CxFloatArray *anisotropyPol, *anisotropyACF, *anisotropyACFSum;
		try { anisotropyPol = new CxFloatArray(); } catch(...) { anisotropyPol = NULL; }
		if(anisotropyPol == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		anisotropyPol->SetSize(((CxFloatArray *)_polarizability[0][0][0])->GetSize() - _derivativeOrder);
		try { anisotropyACF = new CxFloatArray(); } catch(...) { anisotropyACF = NULL; }
		if(anisotropyACF == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		anisotropyACF->SetSize(m_iDepth);
		try { anisotropyACFSum = new CxFloatArray(); } catch(...) { anisotropyACFSum = NULL; }
		if(anisotropyACFSum == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		anisotropyACFSum->SetSize(m_iDepth);
		
		mprintf("    First anisotropic part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < m_iDepth; i++) {
			_anisoACF->m_pData[i] = 0.0f;
		}
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			for(j = 0; j < ((CxFloatArray *)_polarizability[0][0][0])->GetSize() - _derivativeOrder; j++) {
				(*anisotropyPol)[j] = (*((CxFloatArray *)_polarizability[0][0][i]))[j];
				(*anisotropyPol)[j] -= (*((CxFloatArray *)_polarizability[1][1][i]))[j];
			}
			autoCorrelation->AutoCorrelate(anisotropyPol, anisotropyACF);
			for(int j = 0; j < m_iDepth; j++) {
				_anisoACF->m_pData[j] += (*anisotropyACF)[j];
			}
		}
		_anisoACF->NormalizeCached();
		for(i = 0; i < m_iDepth; i++) {
			(*anisotropyACFSum)[i] = 0.5f * _anisoACF->m_pData[i];
		}
		mprintf(WHITE, "]\n");
		
		mprintf("    Second anisotropic part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			for(j = 0; j < ((CxFloatArray *)_polarizability[0][0][0])->GetSize() - _derivativeOrder; j++) {
				(*anisotropyPol)[j] = (*((CxFloatArray *)_polarizability[1][1][i]))[j];
				(*anisotropyPol)[j] -= (*((CxFloatArray *)_polarizability[2][2][i]))[j];
			}
			autoCorrelation->AutoCorrelate(anisotropyPol, anisotropyACF);
			for(j = 0; j < m_iDepth; j++) {
				_anisoACF->m_pData[j] += (*anisotropyACF)[j];
			}
		}
		_anisoACF->NormalizeCached();
		for(i = 0; i < m_iDepth; i++) {
			(*anisotropyACFSum)[i] += 0.5f * _anisoACF->m_pData[i];
		}
		mprintf(WHITE, "]\n");
		
		mprintf("    Third anisotropic part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			for(j = 0; j < ((CxFloatArray *)_polarizability[0][0][0])->GetSize() - _derivativeOrder; j++) {
				(*anisotropyPol)[j] = (*((CxFloatArray *)_polarizability[2][2][i]))[j];
				(*anisotropyPol)[j] -= (*((CxFloatArray *)_polarizability[0][0][i]))[j];
			}
			autoCorrelation->AutoCorrelate(anisotropyPol, anisotropyACF);
			for(j = 0; j < m_iDepth; j++) {
				_anisoACF->m_pData[j] += (*anisotropyACF)[j];
			}
		}
		_anisoACF->NormalizeCached();
		for(i = 0; i < m_iDepth; i++) {
			(*anisotropyACFSum)[i] += 0.5f * _anisoACF->m_pData[i];
		}
		mprintf(WHITE, "]\n");
		
		mprintf("    Fourth anisotropic part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			autoCorrelation->AutoCorrelate(((CxFloatArray *)_polarizability[0][1][i]), anisotropyACF);
			for(j = 0; j < m_iDepth; j++) {
				_anisoACF->m_pData[j] += (*anisotropyACF)[j];
			}
		}
		_anisoACF->NormalizeCached();
		for(i = 0; i < m_iDepth; i++) {
			(*anisotropyACFSum)[i] += 1.5f * _anisoACF->m_pData[i];
		}
		mprintf(WHITE, "]\n");
		
		mprintf("    Fifth anisotropic part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			autoCorrelation->AutoCorrelate(((CxFloatArray *)_polarizability[1][0][i]), anisotropyACF);
			for(j = 0; j < m_iDepth; j++) {
				_anisoACF->m_pData[j] += (*anisotropyACF)[j];
			}
		}
		_anisoACF->NormalizeCached();
		for(i = 0; i < m_iDepth; i++) {
			(*anisotropyACFSum)[i] += 1.5f * _anisoACF->m_pData[i];
		}
		mprintf(WHITE, "]\n");
		
		mprintf("    Sixth anisotropic part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			autoCorrelation->AutoCorrelate(((CxFloatArray *)_polarizability[1][2][i]), anisotropyACF);
			for(j = 0; j < m_iDepth; j++) {
				_anisoACF->m_pData[j] += (*anisotropyACF)[j];
			}
		}
		_anisoACF->NormalizeCached();
		for(i = 0; i < m_iDepth; i++) {
			(*anisotropyACFSum)[i] += 1.5f * _anisoACF->m_pData[i];
		}
		mprintf(WHITE, "]\n");
		
		mprintf("    Seventh anisotropic part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			autoCorrelation->AutoCorrelate(((CxFloatArray *)_polarizability[2][1][i]), anisotropyACF);
			for(j = 0; j < m_iDepth; j++) {
				_anisoACF->m_pData[j] += (*anisotropyACF)[j];
			}
		}
		_anisoACF->NormalizeCached();
		for(i = 0; i < m_iDepth; i++) {
			(*anisotropyACFSum)[i] += 1.5f * _anisoACF->m_pData[i];
		}
		mprintf(WHITE, "]\n");
		
		mprintf("    Eighth anisotropic part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			autoCorrelation->AutoCorrelate(((CxFloatArray *)_polarizability[2][0][i]), anisotropyACF);
			for(j = 0; j < m_iDepth; j++) {
				_anisoACF->m_pData[j] += (*anisotropyACF)[j];
			}
		}
		_anisoACF->NormalizeCached();
		for(i = 0; i < m_iDepth; i++) {
			(*anisotropyACFSum)[i] += 1.5f * _anisoACF->m_pData[i];
		}
		mprintf(WHITE, "]\n");
		
		mprintf("    Ninth anisotropic part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			autoCorrelation->AutoCorrelate(((CxFloatArray *)_polarizability[0][2][i]), anisotropyACF);
			for(j = 0; j < m_iDepth; j++) {
				_anisoACF->m_pData[j] += (*anisotropyACF)[j];
			}
		}
		_anisoACF->NormalizeCached();
		for(i = 0; i < m_iDepth; i++) {
			(*anisotropyACFSum)[i] += 1.5f * _anisoACF->m_pData[i];
		}
		mprintf(WHITE, "]\n");
		
		for(i = 0; i < m_iDepth; i++) {
			_anisoACF->m_pData[i] = (*anisotropyACFSum)[i];
		}
		
#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/acf_aniso_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/acf_aniso_%s.dat", m_sName);
#endif

		mprintf("    Saving ACF as %s...\n", filename);
		acf_file = OpenFileWrite(filename, false);
		for(i = 0; i < m_iDepth; i++)
			fprintf(acf_file, "%10.2f %12.8f\n", i * g_fTimestepLength * g_iStride * g_stride, _anisoACF->m_pData[i]);
		fclose(acf_file);
		
		if(_anisoACF->m_iMirror != 0) {
			mprintf("    Mirroring ACF...\n");
			_anisoACF->Mirror(_anisoACF->m_iMirror);
		}
		if(_anisoACF->m_bWindowFunction != 0) {
			mprintf("    Applying window function to ACF...\n");
			_anisoACF->Window();
		}
		
#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/acf_aniso_%s.mw.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/acf_aniso_%s.mw.dat", m_sName);
#endif

		mprintf("    Saving ACF as %s...\n", filename);
		acf_file = OpenFileWrite(filename, false);
		for(i = 0; i < _anisoACF->m_iSize; i++)
			fprintf(acf_file, "%10.2f %12.8f\n", i * g_fTimestepLength * g_iStride * g_stride, _anisoACF->m_pData[i]);
		fclose(acf_file);
		
		mprintf("    Performing Fourier transformation...\n");
		try { fft = new CFFT(); } catch(...) { fft = NULL; }
		if(fft == NULL) NewException((double)sizeof(CFFT), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		fft->PrepareFFT_C2C(_anisoACF->m_iSize + _anisoACF->m_iZeroPadding);
		_anisoACF->Transform(fft);
		delete fft;
		_anisoACF->m_pSpectrum->SetMaxRWL(1e15f/299792458.0f/100.0f/g_fTimestepLength/g_iStride/g_stride);
		
#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/spectrum_aniso_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/spectrum_aniso_%s.dat", m_sName);
#endif

		mprintf("    Saving spectrum as %s...\n", filename);
		_anisoACF->m_pSpectrum->Write("", filename, "");
		
		int specSize = _isoACF->m_pSpectrum->m_iSize;
		CSpectrum *paraSpectrum, *orthoSpectrum, *sumSpectrum, *depolSpectrum;
		try { paraSpectrum = new CSpectrum(); } catch(...) { paraSpectrum = NULL; }
		if(paraSpectrum == NULL) NewException((double)sizeof(CSpectrum), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		paraSpectrum->m_fWaveNumber = _isoACF->m_fSpecWaveNumber;
		paraSpectrum->Create(specSize);
		paraSpectrum->SetMaxRWL(1e15f/299792458.0f/100.0f/g_fTimestepLength/g_iStride/g_stride);
		try { orthoSpectrum = new CSpectrum(); } catch(...) { orthoSpectrum = NULL; }
		if(orthoSpectrum == NULL) NewException((double)sizeof(CSpectrum), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		orthoSpectrum->m_fWaveNumber = _isoACF->m_fSpecWaveNumber;
		orthoSpectrum->Create(specSize);
		orthoSpectrum->SetMaxRWL(1e15f/299792458.0f/100.0f/g_fTimestepLength/g_iStride/g_stride);
		try { sumSpectrum = new CSpectrum(); } catch(...) { sumSpectrum = NULL; }
		if(sumSpectrum == NULL) NewException((double)sizeof(CSpectrum), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		sumSpectrum->m_fWaveNumber = _isoACF->m_fSpecWaveNumber;
		sumSpectrum->Create(specSize);
		sumSpectrum->SetMaxRWL(1e15f/299792458.0f/100.0f/g_fTimestepLength/g_iStride/g_stride);
		try { depolSpectrum = new CSpectrum(); } catch(...) { depolSpectrum = NULL; }
		if(depolSpectrum == NULL) NewException((double)sizeof(CSpectrum), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		depolSpectrum->m_fWaveNumber = _isoACF->m_fSpecWaveNumber;
		depolSpectrum->Create(specSize);
		depolSpectrum->SetMaxRWL(1e15f/299792458.0f/100.0f/g_fTimestepLength/g_iStride/g_stride);
		
		for(i = 0; i < specSize; i++) {
			paraSpectrum->m_pData[i] = _isoACF->m_pSpectrum->m_pData[i] + 4.0f / 45.0f * _anisoACF->m_pSpectrum->m_pData[i];
		}

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/spectrum_para_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/spectrum_para_%s.dat", m_sName);
#endif

		mprintf("    Saving para spectrum as %s...\n", filename);
		paraSpectrum->Write("", filename, "");
		
		for(i = 0; i < specSize; i++) {
			orthoSpectrum->m_pData[i] = _anisoACF->m_pSpectrum->m_pData[i] / 15.0f;
		}

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/spectrum_ortho_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/spectrum_ortho_%s.dat", m_sName);
#endif

		mprintf("    Saving ortho spectrum as %s...\n", filename);
		orthoSpectrum->Write("", filename, "");
		
		for(i = 0; i < specSize; i++) {
			sumSpectrum->m_pData[i] = paraSpectrum->m_pData[i] + orthoSpectrum->m_pData[i];
		}

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/spectrum_unpol_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/spectrum_unpol_%s.dat", m_sName);
#endif

		mprintf("    Saving unpolarized spectrum as %s...\n", filename);
		sumSpectrum->Write("", filename, "");
		
		for(i = 0; i < specSize; i++) {
			depolSpectrum->m_pData[i] = orthoSpectrum->m_pData[i] / paraSpectrum->m_pData[i];
		}

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/depol_ratio_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/depol_ratio_%s.dat", m_sName);
#endif

		mprintf("    Saving depolarization ratio as %s...\n", filename);
		depolSpectrum->Write("", filename, "");
		
		for(i = 0; i < specSize; i++) {
			float freq = i * paraSpectrum->m_fMaxRWL / paraSpectrum->m_iSize;
			float crossSecFactor = powf(g_laser - freq, 4) / freq / (1 - expf(-1.438777f * freq / g_temp));
			paraSpectrum->m_pData[i] *= crossSecFactor;
			orthoSpectrum->m_pData[i] *= crossSecFactor;
			sumSpectrum->m_pData[i] *= crossSecFactor;
		}

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/crosssec_para_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/crosssec_para_%s.dat", m_sName);
#endif

		mprintf("    Saving para scattering cross section as %s...\n", filename);
		paraSpectrum->Write("", filename, "");

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/crosssec_ortho_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/crosssec_ortho_%s.dat", m_sName);
#endif

		mprintf("    Saving ortho scattering cross section as %s...\n", filename);
		orthoSpectrum->Write("", filename, "");

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/crosssec_unpol_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/crosssec_unpol_%s.dat", m_sName);
#endif

		mprintf("    Saving unpolarized scattering cross section as %s...\n", filename);
		sumSpectrum->Write("", filename, "");
		
		delete anisotropyACFSum;
		delete anisotropyACF;
		delete anisotropyPol;
		
		delete paraSpectrum;
		delete orthoSpectrum;
		delete sumSpectrum;
		delete depolSpectrum;
	} else {
		CxFloatArray *ACF;
		try { ACF = new CxFloatArray(); } catch(...) { ACF = NULL; }
		if(ACF == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		ACF->SetSize(m_iDepth);
		
		mprintf("    Parallel part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < m_iDepth; i++) {
			_isoACF->m_pData[i] = 0.0f;
		}
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			autoCorrelation->AutoCorrelate((CxFloatArray *)_polarizability[0][0][i], ACF);
			for(int j = 0; j < m_iDepth; j++) {
				_isoACF->m_pData[j] += (*ACF)[j];
			}
		}
		_isoACF->NormalizeCached();
		mprintf(WHITE, "]\n");
		
#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/acf_para_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/acf_para_%s.dat", m_sName);
#endif

		mprintf("    Saving ACF as %s...\n", filename);
		FILE *acf_file = OpenFileWrite(filename, false);
		for(i = 0; i < m_iDepth; i++)
			fprintf(acf_file, "%10.2f %12.8f\n", i * g_fTimestepLength * g_iStride * g_stride, _isoACF->m_pData[i]);
		fclose(acf_file);
		
		if(_isoACF->m_iMirror != 0) {
			mprintf("    Mirroring ACF...\n");
			_isoACF->Mirror(_isoACF->m_iMirror);
		}
		if(_isoACF->m_bWindowFunction != 0) {
			mprintf("    Applying window function to ACF...\n");
			_isoACF->Window();
		}
		
#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/acf_para_%s.mw.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/acf_para_%s.mw.dat", m_sName);
#endif

		mprintf("    Saving ACF as %s...\n", filename);
		acf_file = OpenFileWrite(filename, false);
		for(i = 0; i < _isoACF->m_iSize; i++)
			fprintf(acf_file, "%10.2f %12.8f\n", i * g_fTimestepLength * g_iStride * g_stride, _isoACF->m_pData[i]);
		fclose(acf_file);
		
		mprintf("    Performing Fourier transformation...\n");
		CFFT *fft;
		try { fft = new CFFT(); } catch(...) { fft = NULL; }
		if(fft == NULL) NewException((double)sizeof(CFFT), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		fft->PrepareFFT_C2C(_isoACF->m_iSize + _isoACF->m_iZeroPadding);
		_isoACF->Transform(fft);
		delete fft;
		_isoACF->m_pSpectrum->SetMaxRWL(1e15f/299792458.0f/100.0f/g_fTimestepLength/g_iStride/g_stride);
		
#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/spectrum_para_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/spectrum_para_%s.dat", m_sName);
#endif

		mprintf("    Saving spectrum as %s...\n", filename);
		_isoACF->m_pSpectrum->Write("", filename, "");
		
		mprintf("    Orthogonal part...\n");
		mprintf(WHITE, "      [");
		for(i = 0; i < m_iDepth; i++) {
			_anisoACF->m_pData[i] = 0.0f;
		}
		for(i = 0; i < ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize(); i++) {
			if(fmodf((float)i, step) < 1.0f)
				mprintf(WHITE, "#");
			autoCorrelation->AutoCorrelate((CxFloatArray *)_polarizability[1][0][i], ACF);
			for(int j = 0; j < m_iDepth; j++) {
				_anisoACF->m_pData[j] += (*ACF)[j];
			}
		}
		_anisoACF->NormalizeCached();
		mprintf(WHITE, "]\n");
		
#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/acf_ortho_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/acf_ortho_%s.dat", m_sName);
#endif

		mprintf("    Saving ACF as %s...\n", filename);
		acf_file = OpenFileWrite(filename, false);
		for(i = 0; i < m_iDepth; i++)
			fprintf(acf_file, "%10.2f %12.8f\n", i * g_fTimestepLength * g_iStride * g_stride, _anisoACF->m_pData[i]);
		fclose(acf_file);
		
		if(_anisoACF->m_iMirror != 0) {
			mprintf("    Mirroring ACF...\n");
			_anisoACF->Mirror(_isoACF->m_iMirror);
		}
		if(_anisoACF->m_bWindowFunction != 0) {
			mprintf("    Applying window function to ACF...\n");
			_anisoACF->Window();
		}
		
#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/acf_ortho_%s.mw.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/acf_ortho_%s.mw.dat", m_sName);
#endif

		mprintf("    Saving ACF as %s...\n", filename);
		acf_file = OpenFileWrite(filename, false);
		for(i = 0; i < _isoACF->m_iSize; i++)
			fprintf(acf_file, "%10.2f %12.8f\n", i * g_fTimestepLength * g_iStride * g_stride, _anisoACF->m_pData[i]);
		fclose(acf_file);
		
		mprintf("    Performing Fourier transformation...\n");
		try { fft = new CFFT(); } catch(...) { fft = NULL; }
		if(fft == NULL) NewException((double)sizeof(CFFT), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		fft->PrepareFFT_C2C(_anisoACF->m_iSize + _anisoACF->m_iZeroPadding);
		_anisoACF->Transform(fft);
		delete fft;
		_anisoACF->m_pSpectrum->SetMaxRWL(1e15f/299792458.0f/100.0f/g_fTimestepLength/g_iStride/g_stride);
		
#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/spectrum_ortho_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/spectrum_ortho_%s.dat", m_sName);
#endif

		mprintf("    Saving spectrum as %s...\n", filename);
		_anisoACF->m_pSpectrum->Write("", filename, "");
		
		delete ACF;
		
		int specSize = _isoACF->m_pSpectrum->m_iSize;
		CSpectrum *sumSpectrum, *depolSpectrum;
		try { sumSpectrum = new CSpectrum(); } catch(...) { sumSpectrum = NULL; }
		if(sumSpectrum == NULL) NewException((double)sizeof(CSpectrum), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		sumSpectrum->m_fWaveNumber = _isoACF->m_fSpecWaveNumber;
		sumSpectrum->Create(specSize);
		sumSpectrum->SetMaxRWL(1e15f/299792458.0f/100.0f/g_fTimestepLength/g_iStride/g_stride);
		try { depolSpectrum = new CSpectrum(); } catch(...) { depolSpectrum = NULL; }
		if(depolSpectrum == NULL) NewException((double)sizeof(CSpectrum), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		depolSpectrum->m_fWaveNumber = _isoACF->m_fSpecWaveNumber;
		depolSpectrum->Create(specSize);
		depolSpectrum->SetMaxRWL(1e15f/299792458.0f/100.0f/g_fTimestepLength/g_iStride/g_stride);
		
		for(i = 0; i < specSize; i++) {
			sumSpectrum->m_pData[i] = _isoACF->m_pSpectrum->m_pData[i] + _anisoACF->m_pSpectrum->m_pData[i];
		}

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/spectrum_unpol_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/spectrum_unpol_%s.dat", m_sName);
#endif

		mprintf("    Saving unpolarized spectrum as %s...\n", filename);
		sumSpectrum->Write("", filename, "");
		
		for(i = 0; i < specSize; i++) {
			depolSpectrum->m_pData[i] = _anisoACF->m_pSpectrum->m_pData[i] / _isoACF->m_pSpectrum->m_pData[i];
		}

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/depol_ratio_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/depol_ratio_%s.dat", m_sName);
#endif

		mprintf("    Saving depolarization ratio as %s...\n", filename);
		depolSpectrum->Write("", filename, "");
		
		for(i = 0; i < specSize; i++) {
			float freq = i * _isoACF->m_pSpectrum->m_fMaxRWL / _isoACF->m_pSpectrum->m_iSize;
			float crossSecFactor = powf(g_laser - freq, 4) / freq / (1 - expf(-1.438777f * freq / g_temp));
			_isoACF->m_pSpectrum->m_pData[i] *= crossSecFactor;
			_anisoACF->m_pSpectrum->m_pData[i] *= crossSecFactor;
			sumSpectrum->m_pData[i] *= crossSecFactor;
		}

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/crosssec_para_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/crosssec_para_%s.dat", m_sName);
#endif

		mprintf("    Saving para scattering cross section as %s...\n", filename);
		_isoACF->m_pSpectrum->Write("", filename, "");

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/crosssec_ortho_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/crosssec_ortho_%s.dat", m_sName);
#endif

		mprintf("    Saving ortho scattering cross section as %s...\n", filename);
		_anisoACF->m_pSpectrum->Write("", filename, "");

#ifdef TARGET_WINDOWS
		_snprintf(filename, BUF_SIZE, "raman/crosssec_unpol_%s.dat", m_sName);
#else
		snprintf(filename, BUF_SIZE, "raman/crosssec_unpol_%s.dat", m_sName);
#endif

		mprintf("    Saving unpolarized scattering cross section as %s...\n", filename);
		sumSpectrum->Write("", filename, "");
		
		delete sumSpectrum;
		delete depolSpectrum;
	}
	
	delete autoCorrelation;
}

CRamanObservation::CRamanObservation() {
	int i;

	m_bTimeDev = false;
	m_bTimeDiff = false;
	m_bSelf = false;
	m_bOthers = true;
	
	char buf[BUF_SIZE];
	char buf2[BUF_SIZE];
	size_t remaining = BUF_SIZE;
	if(g_oaMolecules.GetSize() > 1) {
#ifdef TARGET_WINDOWS
		remaining -= _snprintf(buf, remaining, "    Which molecule should be observed (");
#else
		remaining -= snprintf(buf, remaining, "    Which molecule should be observed (");
#endif

		for(i = 0; i < g_oaMolecules.GetSize(); i++) {
			if(remaining < 1)
				break;

#ifdef TARGET_WINDOWS
			size_t length = _snprintf(buf2, remaining, "%s=%d", ((CMolecule *)g_oaMolecules[i])->m_sName, i+1);
#else
			size_t length = snprintf(buf2, remaining, "%s=%d", ((CMolecule *)g_oaMolecules[i])->m_sName, i+1);
#endif

			strncat(buf, buf2, remaining - 1);
			remaining -= length;
			if(i < g_oaMolecules.GetSize()) {

#ifdef TARGET_WINDOWS
				length = _snprintf(buf2, remaining, ", ");
#else
				length = snprintf(buf2, remaining, ", ");
#endif

				strncat(buf, buf2, remaining - 1);
				remaining -= length;
			}
		}
		strncat(buf, ")? ", remaining - 1);
		m_iShowMol = AskRangeInteger_ND(buf, 1, g_oaMolecules.GetSize()) - 1;
	} else {
		m_iShowMol = 0;
	}
	m_iShowMolCount = ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize();
	m_bObsCertain = false;
	m_bDecompDist = false;
	m_bDecompType = false;
	
	try { _ramanDyn = new CRamanDyn(m_iShowMol); } catch(...) { _ramanDyn = NULL; }
	if(_ramanDyn == NULL) NewException((double)sizeof(CRamanDyn), __FILE__, __LINE__, __PRETTY_FUNCTION__);
}

CRamanObservation::~CRamanObservation() {
	delete _ramanDyn;
}

void CRamanObservation::initialize() {
	_ramanDyn->initialize();
}

void CRamanObservation::getDipole0() {
	_ramanDyn->getDipole0();
}

void CRamanObservation::calcPolarizability(int fieldDirection) {
	_ramanDyn->calcPolarizability(fieldDirection);
}

void CRamanObservation::finalize() {
	_ramanDyn->finalize();
}

static bool parseSettings(FILE *settingsFile) {
	char buf[BUF_SIZE];
	
	if(fgets(buf, BUF_SIZE, settingsFile) == NULL)
		return false;
	int temp = -1;
	if(sscanf(buf, "%d", &temp) < 1)
		return false;
	if(temp == 0)
		g_orientAvg = false;
	else if(temp == 1)
		g_orientAvg = true;
	else
		return false;
	
	if(fgets(buf, BUF_SIZE, settingsFile) == NULL)
		return false;
	if(sscanf(buf, "%f", &g_fieldStrength) < 1)
		return false;
	
	if(fgets(buf, BUF_SIZE, settingsFile) == NULL)
		return false;
	if(sscanf(buf, "%d", &g_stride) < 1)
		return false;
	
	return true;
}

bool gatherRaman() {
#ifndef TARGET_LINUX
	mprintf(RED, "Raman calculations are currently possible only under Linux.\n");
	return false;
#else
	mprintf("    To calculate Raman spectra, polarizabilities have to be determined.\n");
	mprintf("    For numerical polarizabilities with CP2K, all files are collected in the subdirectory \"raman\".\n\n");
	
	FILE *settingsFile;
	
	if(!FileExist("raman")) {
		g_newRaman = AskYesNo("    A subdirectory \"raman\" was not found. Do you wish to create one? [yes] ", true);
	} else {
		mprintf("    A subdirectory \"raman\" was found. Using data from it.\n\n");
	}
	
	if(g_newRaman) {
		g_bKeepOriginalCoords = true;
		if(mkdir("raman", S_IRWXU) != 0) {
			mprintf(RED, "Directory \"raman\" could not be created: %s\n", strerror(errno));
			return false;
		}
		
		settingsFile = fopen("raman/settings.dat", "w");
		if(settingsFile == NULL) {
			mprintf(RED, "Could not open settings file: %s\n", strerror(errno));
			return false;
		}
		
		g_orientAvg = AskYesNo("\n    Use orientational averaging? [yes] ", true);
		if(g_orientAvg)
			fprintf(settingsFile, "%d\n", 1);
		else
			fprintf(settingsFile, "%d\n", 0);
		
		g_fieldStrength = AskFloat("    Field strength in a. u. [5.0e-4] ", 5.0e-4);
		fprintf(settingsFile, "%.6E\n", g_fieldStrength);
		
		g_stride = AskInteger("    Calculate polarizability for every n-th timestep [5] ", 5);
		fprintf(settingsFile, "%d\n", g_stride);
		
		fclose(settingsFile);
		
		if(mkdir("raman/1", S_IRWXU) != 0) {
			mprintf(RED, "Directory \"1\" could not be created: %s\n", strerror(errno));
			return false;
		}
		if(g_orientAvg) {
			if(mkdir("raman/2", S_IRWXU) != 0) {
				mprintf(RED, "Directory \"2\" could not be created: %s\n", strerror(errno));
				return false;
			}
			if(mkdir("raman/3", S_IRWXU) != 0) {
				mprintf(RED, "Directory \"3\" could not be created: %s\n", strerror(errno));
				return false;
			}
		}
		
		FILE *templateFile;
		templateFile = fopen("raman/template.inp", "w");
		if(templateFile == NULL) {
			mprintf(RED, "Could not open template file: %s\n", strerror(errno));
			return false;
		}
		fprintf(templateFile, "&GLOBAL\n");
		fprintf(templateFile, " PROJECT_NAME polarizability\n");
		fprintf(templateFile, " RUN_TYPE ENERGY\n");
		fprintf(templateFile, " PRINT_LEVEL LOW\n");
		fprintf(templateFile, "&END\n");
		fprintf(templateFile, "&FORCE_EVAL\n");
		fprintf(templateFile, " &DFT\n");
		fprintf(templateFile, "  BASIS_SET_FILE_NAME BASIS_MOLOPT\n");
		fprintf(templateFile, "  POTENTIAL_FILE_NAME POTENTIAL\n");
		fprintf(templateFile, "  &MGRID\n");
		fprintf(templateFile, "   NGRIDS 5\n");
		fprintf(templateFile, "   CUTOFF 280\n");
		fprintf(templateFile, "   REL_CUTOFF 40\n");
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, "  &SCF\n");
		fprintf(templateFile, "   SCF_GUESS RESTART\n");
		fprintf(templateFile, "   MAX_SCF 200\n");
		fprintf(templateFile, "   EPS_SCF 1.0E-5\n");
		fprintf(templateFile, "   &OT\n");
		fprintf(templateFile, "    MINIMIZER DIIS\n");
		fprintf(templateFile, "    PRECONDITIONER FULL_ALL\n");
		fprintf(templateFile, "    ENERGY_GAP 0.001\n");
		fprintf(templateFile, "   &END\n");
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, "  &XC\n");
		fprintf(templateFile, "   &XC_FUNCTIONAL BLYP\n");
		fprintf(templateFile, "   &END\n");
		fprintf(templateFile, "   &XC_GRID\n");
		fprintf(templateFile, "    XC_SMOOTH_RHO NN10\n");
		fprintf(templateFile, "    XC_DERIV NN10_SMOOTH\n");
		fprintf(templateFile, "   &END\n");
		fprintf(templateFile, "   &VDW_POTENTIAL\n");
		fprintf(templateFile, "    POTENTIAL_TYPE PAIR_POTENTIAL\n");
		fprintf(templateFile, "    &PAIR_POTENTIAL\n");
		fprintf(templateFile, "     TYPE DFTD2\n");
		fprintf(templateFile, "     REFERENCE_FUNCTIONAL BLYP\n");
		fprintf(templateFile, "    &END\n");
		fprintf(templateFile, "   &END\n");
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, "  &LOCALIZE\n");
		fprintf(templateFile, "   METHOD CRAZY\n");
		fprintf(templateFile, "   MAX_ITER 2000\n");
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, "  &PERIODIC_EFIELD\n");
		fprintf(templateFile, "   INTENSITY ###!field strength will be put here###\n");
		fprintf(templateFile, "   POLARISATION ###!polarisation vector will be put here###\n");
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, "  &PRINT\n");
		fprintf(templateFile, "   &LOCALIZATION\n");
		fprintf(templateFile, "    &WANNIER_CENTERS\n");
		fprintf(templateFile, "     IONS+CENTERS\n");
		fprintf(templateFile, "     FILENAME =polarizability_wannier.xyz\n");
		fprintf(templateFile, "    &END\n");
		fprintf(templateFile, "   &END\n");
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, " &END\n");
		fprintf(templateFile, " &SUBSYS\n");
		fprintf(templateFile, "  &CELL\n");
		fprintf(templateFile, "   ABC %.6f %.6f %.6f\n", g_fBoxX / 100.0f, g_fBoxY / 100.0f, g_fBoxZ / 100.0f);
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, "  &COORD\n");
		fprintf(templateFile, "###!coordinates will be put here###\n");
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, "  &KIND H\n");
		fprintf(templateFile, "   BASIS_SET DZVP-MOLOPT-SR-GTH\n");
		fprintf(templateFile, "   POTENTIAL GTH-BLYP-q1\n");
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, "  &KIND C\n");
		fprintf(templateFile, "   BASIS_SET DZVP-MOLOPT-SR-GTH\n");
		fprintf(templateFile, "   POTENTIAL GTH-BLYP-q4\n");
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, "  &KIND N\n");
		fprintf(templateFile, "   BASIS_SET DZVP-MOLOPT-SR-GTH\n");
		fprintf(templateFile, "   POTENTIAL GTH-BLYP-q5\n");
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, "  &KIND O\n");
		fprintf(templateFile, "   BASIS_SET DZVP-MOLOPT-SR-GTH\n");
		fprintf(templateFile, "   POTENTIAL GTH-BLYP-q6\n");
		fprintf(templateFile, "  &END\n");
		fprintf(templateFile, " &END\n");
		fprintf(templateFile, "&END\n");
		fclose(templateFile);
		
		mprintf("    An input template for the polarizability calculations has been written to raman/template.inp\n    Please modify it according to your needs.\n    Press any key when you are finished.\n");
		getchar();
	} else {
		settingsFile = fopen("raman/settings.dat", "r");
		if(settingsFile == NULL) {
			mprintf(RED, "Could not open settings file: %s\n", strerror(errno));
			return false;
		}
		if(!parseSettings(settingsFile)) {
			mprintf(RED, "Could not parse settings file\n");
			return false;
		}
		if(g_orientAvg)
			mprintf("    Using orientational averaging\n");
		else
			mprintf("    Not using orientational averaging\n");
		mprintf("    Field strength: %.6e a. u.\n", g_fieldStrength);
		mprintf("    Using every %d-th timestep\n\n", g_stride);
		
		g_laser = AskFloat("    Calculate scattering cross section for which laser frequency (cm^-1)? [20000.0] ", 20000.0f);
		g_temp = AskFloat("    Calculate scattering cross section for which temperature (K)? [300.0] ", 300.0f);
		
		while(true) {
			mprintf(YELLOW, "\n>>> Raman Observation %d >>>\n\n", g_ramObserv.GetSize() + 1);
			
			CRamanObservation *obs;
			try { obs = new CRamanObservation(); } catch(...) { obs = NULL; }
			if(obs == NULL) NewException((double)sizeof(CRamanObservation), __FILE__, __LINE__, __PRETTY_FUNCTION__);
			g_ramObserv.Add(obs);
			
			mprintf(YELLOW, "<<< End of Raman Observation %d <<<\n\n", g_oaObserv.GetSize() + 1);
			
			if(!AskYesNo("    Add another observation (y/n)? [no] ", false))
				break;
			mprintf("\n");
		}
		mprintf("\n");
	}
	
	return true;
#endif
}

bool initializeRaman() {
	int i;

	if(g_newRaman) {
		FILE *templateFile;
		templateFile = fopen("raman/template.inp", "r");
		if(templateFile == NULL) {
			mprintf(RED, "Could not open template file: %s\n", strerror(errno));
			return false;
		}
		fseek(templateFile, 0L, SEEK_END);
		long length = ftell(templateFile);
		if(length < 0) {
			mprintf(RED, "Could not determine size of template file: %s\n", strerror(errno));
			return false;
		}
		
		try { g_inputTemplate = new char[length + 1]; } catch(...) { g_inputTemplate = NULL; }
		if(g_inputTemplate == NULL) NewException((double)(length + 1)*sizeof(char), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		
		rewind(templateFile);
		if((long)fread(g_inputTemplate, sizeof(char), length, templateFile) < length) {
			mprintf(RED, "Could not read template file: %s\n", strerror(errno));
			return false;
		}
		g_inputTemplate[length] = '\0';
		
		fclose(templateFile);
		
		g_templateFieldPos = strstr(g_inputTemplate, "###!field strength will be put here###");
		if(g_templateFieldPos == NULL) {
			mprintf(RED, "Position mark for field strength missing in template.\n");
			return false;
		}
		g_templatePolPos = strstr(g_inputTemplate, "###!polarisation vector will be put here###");
		if(g_templatePolPos == NULL) {
			mprintf(RED, "Position mark for polarisation vector missing in template.\n");
			return false;
		}
		g_templateCoordPos = strstr(g_inputTemplate, "###!coordinates will be put here###");
		if(g_templateCoordPos == NULL) {
			mprintf(RED, "Position mark for coordinates missing in template.\n");
			return false;
		}
		g_templateFieldPos[0] = '\0';
		g_templatePolPos[0] = '\0';
		g_templateCoordPos[0] = '\0';
	} else {
		for(i = 0; i < g_ramObserv.GetSize(); i++) {
			mprintf("Initializing Raman Observation %d...\n", i+1);
			CRamanObservation *obs = (CRamanObservation *)g_ramObserv[i];
			obs->initialize();
		}
		
		char filename[BUF_SIZE];
		for(i = 0; i < (g_orientAvg ? 3 : 1); i++) {

#ifdef TARGET_WINDOWS
			_snprintf(filename, BUF_SIZE, "raman/%d/polarizability_wannier.xyz", i + 1);
#else
			snprintf(filename, BUF_SIZE, "raman/%d/polarizability_wannier.xyz", i + 1);
#endif

			g_polFile[i] = fopen(filename, "r");
			if(g_polFile[i] == NULL) {
				mprintf(RED, "Could not open polarizability file %s: %s\n", filename, strerror(errno));
				return false;
			}
		}
		
		for(i = 0; i < (g_orientAvg ? 3 : 1); i++) {
			try { g_timestep[i] = new CTimeStep(); } catch(...) { g_timestep[i] = NULL; }
			if(g_timestep[i] == NULL) NewException((double)sizeof(CTimeStep), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		}
	}
	
	return true;
}

void processRaman(CTimeStep *ts) {
	int i, j, k, l;
	g_step++;
	if(g_newRaman) {
		if(g_step % g_stride == 0) {
			FILE *inputFile;
			char filename[BUF_SIZE];
			for(i = 1; i < (g_orientAvg ? 4 : 2); i++) {

#ifdef TARGET_WINDOWS
				_snprintf(filename, BUF_SIZE, "raman/%d/polarizability_%08d_%d.inp", i, g_step, i);
#else
				snprintf(filename, BUF_SIZE, "raman/%d/polarizability_%08d_%d.inp", i, g_step, i);
#endif

				inputFile = fopen(filename, "w");
				if(inputFile == NULL) {
					mprintf(RED, "Could not open input file %s: %s\n", filename, strerror(errno));
					abort();
				}
				
				fprintf(inputFile, "%s", g_inputTemplate);
				for(k = 0; k < 3; k++) {
					char *tempPos = g_inputTemplate;
					for(l = 0; l <= k; l++)
						tempPos = strchr(&tempPos[1], '\0');
					if(tempPos == g_templateFieldPos) {
						fprintf(inputFile, "%.6E", g_fieldStrength);
						fprintf(inputFile, "%s", &g_templateFieldPos[38]);
					} else if(tempPos == g_templatePolPos) {
						if(i == 1)
							fprintf(inputFile, "1.0 0.0 0.0");
						if(i == 2)
							fprintf(inputFile, "0.0 1.0 0.0");
						if(i == 3)
							fprintf(inputFile, "0.0 0.0 1.0");
						fprintf(inputFile, "%s", &g_templatePolPos[43]);
					} else if(tempPos == g_templateCoordPos) {
						for(j = 0; j < g_iGesAtomCount; j++) {
							if(g_waAtomMolIndex[j] == 60000)
								continue;
							fprintf(inputFile, "   %s %14.10f %14.10f %14.10f\n", ((CAtom *)g_oaAtoms[g_waAtomRealElement[j]])->m_sName, ts->m_vaCoords_Original[j][0] / 100.0f, ts->m_vaCoords_Original[j][1] / 100.0f, ts->m_vaCoords_Original[j][2] / 100.0f);
						}
						fprintf(inputFile, "%s", &g_templateCoordPos[36]);
					} else {
						mprintf(RED, "Unexpeceted error while processing input template.\n");
						abort();
					}
				}
				fclose(inputFile);
			}
		}
	} else {
		if(g_step % g_stride == 0) {
			for(i = 0; i < g_ramObserv.GetSize(); i++)
				((CRamanObservation *)g_ramObserv[i])->getDipole0();
			for(i = 0; i < (g_orientAvg ? 3 : 1); i++) {
				if(!g_timestep[i]->ReadTimestep(g_polFile[i], false)) {
					mprintf(RED, "Error while reading trajectory for polarizabilities.\n");
					abort();
				}
				if(!g_bSaveCoordsUnchanged) {
					g_timestep[i]->UniteMolecules(false);
					if(g_bRemoveCOM)
						g_timestep[i]->CenterCOM();
				}
				g_timestep[i]->CalcCenters();
				if(g_bWannier)
					g_timestep[i]->ScanWannier(false);
				g_timestep[i]->CalcDipoles();
				for(j = 0; j < g_ramObserv.GetSize(); j++) {
					((CRamanObservation *)g_ramObserv[j])->calcPolarizability(i);
				}
			}
		}
	}
}

void finalizeRaman() {
	int i;
	if(g_newRaman) {
		delete g_inputTemplate;
	} else {
		for(i = 0; i < g_ramObserv.GetSize(); i++) {
			mprintf(YELLOW, ">>> Raman Observation %d >>>\n\n", i + 1);
			((CRamanObservation *)g_ramObserv[i])->finalize();
			mprintf(YELLOW, "\n<<< End of Raman Observation %d <<<\n", i + 1);
		}
		
		for(i = 0; i < (g_orientAvg ? 3 : 1); i++)
			fclose(g_polFile[i]);
		for(i = 0; i < (g_orientAvg ? 3 : 1); i++)
			delete g_timestep[i];
	}
}
