/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2013 Martin Brehm
                  2012-2013 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "ffgen.h"
#include "moltools.h"
#include "globalvar.h"


CFFGen::CFFGen()
{
}


CFFGen::~CFFGen()
{
}


void CFFGen::Parse(CTimeStep *ts)
{
	int z, z2, z3, z4, z5, z6, ti, b1, b2, b3;
	int ba[6], ba2[6]; // Total, H, C, N, O, Het
	CMolecule *m;
	CSingleMolecule *sm;
	CMolBond *mb;
	char buf[16];
	bool ring, ring2, b;
	float tf, xmi, xma, ymi, yma, zmi, zma;
	CxIntArray *rs;
	CFFAtomType *fat;
	CFFAtom *fa;
	CFFBondType *fbt;
	CFFBond *fb;
	CFFAngleType *fnt;
	CFFAngle *fn;
	CFFDihedralType *fdt;
	CFFDihedral *fd;
	CFFImproperType *fit;
	CFFImproper *fi;
	FILE *ffout;
	struct tm *today;
	time_t ltime;

	InitFFParm();

	mprintf(WHITE,"\n>>> Force Field Generator >>>\n\n");

	m_fImproperTol = AskFloat("    Please enter Improper tolerance (in deg): [25.0] ",25.0f);

	if (!g_bResp)
	{
		try { g_pResp = new CResp(); } catch(...) { g_pResp = NULL; }
		if (g_pResp == NULL) NewException((double)sizeof(g_pResp),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		g_pResp->m_faCharges.SetSize(g_iGesAtomCount);
		if (AskYesNo("    Enter partial charges for atoms (y/n)? [yes] ",true))
		{
			for (z=0;z<g_oaMolecules.GetSize();z++)
			{
				m = (CMolecule*)g_oaMolecules[z];
				mprintf(WHITE,"\n  * Molecule %s\n\n",m->m_sName);
				for (z2=0;z2<m->m_baAtomIndex.GetSize();z2++)
				{
					if (m->m_baAtomIndex[z2] == g_iVirtAtomType)
						continue;

					for (z3=0;z3<m->m_waAtomCount[z2];z3++)
					{
						tf = AskFloat("    Enter charge on atom %s%d: [0.0] ",0,((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_sName,z3+1);
						for (z4=0;z4<m->m_laSingleMolIndex.GetSize();z4++)
						{
							sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z4]];
							g_pResp->m_faCharges[((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3)] = tf;
						}
					}
				}
			}
			mprintf("\n");
		} else
		{
			mprintf("\n    Setting all charges to zero...\n\n");
			for (z=0;z<g_iGesAtomCount;z++)
				g_pResp->m_faCharges[z] = 0;
		}
	}


	mprintf("    Assigning unit cell...\n");

	xmi = 1e20f;
	xma = -1e20f;
	ymi = 1e20f;
	yma = -1e20f;
	zmi = 1e20f;
	zma = -1e20f;

	for (z=0;z<g_iGesAtomCount;z++)
	{
		if (ts->m_vaCoords[z][0] < xmi)
			xmi = ts->m_vaCoords[z][0];
		if (ts->m_vaCoords[z][0] > xma)
			xma = ts->m_vaCoords[z][0];

		if (ts->m_vaCoords[z][1] < ymi)
			ymi = ts->m_vaCoords[z][1];
		if (ts->m_vaCoords[z][1] > yma)
			yma = ts->m_vaCoords[z][1];

		if (ts->m_vaCoords[z][2] < zmi)
			zmi = ts->m_vaCoords[z][2];
		if (ts->m_vaCoords[z][2] > zma)
			zma = ts->m_vaCoords[z][2];
	}

	mprintf("      Spatial extent of input structure is %.2f x %.2f x %.2f pm.\n",xma-xmi,yma-ymi,zma-zmi);
	mprintf("      Using box length of %.2f x %.2f x %.2f pm.\n",xma-xmi+250.0f,yma-ymi+250.0f,zma-zmi+250.0f);

	mprintf("    Assigning atom types...\n");

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		mprintf(YELLOW,"\n  * %s:\n",m->m_sName);
		sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[0]];

		try { m->m_sFFLabels = new char*[sm->m_baAtomIndex.GetSize()]; } catch(...) { m->m_sFFLabels = NULL; }
		if (m->m_sFFLabels == NULL) NewException((double)sm->m_baAtomIndex.GetSize()*sizeof(char*),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		for (z2=0;z2<sm->m_baAtomIndex.GetSize();z2++)
		{
			if (sm->m_baAtomIndex[z2] == g_iVirtAtomType)
				continue;

//			mprintf("    Atom type %s:\n",((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z2]])->m_sName);

			try { m->m_sFFLabels[z2] = new char[((CxIntArray*)sm->m_oaAtomOffset[z2])->GetSize()*3]; } catch(...) { m->m_sFFLabels[z2] = NULL; }
			if (m->m_sFFLabels[z2] == NULL) NewException((double)(((CxIntArray*)sm->m_oaAtomOffset[z2])->GetSize()*3)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);

			for (z3=0;z3<((CxIntArray*)sm->m_oaAtomOffset[z2])->GetSize();z3++)
			{
//				ma = (CMolAtom*)sm->m_oaMolAtoms[z3];

				if (sm->m_oaBonds.GetSize() == 0)
				{
					mprintf("      %s%d: No bonds ",((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z2]])->m_sName,z3+1);
					sprintf(buf,"%s",((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z2]])->m_sName);
					mprintf(" --> Type %s\n",buf);
					sprintf(&m->m_sFFLabels[z2][z3*3],"%s",buf);
					continue;
				}

				mprintf("      %s%d: Bound to ",((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z2]])->m_sName,z3+1);

				b = false;

				ScanAtom(sm,((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3),ba);

				ring = false;
				for (z5=0;z5<sm->m_oaRings.GetSize();z5++)
				{
					rs = (CxIntArray*)sm->m_oaRings[z5];
					for (z6=0;z6<rs->GetSize();z6++)
					{
						if (((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3) == rs->GetAt(z6))
						{
							ring = true;
							goto _ridone;
						}
					}
				}
_ridone:

				for (z4=0;z4<sm->m_oaBonds.GetSize();z4++)
				{
					mb = (CMolBond*)sm->m_oaBonds[z4];

					if (mb->m_iAtomOffset[0] == ((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3))
						ti = mb->m_iAtomOffset[1];
					else if (mb->m_iAtomOffset[1] == ((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3))
						ti = mb->m_iAtomOffset[0];
					else continue;

					if (b)
						mprintf(", ");
					b = true;

					mprintf("%s",((CAtom*)g_oaAtoms[g_waAtomRealElement[ti]])->m_sName);

					ScanAtom(sm,ti,ba2);

					ring2 = false;
					for (z5=0;z5<sm->m_oaRings.GetSize();z5++)
					{
						rs = (CxIntArray*)sm->m_oaRings[z5];
						for (z6=0;z6<rs->GetSize();z6++)
						{
							if (ti == rs->GetAt(z6))
							{
								ring2 = true;
								goto _ridone2;
							}
						}
					}
_ridone2:

					if (mystricmp(((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z2]])->m_sName,"H") == 0)
					{
						if (ba[3] == 1) // N
							sprintf(buf,"H");
						else if (ba[2] == 1) // C
						{
							if (ring2 && (ba2[0] == 3))
							{
								if (ba2[5] == 2)
									sprintf(buf,"H5");
								else if (ba2[5] == 1)
									sprintf(buf,"H4");
								else sprintf(buf,"HA");
							} else
							{
								if (ba2[5] == 3)
									sprintf(buf,"H3");
								else if (ba2[5] == 2)
									sprintf(buf,"H2");
								else if (ba2[5] == 1)
									sprintf(buf,"H1");
								else if ((ba2[2] == 1) && (ba2[0] == 2))
									sprintf(buf,"HZ");  // Ethyne
								else sprintf(buf,"HC");
							}
						} else if (ba[4] == 1) // O
							sprintf(buf,"HO");
						else sprintf(buf,"X");
					} else if (mystricmp(((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z2]])->m_sName,"C") == 0)
					{
						if (ba[0] == 4)
							sprintf(buf,"CT");
						else if (ring && (ba[0] == 3) && (ba[1] == 1) && (ba[2] == 2))
							sprintf(buf,"CA");
						else if (ring && (ba[0] == 3) && (ba[1] == 1) && (ba[2] == 1) && (ba[3] == 1))
							sprintf(buf,"CW");
						else if (ring && (ba[0] == 3) && (ba[1] == 1) && (ba[2] == 0) && (ba[3] == 2))
							sprintf(buf,"CR");
						else if ((ba[4] == 2) || ((ba[3] == 1) && (ba[4] == 1)))
							sprintf(buf,"C");
						else if ((ba[0] == 2) && (ba[2] == 1) && (ba[1] == 1))
							sprintf(buf,"CZ"); // Ethyne
						else sprintf(buf,"X");
					} else if (mystricmp(((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z2]])->m_sName,"N") == 0)
					{
						if (ring && (ba[0] == 3))
							sprintf(buf,"NA");
						else if ((ba[2] >= 1) && (ba2[4] == 1))
							sprintf(buf,"N");
						else if (ba[0] >= 3)
							sprintf(buf,"N3");
						else sprintf(buf,"X");
					} else if (mystricmp(((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z2]])->m_sName,"O") == 0)
					{
						if (ba[2] == 2)
							sprintf(buf,"OS");
						else if (ba[1] >= 1)
							sprintf(buf,"OH");
						else if ((ba[0] == 1) && (ba[2] == 1)) // C=O
						{
							if (ba2[4] == 1)
								sprintf(buf,"O");
									else sprintf(buf,"O2");
						} else if ((ba[4] == 1) && (ba[0] == 1))
							sprintf(buf,"OM"); // Molecular O2
						else sprintf(buf,"O");
					} else if (mystricmp(((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z2]])->m_sName,"S") == 0)
						sprintf(buf,"S6");
					else sprintf(buf,"%s",((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z2]])->m_sName);

				} // END FOR z4

				mprintf(" --> Type %s\n",buf);
				sprintf(&m->m_sFFLabels[z2][z3*3],"%s",buf);
			}
		}
	}

	mprintf("\n");
	if (AskYesNo("    Create images of structure formulas with atom labels (y/n)? [no] ",false))
		RenderStructureFormulas();

	mprintf(WHITE,"\n    Setting up atom list...\n");

	for (z=0;z<g_iGesAtomCount;z++)
	{
		m = (CMolecule*)g_oaMolecules[g_waAtomMolIndex[z]];
		sm = (CSingleMolecule*)g_oaSingleMolecules[g_laAtomSMIndex[z]];

		for (z2=0;z2<m_oaAtomTypes.GetSize();z2++)
		{
			if (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],((CFFAtomType*)m_oaAtomTypes[z2])->m_sName) == 0)
			{
				ti = z2;
				goto _atdone;
			}
		}

		try { fat = new CFFAtomType(); } catch(...) { fat = NULL; }
		if (fat == NULL) NewException((double)sizeof(CFFAtomType),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		fat->m_fLJEpsilon = 0;
		fat->m_fLJSigma = 0;
		fat->m_iRealElement = g_waAtomRealElement[z];
		sprintf(fat->m_sName,"%s",&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3]);
		ti = m_oaAtomTypes.GetSize();
		mprintf("      Adding atom type %-2s (%2s)...\n",fat->m_sName,((CAtom*)g_oaAtoms[fat->m_iRealElement])->m_sName);
		m_oaAtomTypes.Add(fat);
_atdone:
		try { fa = new CFFAtom(); } catch(...) { fa = NULL; }
		if (fa == NULL) NewException((double)sizeof(CFFAtom),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		fa->m_iType = ti;
		fa->m_iOffset = z;
		fa->m_iLammpsMolecule = g_waAtomMolUID[z];
		fa->m_iMoleculeType = g_waAtomMolIndex[z];
		fa->m_iMoleculeIndex = g_laAtomSMLocalIndex[z];
		fa->m_iAtomIndex = g_waAtomMolNumber[z];
		m_oaAtoms.Add(fa);
	}
	mprintf("\n    %d atoms and %d atom types added.\n\n",m_oaAtoms.GetSize(),m_oaAtomTypes.GetSize());

	mprintf(WHITE,"\n    Setting up bond list...\n");

	for (z=0;z<g_iGesAtomCount;z++)
	{
		m = (CMolecule*)g_oaMolecules[g_waAtomMolIndex[z]];
		sm = (CSingleMolecule*)g_oaSingleMolecules[g_laAtomSMIndex[z]];

		for (z2=0;z2<sm->m_oaBonds.GetSize();z2++)
		{
			if (((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[0] == z)
				b1 = ((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[1];
			else if (((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[1] == z)
					b1 = ((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[0];
			else continue;

			if (b1 < z)
				continue;

			for (z4=0;z4<m_oaBondTypes.GetSize();z4++)
			{
				if ((mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],((CFFBondType*)m_oaBondTypes[z4])->m_sName) == 0) &&
				    (mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFBondType*)m_oaBondTypes[z4])->m_sName[3]) == 0))
				{
					ti = z4;
					goto _btdone;
				}
				if ((mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFBondType*)m_oaBondTypes[z4])->m_sName[3]) == 0) &&
				    (mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],((CFFBondType*)m_oaBondTypes[z4])->m_sName) == 0))
				{
					ti = z4;
					goto _btdone;
				}
			}

			try { fbt = new CFFBondType(); } catch(...) { fbt = NULL; }
			if (fbt == NULL) NewException((double)sizeof(CFFBondType),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			fbt->m_fK = 0;
			fbt->m_fLength = 0;
			sprintf(fbt->m_sName,"%s",&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3]);
			sprintf(&fbt->m_sName[3],"%s",&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3]);
			sprintf(fbt->m_sRealName,"%-2s-%-2s",&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3]);
			ti = m_oaBondTypes.GetSize();
			mprintf("      Adding bond type %s...\n",fbt->m_sRealName);
			m_oaBondTypes.Add(fbt);
_btdone:
			try { fb = new CFFBond(); } catch(...) { fb = NULL; }
			if (fb == NULL) NewException((double)sizeof(CFFBond),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			fb->m_iType = ti;
			fb->m_iAtoms[0] = z;
			fb->m_iAtoms[1] = b1;
			m_oaBonds.Add(fb);
		}
	}
	mprintf("\n    %d bonds and %d bond types added.\n\n",m_oaBonds.GetSize(),m_oaBondTypes.GetSize());

	mprintf(WHITE,"\n    Setting up angle list...\n");

	for (z=0;z<g_iGesAtomCount;z++)
	{
		m = (CMolecule*)g_oaMolecules[g_waAtomMolIndex[z]];
		sm = (CSingleMolecule*)g_oaSingleMolecules[g_laAtomSMIndex[z]];

		for (z2=0;z2<sm->m_oaBonds.GetSize();z2++)
		{
			if (((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[0] == z)
				b1 = ((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[1];
			else if (((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[1] == z)
					b1 = ((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[0];
			else continue;

			for (z3=z2+1;z3<sm->m_oaBonds.GetSize();z3++)
			{
				if (((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[0] == z)
					b2 = ((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[1];
				else if (((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[1] == z)
						b2 = ((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[0];
				else continue;

				for (z4=0;z4<m_oaAngleTypes.GetSize();z4++)
				{
					if ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],((CFFAngleType*)m_oaAngleTypes[z4])->m_sName) == 0) &&
						 (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFAngleType*)m_oaAngleTypes[z4])->m_sName[3]) == 0) &&
						 (mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&((CFFAngleType*)m_oaAngleTypes[z4])->m_sName[6]) == 0))
					{
						ti = z4;
						goto _ntdone;
					}
					if ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],((CFFAngleType*)m_oaAngleTypes[z4])->m_sName) == 0) &&
						 (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFAngleType*)m_oaAngleTypes[z4])->m_sName[3]) == 0) &&
						 (mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFAngleType*)m_oaAngleTypes[z4])->m_sName[6]) == 0))
					{
						ti = z4;
						goto _ntdone;
					}
				}
				try { fnt = new CFFAngleType(); } catch(...) { fnt = NULL; }
				if (fnt == NULL) NewException((double)sizeof(CFFAngleType),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				fnt->m_fAngle = 0;
				fnt->m_fK = 0;
				sprintf(fnt->m_sName,"%s",&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3]);
				sprintf(&fnt->m_sName[3],"%s",&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3]);
				sprintf(&fnt->m_sName[6],"%s",&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3]);
				sprintf(fnt->m_sRealName,"%-2s-%-2s-%-2s",&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3]);
				ti = m_oaAngleTypes.GetSize();
				mprintf("      Adding angle type %2s...\n",fnt->m_sRealName);
				m_oaAngleTypes.Add(fnt);
_ntdone:
				try { fn = new CFFAngle(); } catch(...) { fn = NULL; }
				if (fn == NULL) NewException((double)sizeof(CFFAngle),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				fn->m_iType = ti;
				fn->m_iAtoms[0] = b1;
				fn->m_iAtoms[1] = z;
				fn->m_iAtoms[2] = b2;
				m_oaAngles.Add(fn);
			}
		}

	}
	mprintf("\n    %d angles and %d angle types added.\n\n",m_oaAngles.GetSize(),m_oaAngleTypes.GetSize());

	mprintf(WHITE,"\n    Setting up dihedral list...\n");

	for (z=0;z<g_iGesAtomCount;z++)
	{
		m = (CMolecule*)g_oaMolecules[g_waAtomMolIndex[z]];
		sm = (CSingleMolecule*)g_oaSingleMolecules[g_laAtomSMIndex[z]];

		for (z2=0;z2<sm->m_oaBonds.GetSize();z2++)
		{
			if (((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[0] == z)
				b1 = ((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[1];
			else if (((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[1] == z)
					b1 = ((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[0];
			else continue;

			for (z3=0;z3<sm->m_oaBonds.GetSize();z3++)
			{
				if (((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[0] == b1)
					b2 = ((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[1];
				else if (((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[1] == b1)
						b2 = ((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[0];
				else continue;

				if (b2 == z)
					continue;

				for (z4=0;z4<sm->m_oaBonds.GetSize();z4++)
				{
					if (((CMolBond*)sm->m_oaBonds[z4])->m_iAtomOffset[0] == b2)
						b3 = ((CMolBond*)sm->m_oaBonds[z4])->m_iAtomOffset[1];
					else if (((CMolBond*)sm->m_oaBonds[z4])->m_iAtomOffset[1] == b2)
							b3 = ((CMolBond*)sm->m_oaBonds[z4])->m_iAtomOffset[0];
					else continue;

					if (b3 == b1)
						continue;

					if (b3 < z)
						continue;

					for (z5=0;z5<m_oaDihedralTypes.GetSize();z5++)
					{
						if ((mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],((CFFDihedralType*)m_oaDihedralTypes[z5])->m_sName) == 0) &&
							(mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFDihedralType*)m_oaDihedralTypes[z5])->m_sName[3]) == 0) &&
							(mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&((CFFDihedralType*)m_oaDihedralTypes[z5])->m_sName[6]) == 0) &&
							(mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],&((CFFDihedralType*)m_oaDihedralTypes[z5])->m_sName[9]) == 0))
						{
							ti = z5;
							goto _dtdone;
						}
						if ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],((CFFDihedralType*)m_oaDihedralTypes[z5])->m_sName) == 0) &&
							(mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&((CFFDihedralType*)m_oaDihedralTypes[z5])->m_sName[3]) == 0) &&
							(mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFDihedralType*)m_oaDihedralTypes[z5])->m_sName[6]) == 0) &&
							(mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFDihedralType*)m_oaDihedralTypes[z5])->m_sName[9]) == 0))
						{
							ti = z5;
							goto _dtdone;
						}
					}

					try { fdt = new CFFDihedralType(); } catch(...) { fdt = NULL; }
					if (fdt == NULL) NewException((double)sizeof(CFFDihedralType),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					fdt->m_iA = 0;
					fdt->m_iN = 0;
					fdt->m_fK = 0;
					sprintf(fdt->m_sName,"%s",&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3]);
					sprintf(&fdt->m_sName[3],"%s",&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3]);
					sprintf(&fdt->m_sName[6],"%s",&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3]);
					sprintf(&fdt->m_sName[9],"%s",&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3]);
					sprintf(fdt->m_sRealName,"%-2s-%-2s-%-2s-%-2s",&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3]);
					ti = m_oaDihedralTypes.GetSize();
					mprintf("      Adding Dihedral type %2s...\n",fdt->m_sRealName);
					m_oaDihedralTypes.Add(fdt);
_dtdone:
					try { fd = new CFFDihedral(); } catch(...) { fd = NULL; }
					if (fd == NULL) NewException((double)sizeof(CFFDihedral),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					fd->m_iType = ti;
					fd->m_iAtoms[0] = z;
					fd->m_iAtoms[1] = b1;
					fd->m_iAtoms[2] = b2;
					fd->m_iAtoms[3] = b3;
					m_oaDihedrals.Add(fd);
				}
			}
		}
	}
	mprintf("\n    %d dihedrals and %d dihedral types added.\n\n",m_oaDihedrals.GetSize(),m_oaDihedralTypes.GetSize());

	mprintf(WHITE,"\n    Setting up improper list...\n");

	try { m_pImproperDF = new CDF(); } catch(...) { m_pImproperDF = NULL; }
	if (m_pImproperDF == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pImproperDF->m_fMinVal = 0;
	m_pImproperDF->m_fMaxVal = 90.0f;
	m_pImproperDF->m_iResolution = 91;
	m_pImproperDF->Create();
	m_pImproperDF->SetLabelX("Improper angle [Deg]");
	m_pImproperDF->SetLabelY("Occurence");

	for (z=0;z<g_iGesAtomCount;z++)
	{
		m = (CMolecule*)g_oaMolecules[g_waAtomMolIndex[z]];
		sm = (CSingleMolecule*)g_oaSingleMolecules[g_laAtomSMIndex[z]];

		for (z2=0;z2<sm->m_oaBonds.GetSize();z2++)
		{
			if (((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[0] == z)
				b1 = ((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[1];
			else if (((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[1] == z)
					b1 = ((CMolBond*)sm->m_oaBonds[z2])->m_iAtomOffset[0];
			else continue;

			for (z3=z2+1;z3<sm->m_oaBonds.GetSize();z3++)
			{
				if (((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[0] == z)
					b2 = ((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[1];
				else if (((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[1] == z)
						b2 = ((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[0];
				else continue;

				for (z4=z3+1;z4<sm->m_oaBonds.GetSize();z4++)
				{
					if (((CMolBond*)sm->m_oaBonds[z4])->m_iAtomOffset[0] == z)
						b3 = ((CMolBond*)sm->m_oaBonds[z4])->m_iAtomOffset[1];
					else if (((CMolBond*)sm->m_oaBonds[z4])->m_iAtomOffset[1] == z)
							b3 = ((CMolBond*)sm->m_oaBonds[z4])->m_iAtomOffset[0];
					else continue;

					tf = Dihedral(ts->m_vaCoords[b2]-ts->m_vaCoords[b1],ts->m_vaCoords[b3]-ts->m_vaCoords[z],ts->m_vaCoords[z]-ts->m_vaCoords[b2],true);

					m_pImproperDF->AddToBin(tf);

					if (tf > m_fImproperTol)
						continue;

					for (z5=0;z5<m_oaImproperTypes.GetSize();z5++)
					{
						if (((mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)) ||

					       ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)) ||

					       ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)) ||

					       ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)) ||

					       ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)) ||

					       ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)) ||

						    ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)) ||

					       ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)) ||

					       ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)) ||

					       ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)) ||

					       ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)) ||

					       ((mystricmp(&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],((CFFImproperType*)m_oaImproperTypes[z5])->m_sName) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[3]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[6]) == 0) &&
						     (mystricmp(&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3],&((CFFImproperType*)m_oaImproperTypes[z5])->m_sName[9]) == 0)))
						{
							ti = z5;
							goto _itdone;
						}
					}

					try { fit = new CFFImproperType(); } catch(...) { fit = NULL; }
					if (fit == NULL) NewException((double)sizeof(CFFImproperType),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					fit->m_fDihedral = 0;
					fit->m_fK = 0;
					sprintf(fit->m_sName,"%s",&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3]);
					sprintf(&fit->m_sName[3],"%s",&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3]);
					sprintf(&fit->m_sName[6],"%s",&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3]);
					sprintf(&fit->m_sName[9],"%s",&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3]);
					sprintf(fit->m_sRealName,"%-2s-%-2s-%-2s-%-2s",&m->m_sFFLabels[g_waAtomElement[b1]][g_waAtomMolNumber[b1]*3],&m->m_sFFLabels[g_waAtomElement[b2]][g_waAtomMolNumber[b2]*3],&m->m_sFFLabels[g_waAtomElement[z]][g_waAtomMolNumber[z]*3],&m->m_sFFLabels[g_waAtomElement[b3]][g_waAtomMolNumber[b3]*3]);
					ti = m_oaImproperTypes.GetSize();
					mprintf("      Adding Improper type %2s...\n",fit->m_sRealName);
					m_oaImproperTypes.Add(fit);
_itdone:
					try { fi = new CFFImproper(); } catch(...) { fi = NULL; }
					if (fi == NULL) NewException((double)sizeof(CFFImproper),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					fi->m_iType = ti;
					fi->m_iAtoms[0] = b1;
					fi->m_iAtoms[1] = b2;
					fi->m_iAtoms[2] = z;
					fi->m_iAtoms[3] = b3;
					m_oaImpropers.Add(fi);
				}
			}
		}
	}

	mprintf("\n    %d impropers and %d improper types added.\n\n",m_oaImpropers.GetSize(),m_oaImproperTypes.GetSize());

	mprintf("    Saving Improper Distribution function as ffgen_improper_df.csv ...\n");
	m_pImproperDF->Write("","ffgen_improper_df.csv","",false);
	mprintf("\n");

	AssignParms();

	mprintf(WHITE,"\n    Saving LAMMPS data file as lammps.dat ...\n");

	ffout = OpenFileWrite("lammps.dat",true);

	time(&ltime);
	today = localtime(&ltime);
	strcpy(buf,asctime(today));
	buf[strlen(buf)-1] = 0;

	mprintf("      Header...\n");

	mfprintf(ffout,"LAMMPS data file. CGCMM style. atom_style full generated by TRAVIS on %s\n",buf);
	mfprintf(ffout," %d atoms\n",m_oaAtoms.GetSize());
	mfprintf(ffout," %d bonds\n",m_oaBonds.GetSize());
	mfprintf(ffout," %d angles\n",m_oaAngles.GetSize());
	mfprintf(ffout," %d dihedrals\n",m_oaDihedrals.GetSize());
	mfprintf(ffout," %d impropers\n",m_oaImpropers.GetSize());
	mfprintf(ffout," %d atom types\n",m_oaAtomTypes.GetSize());
	mfprintf(ffout," %d bond types\n",m_oaBondTypes.GetSize());
	mfprintf(ffout," %d angle types\n",m_oaAngleTypes.GetSize());
	mfprintf(ffout," %d dihedral types\n",m_oaDihedralTypes.GetSize());
	mfprintf(ffout," %d improper types\n",m_oaImproperTypes.GetSize());
	mfprintf(ffout," %f  %f  xlo xhi\n",0.0f,(xma-xmi+250.0f)/100.0f);
	mfprintf(ffout," %f  %f  ylo yhi\n",0.0f,(yma-ymi+250.0f)/100.0f);
	mfprintf(ffout," %f  %f  zlo zhi\n",0.0f,(zma-zmi+250.0f)/100.0f);
	mfprintf(ffout,"\n");

	mprintf("      Masses...\n");
	mfprintf(ffout," Masses\n\n");
	fflush(ffout);
	for (z=0;z<m_oaAtomTypes.GetSize();z++)
	{
		fat = (CFFAtomType*)m_oaAtomTypes[z];
		mfprintf(ffout," %2d  %7.3f  # %s\n",z+1,((CAtom*)g_oaAtoms[fat->m_iRealElement])->m_pElement->m_fMass,fat->m_sName);
		fflush(ffout);
	}

	mprintf("      Pair Coeffs...\n");
	mfprintf(ffout,"\n Pair Coeffs\n\n"); //   # epsilon (Kcal/mol),  sigma (Angstrom)
	fflush(ffout);
	for (z=0;z<m_oaAtomTypes.GetSize();z++)
	{
		fat = (CFFAtomType*)m_oaAtomTypes[z];
		mfprintf(ffout," %2d  %7.5f  %6.4f  # %s\n",z+1,fat->m_fLJEpsilon/4.184,fat->m_fLJSigma,fat->m_sName);
		fflush(ffout);
	}

	mprintf("      Bond Coeffs...\n");
	mfprintf(ffout,"\n Bond Coeffs\n\n"); //   # K (Kcal*Angstrom^2/mol),  eq. length (Angstrom)
	fflush(ffout);
	for (z=0;z<m_oaBondTypes.GetSize();z++)
	{
		fbt = (CFFBondType*)m_oaBondTypes[z];
		mfprintf(ffout," %2d  %8.3f  %7.5f  # %s\n",z+1,fbt->m_fK/4.184,fbt->m_fLength,fbt->m_sRealName);
		fflush(ffout);
	}

	if (m_oaAngleTypes.GetSize() > 0)
	{
		mprintf("      Angle Coeffs...\n");
		mfprintf(ffout,"\n Angle Coeffs\n\n"); //   # K (Kcal*rad^2/mol),  eq. angle (deg)
		fflush(ffout);
		for (z=0;z<m_oaAngleTypes.GetSize();z++)
		{
			fnt = (CFFAngleType*)m_oaAngleTypes[z];
			mfprintf(ffout," %2d  %8.3f  %7.3f  # %s\n",z+1,fnt->m_fK/4.184,fnt->m_fAngle,fnt->m_sRealName);
			fflush(ffout);
		}
	}

	if (m_oaDihedralTypes.GetSize() > 0)
	{
		mprintf("      Dihedral Coeffs...\n");
		mfprintf(ffout,"\n Dihedral Coeffs\n\n"); //   # K/2 (Kcal/mol),  d,  n
		fflush(ffout);
		for (z=0;z<m_oaDihedralTypes.GetSize();z++)
		{
			fdt = (CFFDihedralType*)m_oaDihedralTypes[z];
			mfprintf(ffout," %2d  %7.3f  %2d  %2d  # %s\n",z+1,fdt->m_fK/4.184,fdt->m_iA,fdt->m_iN,fdt->m_sRealName);
			fflush(ffout);
		}
	}

	if (m_oaImproperTypes.GetSize() > 0)
	{
		mprintf("      Improper Coeffs...\n");
		mfprintf(ffout,"\n Improper Coeffs\n\n");
		fflush(ffout);
		for (z=0;z<m_oaImproperTypes.GetSize();z++)
		{
			fit = (CFFImproperType*)m_oaImproperTypes[z];
			mfprintf(ffout," %2d  %7.3f  %7.3f  # %s\n",z+1,fit->m_fK/4.184,fit->m_fDihedral,fit->m_sRealName);
			fflush(ffout);
		}
	}

	mprintf("      Atoms...\n");
	mfprintf(ffout,"\n Atoms\n\n");
	fflush(ffout);
	for (z=0;z<m_oaAtoms.GetSize();z++)
	{
		fa = (CFFAtom*)m_oaAtoms[z];
		tf = g_pResp->m_faCharges[z];
		mfprintf(ffout," %5d  %3d  %2d  %8.5f  %10.5f  %10.5f  %10.5f  # %-2s, %s[%d] %s%d\n",z+1,fa->m_iLammpsMolecule+1,fa->m_iType+1,tf,ts->m_vaCoords[z][0]/100.0,ts->m_vaCoords[z][1]/100.0,ts->m_vaCoords[z][2]/100.0,((CFFAtomType*)m_oaAtomTypes[fa->m_iType])->m_sName,((CMolecule*)g_oaMolecules[fa->m_iMoleculeType])->m_sName,fa->m_iMoleculeIndex+1,((CAtom*)g_oaAtoms[((CFFAtomType*)m_oaAtomTypes[fa->m_iType])->m_iRealElement])->m_sName,fa->m_iAtomIndex+1);
		fflush(ffout);
	}

	mprintf("      Bonds...\n");
	mfprintf(ffout,"\n Bonds\n\n");
	fflush(ffout);
	for (z=0;z<m_oaBonds.GetSize();z++)
	{
		fb = (CFFBond*)m_oaBonds[z];
		mfprintf(ffout," %5d  %2d  %5d  %5d  # %s, %s[%d] %s%d - %s%d\n",z+1,fb->m_iType+1,fb->m_iAtoms[0]+1,fb->m_iAtoms[1]+1,((CFFBondType*)m_oaBondTypes[fb->m_iType])->m_sRealName,((CMolecule*)g_oaMolecules[g_waAtomMolIndex[fb->m_iAtoms[0]]])->m_sName,g_laAtomSMLocalIndex[fb->m_iAtoms[0]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fb->m_iAtoms[0]]])->m_sName,g_waAtomMolNumber[fb->m_iAtoms[0]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fb->m_iAtoms[1]]])->m_sName,g_waAtomMolNumber[fb->m_iAtoms[1]]+1);
		fflush(ffout);
	}

	if (m_oaAngleTypes.GetSize() > 0)
	{
		mprintf("      Angles...\n");
		mfprintf(ffout,"\n Angles\n\n");
		fflush(ffout);
		for (z=0;z<m_oaAngles.GetSize();z++)
		{
			fn = (CFFAngle*)m_oaAngles[z];
			mfprintf(ffout," %5d  %2d  %5d  %5d  %5d  # %s, %s[%d] %s%d - %s%d - %s%d\n",z+1,fn->m_iType+1,fn->m_iAtoms[0]+1,fn->m_iAtoms[1]+1,fn->m_iAtoms[2]+1,((CFFAngleType*)m_oaAngleTypes[fn->m_iType])->m_sRealName,((CMolecule*)g_oaMolecules[g_waAtomMolIndex[fn->m_iAtoms[0]]])->m_sName,g_laAtomSMLocalIndex[fn->m_iAtoms[0]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fn->m_iAtoms[0]]])->m_sName,g_waAtomMolNumber[fn->m_iAtoms[0]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fn->m_iAtoms[1]]])->m_sName,g_waAtomMolNumber[fn->m_iAtoms[1]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fn->m_iAtoms[2]]])->m_sName,g_waAtomMolNumber[fn->m_iAtoms[2]]+1);
			fflush(ffout);
		}
	}

	if (m_oaDihedralTypes.GetSize() > 0)
	{
		mprintf("      Dihedrals...\n");
		mfprintf(ffout,"\n Dihedrals\n\n");
		fflush(ffout);
		for (z=0;z<m_oaDihedrals.GetSize();z++)
		{
			fd = (CFFDihedral*)m_oaDihedrals[z];
			mfprintf(ffout," %5d  %2d  %5d  %5d  %5d  %5d  # %s, %s[%d] %s%d - %s%d - %s%d - %s%d\n",z+1,fd->m_iType+1,fd->m_iAtoms[0]+1,fd->m_iAtoms[1]+1,fd->m_iAtoms[2]+1,fd->m_iAtoms[3]+1,((CFFDihedralType*)m_oaDihedralTypes[fd->m_iType])->m_sRealName,((CMolecule*)g_oaMolecules[g_waAtomMolIndex[fd->m_iAtoms[0]]])->m_sName,g_laAtomSMLocalIndex[fd->m_iAtoms[0]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fd->m_iAtoms[0]]])->m_sName,g_waAtomMolNumber[fd->m_iAtoms[0]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fd->m_iAtoms[1]]])->m_sName,g_waAtomMolNumber[fd->m_iAtoms[1]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fd->m_iAtoms[2]]])->m_sName,g_waAtomMolNumber[fd->m_iAtoms[2]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fd->m_iAtoms[3]]])->m_sName,g_waAtomMolNumber[fd->m_iAtoms[3]]+1);
			fflush(ffout);
		}
	}

	if (m_oaImproperTypes.GetSize() > 0)
	{
		mprintf("      Impropers...\n");
		mfprintf(ffout,"\n Impropers\n\n");
		fflush(ffout);
		for (z=0;z<m_oaImpropers.GetSize();z++)
		{
			fi = (CFFImproper*)m_oaImpropers[z];
			mfprintf(ffout," %5d  %2d  %5d  %5d  %5d  %5d  # %s, %s[%d] %s%d - %s%d - %s%d - %s%d\n",z+1,fi->m_iType+1,fi->m_iAtoms[0]+1,fi->m_iAtoms[1]+1,fi->m_iAtoms[2]+1,fi->m_iAtoms[3]+1,((CFFImproperType*)m_oaImproperTypes[fi->m_iType])->m_sRealName,((CMolecule*)g_oaMolecules[g_waAtomMolIndex[fi->m_iAtoms[0]]])->m_sName,g_laAtomSMLocalIndex[fi->m_iAtoms[0]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fi->m_iAtoms[0]]])->m_sName,g_waAtomMolNumber[fi->m_iAtoms[0]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fi->m_iAtoms[1]]])->m_sName,g_waAtomMolNumber[fi->m_iAtoms[1]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fi->m_iAtoms[2]]])->m_sName,g_waAtomMolNumber[fi->m_iAtoms[2]]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[fi->m_iAtoms[3]]])->m_sName,g_waAtomMolNumber[fi->m_iAtoms[3]]+1);
			fflush(ffout);
		}
	}

	fclose(ffout);

	mprintf("\n    Finished.\n");

	mprintf(WHITE,"\n<<< End of Force Field Generator <<<\n\n");
}


void CFFGen::ScanAtom(CSingleMolecule *sm, int o, int *ba)
{
	int z, z2, z3, z4, ti;
	CMolBond *mb;

	for (z=0;z<6;z++)
		ba[z] = 0;

	for (z2=0;z2<sm->m_baAtomIndex.GetSize();z2++)
	{
		if (sm->m_baAtomIndex[z2] == g_iVirtAtomType)
			continue;

		for (z3=0;z3<((CxIntArray*)sm->m_oaAtomOffset[z2])->GetSize();z3++)
		{
			if (((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3) != o)
				continue;

			for (z4=0;z4<sm->m_oaBonds.GetSize();z4++)
			{
				mb = (CMolBond*)sm->m_oaBonds[z4];

				if (mb->m_iAtomOffset[0] == ((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3))
					ti = mb->m_iAtomOffset[1];
				else if (mb->m_iAtomOffset[1] == ((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3))
					ti = mb->m_iAtomOffset[0];
				else continue;

				ba[0]++;

				if (mystricmp(((CAtom*)g_oaAtoms[g_waAtomRealElement[ti]])->m_sName,"H") == 0)
					ba[1]++;
				else if (mystricmp(((CAtom*)g_oaAtoms[g_waAtomRealElement[ti]])->m_sName,"C") == 0)
					ba[2]++;
				else if (mystricmp(((CAtom*)g_oaAtoms[g_waAtomRealElement[ti]])->m_sName,"N") == 0)
				{ ba[3]++; ba[5]++; }
				else if (mystricmp(((CAtom*)g_oaAtoms[g_waAtomRealElement[ti]])->m_sName,"O") == 0)
				{ ba[4]++; ba[5]++; }
				else ba[5]++;
			}
		}
	}
}


void CFFGen::RenderStructureFormulas()
{
	int z, z2, z3;
	FILE *a;
	char buf[256];
	CMolecule *m;
	CSingleMolecule *sm;
	CMolAtom *ma1, *ma2;

	mprintf("\n");
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[0]];

		sprintf(buf,"ff_mol%d_%s.dot",z+1,m->m_sName);
		mprintf("Molecule %d: %s\n",z+1,m->m_sName);
		mprintf("    Writing dot file %s...\n",buf);
		a = OpenFileWrite(buf,true);
		mfprintf(a,"graph molecule {\n");
		mfprintf(a,"  graph [pack=true,splines=true,overlap=false];\n");
		mfprintf(a,"  node [shape=none,fontsize=16,fontname=\"Arial\",margin=0,fixedsize,height=0.35];\n");
		mfprintf(a,"  edge [style=bold,len=0.70];\n");

		for (z2=0;z2<sm->m_oaMolAtoms.GetSize();z2++)
		{
			ma1 = (CMolAtom*)sm->m_oaMolAtoms[z2];
			mfprintf(a,"  %s%d [label=\"%s\",width=%.2f];\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[ma1->m_iType]])->m_sName,ma1->m_iNumber+1,&m->m_sFFLabels[ma1->m_iType][ma1->m_iNumber*3],(strlen(&m->m_sFFLabels[ma1->m_iType][ma1->m_iNumber*3]))*0.25f);
		}

		for (z2=0;z2<sm->m_oaMolAtoms.GetSize();z2++)
		{
			ma1 = (CMolAtom*)sm->m_oaMolAtoms[z2];
			for (z3=0;z3<ma1->m_oaBonds.GetSize();z3++)
			{
				ma2 = (CMolAtom*)ma1->m_oaBonds[z3];
				if (ma2->m_iMolAtomNumber < ma1->m_iMolAtomNumber)
					continue;
				mfprintf(a,"  %s%d -- %s%d [weight=%d];\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[ma1->m_iType]])->m_sName,ma1->m_iNumber+1,((CAtom*)g_oaAtoms[m->m_baAtomIndex[ma2->m_iType]])->m_sName,ma2->m_iNumber+1,(int)(((CAtom*)g_oaAtoms[m->m_baAtomIndex[ma1->m_iType]])->m_pElement->m_fMass+((CAtom*)g_oaAtoms[m->m_baAtomIndex[ma2->m_iType]])->m_pElement->m_fMass));
			}
		}
		mfprintf(a,"}\n");
		fclose(a);
		sprintf(buf,"dot ff_mol%d_%s.dot -Tpng -off_mol%d_%s.png -Kneato -Gepsilon=0.000001 -Gnslimit=5000 -Gmclimit=5000",z+1,m->m_sName,z+1,m->m_sName);
		mprintf("    Executing \"%s\"...\n",buf);
		system(buf);
		mprintf("\n");
	}
	mprintf("    You can now view the PNG files that have been created.\n\n");
}


void CFFGen::AddAtomParm(const char *s, float sigma, float epsilon)
{
	CFFAtomType *at;

	try { at = new CFFAtomType(); } catch(...) { at = NULL; }
	if (at == NULL) NewException((double)sizeof(CFFAtomType),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	at->m_fLJEpsilon = epsilon;
	at->m_fLJSigma = sigma;
	strcpy(at->m_sName,s);

	m_oaAtomParm.Add(at);
}


void CFFGen::AddBondParm(const char *s1, const char *s2, float length, float k)
{
	CFFBondType *bt;

	try { bt = new CFFBondType(); } catch(...) { bt = NULL; }
	if (bt == NULL) NewException((double)sizeof(CFFBondType),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	bt->m_fLength = length;
	bt->m_fK = k;
	strcpy(bt->m_sName,s1);
	strcpy(&bt->m_sName[3],s2);
	sprintf(bt->m_sRealName,"%-2s-%-2s",s1,s2);

	m_oaBondParm.Add(bt);
}


void CFFGen::AddAngleParm(const char *s1, const char *s2, const char *s3, float angle, float k)
{
	CFFAngleType *nt;

	try { nt = new CFFAngleType(); } catch(...) { nt = NULL; }
	if (nt == NULL) NewException((double)sizeof(CFFAngleType),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	nt->m_fAngle = angle;
	nt->m_fK = k;
	strcpy(nt->m_sName,s1);
	strcpy(&nt->m_sName[3],s2);
	strcpy(&nt->m_sName[6],s3);
	sprintf(nt->m_sRealName,"%-2s-%-2s-%-2s",s1,s2,s3);

	m_oaAngleParm.Add(nt);
}


void CFFGen::AddDihedralParm(const char *s1, const char *s2, const char *s3, const char *s4, float dih, int n, float k)
{
	CFFDihedralType *dt;

	try { dt = new CFFDihedralType(); } catch(...) { dt = NULL; }
	if (dt == NULL) NewException((double)sizeof(CFFDihedralType),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	dt->m_fK = k;
	dt->m_fDih = dih;
	dt->m_iN = n;
	if (dih == 0)
		dt->m_iA = 1;
	else if (dih == 180.0f)
		dt->m_iA = -1;
	else eprintf("AddDihedralParm(): Error in parameter set for %s-%s-%s-%s.\n",s1,s2,s3,s4);
	strcpy(dt->m_sName,s1);
	strcpy(&dt->m_sName[3],s2);
	strcpy(&dt->m_sName[6],s3);
	strcpy(&dt->m_sName[9],s4);
	sprintf(dt->m_sRealName,"%-2s-%-2s-%-2s-%-2s",s1,s2,s3,s4);

	m_oaDihedralParm.Add(dt);
}


void CFFGen::AddImproperParm(const char *s1, const char *s2, const char *s3, const char *s4, float dih, int n, float k)
{
	CFFImproperType *it;

	try { it = new CFFImproperType(); } catch(...) { it = NULL; }
	if (it == NULL) NewException((double)sizeof(CFFImproperType),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	it->m_fK = k;
	it->m_fDihedral = dih;
	it->m_iN = n;
	strcpy(it->m_sName,s1);
	strcpy(&it->m_sName[3],s2);
	strcpy(&it->m_sName[6],s3);
	strcpy(&it->m_sName[9],s4);
	sprintf(it->m_sRealName,"%-2s-%-2s-%-2s-%-2s",s1,s2,s3,s4);

	m_oaImproperParm.Add(it);
}


void CFFGen::AssignParms()
{
	int z, z2;
	bool b, c;
	CFFAtomType *at, *at2;
	CFFBondType *bt, *bt2;
	CFFAngleType *nt, *nt2;
	CFFDihedralType *dt, *dt2;
	CFFImproperType *it, *it2;

	mprintf("    Assigning force field parameters...\n");

	mprintf(WHITE,"\n    Atoms...\n");
	b = false;
	c = false;
	for (z=0;z<m_oaAtomTypes.GetSize();z++)
	{
		at = (CFFAtomType*)m_oaAtomTypes[z];

		for (z2=0;z2<m_oaAtomParm.GetSize();z2++)
		{
			at2 = (CFFAtomType*)m_oaAtomParm[z2];
			if (mystricmp(at->m_sName,at2->m_sName) == 0)
			{
				at->m_fLJSigma = at2->m_fLJSigma;
				at->m_fLJEpsilon = at2->m_fLJEpsilon;
				goto _atdone;
			}
		}
		eprintf("      No atom parameter for %s.\n",at->m_sName);
		b = true;
		c = true;
_atdone:;
	}
	if (b)
		mprintf("\n");

	mprintf(WHITE,"    Bonds...\n");
	b = false;
	for (z=0;z<m_oaBondTypes.GetSize();z++)
	{
		bt = (CFFBondType*)m_oaBondTypes[z];

		for (z2=0;z2<m_oaBondParm.GetSize();z2++)
		{
			bt2 = (CFFBondType*)m_oaBondParm[z2];

			if (((mystricmp(&bt->m_sName[0],&bt2->m_sName[0]) == 0) && (mystricmp(&bt->m_sName[3],&bt2->m_sName[3]) == 0)) ||
			    ((mystricmp(&bt->m_sName[3],&bt2->m_sName[0]) == 0) && (mystricmp(&bt->m_sName[0],&bt2->m_sName[3]) == 0)))
			{
				bt->m_fLength = bt2->m_fLength;
				bt->m_fK = bt2->m_fK;
				memcpy(bt->m_sName,bt2->m_sName,6);
				strcpy(bt->m_sRealName,bt2->m_sRealName);
				goto _btdone;
			}
		}
		eprintf("      No bond parameter for %s.\n",bt->m_sRealName);
		b = true;
		c = true;
_btdone:;
	}
	if (b)
		mprintf("\n");

	mprintf(WHITE,"    Angles...\n");
	b = false;
	for (z=0;z<m_oaAngleTypes.GetSize();z++)
	{
		nt = (CFFAngleType*)m_oaAngleTypes[z];

		for (z2=0;z2<m_oaAngleParm.GetSize();z2++)
		{
			nt2 = (CFFAngleType*)m_oaAngleParm[z2];

			if (((mystricmp(&nt->m_sName[0],&nt2->m_sName[0]) == 0) && (mystricmp(&nt->m_sName[3],&nt2->m_sName[3]) == 0) && (mystricmp(&nt->m_sName[6],&nt2->m_sName[6]) == 0)) ||
			    ((mystricmp(&nt->m_sName[0],&nt2->m_sName[6]) == 0) && (mystricmp(&nt->m_sName[3],&nt2->m_sName[3]) == 0) && (mystricmp(&nt->m_sName[6],&nt2->m_sName[0]) == 0)))
			{
				nt->m_fAngle = nt2->m_fAngle;
				nt->m_fK = nt2->m_fK;
				memcpy(nt->m_sName,nt2->m_sName,9);
				strcpy(nt->m_sRealName,nt2->m_sRealName);
				goto _ntdone;
			}
		}
		eprintf("      No angle parameter for %s.\n",nt->m_sRealName);
		b = true;
		c = true;
_ntdone:;
	}
	if (b)
		mprintf("\n");

	mprintf(WHITE,"    Dihedrals...\n");
	b = false;
	for (z=0;z<m_oaDihedralTypes.GetSize();z++)
	{
		dt = (CFFDihedralType*)m_oaDihedralTypes[z];

		for (z2=0;z2<m_oaDihedralParm.GetSize();z2++)
		{
			dt2 = (CFFDihedralType*)m_oaDihedralParm[z2];

			if (((mystricmp(&dt->m_sName[0],&dt2->m_sName[0]) == 0) && (mystricmp(&dt->m_sName[3],&dt2->m_sName[3]) == 0) && (mystricmp(&dt->m_sName[6],&dt2->m_sName[6]) == 0) && (mystricmp(&dt->m_sName[9],&dt2->m_sName[9]) == 0)) ||
			    ((mystricmp(&dt->m_sName[0],&dt2->m_sName[9]) == 0) && (mystricmp(&dt->m_sName[3],&dt2->m_sName[6]) == 0) && (mystricmp(&dt->m_sName[6],&dt2->m_sName[3]) == 0) && (mystricmp(&dt->m_sName[9],&dt2->m_sName[0]) == 0)))
			{
				dt->m_fDih = dt2->m_fDih;
				dt->m_fK = dt2->m_fK;
				dt->m_iA = dt2->m_iA;
				dt->m_iN = dt2->m_iN;
				memcpy(dt->m_sName,dt2->m_sName,12);
				strcpy(dt->m_sRealName,dt2->m_sRealName);
				goto _dtdone;
			}
		}
		eprintf("      No dihedral parameter for %s.\n",dt->m_sRealName);
		b = true;
		c = true;
_dtdone:;
	}
	if (b)
		mprintf("\n");

	mprintf(WHITE,"    Impropers...\n");
	b = false;
	for (z=0;z<m_oaImproperTypes.GetSize();z++)
	{
		it = (CFFImproperType*)m_oaImproperTypes[z];

		for (z2=0;z2<m_oaImproperParm.GetSize();z2++)
		{
			it2 = (CFFImproperType*)m_oaImproperParm[z2];

			if (((mystricmp(&it->m_sName[0],&it2->m_sName[0]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[3]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[9],&it2->m_sName[9]) == 0)) ||
			    ((mystricmp(&it->m_sName[0],&it2->m_sName[3]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[9]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[9],&it2->m_sName[0]) == 0)) ||
			    ((mystricmp(&it->m_sName[0],&it2->m_sName[9]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[0]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[9],&it2->m_sName[3]) == 0)) ||
			    ((mystricmp(&it->m_sName[0],&it2->m_sName[0]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[9]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[9],&it2->m_sName[3]) == 0)) ||
			    ((mystricmp(&it->m_sName[0],&it2->m_sName[3]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[0]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[9],&it2->m_sName[0]) == 0)) ||
			    ((mystricmp(&it->m_sName[0],&it2->m_sName[9]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[3]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[9],&it2->m_sName[9]) == 0)) ||
			    ((mystricmp(&it->m_sName[9],&it2->m_sName[0]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[3]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[0],&it2->m_sName[9]) == 0)) ||
			    ((mystricmp(&it->m_sName[9],&it2->m_sName[3]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[9]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[0],&it2->m_sName[0]) == 0)) ||
			    ((mystricmp(&it->m_sName[9],&it2->m_sName[9]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[0]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[0],&it2->m_sName[3]) == 0)) ||
			    ((mystricmp(&it->m_sName[9],&it2->m_sName[0]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[9]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[0],&it2->m_sName[3]) == 0)) ||
			    ((mystricmp(&it->m_sName[9],&it2->m_sName[3]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[0]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[0],&it2->m_sName[0]) == 0)) ||
			    ((mystricmp(&it->m_sName[9],&it2->m_sName[9]) == 0) && (mystricmp(&it->m_sName[6],&it2->m_sName[3]) == 0) && (mystricmp(&it->m_sName[3],&it2->m_sName[6]) == 0) && (mystricmp(&it->m_sName[0],&it2->m_sName[9]) == 0)))
			{
				it->m_fDihedral = it2->m_fDihedral;
				it->m_fK = it2->m_fK;
				it->m_iN = it2->m_iN;
				memcpy(it->m_sName,it2->m_sName,12);
				strcpy(it->m_sRealName,it2->m_sRealName);
				goto _itdone;
			}
		}
		eprintf("      No improper parameter for %s.\n",it->m_sRealName);
		b = true;
		c = true;
_itdone:;
	}

	if (c)
	{
		mprintf(WHITE,"\n    Note: ");
		mprintf("You can add the missing parameters either in the output file \"lammps.dat\"\n");
		mprintf("          or in the TRAVIS source code in \"ffparm.cpp\".\n");
	}
}


void CFFAtomNode::REC_Parse(char *s, int depth)
{
	(void)s;
	(void)depth;
/*	char *p, *q, *r, buf[256];
	bool top;
	int i, z;
	CFFAtomNode *c;

	RecHeader(depth);
	mprintf("REC_Parse(\"%s\", %d)\n",s,depth);
	if ((p=strchr(s,'=')) != NULL)
	{
		top = true;
		m_sDefineType = new char[p-s+1];
		memcpy(m_sDefineType,s,p-s);
		m_sDefineType[p-s] = 0;
		p++;
		RecHeader(depth);
		mprintf("  Toplevel, definiere \"%s\".\n",m_sDefineType);
	} else
	{
		p = s;
		top = false;

		while ((*p != '(') && (*p != 0))
			p++;

		if (*p == 0)
		{
			eprintf("Error 1.\n");
			abort();
		}
		p++;
		RecHeader(depth);
		mprintf("  Non-Toplevel: \"%s\".\n",p);
	}

_nextlabel:
	q = p;
	while ((*q != '(') && (*q != '[') && (*q != 0) && (*q != ','))
		q++;

	r = new char[q-p+1];
	memcpy(r,p,q-p);
	r[q-p] = 0;
	m_paLabel.Add(r);
	p = q+1;
	RecHeader(depth);
	mprintf("  Atomtyp \"%s\".\n",r);

	if (*q == ',')
	{
		RecHeader(depth);
		mprintf("  Noch ein Atomtyp...\n");
		goto _nextlabel;
	}

_nextsub:
	if (*q == '(')
	{
		i = 1;
		p = q;
		while (true)
		{
			q++;
			if (*q == '(')
				i++;
			if (*q == ')')
				i--;
			if (i == 0)
				break;
			if (*q == 0)
			{
				eprintf("Error 2.\n");
				abort();
			}
		}
		if (*(q+1) == '^')
		{
			q+=2;
			if (*q == '(')
			{
				while (*q != ')')
					q++;
			}
		}
		RecHeader(depth);
		memcpy(buf,p,q-p);
		buf[q-p] = 0;
		mprintf("  Found substituent \"%s\".\n",buf);

		c = new CFFAtomNode();
		m_oaChildren.Add(c);
		c->m_pParent = this;
		c->REC_Parse(buf,depth+1);

		q++;
		goto _nextsub;
	}

	if (*q == '^')
	{
		q++;
		p = q;
		if (*q == '(')
		{
			p++;
			while (*q != ')')
				q++;
			memcpy(buf,p,q-p);
			buf[q-p] = 0;
			RecHeader(depth);
			mprintf("  Exponent found: \"%s\".\n",buf);
			ParseIntList(buf,&m_iaMultiplicity);
		} else if (*q == '*')
		{
			for (z=0;z<10;z++)
				m_iaMultiplicity.Add(z);
		} else
		{
			buf[0] = *p;
			buf[1] = 0;
			m_iaMultiplicity.Add(atoi(buf));
		}
	}

	if (m_iaMultiplicity.GetSize() == 0)
		m_iaMultiplicity.Add(1);

	RecHeader(depth);
	mprintf("  Multiplicity: %d",m_iaMultiplicity[0]);
	for (z=1;z<m_iaMultiplicity.GetSize();z++)
		mprintf(", %d",m_iaMultiplicity[z]);
	mprintf("\n");

	RecHeader(depth);
	mprintf("REC_Parse(\"%s\", %d) done.\n",s,depth);*/
}


void CFFAtomNode::RecHeader(int depth)
{
	int z;

	for (z=0;z<depth;z++)
		mprintf(" (%d) ",depth);
}


void CFFAtomNode::REC_DumpTree(int depth, unsigned long bitmask, bool last)
{
	BTIN;
	int z, z2;
	
	mprintf("  ");
	for (z=0;z<depth-1;z++)
	{
		if ((bitmask & (unsigned long)pow(2.0,z)) != 0)
			mprintf("|   ");
				else mprintf("    ");
	}
	if (depth != 0)
	{
		if (last)
			mprintf("\\---");
				else mprintf("|---");
	}
	mprintf("[%s",(const char*)m_paLabel[0]);
	for (z=1;z<m_paLabel.GetSize();z++)
		mprintf(",%s",(const char*)m_paLabel[z]);
	mprintf("]\n");
	
	if (m_oaChildren.GetSize() != 0)
	{
		mprintf("  ");
		for (z2=0;z2<depth;z2++)
			if ((bitmask & (unsigned long)pow(2.0,z2)) != 0)
				mprintf("|   ");
					else mprintf("    ");
		mprintf("|\n");
	}
	
	for (z=0;z<m_oaChildren.GetSize();z++)
	{
		if (z+1 == m_oaChildren.GetSize())
			bitmask -= (unsigned long)pow(2.0,depth);
		((CFFAtomNode*)m_oaChildren[z])->REC_DumpTree(depth+1,bitmask,z+1==m_oaChildren.GetSize());
	}

	if ((m_oaChildren.GetSize() == 0) && last)
	{
		mprintf("  ");
		for (z2=0;z2<depth;z2++)
			if ((bitmask & (unsigned long)pow(2.0,z2)) != 0)
				mprintf("|   ");
					else mprintf("    ");
		mprintf("\n");
	}
}
