#include "tra.h"

int expectconflicts;
int debug;
int coverage;
int res;
int nflag;
int autosame = 1;
Chan *c;
int printwork;
int printstats;
char *fromname;
char *toname;
Sync *sync;		/* for our copy of sysfatal */
int flate;

static Path*
strtopath(char *s)
{
	char *f[20];
	int i, nf;
	Path *p;

	nf = getfields(s, f, nelem(f), 1, "/");
	p = nil;
	for(i=0; i<nf; i++)
		p = mkpath(p, f[i]);
	return p;
}

void
usage(void)
{
	fprint(2, "usage: minisync [-nftv] [-z level] from to [path...]\n");
	exits("usage");
}

void
sysfatal(char *fmt, ...)
{
	char buf[256];
	va_list arg;

	va_start(arg, fmt);
	vseprint(buf, buf+sizeof buf, fmt, arg);
	va_end(arg);

	fprint(2, "%s; %s\n", argv0 ? argv0 : "<prog>", buf);
	if(sync->from)
		replclose(sync->from);
	if(sync->to)
		replclose(sync->to);
	sync->from = sync->to = nil;
	fprint(2, "hangup\n");
	exits("fatal");
}

static char* conflictstr[] =
{
	"no conflict",
	"update/update",
	"update/delete",
	"delete/update",
};

static char*
conflicttostr(int c)
{
	static char buf[32];

	if(0 <= c && c < nelem(conflictstr))
		return conflictstr[c];

	snprint(buf, sizeof buf, "unknown %d", c);
	return buf;
}

static char* statestr[] = 
{
	"start",
	"copy",
	"remove",
	"kids",
	"error",
	"conflict",
	"complete",
	"incomplete",
};

static void
noworkthread(void *a)
{
	Queue *q;
	Syncpath *p;

	q = a;
	while((p = qrecv(q)) != nil)
		syncfinish(p, 0);
}

static void
printworkfn(Syncpath *s)
{
	if(s->state != SyncCopy || s->t->state != SFile || s->f->state != SFile
	|| memcmp(s->f->sha1, s->t->sha1, 20) != 0)
		print("%s %P%s\n", statestr[s->state], s->p, s->f->state == SDir ? "/" : "");
}

static char*
thetime(long t)
{
	static char buf[32];

	strcpy(buf, sysctime(t));
	buf[strlen(buf)-1] = '\0';
	return buf;
}

static void
printmtime(int fd, char *host, Stat *s)
{
	int i;
	char *by;
	char *muid;
	Vtime *m;

	m = s->mtime;
	switch(m->nl){
	case 1:
		muid = s->muid;
		if(muid && muid[0])
			by = " by ";
		else{
			muid = "";
			by = "";
		}
		fprint(fd, "\t%s: last modified %s (#%lud) on %s%s%s\n", host,
			thetime(m->l[0].wall), m->l[0].t, m->l[0].m, by, muid);
		break;
	default:
		if(m->nl < 1){
			fprint(fd, "\t%s: unexpected mtime %V\n", host, m);
			break;
		}
		fprint(fd, "\t%s: current file is resolution of modifications\n", host);
		for(i=0; i<m->nl; i++)
			fprint(fd, "\t\tat %s (#%lud) on %s\n", thetime(m->l[i].wall), m->l[0].t, m->l[i].m);
		break;
	}
}

void
threadmain(int argc, char **argv)
{
	char *cmd, *fstr, *tstr;
	int i, npending, nfinish, nsuccess;
	Syncpath *s;
	Queue *workout;
	Fmt fmt;

	quotefmtinstall();
	fmtinstall('H', encodefmt);
	fmtinstall('P', pathfmt);
	fmtinstall('R', rpcfmt);
	fmtinstall('$', statfmt);
	fmtinstall('V', vtimefmt);

	fmtstrinit(&fmt);
	for(i=0; i<argc; i++)
		fmtprint(&fmt, " %q", argv[i]);
	cmd = fmtstrflush(&fmt);
	tlog("cmd %s\n", cmd);

	ARGBEGIN{
	default:
		usage();
	case 'A':
		autosame = 0;
		break;
	case 'C':
		expectconflicts = 1;
		break;
	case 'D':
		debug |= dbglevel(EARGF(usage()));
		break;
	case 'V':
		traversion();
	case 'n':
		nflag = 1;
		printwork = 1;
		break;
	case 't':
	case 'f':
		res = ARGC();
		break;
	case 'v':
		printwork = 1;
		break;
	case 's':
		printstats = 1;
		break;
	case 'z':
		flate = atoi(EARGF(usage()));
		break;
	}ARGEND
	
	if(argc < 2)
		usage();

	fromname = argv[0];
	toname = argv[1];

	startclient();
	sync = emalloc(sizeof(Sync));
	sync->from = dialreplica(fromname);
	sync->to = dialreplica(toname);
	if(clientbanner(sync->from, fromname) < 0)
		sysfatal("%s handshake: %r", fromname);
	threadcreate(replthread, sync->from);
	if(clientbanner(sync->to, toname) < 0)
		sysfatal("%s handshake: %r", toname);
	threadcreate(replthread, sync->to);

	if(rpcdebug(sync->from, debug) < 0)
		sysfatal("%s debug: %r", fromname);
	if(rpcdebug(sync->to, debug) < 0)
		sysfatal("%s debug: %r", toname);

	if((fstr = rpcmeta(sync->from, "sysname")) == nil)
		sysfatal("%s sysname: %r", fromname);
	if((tstr = rpcmeta(sync->to, "sysname")) == nil)
		sysfatal("%s sysname: %r", toname);
	if(strcmp(fstr, tstr) == 0)
		sysfatal("%s and %s are same replica", fstr, tstr);
	tlog("sync from %q to %q\n", fstr, tstr);

	if(flate){
		if(rpcflate(sync->from, flate) < 0)
			sysfatal("starting compression on %s: %r", fromname);
		if(rpcflate(sync->to, flate) < 0)
			sysfatal("starting compression on %s: %r", toname);
	}

	if(rpcreadonly(sync->from, nflag) < 0)
		sysfatal("setting %s to readonly: %r", fromname);
	if(nflag && rpcreadonly(sync->to, 1) < 0)
		sysfatal("setting %s to readonly: %r", toname);

	sync->syncq = mkstack();	/* stack gets things out of system fast */
	sync->workq = mkqueue();		/* depends on ordering */
	workout = sync->workq;
	sync->eventq = mkqueue();	/* probably depends on ordering */

	threadcreate(syncthread, sync->syncq);

	if(printwork)
		sync->workq->printrecv = printworkfn;

	if(nflag)
		threadcreate(noworkthread, workout);
	else
		threadcreate(workthread, workout);

	argc -= 2;
	argv += 2;
	if(argc==0){
		s = emalloc(sizeof(Syncpath));
		s->sync = sync;
		s->p = nil;
		qsend(sync->syncq, s);
		npending = 1;
	}else{
		for(i=0; i<argc; i++){
			s = emalloc(sizeof(Syncpath));
			s->sync = sync;
			s->p = strtopath(argv[i]);
			qsend(sync->syncq, s);
		}
		npending = argc;
	}

	nfinish = 0;
	nsuccess = 0;
	while(nfinish < npending){
		s = qrecv(sync->eventq);
	//	print("%P: %s\n", s->p, statestr[s->state]);
		switch(s->state){
		case SyncComplete:
			nfinish++;
			nsuccess++;
			break;
		case SyncIncomplete:
			nfinish++;
			break;
		case SyncError:
			print("%P: %s\n", s->p, s->err);
			syncfinish(s, 0);
			break;
		case SyncConflict:
			if(res)
				resolve(s, res);
			else if(autosame && s->f && s->t && memcmp(s->f->sha1, s->t->sha1, sizeof s->f->sha1)==0)
				resolve(s, 'f');
			else{
				print("%P: %s conflict\n", s->p, conflicttostr(s->conflict));
				printmtime(1, fromname, s->f);
				printmtime(1, toname, s->t);
				syncfinish(s, 0);
			}
			break;
		default:
			print("%P: state %d\n", s->p, s->state);
			break;
		}
	}

dbg(DbgRpc, "hangup from\n");
	rpchangup(sync->from);
dbg(DbgRpc, "hangup to\n");
	rpchangup(sync->to);
dbg(DbgRpc, "close from\n");
	replclose(sync->from);
dbg(DbgRpc, "close to\n");
	replclose(sync->to);

if(printstats){
	print("queue highwater sync %d work %d event %d rpc %d\n", sync->syncq->m,
		sync->workq->m, sync->eventq->m, mrpc);
	print("%.2fs\n", ((ulong)(nsec()/1000000)-start)/100.0);
	print("rpc %d in %d out\n", inrpctot, outrpctot);
	if(inzrpctot)
		print("compressed rpc %d in %d out\n", inzrpctot, outzrpctot);
}
	if(expectconflicts && nsuccess==nfinish)
		exits("complete");
	else if(!expectconflicts && nsuccess!=nfinish)
		exits("incomplete");
	exits(nil);
}

