/* wmtuxtime.c -- WindowMaker dockable applet portion of TuxTime
 *
 * Copyright (c) 1997-2001  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * Based on
 *       wmapm by Chris D. Faulhaber <jedgar@speck.ml.org> and
 *                Michael G. Henderson <mghenderson@lanl.gov>
 *       mount.app by Steve Borho <borho@stlnet.com>
 *
 * $Log: wmtuxtime.c,v $
 * Revision 1.12  2002/01/27 13:22:40  jab
 * updated list of machine ID's
 *
 * Revision 1.11  2001/10/05 13:02:01  jab
 * fixed argv cast problem in ProcessComandLine
 *
 * Revision 1.10  2001/07/15 08:56:28  jab
 * added -h and -v command line options plus a few machine ID's
 *
 * Revision 1.9  2001/03/21 13:40:35  jab
 * fixed bug with charging C in combination with low contrast option
 * fixed display corruption when changing battery
 *
 * Revision 1.8  2001/02/04 16:39:47  jab
 * added a pile more machine ID's
 * added notice about kernel module to error message
 * moved the lock file to /var/tmp
 * don't display battery warning messages when on mains power
 * added -c option for higher contrast for DSTN screens
 *
 * Revision 1.7  2000/06/23 21:47:25  jab
 * moved all the calls to determine power status into a single routine
 * various code changes to make it more Libretto friendly
 * added a good few machine ID's
 *
 * Revision 1.6  2000/02/12 21:33:31  jab
 * added patch from Stephen Morphet <smorphet@iee.org> to use a fallback to
 * /proc/apm for time remaining when SCI_BATTERY_TIME does not work
 *
 * Revision 1.5  2000/02/12 10:55:55  jab
 * patchs to use waitpid to prevent possible race conditions, update the
 * display less franetically and get the CPU speed working courtesy
 * of Tom May <tom@you-bastards.com>
 *
 * Revision 1.4  2000/01/21 22:16:25  jab
 * fixed bug with blank xmessage windows when launched from dock
 *
 * Revision 1.3  1999/12/17 12:32:28  jab
 * added lots of machine ID's to stop emails with already known models
 *
 * Revision 1.2  1999/12/04 13:42:38  jab
 * added titles to battery warning dialog boxes
 *
 * Revision 1.1  1999/12/01 19:55:28  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

static char const rcsid[]="$Id: wmtuxtime.c,v 1.12 2002/01/27 13:22:40 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#include<fcntl.h>
#include<sys/stat.h>
#include<sys/wait.h>
#include<errno.h>
#include<signal.h>
#include<paths.h>
#include<pwd.h>
#include<features.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif
#include<X11/X.h>
#include<X11/xpm.h>


#include"wmtuxtime.h"
#include"wmgeneral.h"
#include"wmtuxtime.xpm"
#include"wmtuxtime_mask.xbm"
#include"sci.h"
#include"hci.h"
#include"config.h"

#define VERSION_STRING "\
wmTuxTime version 1.4\n\
Copyright (c) 1998-2001 Jonathan A. Buzzard <jonathan@buzzard.org.uk>\n"

#define USAGE_STRING "\
Usage: wmtuxtime [-c] [-n] [-display] [-geometry] [-v] [-h]\n\
View battery status and set power saving options on a Toshiba laptop\n\n\
  -c,--contrast      high contrast option for DSTN screens\n\
  -n,--normal        normal window attributes instead of dock-app\n\
  -display           specify the X server to contact\n\
  -geometry          specify the preferred position of the window\n\
  -h,--help          display this help message\n\
  -v,--version       display version\n\
Report bugs to jonathan@buzzard.org.uk\n"



enum { EMPTY, PRIMARY, AUXILLARY };

static int id;
static int power = SCI_MAINS;
static int selectbay = EMPTY;
static int oldpercent = -1;
static int alarms;
static char *config;
static BatteryState low,critical;


/*
 * Catch any signals and exit the program nicely
 */
void CatchSignal(int x)
{
	fprintf(stderr, "wmtuxtime: caught signal %d, cleaning up...\n", x);

	SciCloseInterface();
	unlink(PID_FILE);
	exit(0);
}


/*
 * Install signal handlers so we can clean up nicely on an error
 */
void HandleSignals(void)
{
	signal(SIGHUP, CatchSignal);
	signal(SIGINT, CatchSignal);
	signal(SIGQUIT, CatchSignal);
	signal(SIGILL, CatchSignal);
	signal(SIGTRAP, CatchSignal);
	signal(SIGABRT, CatchSignal);
	signal(SIGIOT, CatchSignal);
	signal(SIGFPE, CatchSignal);
	signal(SIGKILL, CatchSignal);
	signal(SIGSEGV, CatchSignal);
	signal(SIGPIPE, SIG_IGN);
	signal(SIGTERM, CatchSignal);
	signal(SIGSTOP, CatchSignal);
	signal(SIGTTIN, CatchSignal);
	signal(SIGTTOU, CatchSignal);
	signal(SIGUSR1, SIG_IGN);

	return;
}

/*
 * Process the command line arguments
 */
void ProcessComandLine(int *argc, char ***argv)
{
	int i,j;

	for (i=1;i<*argc;i++) {
		if ((!strcmp((*argv)[i], "-c")) || (!strcmp((*argv)[i], "--contrast"))) {
			for (j=5;j<11;j++)
				wmtuxtime_xpm[3][j] = '0';
			for (j=5;j<11;j++)
				wmtuxtime_xpm[4][j] = '0';
			for (j=5;j<11;j++)
				wmtuxtime_xpm[7][j] = '0';
			for (j=5;j<11;j++)
				wmtuxtime_xpm[9][j] = '0';
		}
		if ((!strcmp((*argv)[i], "-h")) || (!strcmp((*argv)[i], "--help"))) {
			printf(USAGE_STRING);
			SciCloseInterface();
			unlink(PID_FILE);
			exit(0);
		}
		if ((!strcmp((*argv)[i], "-v")) || (!strcmp((*argv)[i], "--version"))) {
			printf(VERSION_STRING);
			SciCloseInterface();
			unlink(PID_FILE);
			exit(0);
		}

	}

	return;
}


/*
 * Use xmessage to display the battery alert messages
 */
int DisplayMessage(char *message, char *title)
{
	int pid,lead[2];

	/* create the pipe */

	if (pipe(lead)<0) {
		fprintf(stderr, "wmtuxtime: unable to create pipe to "
			"xmessage\n");
		return -1;
	}

	if (fcntl(lead[0], F_SETFL, O_NDELAY)<0)
		fprintf(stderr, "wmtuxtime: fatal fcntl call\n");

	pid = fork();
	if (pid>0) {
		write(lead[1], message, strlen(message));
		close(lead[0]);
		close(lead[1]);
		return 127;
	} else if (pid==0) {
		char *argv[9] = {"xmessage", "-buttons", "Ok", "-title", title,
			"-center", "-file", "-", NULL};
		close(0);
		dup(lead[0]);
		close(lead[0]);
		close(lead[1]);
		execv(XMESSAGE, argv);
		exit(127);
	}

	return -1;
}


/*
 * Call the TuxTime configuration manager
 */
int CallConfigApp(char *settings)
{
	if (!fork()) {
		char *argv[3] = {"tuxtime-conf", settings, NULL};

		execv(BINDIR"/tuxtime-conf", argv);
		exit(127);
	}

	return -1;
}


/*
 * Trigger a software suspend sequence using the HCI
 */
void SoftwareSuspend(void)
{
	SMMRegisters reg;

	reg.eax = HCI_SET;
	reg.ebx = HCI_SOFTWARE_SUSPEND;
	reg.ecx = HCI_ENABLE;

	HciFunction(&reg);

	return;
}


/*
 * Handle button presses
 */
void PressEvent(XButtonEvent *event)
{	
	int x,y;
	
	x = event->x;
	y = event->y;

	if(x>=6 && y>=48 && x<=18 && y<=58) {
		/* next arrow button */

		if ((selectbay!=EMPTY) && (power==SCI_BATTERY)) {
		 	copyXPMArea(66, 44, 13, 11, 6, 48);
			switch (selectbay) {
				case PRIMARY:
					copyXPMArea(79, 93, 15, 10, 44, 33);
					selectbay = AUXILLARY;
					break;
			 	case AUXILLARY:
					copyXPMArea(96, 93, 15, 10, 44, 33);
					selectbay = BOTH;
					break;
				case BOTH:
					copyXPMArea(62, 93, 15, 10, 44, 33);
					selectbay = PRIMARY;
					break;
			}
		}
	} else if (x>=19 && y>=48 && x<=31 && y<=58){
		/* suspend call */

		copyXPMArea(79, 44, 13, 11, 19, 48);
		SoftwareSuspend();
	} else if (x>=32 && y>=48 && x<=44 && y<=58){
		/* alarm settings */

		copyXPMArea(92, 44, 13, 11, 32, 48);
		CallConfigApp("--alarms");
	} else if (x>=45 && y>=48 && x<=57 && y<=58){
		/* machine settings */

		copyXPMArea(105, 44, 13, 11, 45, 48);
		CallConfigApp("--settings");
	}

	RedrawWindow();

	return;
}


/*
 * Repaint the buttons in the non-depressed state on release
 */
void ReleaseEvent(XButtonEvent *event)
{
	copyXPMArea(66, 56, 52, 11, 6, 48);
	RedrawWindow();

	return;
}


/*
 * Convert a single character to an interger, similar to atoi 
 */
inline int ctoi(char *s)
{
	return ((*s>='0') && (*s<='9')) ? *s-'0' : -1;
}


/*
 * Convert the time string to hours and minutes in the 24 hour format
 */
int ConvertTime(char *time, int *hour, int *minute)
{
	/* we can't rely on short circuit boolean evaluation */

	if (*time=='\0')
		return 0;
	if (*(time+1)=='\0')
		return 0;
	if (*(time+2)=='\0')
		return 0;

	/* have we a dropped leading zero? */

	if (*(time+1)==':') {
		*hour = ctoi(time);
		time+=2;
	}
	else if (*(time+2)!=':')
		return 0;
	else {
		*hour = (ctoi(time+1))+(10*ctoi(time));
		time+=3;
	}

	if (*time=='\0')
		return 0;
	if (*(time+1)=='\0')
		return 0;

	*minute = ctoi(time+1)+(10*ctoi(time));

	/* deal with any am or pm switch */

	if (*(time+2)!='\0') {
		if ((*(time+2)=='p') || (*(time+2)=='P'))
			*hour+=12;
		else if ((*(time+2)!='h') && (*(time+2)!='H') &&
			(*(time+2)!='a') && (*(time+2)!='A'))
			return 0;
	}

	/* check to see if the converted time is valid */

	if ((*hour>=0) && (*hour<24) && (*minute>=0) && (*minute<60))
		return 1;
	else
		return 0;
}


/*
 * Create the default .tuxtimerc file at the given location
 */
void CreateDefaultRC(char *file)
{
	WriteConfigString("TuxTime", "Left", "300", file);
	WriteConfigString("TuxTime", "Top", "200", file);
	WriteConfigString("TuxTime", "Title", "1", file);
	WriteConfigString("TuxTime", "Iconic", "0", file);
	WriteConfigString("TuxTime", "AlarmPowerOn", "12:00H", file);
	WriteConfigString("TuxTime", "PollInterval", "10", file);
	WriteConfigString("TuxTime", "AutoSetCharge", "1", file);

	WriteConfigString("Display", "Size", "1", file);
	WriteConfigString("Display", "Modules", "1", file);
	WriteConfigString("Display", "Style", "0", file);
	WriteConfigString("Display", "Icon", "1", file);
	WriteConfigString("Display", "Cycle", "0", file);
	WriteConfigString("Display", "CycleButton", "0", file);
	WriteConfigString("Display", "CycleInterval", "15", file);
	WriteConfigString("Display", "CycleCrnt", "1", file);
	WriteConfigString("Display", "Title", "Justify", file);
	WriteConfigString("Display", "RightOffset", "0", file);
	WriteConfigString("Display", "LeftOffset", "0", file);
	WriteConfigString("Display", "ShowAllBatteries", "0", file);

	WriteConfigString("BatPowerSettings", "HDDAutoOff", "8", file);
	WriteConfigString("BatPowerSettings", "DisplayAutoOff", "8", file);
	WriteConfigString("BatPowerSettings", "SleepMode", "1", file);
	WriteConfigString("BatPowerSettings", "CPUSpeed", "0", file);
	WriteConfigString("BatPowerSettings", "LCDBrightness", "1", file);
	WriteConfigString("BatPowerSettings", "BatterySaveMode", "2", file);
	WriteConfigString("BatPowerSettings", "SystemAutoOff", "512", file);
	WriteConfigString("BatPowerSettings", "CoolingMethod", "1", file);

	WriteConfigString("ExtPowerSettings", "HDDAutoOff", "8", file);
	WriteConfigString("ExtPowerSettings", "DisplayAutoOff", "8", file);
	WriteConfigString("ExtPowerSettings", "SleepMode", "1", file);
	WriteConfigString("ExtPowerSettings", "CPUSpeed", "0", file);
	WriteConfigString("ExtPowerSettings", "LCDBrightness", "2", file);
	WriteConfigString("ExtPowerSettings", "BatterySaveMode", "0", file);
	WriteConfigString("ExtPowerSettings", "SystemAutoOff", "512", file);
	WriteConfigString("ExtPowerSettings", "CoolingMethod", "1", file);

	WriteConfigString("AlarmLow", "Percent", "20", file);
	WriteConfigString("AlarmLow", "Time", "00:20", file);
	WriteConfigString("AlarmLow", "Trigger", "1", file);

	WriteConfigString("AlarmCritical", "Percent", "10", file);
	WriteConfigString("AlarmCritical", "Time", "00:10", file);
	WriteConfigString("AlarmCritical", "Trigger", "1", file);

	return;
}


/*
 * Setup the alarm values by reading them in from the config file
 */
void SetupAlarms(char *config)
{
	char buffer[8];

	low.percent = GetConfigInt("AlarmLow", "Percent", 20, config);
	low.trigger = GetConfigInt("AlarmLow", "Trigger", 3, config);
	GetConfigString("AlarmLow", "Time", "00:20", buffer, 8, config);
	ConvertTime(buffer, &(low.hour), &(low.minute));
	critical.percent = GetConfigInt("AlarmCritical", "Percent", 10, config);
	critical.trigger = GetConfigInt("AlarmCritical", "Trigger", 3, config);
	GetConfigString("AlarmCritical", "Time", "00:10", buffer, 8, config);
	ConvertTime(buffer, &(critical.hour), &(critical.minute));

	return;
}


/*
 * Test to see what condition the battery is in (i.e. normal, low, critial)
 */
int BatteryTest(BatteryState *settings, int percent, int time)
{
	int flag;

	flag = 0;
	if ((settings->trigger==TIME) || (settings->trigger==BOTH))
		if (time<=MINUTES(settings->hour, settings->minute))
			flag |= TIME;
			
	if ((settings->trigger==PERCENT) || (settings->trigger==BOTH))
		if (percent<=settings->percent)
			flag |= PERCENT;

	return flag;
}


/*
 * Change the power saving settings to reflect the power source
 */
void PowerChanged(int source, char *file)
{
	unsigned short save,hdd,display,sleep,speed;
	unsigned short lcd,autooff,cooling,standby;
	unsigned short lcdtype;
	SMMRegisters reg;
	int user;
	char *section;

	/* read the settings from the config file */

        if (source==SCI_MAINS)
                section = "ExtPowerSettings";
        else
                section = "BatPowerSettings";

        /* get the settings from the configuration file */

        save = GetConfigInt(section, "BatterySaveMode", 2, file);
	hdd = GetConfigInt(section, "HDDAutoOff", 512, file);
	display = GetConfigInt(section, "DisplayAutoOff", 1, file);
	sleep = GetConfigInt(section, "SleepMode", 0, file);
	speed = GetConfigInt(section, "CPUSpeed", 1, file);
	lcd = GetConfigInt(section, "LCDBrightness", 0, file);
	autooff = GetConfigInt(section, "SystemAutoOff", 1, file);
	cooling = GetConfigInt(section, "CoolingMethod", 0, file);
	standby = GetConfigInt(section, "StandbyTime", 0, file);

	SciOpenInterface();

	/* try and set the battery save mode */

	reg.ebx = SCI_BATTERY_SAVE;
	reg.ecx = save;
	if (SciSet(&reg)==SCI_SUCCESS) {
		user = 0;
	} else {
		user = 1;
	}

	/* determine whether to make LCD backlight or brightness call */

	reg.ebx = SCI_LCD_BRIGHTNESS;
	if (SciGet(&reg)==SCI_SUCCESS) {
		lcdtype = SCI_LCD_BRIGHTNESS;
	} else {
		lcdtype = SCI_LCD_BACKLIGHT;
	}
	reg.ebx = SCI_LCD_MAXBRIGHT;
	if (SciGet(&reg)==SCI_SUCCESS) {
		lcdtype = SCI_LCD_MAXBRIGHT;
		if (lcd>1)
			lcd = 1;
	}

	/* set the other settings if user mode or new style machine */

	if ((save==SCI_USER_SETTINGS) || (user==1)) {
		reg.ebx = SCI_HDD_AUTO_OFF;
		reg.ecx = hdd;
		if (SciSet(&reg)==SCI_FAILURE) {
			fprintf(stderr, "wmtuxtime: unable to set HDD "
				"auto off\n");
		}
		reg.ebx = SCI_DISPLAY_AUTO;
		reg.ecx = display;
		if (SciSet(&reg)==SCI_FAILURE) {
			fprintf(stderr, "wmtuxtime: unable to set display "
				"auto off\n");
		}
		reg.ebx = SCI_SLEEP_MODE;
		reg.ecx = sleep;
		if (SciSet(&reg)==SCI_FAILURE) {
			fprintf(stderr, "wmtuxtime: unable to set sleep "
				"mode\n");
		}
		reg.ebx = SCI_PROCESSING;
		reg.ecx = speed;
		if (SciSet(&reg)==SCI_FAILURE) {
			fprintf(stderr, "wmtuxtime: unable to set CPU "
				"speed\n");
		}
		reg.ebx = lcdtype;
		reg.ecx = lcd;
		if (SciSet(&reg)==SCI_FAILURE) {
			fprintf(stderr, "wmtuxtime: unable to set LCD "
				"brightness\n");
		}
		reg.ebx = SCI_SYSTEM_AUTO;
		reg.ecx = autooff;
		if (SciSet(&reg)==SCI_FAILURE) {
			fprintf(stderr, "wmtuxtime: unable to set system "
				"auto off\n");
		}
		reg.ebx = SCI_COOLING_METHOD;
		reg.ecx = cooling;
		if (SciSet(&reg)==SCI_FAILURE) {
			fprintf(stderr, "wmtuxtime: unable to set cooling "
				"method\n");
		}

		/* toggle battery save mode to ensure settings take effect */

		if (user==0) {
			reg.ebx = SCI_BATTERY_SAVE;
			reg.ecx = 1;
			if (SciSet(&reg)==SCI_FAILURE) {
				fprintf(stderr, "wmtuxtime: unable to set "
					"battery save mode (1)\n");
			}

			reg.ebx = SCI_BATTERY_SAVE;
			reg.ecx = save;
			if (SciSet(&reg)==SCI_FAILURE) {
				fprintf(stderr, "wmtuxtime: unable to set "
					"battery save mode (save)\n");
			}
		}
	}

	SciCloseInterface();

	return;
}


/*
 * Display the battery alarm messages using xmessage or if avaliable
 * wmessage
 */
void DisplayAlarm(int state, int percent, int time)
{
	int i,trigger,hour,minute;
 	char message[128];


	/* Don't bother with the alarms if we are on mains power */

	i = SciACPower();
	if ((i==SCI_MAINS) && (i!=0))
		return;

	/* send out the emergency alarm first */

	if (state==EMERGENCY) {
		XBell(display, 100);
		sprintf(message, "%s", EMERGENCY_ALERT);
		DisplayMessage(message, "Emergency Battery Alert");
		return;
	}

	hour = SCI_HOUR(time);
	minute = SCI_MINUTE(time);
	trigger = NEVER;

	/* determine what triggered the alarm and from the correct message */

	if (state==LOW)
		trigger = BatteryTest(&low, percent, MINUTES(hour, minute));
	else if (state==CRITICAL)
		trigger = BatteryTest(&critical, percent, MINUTES(hour, minute));

	/* form the correct message */

	if (trigger & TIME)
		sprintf(message, TIME_ALERT, MINUTES(hour, minute));
	else if (trigger & PERCENT)
		sprintf(message, PERCENT_ALERT, percent);
	
	/* display the message */

	if (trigger!=NEVER) {
		XBell(display, 100);
		if (state==LOW)
			DisplayMessage(message, "Low Battery Alarm");
		else if (state==CRITICAL)
			DisplayMessage(message, "Critical Battery Alarm");
	}

	return;
}


/*
 * Get the power status i.e. percent, time remaining, battery charging etc.
 */
int PowerStatus(int *charging, int *time, int *percent, int *second)
{
	FILE *str;
	SMMRegisters reg;
	int hours,minutes;
	char version[10],buffer[64];


	if (access("/proc/apm", R_OK)) {
		/* cannot find /proc/apm */
		fprintf(stderr, "wmtuxtime: unable to access /proc/apm\n");
		return 0;
	} else {

		/* open /proc/apm for reading */
		if (!(str = fopen("/proc/apm", "r"))) {
			fprintf(stderr, "wmtuxtime: unable to open /proc/apm\n");
			return 0;
		}

		/* scan in the information */
		fgets(buffer, sizeof(buffer)-1, str);
		fclose(str);
		buffer[sizeof(buffer)-1] = '\0';
		sscanf(buffer, "%s %*d.%*d %*x %*x %x %*x %d%% %d %*s\n",
			version, charging, percent, time);

		/* if old style driver return */
		if (version[0]=='B') {
			fprintf(stderr, "wmtuxtime: old style apm driver\n");
			return 0;
		}
	}

	/* get the percent and time remaining using SCI method */

	if (SciOpenInterface()==SCI_SUCCESS) {
		reg.ebx = SCI_BATTERY_PERCENT;
		if (SciGet(&reg)==SCI_SUCCESS)
			*percent = ((100*(reg.ecx & 0xffff))/(reg.edx & 0xffff));
		else
			*percent = -1;

		/* some times percent fails so catch to prevent display flickering */

		if ((*percent==-1) && (oldpercent!=-1)) {
			*percent = oldpercent;
			oldpercent = -1;
		} else {
			oldpercent = *percent;
		}

		reg.ebx = SCI_BATTERY_TIME;
		if (SciGet(&reg)==SCI_SUCCESS) {
			*time = (reg.ecx & 0xffff);
		} else {
			/* fall back to using the /proc/apm time data */
			hours = *time/60;
			minutes = *time-(60*hours);
			*time = SCI_TIME(hours, minutes);
		}

		/* check to see if their is a battery in the SelectBay */

		reg.ebx = SCI_2ND_BATTERY;
		if (SciGet(&reg)==SCI_SUCCESS) {
			if ((reg.ecx & 0xffff)!=0xffff)
				*second = ((100*(reg.ecx & 0xffff))/(reg.edx & 0xffff));
			else
				*second = -1;
		} else {
			selectbay = EMPTY;
			*second = -1;
		}

		SciCloseInterface();
	} else {
		return 0;
	}

	return 1;
}


/*
 * Update the battery life remaining display, and switch settings if the
 * power source has changed
 */
void Update(void)
{
	int time,percent,second,charging;
	int hour,minute,state,battery,i;


	/* get the power status of the laptop */

	if (PowerStatus(&charging, &time, &percent, &second)==0)
		return;

	/* activate button if battery in the SelectBay */

	if (second>=0) {
		if ((power==SCI_BATTERY) && (selectbay==EMPTY))
			copyXPMArea(62, 93, 15, 10, 44, 33);
		if (selectbay==EMPTY) {
			selectbay = PRIMARY;
			/* activate the arrow button */
			copyXPMArea(119, 25, 4, 7,  11, 50);
			copyXPMArea(119, 25, 4, 7,  71, 58);
			copyXPMArea(119, 25, 4, 7,  71, 46);
		}
	} else {
		selectbay = EMPTY;
		/* grey out the arrow button */
		copyXPMArea(119, 35, 4, 7,  11, 50);
		copyXPMArea(119, 35, 4, 7,  71, 58);
		copyXPMArea(119, 35, 4, 7,  71, 46);
	}

	/* calculate the battery condition */

	state = NORMAL;
	hour = SCI_HOUR(time);
	minute = SCI_MINUTE(time);
	
	/* make sure the times are sensible */
	if (hour<0) {
		hour = 0;
		time = -1;
	}

	if ((minute<0) || (minute>59)) {
		minute = 0;
		time = -1;
	}

	
	if ((percent>=0) && (time>=0)) {
		if ((percent<3) && (percent>=0))
			state = EMERGENCY;
		else if (BatteryTest(&critical, percent, MINUTES(hour, minute))!=0)
			state = CRITICAL;
		else if (BatteryTest(&low, percent, MINUTES(hour, minute))!=0)
			state = LOW;
	}

	/* send out the battery alarms */

	if ((state==EMERGENCY) && ((alarms & EMERGENCY)==0)) {
		alarms |= EMERGENCY;
		DisplayAlarm(state, percent, time);
	}
	if ((state==LOW) && ((alarms & LOW)==0)) {
		alarms |= LOW;
		DisplayAlarm(state, percent, time);
	}
	if ((state==CRITICAL) && ((alarms & CRITICAL)==0)) {
		alarms |= CRITICAL;
		DisplayAlarm(state, percent, time);
	}

	/* turn off alarm states as necessary */

	if (state==NORMAL)
		alarms = NORMAL;
	else if (state==LOW)
		alarms = NORMAL | LOW;
	else if (state==CRITICAL)
		alarms = NORMAL | LOW | CRITICAL;

	/* check AC status. */

	i = SciACPower();
	if ((i!=power) && (i!=0)) {
		if (power==SCI_MAINS) {
			power = SCI_BATTERY;
			if (selectbay==EMPTY) {
				copyXPMArea(45, 93, 15, 10, 44, 33);
			} else {
				selectbay = PRIMARY;
				copyXPMArea(62, 93, 15, 10, 44, 33);
			}
			PowerChanged(power, config);
		} else {
			power = SCI_MAINS;
			if (selectbay!=EMPTY)
				selectbay = BOTH;
			copyXPMArea(28, 93, 15, 10, 44, 33);
			PowerChanged(power, config);
		}
	}

	/* paste up the default charge status and time */

	copyXPMArea(119, 15, 5, 7, 6, 7);
	copyXPMArea(83, 81, 41, 9, 15, 7);

	/* check to see if the batteries are charging */

	if (charging==3) {
		copyXPMArea(119, 5, 5, 7,  6, 7);
		copyXPMArea(83, 69, 5, 9, 15, 7);
	}

	/* paste up the "Time Left", this time means:
	     If not charging: Time left before battery drains to 0% 
	     If charging:     Time left before battery gets to maximum */

	if (time<0) {
		/* dodgy time so just display a "null" indicator (--:--) */

		copyXPMArea(83, 69, 41, 9, 15, 7);
	} else {
		copyXPMArea((hour/10)*7+5, 81, 7, 9, 21, 7);
		copyXPMArea((hour%10)*7+5, 81, 7, 9, 29, 7);
		copyXPMArea(76, 81, 2, 9, 38, 7);
		copyXPMArea((minute/10)*7+5, 81, 7, 9, 42, 7);
		copyXPMArea((minute%10)*7+5, 81, 7, 9, 50, 7);
	}
	
	/*  do battery percentage */

	switch (selectbay) {
		case EMPTY: case PRIMARY:
			battery = percent;
			break;
		case AUXILLARY:
			battery = second;
			break;
		case BOTH:
			battery = (percent+second)/2;
			break;
		default:
			battery = percent;
			break;
	}

	/* make sure the battery percentage is something sensible */

	if (battery<0)
		battery = 0;
	if (battery>100)
		battery = 100;

	copyXPMArea(5, 93, 19, 7, 7, 34);   /* Show Default % */
	copyXPMArea(66, 4, 49, 9, 7, 21);    /* Show Default Meter */
	if (battery==100) {
		copyXPMArea(15, 71, 1, 7,  7, 34);    /* If 100%, show 100% */
		copyXPMArea(5, 71, 6, 7,  9, 34);
		copyXPMArea(5, 71, 6, 7, 15, 34);
		copyXPMArea(65, 71, 7, 7, 21, 34);    /* Show '%' */
		copyXPMArea(66, 14, 49, 9, 7, 21);    /* Show Meter */
	} else if (battery>=0) {
		if (battery>=10)
			copyXPMArea((battery/10)*6+4, 71, 6, 7, 9, 34);
		copyXPMArea((battery%10)*6+4, 71, 6, 7, 15, 34);
		copyXPMArea(65, 71, 7, 7, 22, 34);    /* Show '%' */

		/* Show Meter */

		switch (state) {
			case NORMAL:
				i = 14;
				break;
			case LOW:
				i = 24;
				break;
			case CRITICAL:
				i = 34;
				break;
			default:
				i = 14;
		}

		copyXPMArea(66, i, (battery*49)/100, 9, 7, 21);
	}
		
	return;
}


/*
 * Determine if configuration information has changed and needs to be re-read
 */
void HandleDeadChild(pid_t pid, int status)
{
	if (WIFEXITED(status)) {
		if (WEXITSTATUS(status)==0x10) {
			/* reread the alarm settings */
			SetupAlarms(config);
			Update();
		}	
		if (WEXITSTATUS(status)==0x20) {
			/* set the battery save options */
			PowerChanged(power, config);
		}
	}

	return;
}


/*
 * Called from the main loop, it's job is to find dead children.
 */
void ShovelDeadChildren()
{
	pid_t childpid;
	int status;

	while ((childpid = waitpid(-1, &status, WNOHANG))>0) {
		HandleDeadChild(childpid, status);
 	}

	return;
}


/*
 * the entry point of wmTuxTime
 */
int main(int argc, char *argv[])
{

	int pid,version,bios,poll,i;
	XEvent event;
	FILE *str;
	struct passwd *pw;
	struct stat info;


	/* do some quick checks on the laptop */

	if (SciSupportCheck(&version)==1) {
		fprintf(stderr, "wmtuxtime: this computer is not supported "
			"or the kernel module not installed.\n");
		return 1;
	}

	bios = HciGetBiosVersion();
	if (bios==0) {
		fprintf(stderr, "wmtuxtime: unable to get BIOS version\n");
		return 1;
	}

	if (HciGetMachineID(&id)==HCI_FAILURE) {
		fprintf(stderr, "wmtuxtime: unable to get machine "
			"identification\n");
		return 1;
	}

        /* drop root priveleges to minimize the risk of running suid root */

        seteuid(getuid());
        setegid(getgid());

	switch (id) {
		case 0xfc00: /* Satellite 2140CDS/2180CDT/2675DVD */
		case 0xfc01: /* Satellite 2710xDVD */
		case 0xfc02: /* Satellite Pro 4270CDT//4280CDT/4300CDT/4340CDT */
		case 0xfc04: /* Portege 3410CT, 3440CT */
		case 0xfc08: /* Satellite 2100CDS/CDT 1550CDS */
		case 0xfc09: /* Satellite 2610CDT, 2650XDVD */
		case 0xfc0a: /* Portage 7140 */
		case 0xfc0b: /* Satellite Pro 4200 */
		case 0xfc0c: /* Tecra 8100x */
		case 0xfc0f: /* Satellite 2060CDS/CDT */
		case 0xfc10: /* Satellite 2550/2590 */
		case 0xfc11: /* Portage 3110CT */
		case 0xfc12: /* Portage 3300CT */
		case 0xfc13: /* Portage 7020CT */
		case 0xfc15: /* Satellite 4030/4030X/4050/4060/4070/4080/4090/4100X CDS/CDT */
		case 0xfc17: /* Satellite 2520/2540 CDS/CDT */
		case 0xfc18: /* Satellite 4000/4010 XCDT */
		case 0xfc19: /* Satellite 4000/4010/4020 CDS/CDT */
		case 0xfc1a: /* Tecra 8000x */
		case 0xfc1c: /* Satellite 2510CDS/CDT */
		case 0xfc1d: /* Portage 3020x */
		case 0xfc1f: /* Portage 7000CT/7010CT */
		case 0xfc39: /* T2200SX */
		case 0xfc40: /* T4500C */
		case 0xfc41: /* T4500 */
		case 0xfc45: /* T4400SX/SXC */
		case 0xfc51: /* Satellite 2210CDT, 2770XDVD */
		case 0xfc52: /* Satellite 2775DVD, Dynabook Satellite DB60P/4DA */
		case 0xfc53: /* Portage 7200CT/7220CT, Satellite 4000CDT */
		case 0xfc54: /* Satellite 2800DVD */
		case 0xfc56: /* Portage 3480CT */
		case 0xfc57: /* Satellite 2250CDT*/
		case 0xfc5a: /* Satellite Pro 4600 */
		case 0xfc5d: /* Satellite 2805 */
		case 0xfc5f: /* T3300SL */
		case 0xfc61: /* Tecra 8200 */
		case 0xfc64: /* Satellite 1800 */
		case 0xfc69: /* T1900C */
		case 0xfc6a: /* T1900 */
		case 0xfc6d: /* T1850C */
		case 0xfc6e: /* T1850 */
		case 0xfc6f: /* T1800 */
		case 0xfc72: /* Satellite 1800 */
		case 0xfc7e: /* T4600C */
		case 0xfc7f: /* T4600 */
		case 0xfc8a: /* T6600C */
		case 0xfc91: /* T2400CT */
		case 0xfc97: /* T4800CT */
		case 0xfc99: /* T4700CS */
		case 0xfc9b: /* T4700CT */
		case 0xfc9d: /* T1950 */
		case 0xfc9e: /* T3400/T3400CT */
		case 0xfcb2: /* Libretto 30CT */
		case 0xfcba: /* T2150 */
		case 0xfcbe: /* T4850CT */
		case 0xfcc0: /* Satellite Pro 420x */
		case 0xfcc1: /* Satellite 100x */
		case 0xfcc3: /* Tecra 710x/720x */
		case 0xfcc6: /* Satellite Pro 410x */
		case 0xfcca: /* Satellite Pro 400x */
		case 0xfccb: /* Portage 610CT */
		case 0xfccc: /* Tecra 700x */
		case 0xfccf: /* T4900CT */
		case 0xfcd0: /* Satellite 300x */
		case 0xfcd1: /* Tecra 750CDT */
		case 0xfcd2: /* Vision Connect -- what is this??? */
		case 0xfcd3: /* Tecra 730XCDT*/
		case 0xfcd4: /* Tecra 510x */
		case 0xfcd5: /* Satellite 200x */
		case 0xfcd7: /* Satellite Pro 430x */
		case 0xfcd8: /* Tecra 740x */
		case 0xfcd9: /* Portage 660CDT */
		case 0xfcda: /* Tecra 730CDT */
		case 0xfcdb: /* Portage 620CT */
		case 0xfcdc: /* Portage 650CT */
		case 0xfcdd: /* Satellite 110x */
		case 0xfcdf: /* Tecra 500x */
		case 0xfce0: /* Tecra 780DVD */
		case 0xfce2: /* Satellite 300x */
		case 0xfce3: /* Satellite 310x */
		case 0xfce4: /* Satellite Pro 490x */
		case 0xfce5: /* Libretto 100CT */
		case 0xfce6: /* Libretto 70CT */
		case 0xfce7: /* Tecra 540x/550x */
		case 0xfce8: /* Satellite Pro 470x/480x */
		case 0xfce9: /* Tecra 750DVD */
		case 0xfcea: /* Libretto 60 */
		case 0xfceb: /* Libretto 50CT */
		case 0xfcec: /* Satellite 320x/330x, Satellite 2500CDS */
		case 0xfced: /* Tecra 520x/530x */
		case 0xfcef: /* Satellite 220x, Satellite Pro 440x/460x */
			printf("wmtuxtime: machine id: 0x%04x    BIOS version:"
				" %d.%d    SCI version: %d.%d\n", id,
				(bios & 0xff00)>>8, bios & 0xff,
				(version & 0xff00)>>8, version & 0xff);
			break;
		default:
			printf("wmtuxtime: unrecognized machine "
				"identification:\n");
			printf("machine id : 0x%04x    BIOS version : %d.%d"
				"    SCI version: %d.%d\n", id,
				(bios & 0xff00)>>8, bios & 0xff,
				(version & 0xff00)>>8, version & 0xff);
			printf("\nplease email this information to "
				"jonathan@buzzard.org.uk and include the "
				"model\nand model number, eg. Libretto 50CT "
				" model number PA1249E\n");
	}

	/* check to see if a copy of wmTuxTime is already running */

	if (!access(PID_FILE, R_OK)) {
		if ((str = fopen(PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);

			/* test to see if the other TuxTime died unexpectedly */

			if (kill(pid, SIGUSR1)==0) {
				fprintf(stderr, "wmtuxtime: already running as "
					"process %d.\n", pid);
				exit(1);
			}

			fprintf(stderr, "wmtuxtime: process %d appears to have "
				"died, continuing\n", pid);
			unlink(PID_FILE);
		}
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}

	/* get the path of the tuxtimerc file */

	pw = getpwuid(getuid());
	config = (char *) malloc(strlen(pw->pw_dir)+12);
	if (config==NULL) {
		fprintf(stderr, "wmtuxtime: unable to allocate sufficent "
			"memory, exiting\n");
		exit(1);
		}
	strcpy(config, pw->pw_dir);
	strcat(config, "/.tuxtimerc");

	/* test to see if tuxtimerc file exists and if not create one */

	if ((stat(config, &info)!=0) && (errno==ENOENT)) {
		CreateDefaultRC(config);
	}

	HandleSignals();
	ProcessComandLine(&argc, &argv);

	/* read the config file and initilize state */

	SetupAlarms(config);
	alarms = NORMAL;
	if (SciACPower()==SCI_MAINS)
		power = SCI_BATTERY;
	else
		power = SCI_MAINS;
	poll = GetConfigInt("TuxTime", "PollInterval", 10, config);


	/* open the X11 windows */

	openXwindow(argc, argv, wmtuxtime_xpm, wmtuxtime_mask_bits,
		wmtuxtime_mask_width, wmtuxtime_mask_height);

	/* update immediately to make sure it's right */

	Update();
	RedrawWindow();

	/* loop until we die... */

	i = 0;
	for (;;) {

		/*  only process battery state every poll cycles of this loop.
		    We run it faster to catch the expose events, etc... */ 

		if (i++>poll) {
			Update();
			RedrawWindow();
			i = 0;

			/* handle dead child processes */

			ShovelDeadChildren();
		}

		/* process any pending X events */

		while (XPending(display)) {
			XNextEvent(display, &event);
			switch(event.type){
				case Expose:
					RedrawWindow();
					break;
				case ButtonPress:
					PressEvent(&event.xbutton);
					break;
				case ButtonRelease:
					ReleaseEvent(&event.xbutton);
                        		break;
			}
		}


		/* wait for next update  */

		usleep(DELAY);
	}

	return 0;
}
