-- Topal: GPG/GnuPG and Alpine/Pine integration
-- Copyright (C) 2001--2008  Phillip J. Brooke
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License version 3 as
-- published by the Free Software Foundation.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

with Ada.Command_Line;
with Ada.Text_IO;
with Attachments;
with Configuration;
with Externals;             use Externals;
with Externals.GPG;
with Externals.Mail;
with Externals.Simple;      use Externals.Simple;
with Keys;
with Menus;                 use Menus;
with Misc;                  use Misc;
with Readline;
with Remote_Mode;

package body Sending is

   Exit_Send_Loop : exception; -- Subroutine says `bail out, please'.

   procedure Key_Preselect (K          : in out Keys.Key_List;
                            Recipients : in     UBS_Array;
                            Alt_Sign   : in     String       := "") is
   begin
      -- Can we preselect any keys?
      Keys.Empty_Keylist(K);
      Keys.Use_Keylist(Recipients, K);
      -- Add our own key.  If Alt_Sign is non-empty, then try and add
      --  keys for that instead.
      declare
         AKL     : Keys.Key_List;
         Found   : Boolean       := False;
         KC      : Natural;
         The_Key : UBS;
      begin
         if Alt_Sign /= "" then
            Ada.Text_IO.Put_Line("Looking for signing key for `"&Alt_Sign&"' on SAKE list...");
            for J in 1..Config.SAKE_Count loop
               if ToStr(UAP.Value(Config.SAKE_Email, J)) = Alt_Sign then
                  Keys.Add_Keys_By_Fingerprint(UAP.Value(Config.SAKE_Key, J),
                                               K, Keys.Both, Found);
                  if Found then
                     Ada.Text_IO.Put_Line("Added key `"&ToStr(UAP.Value(Config.SAKE_Key, J))&"' for signing...");
                     Config.My_Key := UAP.Value(Config.SAKE_Key, J);
                  else
                     Ada.Text_IO.Put_Line("Couldn't add key `"&ToStr(UAP.Value(Config.SAKE_Key, J))&"' for signing...");
                  end if;
               end if;
            end loop;
         end if;
         if Alt_Sign /= "" and (not Found) then
            Keys.Empty_Keylist(AKL);
            Ada.Text_IO.Put_Line("Looking for signing key for `"&Alt_Sign&"' via keyring (& XK list)...");
            Keys.Add_Secret_Keys(ToUBS(Alt_Sign),
                                 AKL,
                                 Keys.Both);
            -- Any keys to remove on the SXK list?
            for J in 1..Config.SXK_Count loop
               Keys.Remove_Key(UAP.Value(Config.SXK_Key, J), AKL);
            end loop;
            KC := Keys.Count(AKL);
            Found := KC > 0;
            if Found then
               if KC = 1 then
                  Ada.Text_IO.Put_Line("Added one signing key...");
                  -- Add this one.
                  Keys.First_Key_From_List(AKL, The_Key);
                  Keys.Add_Key(The_Key, K, Keys.Both);
                  Config.My_Key := The_Key;
               else
                  -- Multiple keys...
                  Ada.Text_IO.Put_Line("Found multiple possible signing keys...");
                  -- Choose one.  If aborted, fall-back to my-key.
                  Keys.Select_Key_From_List(AKL, The_Key, Found);
                  -- Found is not aborted...
                  Found := not Found;
                  if Found then
                     Keys.Add_Key(The_Key, K, Keys.Both);
                     Config.My_Key := The_Key;
                  else
                     Ada.Text_IO.Put_Line("Aborted selection of secret key, will consider my-key instead");
                  end if;
               end if;
            else
               Ada.Text_IO.Put_Line("Can't find alternative key, trying my-key...");
            end if;
         end if;
         if Alt_Sign = "" or (not Found) then
            if ToStr(Config.My_Key) = "" then
               Ada.Text_IO.Put_Line("my-key is empty; not selecting any keys for self");
            else
               Keys.Add_Keys_By_Fingerprint(Value_Nonempty(Config.My_Key),
                                            K,
                                            Keys.Encrypter,
                                            Found);
               if Found then
                  Ada.Text_IO.Put_Line("Added `my-key' key(s) for self");
               else
                  Ada.Text_IO.Put_Line("Warning: my-key="
                                       & ToStr(Config.My_key)
                                       & " does not find keys");
               end if;
            end if;
         end if;
      end;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Sending.Key_Preselect");
         raise;
   end Key_Preselect;

   procedure Check_Send (Tmpfile   : in String;
                         Non_Pine  : in Boolean;
                         Mime      : in Boolean;
                         Mimefile  : in String) is
   begin
      if not Non_Pine then
         if Mime then
            if Externals.Simple.Test_R(Mimefile) then
               Ada.Text_IO.Put_Line("The Content-Type being returned is: ");
               Cat(Mimefile);
            else
               Ada.Text_IO.Put_Line("Unchanged Content-Type.");
            end if;
         end if;
         Wait;
         Pager(Tmpfile);
         Ada.Text_IO.New_Line(5);
         if YN_Menu("Okay to send? ") = No then
            Ada.Command_Line.Set_Exit_Status(Ada.Command_Line.Failure);
         end if;
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Sending.Check_Send");
         raise;
   end Check_Send;

   procedure Encrypt (Tmpfile   : in String;
                      Non_Pine  : in Boolean;
                      Mime      : in Boolean;
                      Mimefile  : in String;
                      Send_Keys : in Keys.Key_List;
                      Selection : in Send_Index;
                      AL        : in Attachments.Attachment_List) is
      Out_File       : constant String := Temp_File_Name("out");
      -- PCT = Prepend content-type.
      Mime_Selection : MIME_Send_Index;
   begin
      if Mime then
         begin
            Ada.Text_IO.New_Line(3);
            Mime_Selection := MIME_Send_Menu;
            if Mime_Selection = AAbort then
               Ada.Command_Line.Set_Exit_Status(Ada.Command_Line.Failure);
               raise Exit_Send_Loop;
            elsif Mime_Selection = Multipart then
               declare
                  PCT : constant String := Temp_File_Name("pct");
               begin
                  Echo_Out("Content-Type: text/plain", PCT);
                  Echo_Append("", PCT);
                  Cat_Append(Tmpfile, PCT);
                  Mv_F(PCT, Tmpfile);
                  -- Call out to attachments in case we have to modify Tmpfile.
                  Attachments.Replace_Tmpfile(Tmpfile, AL);
               end;
            end if;
         exception
            when others =>
               Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                    "Exception raised in Sending.Encrypt (MIME block 1)");
               raise;
         end;
      end if;

      if not (Mime and then Mime_Selection = Multipart) then
         if Attachments.Count(AL) > 0 then
            Ada.Text_IO.Put_Line("WARNING: attachments not included in non-MIME emails!");
            Ada.Text_IO.New_Line;
         end if;
      end if;

      begin
         -- Run GPG.
         Externals.GPG.GPG_Wrap(ToStr(Config.Gpg_Options)
                      & " --encrypt --armor "
                      & ToStr(Config.general_options)
                      & " "
                      & ToStr(Config.sending_options)
                      & " "
                      & " --output "
                      & Out_File
                      & " "
                      & Keys.Processed_Recipient_List(Send_Keys)
                      & " "
                      & Tmpfile,
                      Out_File);
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Sending.Encrypt (GPG block)");
            raise;
      end;

      begin
         if Selection = EncryptPO then
            -- Rename the file appropriately.
            Mv_F(Out_File, Tmpfile & ".asc");
         else
            Mv_F(Out_File, Tmpfile);
         end if;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Sending.Encrypt (mv block)");
            raise;
      end;

      if Mime then
         begin
            case Mime_Selection is
               when AAbort => -- Abort
                  Ada.Command_Line.Set_Exit_Status(Ada.Command_Line.Failure);
                  raise Exit_Send_Loop;
               when InlinePlain => -- Inline plain text
                  Echo_Out(" Content-Type: text/plain ",
                           Mimefile);
               when AppPGP => -- application/pgp
                  Echo_Out_N("Content-Type: application/pgp; format=text; x-action=encrypt ",
                             Mimefile);
               when Multipart => -- RFC2015 multipart
                                 -- This is the nasty one.
                  declare
                     Blk1   : constant String := Temp_File_Name("mp1");
                     Blk2   : constant String := Temp_File_Name("mp2");
                     MC     : constant String := Temp_File_Name("mc");
                  begin
                     -- We first create the two blocks.
                     -- The first block is easy:
                     Echo_Out("Content-Type: application/pgp-encrypted",
                              Blk1);
                     Echo_Append("", Blk1);
                     Echo_Append("Version: 1", Blk1);
                     -- The second block starts with a
                     -- content-type, then is the tmpfile we've
                     -- put together.
                     Echo_Out("Content-Type: application/octet-stream",
                              Blk2);
                     Echo_Append("", Blk2);
                     Cat_Append(Tmpfile, Blk2);
                     -- Now we put them together.
                     Externals.Mail.Mimeconstruct2(Part1_Filename  => Blk1,
                                         Part2_Filename  => Blk2,
                                         Output_Filename => MC,
                                         Content_Type    => "multipart/encrypted; protocol=""application/pgp-encrypted""");
                     -- Now we need to split these up.
                     Mail.Extract_Content_Type_From_Header(MC, Mimefile);
                     Mail.Extract_Body(MC, Tmpfile);
                  end;
               when MultipartEncap =>
                  Error("Menu should not have allowed MultipartEncap here");
            end case;
         exception
            when others =>
               Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                    "Exception raised in Sending.Encrypt (MIME block 2)");
               raise;
         end;
      end if;

      Check_Send(Tmpfile, Non_Pine, Mime, Mimefile);

   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Sending.Encrypt");
         raise;
   end Encrypt;

   procedure Sign_Encrypt  (Tmpfile   : in String;
                            Non_Pine  : in Boolean;
                            Mime      : in Boolean;
                            Mimefile  : in String;
                            Send_Keys : in Keys.Key_List;
                            Selection : in Send_Index;
                            AL        : in Attachments.Attachment_List) is
      Out_File       : constant String   := Temp_File_Name("out");
      DTBL_File      : constant String   := Temp_File_Name("dtbl");
      QP_File        : constant String   := Temp_File_Name("qp");
      TDS_File       : constant String   := Temp_File_Name("tds");
      Mime_Selection : MIME_Send_Index;
   begin
      if Mime then
         begin
            Ada.Text_IO.New_Line(3);
            Mime_Selection := MIME_Send_Menu2;
            if Mime_Selection = AAbort then
               Ada.Command_Line.Set_Exit_Status(Ada.Command_Line.Failure);
               raise Exit_Send_Loop;
            end if;
         exception
            when others =>
               Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                    "Exception raised in Sending.Sign_Encrypt (MIME block 1)");
               raise;
         end;
      end if;

      if not (Mime
              and then (Mime_Selection = MultipartEncap
                        or Mime_Selection = Multipart)) then
         if Attachments.Count(AL) > 0 then
            Ada.Text_IO.Put_Line("WARNING: attachments not included in non-MIME emails!");
            Ada.Text_IO.New_Line;
         end if;
      end if;

      -- Run GPG.
      if Mime and then Mime_Selection = MultipartEncap then
         -- We need to do something different with Tmpfile.
         -- First, we delete all trailing blank lines.
         begin
            Mail.Delete_Trailing_Blank_Lines(Infile  => Tmpfile,
                                             Outfile => DTBL_File);
            -- Then we turn it into quoted printable.
            -- Also turn it into DOS line endings.
            Externals.Mail.Mimeconstruct_Subpart(Infile       => DTBL_File,
                                       Outfile      => QP_File,
                                       Content_Type => "text/plain",
                                       Dos2UnixU    => True,
                                       Use_Header   => True,
                                       Header       => "Content-Type: text/plain",
                                       Use_Encoding => True,
                                       Encoding     => "quoted-printable");
            -- Call out to attachments in case we have to modify QP_File.
            Attachments.Replace_Tmpfile(QP_File, AL);
            -- Then we clearsign it.
            Externals.GPG.GPG_Wrap(ToStr(Config.Gpg_Options)
                         & " --detach-sign --armor --local-user "
                         & Value_Nonempty(Config.my_key)
                         & " "
                         & ToStr(Config.general_options)
                         & " "
                         & ToStr(Config.sending_options)
                         & " "
                         & " --output "
                         & TDS_File
                         & " "
                         & QP_File,
                         TDS_File);
         exception
            when others =>
               Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                    "Exception raised in Sending.Sign_Encrypt (GPG MIME/4 block)");
               raise;
         end;

      else

         if Mime and then Mime_Selection = Multipart then
            -- We need to do something different with Tmpfile.
            -- First, we delete all trailing blank lines.
            begin
               Mail.Delete_Trailing_Blank_Lines(Infile  => Tmpfile,
                                                Outfile => DTBL_File);
               -- Then we turn it into quoted printable.
               -- Also turn it into DOS line endings.
               Externals.Mail.Mimeconstruct_Subpart(Infile       => DTBL_File,
                                          Outfile      => QP_File,
                                          Content_Type => "text/plain",
                                          Dos2UnixU    => True,
                                          Use_Header   => True,
                                          Header       => "Content-Type: text/plain",
                                          Use_Encoding => True,
                                          Encoding     => "quoted-printable");
               -- Call out to attachments in case we have to modify QP_File.
               Attachments.Replace_Tmpfile(QP_File, AL);
               -- Then rename QP_File to Tmpfile...
               Mv_F(QP_File, Tmpfile);
            exception
               when others =>
                  Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                       "Exception raised in Sending.Sign_Encrypt (Pre-GPG MIME/3 block)");
                  raise;
            end;
         end if;

         begin
            Externals.GPG.GPG_Wrap(ToStr(Config.Gpg_Options)
                         & " --encrypt --sign --armor --local-user "
                         & Value_Nonempty(Config.my_key)
                         & " "
                         & ToStr(Config.general_options)
                         & " "
                         & ToStr(Config.sending_options)
                         & " "
                         & " --output "
                         & Out_File
                         & " "
                         & Keys.Processed_Recipient_List(Send_Keys)
                         & " "
                         & Tmpfile,
                         Out_File);
         exception
            when others =>
               Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                    "Exception raised in Sending.Sign_Encrypt (GPG block)");
               raise;
         end;

      end if;

      begin
         if not (Mime and then Mime_Selection = MultipartEncap) then
            if Selection = SignEncryptPO then
               -- Rename the file appropriately.
               Mv_F(Out_File, Tmpfile & ".asc");
            else
               Mv_F(Out_File, Tmpfile);
            end if;
         end if;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Sending.Sign_Encrypt (MV block)");
            raise;
      end;

      begin
         if Mime then
            case Mime_Selection is
               when AAbort => -- Abort
                  Ada.Command_Line.Set_Exit_Status(Ada.Command_Line.Failure);
                  raise Exit_Send_Loop;
               when InlinePlain => -- Inline plain text
                  Echo_Out("Content-Type: text/plain ",
                           Mimefile);
               when AppPGP => -- application/pgp
                  Echo_Out("Content-Type: application/pgp; format=text; x-action=encrypt",
                           Mimefile);
               when Multipart => -- RFC2015 multipart
                                 -- This is the nasty one.
                                 -- We are using the combined method allowed in section 6.2.
                  declare
                     Blk1   : constant String := Temp_File_Name("mp1");
                     Blk2   : constant String := Temp_File_Name("mp2");
                     MC     : constant String := Temp_File_Name("mc");
                  begin
                     -- We first create the two blocks.
                     -- The first block is easy:
                     Echo_Out("Content-Type: application/pgp-encrypted",
                              Blk1);
                     Echo_Append("", Blk1);
                     Echo_Append("Version: 1", Blk1);
                     -- The second block starts with a
                     -- content-type, then is the tmpfile we've
                     -- put together.
                     Echo_Out("Content-Type: application/octet-stream",
                              Blk2);
                     Echo_Append("", Blk2);
                     Cat_Append(Tmpfile, Blk2);
                     -- Now we put them together.
                     Externals.Mail.Mimeconstruct2(Part1_Filename  => Blk1,
                                         Part2_Filename  => Blk2,
                                         Output_Filename => MC,
                                         Content_Type    => "multipart/encrypted; protocol=""application/pgp-encrypted""");
                     -- Now we need to split these up.
                     Mail.Extract_Content_Type_From_Header(MC, Mimefile);
                     Mail.Extract_Body(MC, Tmpfile);
                  end;
               when MultipartEncap => -- RFC2015 multipart encapsulated
                                      -- This is the REALLY nasty one.  We are
                                      -- encapsulating, so we have to sign then
                                      -- encrypt.
                  declare
                     Blk1   : constant String := Temp_File_Name("mp1");
                     Blk2   : constant String := Temp_File_Name("mp2");
                     MCS    : constant String := Temp_File_Name("mcs");
                     MC     : constant String := Temp_File_Name("mc");
                  begin
                     -- We first create the two blocks.
                     Externals.Mail.Mimeconstruct_Subpart(Infile       => TDS_File,
                                                Outfile      => Blk2,
                                                Content_Type => "application/pgp-signature",
                                                Dos2UnixU    => False,
                                                Use_Header   => False,
                                                Use_Encoding => False);
                     Externals.Mail.Mimeconstruct2(Part1_Filename  => QP_File,
                                         Part2_Filename  => Blk2,
                                         Output_Filename => MCS,
                                         Content_Type    => "multipart/signed; protocol=""application/pgp-signature""; micalg="""
                                         & Externals.GPG.Micalg_From_Filename(TDS_File)
                                         & """");
                     -- Now the encrypt bit....
                     Rm_File(Tmpfile);
                     Externals.GPG.GPG_Wrap(ToStr(Config.Gpg_Options)
                                  & " --encrypt --armor "
                                  & ToStr(Config.general_options)
                                  & " "
                                  & ToStr(Config.sending_options)
                                  & " "
                                  & " --output "
                                  & Tmpfile
                                  & " "
                                  & Keys.Processed_Recipient_List(Send_Keys)
                                  & " "
                                  & MCS,
                                  Tmpfile);
                     -- The first block is easy:
                     Echo_Out("Content-Type: application/pgp-encrypted",
                              Blk1);
                     Echo_Append("", Blk1);
                     Echo_Append("Version: 1", Blk1);
                     -- The second block starts with a
                     -- content-type, then is the tmpfile we've
                     -- put together.
                     Echo_Out("Content-Type: application/octet-stream",
                              Blk2);
                     Echo_Append("", Blk2);
                     Cat_Append(Tmpfile, Blk2);
                     -- Now we put them together.
                     Externals.Mail.Mimeconstruct2(Part1_Filename => Blk1,
                                         Part2_Filename => Blk2,
                                         Output_Filename => MC,
                                         Content_Type => "multipart/encrypted; protocol=""application/pgp-encrypted""");
                     -- Now we need to split these up.
                     Mail.Extract_Content_Type_From_Header(MC, Mimefile);
                     Mail.Extract_Body(MC, Tmpfile);
                  end;
            end case;
         end if;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Sending.Sign_Encrypt (MIME block 2)");
            raise;
      end;

      Check_Send(Tmpfile, Non_Pine, Mime, Mimefile);

   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Sending.Sign_Encrypt");
         raise;
   end Sign_Encrypt;

   procedure Clearsign  (Tmpfile   : in String;
                         Non_Pine  : in Boolean;
                         Mime      : in Boolean;
                         Mimefile  : in String;
                         Selection : in Send_Index;
                         AL        : in Attachments.Attachment_List) is
      Out_File       : constant String   := Temp_File_Name("out");
      DTBL_File      : constant String   := Temp_File_Name("dtbl");
      QP_File        : constant String   := Temp_File_Name("qp");
      TDS_File       : constant String   := Temp_File_Name("tds");
      Mime_Selection : MIME_Send_Index;
   begin
      if Mime then
         begin
            Ada.Text_IO.New_Line(3);
            Mime_Selection := MIME_Send_Menu;
            if Mime_Selection = AAbort then
               Ada.Command_Line.Set_Exit_Status(Ada.Command_Line.Failure);
               raise Exit_Send_Loop;
            end if;
         exception
            when others =>
               Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                    "Exception raised in Sending.Clearsign (MIME block 1)");
               raise;
         end;
      end if;

      -- Run GPG.
      if Mime and then Mime_Selection = Multipart then
         -- We need to do something different with Tmpfile.
         -- First, we delete all trailing blank lines.

         begin
            Mail.Delete_Trailing_Blank_Lines(Tmpfile, DTBL_File);
            -- Then we turn it into quoted printable.
            -- Also turn it into DOS line endings.
            Externals.Mail.Mimeconstruct_Subpart(Infile       => DTBL_File,
                                       Outfile      => QP_File,
                                       Content_Type => "text/plain",
                                       Dos2UnixU    => True,
                                       Use_Header   => True,
                                       Header       => "Content-Type: text/plain",
                                       Use_Encoding => True,
                                       Encoding     => "quoted-printable");
            -- Call out to attachments in case we have to modify QP_File.
            Attachments.Replace_Tmpfile(QP_File, AL);
            -- Then we clearsign it.
            Externals.GPG.GPG_Wrap(ToStr(Config.Gpg_Options)
                         & " --detach-sign --armor --local-user "
                         & Value_Nonempty(Config.my_key)
                         & " "
                         & ToStr(Config.general_options)
                         & " "
                         & ToStr(Config.sending_options)
                         & " "
                         & " --output "
                         & TDS_File
                         & " "
                         & QP_File,
                         TDS_File);
         exception
            when others =>
               Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                    "Exception raised in Sending.Clearsign (GPG MIME/3 block 1)");
               raise;
         end;
      else
         if Attachments.Count(AL) > 0 then
            Ada.Text_IO.Put_Line("WARNING: attachments not included in non-MIME emails!");
            Ada.Text_IO.New_Line;
         end if;
         begin
            Externals.GPG.GPG_Wrap(ToStr(Config.Gpg_Options)
                         & " --clearsign --local-user "
                         & Value_Nonempty(Config.my_key)
                         & " "
                         & ToStr(Config.general_options)
                         & " "
                         & ToStr(Config.sending_options)
                         & " "
                         & " --output "
                         & Out_File
                         & " "
                         & Tmpfile,
                         Out_File);
         exception
            when others =>
               Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                    "Exception raised in Sending.Clearsign (GPG block)");
               raise;
         end;
      end if;

      begin
         if not (Mime and then Mime_Selection = Multipart) then
            if Selection = ClearsignPO then
               -- Rename the file appropriately.
               Mv_F(Out_File, Tmpfile & ".asc");
            else
               Mv_F(Out_File, Tmpfile);
            end if;
         end if;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Sending.Clearsign (MV block)");
            raise;
      end;

      begin
         if Mime then
            case Mime_Selection is
               when AAbort => -- Abort
                  Ada.Command_Line.Set_Exit_Status(Ada.Command_Line.Failure);
                  raise Exit_Send_Loop;
               when InlinePlain => -- Inline plain text
                  Echo_Out("Content-Type: text/plain", Mimefile);
               when AppPGP => -- application/pgp
                  Echo_Out("Content-Type: application/pgp; format=text; x-action=sign",
                           Mimefile);
               when Multipart => -- RFC2015 multipart
                                 -- This is the _really_ nasty one.
                                 -- At this point, we have QP_File, the quoted-printable, and TDS_File, then detached signature.
                  declare
                     Blk2   : constant String := Temp_File_Name("mp2");
                     MC     : constant String := Temp_File_Name("mc");
                  begin
                     Externals.Mail.Mimeconstruct_Subpart(Infile       => TDS_File,
                                                Outfile      => Blk2,
                                                Content_Type => "application/pgp-signature",
                                                Dos2UnixU    => False,
                                                Use_Header   => False,
                                                Use_Encoding => False);
                     Externals.Mail.Mimeconstruct2(Part1_Filename  => QP_File,
                                         Part2_Filename  => Blk2,
                                         Output_Filename => MC,
                                         Content_Type    => "multipart/signed; protocol=""application/pgp-signature""; micalg="""
                                         & Externals.GPG.Micalg_From_Filename(TDS_File)
                                         & """");
                     -- Now we need to split these up.
                     Mail.Extract_Content_Type_From_Header(MC, Mimefile);
                     Mail.Extract_Body(MC, Tmpfile);
                  end;
               when MultipartEncap =>
                  Error("Menu should not have allowed MultipartEncap here");
            end case;
         end if;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Sending.Clearsign (MIME block 2)");
            raise;
      end;

      Check_Send(Tmpfile, Non_Pine, Mime, Mimefile);

   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Sending.Clearsign");
         raise;
   end Clearsign;

   procedure Detached_Sig  (Tmpfile   : in String) is
   begin
      -- Run GPG.
      Externals.GPG.GPG_Wrap(ToStr(Config.Gpg_Options)
                   & " --detach-sign --armor --local-user "
                   & Value_Nonempty(Config.my_key)
                   & " "
                   & ToStr(Config.general_options)
                   & " "
                   & ToStr(Config.sending_options)
                   & " "
                   & " --output "
                   & Tmpfile
                   & ".asc "
                   & Tmpfile,
                   Tmpfile & ".asc");
      Wait;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Sending.Detached_Sig");
         raise;
   end Detached_Sig;

   procedure Send_Unchanged  (Tmpfile   : in String;
                              Non_Pine  : in Boolean;
                              Mime      : in Boolean;
                              Mimefile  : in String) is
   begin
      Check_Send(Tmpfile, Non_Pine, Mime, Mimefile);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Sending.Send_Unchanged");
         raise;
   end Send_Unchanged;

   -- We want to send a message.
   procedure Send (Tmpfile    : in String;
                   Recipients : in UBS_Array;
                   Non_Pine   : in Boolean   := False;
                   Mime       : in Boolean   := False;
                   Mimefile   : in String    := "") is
      Send_Keys      : Keys.Key_List;
      Selection      : Send_Index;
      Originfile     : constant String := Temp_File_Name("origin");
      Hdrfile        : constant String := Temp_File_Name("hdr");
      Tmpfile2       : constant String := Temp_File_Name("tmp2");
      Fromfile       : constant String := Temp_File_Name("from");
      Fromfile2      : constant String := Temp_File_Name("from2");
      Alt_Sign       : UBS;
      AL             : Attachments.Attachment_List;
   begin
      Debug("+Send");
      -- Save the original input.
      Externals.Simple.Cat_Out(Tmpfile, Originfile);
      -- If Config.All_Headers is set, then split tmpfile into header
      --  and body.  But we'll copy the tmpfile into another tmp
      --  first, then put the body alone back in Tmpfile.
      if Config.All_Headers then
         Externals.Simple.Mv_F(Tmpfile, Tmpfile2);
         -- Make sure the line endings are correct.
         Externals.Simple.Dos2Unix(Tmpfile2);
         Externals.Mail.Extract_Header(Email_Filename => Tmpfile2,
                                       Target_Filename => Hdrfile);
         Externals.Mail.Extract_Body(Email_Filename => Tmpfile2,
                                     Target_Filename => Tmpfile);
      end if;
      -- If Config.Read_From is set, then parse the header to find the
      --  fromline and set my-key appropriately.  This implies that
      --  All_Headers was set, too.
      if Config.Read_From then
         Externals.Mail.Formail_Concat_Extract_InOut("From:",
                                                     Hdrfile,
                                                     Fromfile);
         Externals.Simple.Sed_InOut("s/^.*<//; s/>.*$//",
                                    Fromfile, Fromfile2);
         declare
            F : Ada.Text_IO.File_Type;
         begin
            Ada.Text_IO.Open(File => F,
                             Mode => Ada.Text_IO.In_File,
                             Name => Fromfile2);
            Alt_Sign := Unbounded_Get_Line(F);
            Ada.Text_IO.Close(F);
         end;
      end if;
      -- Sort out the send menus/msgs.
      Key_Preselect(Send_Keys, Recipients, ToStr(Alt_Sign));
      -- Initialise attachments.
      Attachments.Empty_Attachment_List(AL);
      -- Loop around doingstuff, now.
  Send_Loop:
      loop
         Ada.Text_IO.New_Line(5);
         Ada.Text_IO.Put_Line("** Main send menu:");
         Ada.Text_IO.Put(Integer'Image(Keys.Count(Send_Keys))
                         & " key(s) in key list     ");
         if Non_Pine then
            Ada.Text_IO.New_Line;
            Selection := Send_Menu_NP;
         else
            Ada.Text_IO.Put_Line(Integer'Image(Attachments.Count(AL))
                                 & " attachment(s)");
            Selection := Send_Menu_Pine;
         end if;
         begin
            case Selection is
               when AAbort => -- Abort
                  Ada.Text_IO.Put_Line("Aborting");
                  Ada.Command_Line.Set_Exit_Status(Ada.Command_Line.Failure);
                  exit Send_Loop;
               when AddOwn => -- Add own key
                  if ToStr(Config.My_Key) = "" then
                  Ada.Text_IO.Put_Line("my-key is empty; not selecting any keys for self");
                  else
                     Keys.Add_Keys_By_Fingerprint(Value_Nonempty(Config.My_Key),
                                                  Send_Keys,
                                                  Keys.Encrypter);
                  end if;
               when EditOwn => -- Change own key
                  Configuration.Edit_Own_Key;
               when ViewMail => -- View the input email.
                  Pager(Tmpfile);
               when EditMail => -- Edit the input email.
                  declare
                     Editor_Command : UBS;
                     Dummy          : Integer;
                     use Ada.Text_IO;
                     use type UBS;
                     pragma Unreferenced(Dummy);
                  begin
                     New_Line(1);
                     Put_Line("Edit input email:");
                     Editor_Command
                       := ToUBS(Readline.Get_String("Which editor command? (filename will be appended) "));
                     if ToStr(Editor_Command)'Length > 0 then
                        Dummy := Externals.Simple.System(Editor_Command & ToUBS(" " & Tmpfile));
                     end if;
                  end;
               when AttachEdit => -- Edit the attachment list.
                  Attachments.List(AL);
               when Encrypt | EncryptPO => -- Do GPG: encrypt
                  Encrypt(Tmpfile, Non_Pine,
                          Mime, Mimefile,
                          Send_Keys, Selection,
                          AL);
                  exit Send_Loop;
               when SignEncrypt | SignEncryptPO => -- Do GPG:  sign-encrypt
                  Sign_Encrypt(Tmpfile, Non_Pine,
                               Mime, Mimefile,
                               Send_Keys, Selection,
                               AL);
                  exit Send_Loop;
               when Clearsign | ClearSignPO => -- Do GPG:  clearsign
                  Clearsign(Tmpfile, Non_Pine,
                            Mime, Mimefile,
                            Selection,
                            AL);
                  exit Send_Loop;
               when DetachSignPO => -- Do GPG:  create detached signature
                  Detached_Sig(Tmpfile);
                  exit Send_Loop;
               when Configure => -- Configuration.
                  Configuration.Edit_Configuration;
               when ListEdit => -- List keys
                  Keys.List_Keys(Send_Keys);
               when NoGPG => -- Pass thu' unchanged.
                  Send_Unchanged(Tmpfile, Non_Pine, Mime, Mimefile);
                  exit Send_Loop;
               when Remote => -- Run remote connection
                              -- Restore original tmpfile from origin.
                  Externals.Simple.Cat_Out(Originfile, Tmpfile);
                  Remote_Mode.Send(Tmpfile, Mime, Mimefile, Recipients);
                  exit Send_Loop;
            end case;
         exception
            when Exit_Send_Loop =>
               exit Send_Loop;
            when others =>
               Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                    "Exception raised in Sending.Send (Send_Loop)");
               raise;
         end;
      end loop Send_Loop;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Sending.Send");
         raise;
   end Send;

end Sending;

