# $Id: ispell.tcl,v 1.10 2005/07/26 23:22:17 aleksey Exp $

if {![info exists ::use_ispell] || !$::use_ispell} {
    return
}

package require textutil

namespace eval ispell {
    custom::defgroup Plugins [::msgcat::mc "Plugins options."] -group Tkabber
    custom::defgroup Ispell [::msgcat::mc "Spell check options."] -group Plugins

    variable options
    #set options(executable) /usr/bin/ispell
    custom::defvar options(executable) /usr/bin/ispell \
	[::msgcat::mc "Path to the ispell executable."] -group Ispell
    #set options(check_every_symbol) 0
    custom::defvar options(check_every_symbol) 0 \
	[::msgcat::mc "Check spell after every entered symbol."] -type boolean -group Ispell
    custom::defvar options(dictionary) "" \
	[::msgcat::mc "Ispell dictionary. If it is empty, default dictionary is used."] -type string -group Ispell
    custom::defvar options(dictionary_encoding) "" \
	[::msgcat::mc "Ispell dictionary encoding. If it is empty, system encoding is used."] -type string -group Ispell
    variable misspelled
    variable word_id 0
    option add *Chat.errorColor Red widgetDefault
    option add *Chat.comboColor Blue widgetDefault
}

proc ispell::start {} {
    variable options
    variable pipe

    if {[info exists options(dictionary)] \
	    && ![cequal $options(dictionary)  ""]} {
	set dict_string "-d $options(dictionary)"
    } else {
	set dict_string ""
    }
    set pipe [open "|$options(executable) -a $dict_string" r+]
    set version [gets $pipe]
    if {[cequal $version ""]} {
	debugmsg plugins "ISPELL: Could not start ispell server. Check your dictionary name."
	return
    }
    if {[info exists options(dictionary_encoding)] \
	    && ![cequal $options(dictionary_encoding) ""]} {
	fconfigure $pipe -blocking off -buffering line -encoding $options(dictionary_encoding)
    } else {
	fconfigure $pipe -blocking off -buffering line
    }
    fileevent $pipe readable [namespace current]::process_filter
}

proc ispell::process_filter {} {
    variable pipe
    variable response
    variable current_word
    variable input_window
    variable misspelled

    set word [read $pipe]
    if {[string length $word] <= 1} {
	set response $word
	return
    }
    switch -- [string index $word 0] {
	\- {
	    set misspelled($current_word) combo
	}
	\& -
	\? -
	\# {
	    set misspelled($current_word) err
	}
	default {
	    set misspelled($current_word) ok
	}
    }
    set response $word
}

proc ispell::pipe_word {word} {
    variable pipe
    variable response
    variable current_word
    variable misspelled

    set current_word $word

    if {![info exist pipe]} {
	start
    }
    if {[string length $word] <= 1} {
	set misspelled($word) ok
	return
    }
    puts $pipe $word
    vwait [namespace current]::response
}

proc ispell::process_word {iw insind} {
    variable input_window
    variable misspelled
    variable word_id

    set wid $word_id
    incr word_id

    set ins [lindex [split $insind .] 1]
    set line [$iw get "$insind linestart" "$insind lineend"]
    set wordstart [string wordstart $line $ins]
    set wordend   [expr {[string wordend $line $ins] - 1}]
    set w [crange $line $wordstart $wordend]
    $iw mark set ispell_wordstart$wid "insert linestart +$wordstart chars"
    $iw mark set ispell_wordend$wid \
	"insert linestart +$wordend chars +1 chars"
    if {[info exists misspelled($w)]} {
	$iw tag remove err ispell_wordstart$wid ispell_wordend$wid
	$iw tag remove combo ispell_wordstart$wid ispell_wordend$wid
	$iw tag add $misspelled($w) \
	    ispell_wordstart$wid ispell_wordend$wid
    } elseif {[string length $w] > 1} {
	pipe_word $w
	if {![winfo exists $iw]} {
	    return 0
	}
	$iw tag remove err ispell_wordstart$wid ispell_wordend$wid
	$iw tag remove combo ispell_wordstart$wid ispell_wordend$wid
	if {[info exists misspelled($w)]} {
	    $iw tag add $misspelled($w) \
		ispell_wordstart$wid ispell_wordend$wid
	}
    } else {
	$iw tag remove err ispell_wordstart$wid ispell_wordend$wid
	$iw tag remove combo ispell_wordstart$wid ispell_wordend$wid
	$iw mark unset ispell_wordstart$wid
	$iw mark unset ispell_wordend$wid
	return 0
    }
    $iw mark unset ispell_wordstart$wid
    $iw mark unset ispell_wordend$wid
    return 1
}

proc ispell::process_line {iw sym} {
    variable state
    variable insert_prev
    variable options

    if {![winfo exists $iw]} {
	return
    }

    switch -- $state($iw) {
	0 {
	    if {[cequal $sym ""]} {
		set state($iw) 1
		# in state 0 it's more likely that the word is to the left of cursor position
		set leftword [process_word $iw [$iw index "$insert_prev -1 chars"]]
		# but in rare cases (BackSpace) the word could be to the right
		if {!$leftword} {
		    process_word $iw [$iw index "$insert_prev +0 chars"]
		}
	    } elseif {![string is wordchar $sym] && ($sym != "\u0008")} {
		set state($iw) 1
		process_word $iw [$iw index "$insert_prev -1 chars"]
		process_word $iw [$iw index "insert +0 chars"]
	    } elseif {$options(check_every_symbol)} {
	        process_word $iw [$iw index "insert -1 chars"]
	    }
	}
	1 {
	    if {[cequal $sym ""]} {
		# do nothing
	    } elseif {![string is wordchar $sym]} {
		process_word $iw [$iw index "$insert_prev -1 chars"]
		process_word $iw [$iw index "insert +0 chars"]
		process_word $iw [$iw index "insert -1 chars"]
	    } else {
		set leftword [process_word $iw [$iw index "insert -1 chars"]]
		set cur_sym [$iw get "insert" "insert +1 chars"]
		if {!$leftword && ![string is wordchar $cur_sym]} {
		    set state($iw) 0
		}
	    }
	    
	}
    }

    set insert_prev [$iw index "insert"]

    variable after_id
    unset after_id($iw)
}

proc ispell::clear_ispell {iw} {
    variable misspelled
    variable state
    variable insert_prev

    set insert_prev [$iw index "insert"]
    if {[info exists misspelled] && \
	    ([llength [array names misspelled]] > 2048)} {
	unset misspelled
    }
    set state($iw) 0
}
    
proc ispell::after_process {iw sym} {
    variable after_id
    if {![info exists after_id($iw)]} {
	set after_id($iw) \
	    [after idle [list [namespace current]::process_line $iw $sym]]
    }
}

proc ispell::popup_menu {iw x y} {
    variable response

    set ind [$iw index @$x,$y]
    lassign [split $ind .] l i
    set line [$iw get "$ind linestart" "$ind lineend"]
    set wordstart [string wordstart $line $i]
    set wordend   [expr {[string wordend $line $i] - 1}]
    set w [crange $line $wordstart $wordend]
    pipe_word $w
    if {[catch { string trim $response } r]} {
	return
    }
    if {[winfo exists [set m .ispellpopupmenu]]} {
	destroy $m
    }
    switch -- [string index $r 0] {
	\& -
	\? {
	    regsub -all {: } $r {:} r
	    regsub -all {, } $r {,} r
	    set variants [split [lindex [split $r ":"] 1] ","]
	    menu $m -tearoff 0
	    foreach var $variants {
		$m add command -label "$var" \
		    -command [list [namespace current]::substitute $iw $l.$wordstart $l.[expr $wordend + 1] $var]
	    }
	    tk_popup $m [winfo pointerx .] [winfo pointery .]
	}
	\# {
	    menu $m -tearoff 0
	    $m add command -label [::msgcat::mc "- nothing -"] -command {}
	    tk_popup $m [winfo pointerx .] [winfo pointery .]
	}
	default {}
    }
    
}

proc ispell::substitute {iw wordstart wordend sub} {
    $iw delete $wordstart $wordend
    $iw insert $wordstart $sub
}

proc ispell::setup_bindings {chatid type} {
    global usetabbar
    variable history

    set iw [chat::input_win $chatid]
    clear_ispell $iw
    bind $iw <Return> +[list [namespace current]::clear_ispell $iw]
    bind $iw <Key> +[list [namespace current]::after_process $iw %A]
    bind $iw <3> [list [namespace current]::popup_menu $iw %x %y]
    $iw tag configure err -foreground [option get [winfo parent $iw] errorColor Chat]
    $iw tag configure combo -foreground [option get [winfo parent $iw] comboColor Chat]
}

hook::add open_chat_post_hook [namespace current]::ispell::setup_bindings

