# $Id: gpgme.tcl,v 1.29 2005/07/31 21:09:36 aleksey Exp $

if {[catch { package require gpgme }]} {
    debugmsg ssj "unable to load the GPGME package, so no crypto!"
    set have_gpgme 0
    return
} else {
    set have_gpgme 1
}

namespace eval ssj {
    variable options

    custom::defgroup GPGME [::msgcat::mc "GPGME options (signing and encryption)."] \
	-group Tkabber

    custom::defvar options(one-passphrase) 1 \
	[::msgcat::mc "Use the same passphrase for signing and decrypting messages."] \
	-group GPGME -type boolean

    custom::defvar options(sign-traffic) 0 \
	[::msgcat::mc "GPG-sign outgoing messages and presence updates."] \
	-group GPGME -type boolean

    custom::defvar options(encrypt-traffic) 0 \
	[::msgcat::mc "GPG-encrypt outgoing messages where possible."] \
	-group GPGME -type boolean

    custom::defvar options(key) "" \
	[::msgcat::mc "Use specified key ID for signing and decrypting messages."] \
	-group GPGME -type string
}


package require base64


namespace eval ssj {
    variable ctx
    variable e4me
    variable j2k
    variable options
    variable passphrase
    variable s2e
    variable signers
    variable warnings
    variable gpg_error_id 0

    array set ctx {}

    array set j2k {}

    array set options {}

    array set passphrase {}

    array set s2e \
          [list none       "No information available"                         \
                bad        "Invalid signature"                                \
                nokey      "Signature not processed due to missing key"       \
                nosig      "Malformed signature block"                        \
                error      "Error in signature processing"                    \
                diff       "Multiple signatures having different authenticity"\
                expired    "The signature is good but has expired"            \
                expiredkey "The signature is good but the key has expired"]

    catch { unset warnings }
    array set warnings {}
}


proc ssj::once_only {connid {armorP 0}} {
    global env
    variable options
    variable ctx

    if {[info exists ctx($connid)] && ![cequal $ctx($connid) ""]} {
        $ctx($connid) -operation set   \
		      -property  armor \
		      -value     $armorP

        return
    }

    set ctx($connid) [gpgme::context]
    $ctx($connid) -operation set   \
		  -property  armor \
		  -value     $armorP


    if {![info exists env(GPG_AGENT_INFO)]} {
        $ctx($connid) -operation set                 \
		      -property  passphrase-callback \
		      -value     [list [namespace current]::passphrase $connid]
    }

    set pattern [jlib::connection_bare_jid $connid]

    set firstP 1
    if {$options(key) != ""} {
	set patterns [list $options(key)]
    } else {
	set patterns {}
    }
    lappend patterns $pattern ""
    foreach p $patterns {
        set command [list $ctx($connid) -operation start-key -secretonly true]
        if {![cequal $p ""]} {
            lappend command -patterns [list $p]
        }
        eval $command

        for {set keys {}} \
            {![cequal [set key [$ctx($connid) -operation next-key]] ""]} \
            {lappend keys $key} {}
        $ctx($connid) -operation done-key

        if {[llength $keys] > 0} {
            break
        }
        if {[cequal $p ""]} {
            return
        }
        set firstP 0
    }

    switch -- [llength $keys] {
        0 {
            return
        }

        1 {
            if {$firstP} {
                e4meP $connid $keys
                return
            }
        }

        default {
        }
    }

    set dw .selectkey$connid
    catch { destroy $dw }

    set titles {}
    set balloons {}
    foreach key $keys {
        foreach {k v} [$ctx($connid) -operation info-key -key $key] {
            if {![cequal $k subkeys]} {
                continue
            }
            foreach subkey $v {
                catch { unset params }
                array set params $subkey
                if {![info exists params(email)]} {
                    continue
                }
		lappend titles $key $params(email)
                if {![catch { format "%d%s/%s %s %s" $params(length)         \
                                     [string range $params(algorithm) 0 0]   \
                                     [string range $params(keyid) end-7 end] \
                                     [clock format $params(created)          \
                                            -format "%Y-%m-%d"]              \
                                     $params(userid) } text]} {
		    lappend balloons $key $text
		}
	    }
	}
    }

    CbDialog $dw [format [::msgcat::mc "Select Key for Signing %s Traffic"] $pattern] \
        [list [::msgcat::mc "Select"] "[namespace current]::once_only_aux $dw $connid" \
	      [::msgcat::mc "Cancel"] "destroy $dw"] \
	[namespace current]::selectkey$connid $titles $balloons \
	-modal local
}

proc ssj::once_only_aux {dw connid} {
    variable selectkey$connid

    set keys {}
    foreach key [array names selectkey$connid] {
        if {[set selectkey${connid}($key)]} {
            lappend keys $key
        }
    }

    destroy $dw

    if {[llength $keys] > 0} {
        e4meP $connid $keys
    }
}


proc ssj::passphrase {connid data} {
    variable passphrase
    variable options

    array set params $data
    set lines [split [string trimright $params(description)] "\n"]
    set text [lindex $lines 0]

    if {[set x [string first " " [set keyid [lindex $lines 1]]]] > 0} {
        set userid [string range $keyid [expr $x+1] end]
        if {!$options(one-passphrase)} {
            set keyid [string range $keyid 0 [expr $x-1]]
        } else {
            regexp { +([^ ]+)} [lindex $lines 2] ignore keyid
        }
    } else {
        set userid unknown!
    }

    if {([cequal $text ENTER]) \
            && ([info exists passphrase($keyid)]) \
            && (![cequal $passphrase($keyid) ""])} {
        return $passphrase($keyid)
    }

    set pw .passphrase$connid
    if {[winfo exists $pw]} {
        destroy $pw
    }

    set title [::msgcat::mc "Please enter passphrase"]
    switch -- $text {
        ENTER {
        }

        TRY_AGAIN {
            set title [::msgcat::mc "Please try again"]
        }

        default {
            append title ": " $text
        }
    }
    Dialog $pw -title $title -separator 1 -anchor e -default 0 -cancel 1

    set pf [$pw getframe]
    grid columnconfigure $pf 1 -weight 1

    foreach {k v} [list keyid  [::msgcat::mc "Key ID"] \
			userid [::msgcat::mc "User ID"]] {
        label $pf.l$k -text ${v}:
        entry $pf.$k
        $pf.$k insert 0 [set $k]
        if {[string length [set $k]] <= 72} {
            $pf.$k configure -width 0
        }
        if {[info tclversion] >= 8.4} {
            set bgcolor [lindex [$pf.$k configure -background] 4]
            $pf.$k configure -state readonly -readonlybackground $bgcolor
        } else {
            $pf.$k configure -state disabled
        }
    }

    label $pf.lpassword -text [::msgcat::mc "Passphrase:"]
    entry $pf.password  \
	  -textvariable [namespace current]::passphrase($connid,$keyid) \
          -show *
    set passphrase($connid,$keyid) ""

    grid $pf.lkeyid    -row 0 -column 0 -sticky e
    grid $pf.keyid     -row 0 -column 1 -sticky ew
    grid $pf.luserid   -row 1 -column 0 -sticky e
    grid $pf.userid    -row 1 -column 1 -sticky ew
    grid $pf.lpassword -row 2 -column 0 -sticky e
    grid $pf.password  -row 2 -column 1 -sticky ew

    $pw add -text [::msgcat::mc "OK"] -command "$pw enddialog 0"
    $pw add -text [::msgcat::mc "Cancel"] -command "$pw enddialog 1"

    if {[set abort [$pw draw $pf.password]]} {
        $params(token) -operation cancel
	# TODO: unset options(sign-traffic) etc. ?
    }

    destroy $pw

    if {!$abort} {
	set passphrase($keyid) $passphrase($connid,$keyid)
	unset passphrase($connid,$keyid)
        return $passphrase($keyid)
    }
}


proc ssj::armor:encode {text} {
    if {[set x [string first "\n\n" $text]] >= 0} {
        set text [string range $text [expr $x+2] end]
    }
    if {[set x [string first "\n-----" $text]] > 0} {
        set text [string range $text 0 [expr $x-1]]
    }

    return $text
}

proc ssj::armor:decode {text} {
    return "-----BEGIN PGP MESSAGE-----\n\n$text\n-----END PGP MESSAGE-----"
}

proc ssj::signed:input {connid from signature data what} {
    variable ctx
    variable j2k
    variable s2e
    variable warnings
    variable gpg_error_id

    once_only $connid

    if {[catch { $ctx($connid) -operation verify \
			       -input     [binary format a* [encoding convertto utf-8 $data]]  \
			       -signature [armor:decode $signature] } result]} {
        debugmsg ssj "verify processing error ($connid): $result ($from)"

        if {![info exists warnings(verify-traffic,$connid)]} {

            set warnings(verify-traffic,$connid) 1
            after idle [list NonmodalMessageDlg .verify_error$connid -aspect 50000 -icon error \
                -message [format [::msgcat::mc "Error in signature verification software: %s."] \
				 $result]]
        }

        set params(reason) $result

        return [array get params]
    }

    debugmsg ssj "VERIFY: $connid $from ($data); $result"

    array set params $result
    set result $params(status)

    set signatures {}
    foreach signature $params(signatures) {
        catch { unset sparams }
        array set sparams $signature

        if {[info exists sparams(key)]} {
            set sparams(key) [$ctx($connid) -operation info-key -key $sparams(key)]
            foreach {k v} $sparams(key) {
                if {![cequal $k subkeys]} {
                    continue
                }
                foreach subkey $v {
                    catch { unset kparams }
                    array set kparams $subkey
                    if {[info exists kparams(keyid)]} {
                        set j2k($from) $kparams(keyid)
                        break
                    }
                }
            }
        }

        lappend signatures [array get sparams]
    }
    catch { unset params }
    array set params [list signatures $signatures]

    if {![cequal $result good]} {
        if {[info exists s2e($result)]} {
            set result $s2e($result)
        }
        set params(reason) $result

        if {![info exists warnings(verify,$from)]} {
            set warnings(verify,$from) 1
            incr gpg_error_id
            after idle [list NonmodalMessageDlg .verify_error$gpg_error_id -aspect 50000 -icon error \
                -message [format \
			      [::msgcat::mc "%s purportedly signed by %s can't be verified.\n\n%s."] \
			      $what $from $result]]
        }
    }

    return [array get params]
}


proc ssj::signed:output {connid data args} {
    variable ctx
    variable options
    variable warnings
    variable gpg_error_id

    if {(!$options(sign-traffic)) || ([cequal $data ""])} {
        return
    }

    once_only $connid 1

    if {[catch { $ctx($connid) -operation sign  \
			       -input     [binary format a* [encoding convertto utf-8 $data]] \
			       -mode      detach } result]} {
        set options(sign-traffic) 0

        debugmsg ssj "signature processing error ($connid): $result ($data)"

        if {[llength $args] == 0} {
            set buttons ok
            set cancel 0
            set message [format [::msgcat::mc "Unable to sign presence information: %s.\n\nPresence will be sent, but signing traffic is now disabled."] $result]
        } else {
            set buttons {ok cancel}
            set cancel 1
            set message [format [::msgcat::mc "Unable to sign message body: %s.\n\nSigning traffic is now disabled.\n\nSend it WITHOUT a signature?"] $result]
        }

        incr gpg_error_id
        if {[MessageDlg .sign_error$gpg_error_id -aspect 50000 -icon error -type user \
                        -buttons $buttons -default 0 -cancel $cancel \
                        -message $message]} {
            error ""
        }           

        return
    }
    set result [armor:encode $result]

    debugmsg ssj "SIGN: $data; $result"
    whichkeys $connid sign

    return $result
}

proc ssj::signed:info {pinfo} {

    set text ""
    array set params $pinfo

    foreach {k v} $pinfo {
	if {![cequal $k signatures]} {
	    if {![cequal $v ""]} {
		append text [format "%s: %s\n" $k $v]
	    }
	}
    }

    foreach signature $params(signatures) {
	set info ""
	set addrs ""
	set s ""
	foreach {k v} $signature {
	    switch -- $k {
		key {
		    foreach {k v} $v {
			if {![cequal $k subkeys]} {
			    continue
			}
			foreach subkey $v {
			    catch { unset sparams }
			    array set sparams $subkey
			    if {[info exists sparams(email)]} {
				append addrs $s $sparams(email)
                                set s "\n     "
			    }
			}
		    }
		}

		created {
		    append info "created: [clock format $v]\n"
		}

		fingerprint {
		    append info [format "keyid: 0x%s\n" [string range $v end-7 end]]
		    append info [format "%s: %s\n" $k $v]
		}

		default {
		    if {![cequal $v ""]} {
			append info [format "%s: %s\n" $k $v]
		    }
		}
	    }
	}

	if {![cequal $addrs ""]} {
	    set info "email: $addrs\n$info"
	}
	if {![cequal $info ""]} {
	    append text "\n" [string trimright $info]
	}
    }

    return [string trimleft $text]
}

proc ssj::signed:Label {lb pinfo} {
    global messageicon

    array set params $pinfo
    if {[info exists params(reason)]} {
	set args [list -image $messageicon(badsigned)]
    } else {
	set args [list -image $messageicon(signed)]
    }

    if {![cequal [set info [signed:info $pinfo]] ""]} {
	lappend args -helptext $info -helptype balloon
    }

    eval [list Label $lb] $args -cursor arrow \
	 -padx 0 -pady 0 -borderwidth 0 -highlightthickness 0

    if {[info exists params(reason)] && [cequal $params(reason) nokey]} {
	bind $lb <3> [list [namespace current]::signed:popup $pinfo]
    }
    return $lb
}

proc ssj::signed:popup {pinfo} {
    set m .signed_label_popupmenu
    if {[winfo exists $m]} {
	destroy $m
    }
    menu $m -tearoff 0
    $m add command -label [::msgcat::mc "Fetch GPG key"] \
	-command [list [namespace current]::fetchkeys $pinfo]
    tk_popup $m [winfo pointerx .] [winfo pointery .]
}

proc ssj::signed:user_menu {m connid jid} {
    global presence
    global curuser

    if {[cequal $jid "\$curuser"]} {
	set jid $curuser
    }
    if {[info exists presence(signed,$connid,$jid)]} {
	array set params $presence(signed,$connid,$jid)
	if {[info exists params(status)] && [cequal $params(status) nokey]} {
	    $m add command -label [::msgcat::mc "Fetch GPG key"] \
		-command [list [namespace current]::fetchkeys \
			       $presence(signed,$connid,$jid)]
	}
    }
}

hook::add roster_create_user_menu_edit_hook \
    [namespace current]::ssj::signed:user_menu 60

proc ssj::fetchkeys {pinfo} {
    variable gpg_error_id

    array set params $pinfo

    set keyids {}        
    foreach signature $params(signatures) {
	catch { unset sparams }
	array set sparams $signature

	if {[info exists sparams(fingerprint)]} {
	    lappend keyids [string range $sparams(fingerprint) end-7 end]
	}
    }
    set res [catch {set output [eval [list exec gpg --recv-keys] $keyids]} errMsg]
    incr gpg_error_id
    if {$res} {
        NonmodalMessageDlg .keyfetch_ok$gpg_error_id -aspect 50000 -icon error \
            -message "Key fetch error\n\n$errMsg"
    } else {
        NonmodalMessageDlg .keyfetch_error$gpg_error_id -aspect 50000 -icon info \
            -message "Key fetch result\n\n$output"
    }
}

proc ssj::encrypted:input {connid from data} {
    variable ctx
    variable warnings
    variable gpg_error_id

    once_only $connid

    if {[catch { $ctx($connid) -operation decrypt \
			       -input     [armor:decode $data] } result]} {
        debugmsg ssj "decryption processing error ($connid): $result ($from)"

        if {![info exists warnings(decrypt,$from)]} {
            set warnings(decrypt,$from) 1
            incr gpg_error_id
            after idle [list NonmodalMessageDlg .decrypt_error$gpg_error_id -aspect 50000 -icon error \
                -message [format \
			      [::msgcat::mc "Data purported sent by %s can't be deciphered.\n\n%s."] \
			      $from $result]]
        }

        error $result
    }

    debugmsg ssj "DECRYPT: $connid; $from; $result"

    array set params $result
    binary scan $params(plaintext) a* temp_utf8
    return [encoding convertfrom utf-8 $temp_utf8]
}


proc ssj::encrypted:output {connid data to} {
    variable ctx
    variable e4me
    variable j2k
    variable options
    variable gpg_error_id

    if {[cequal $data ""]} {
        return
    }

    #if {[cequal [set jid [roster::find_jid $to]] ""]} {
    #    set jid $to
    #}
    set jid [node_and_server_from_jid $to]
    if {![encryptP $connid $jid]} {
        return
    }

    if {[info exists j2k($to)]} {
        set name $j2k($to)
    } elseif {[llength [set k [array names j2k $to/*]]] > 0} {
        set name $j2k([lindex $k 0])
    } else {
        set name $jid
    }

    set recipient [gpgme::recipient]
    $recipient -operation add   \
	       -name      $name \
               -validity  full
    foreach signer $e4me($connid) {
        $recipient -operation add \
                   -name      $signer \
                   -validity  full
    }

    once_only $connid 1

    set code [catch { $ctx($connid) -operation   encrypt \
				    -input       [binary format a* [encoding convertto utf-8 $data]] \
				    -recipients $recipient } result]

    rename $recipient {}

    if {$code} {
        debugmsg ssj "encryption processing error ($connid): $result ($data)"

        set options(encrypt,$connid,$jid) 0
        incr gpg_error_id
        if {[MessageDlg .encrypt_error$gpg_error_id -aspect 50000 -icon error -type user \
                -buttons {ok cancel} -default 0 -cancel 1 \
                -message [format [::msgcat::mc "Unable to encipher data for %s: %s.\n\nEncrypting traffic to this user is now disabled.\n\nSend it as PLAINTEXT?"] $to $result]]} {
            error ""
        }

        return
    }
    set result [armor:encode $result]

    debugmsg ssj "ENCRYPT: $connid; $data; $result"

    return $result
}

proc ssj::whichkeys {connid what} {
    variable ctx
    variable warnings

    set s [$ctx($connid) -operation get -property last-op-info]
    if {[cequal $s ""]} {
        return
    }

    set keys {}
    while {([set x [string first <fpr> $s]] > 0) \
                && ([set y [string first </fpr> $s]] > $x) \
                && ($x+45 == $y)} {
        lappend keys [string range $s [expr $x+20] [expr $y-1]]
        set s [string range $s $y end]
    }

    if {![info exists warnings($what)]} {
        set warnings($what) ""
    } elseif {[cequal $warnings($what) $keys]} {
        return
    }

    set warnings($what) $keys
    debugmsg ssj "${what}ing with $keys"
}


proc ssj::prefs {connid jid} {
    variable ctx
    variable options
    variable optionsX

    set w [win_id security_preferences [list $connid $jid]]

    if {[winfo exists $w]} {
        focus -force $w
        return
    }

    Dialog $w \
	   -title [format [::msgcat::mc "Change security preferences for %s"] $jid] \
	   -separator 1 -anchor e -default 0 -cancel 1

    $w add -text [::msgcat::mc "OK"] \
	   -command [list [namespace current]::prefs_ok $w $connid $jid]
    $w add -text [::msgcat::mc "Cancel"] -command [list destroy $w]

    set f [$w getframe]

    if {![info exists options(encrypt,$connid,$jid)]} {
        set options(encrypt,$connid,$jid) [encryptP $connid $jid]
    }

    set optionsX(encrypt,$connid,$jid) $options(encrypt,$connid,$jid)
    checkbutton $f.encrypt \
        -text     [::msgcat::mc "Encrypt traffic"] \
        -variable [namespace current]::optionsX(encrypt,$connid,$jid)

    pack $f.encrypt -side left
    pack [frame $f.f -width 9c -height 2c]

    $w draw $f.name
}

proc ssj::prefs_ok {w connid jid} {
    variable options
    variable optionsX

    set options(encrypt,$connid,$jid) $optionsX(encrypt,$connid,$jid)

    destroy $w
}

proc ssj::prefs_user_menu {m connid jid} {
    global curuser

    if {[cequal $jid "\$curuser"]} {
	set jid $curuser
    }
    $m add command -label [::msgcat::mc "Edit security..."] \
	-command [list [namespace current]::prefs $connid $jid]
}

hook::add roster_create_user_menu_edit_hook \
    [namespace current]::ssj::prefs_user_menu

proc ssj::signP {} {
    variable options

    return $options(sign-traffic)
}

proc ssj::encryptP {connid jid} {
    variable ctx
    variable j2k
    variable options

    if {[cequal $jid ""]} {
	return $options(encrypt-traffic)
    }

    lassign [roster::get_category_and_subtype $connid $jid] \
            category subtype
    switch -- $category {
	conference
	    -
	service {
	    set resP 0
	}

	default {
	    set resP 1
	}
    }

    if {[info exists options(encrypt,$connid,$jid)]} {
        return $options(encrypt,$connid,$jid)
    } elseif {[info exists options(encrypt,$jid)]} {
	return $options(encrypt,$jid)
    }

    if {!$options(encrypt-traffic)} {
        return 0
    }

    if {[info exists options(encrypt-tried,$connid,$jid)]} {
        return $options(encrypt-tried,$connid,$jid)
    }

    once_only $connid

    if {[info exists j2k($jid)]} {
        set name $j2k($jid)
    } elseif {($resP) && ([llength [set k [array names j2k $jid/*]]] > 0)} {
        set name $j2k([lindex $k 0])
    } else {
        set name $jid
    }

    [set recipient [gpgme::recipient]] \
            -operation add   \
            -name      $name \
            -validity  full

    if {[catch { $ctx($connid) -operation  encrypt        \
			       -input      "Hello world." \
			       -recipients $recipient }]} {
        set options(encrypt-tried,$connid,$jid) 0
    } else {
        set options(encrypt-tried,$connid,$jid) 1
    }

    rename $recipient {}

    return $options(encrypt-tried,$connid,$jid)
}


proc ssj::e4meP {connid keys} {
    variable ctx
    variable e4me
    variable signers

    $ctx($connid) -operation set     \
		  -property  signers \
		  -value     [set signers($connid) $keys]

    set e4me($connid) {}
    foreach signer $signers($connid) {
        [set recipient [gpgme::recipient]] \
                -operation add     \
                -name      $signer \
                -validity  full

        if {![catch { $ctx($connid) -operation  encrypt        \
				    -input      "Hello world." \
				    -recipients $recipient } result]} {
            lappend e4me($connid) $signer
        }

        rename $recipient {}
    }
}

proc ssj::sign:toggleP {} {
    variable options

    set options(sign-traffic) [expr {!$options(sign-traffic)}]
}

proc ssj::encrypt:toggleP {{connid ""} {jid ""}} {
    variable options

    if {[cequal $jid ""]} {
	set options(encrypt-traffic) [expr {!$options(encrypt-traffic)}]
        return
    }

    if {![cequal $connid ""]} {
	if {![info exists options(encrypt,$connid,$jid)]} {
	    set options(encrypt,$connid,$jid) [encryptP $connid $jid]
	}
	set options(encrypt,$connid,$jid) [expr {!$options(encrypt,$connid,$jid)}]
    } else {
	return -code error "ssj::encrypt:toggleP: connid is empty and jid is not"
    }
}


proc ssj::signed:trace {script} {
    variable options
    variable trace

    if {![info exists trace(sign-traffic)]} {
        set trace(sign-traffic) {}

        ::trace variable [namespace current]::options(sign-traffic) w \
                [namespace current]::trace
    }

    lappend trace(sign-traffic) $script
}

proc ssj::encrypted:trace {script {connid ""} {jid ""}} {
    variable options
    variable trace

    if {[cequal $jid ""]} {
	set k encrypt-traffic
    } else {
	if {![cequal $connid ""]} {
	    set k encrypt,$connid,$jid
	} else {
	    return -code error "ssj::encrypted:trace: connid is empty and jid is not"
	}
    }
    if {![info exists trace($k)]} {
        set trace($k) {}

        ::trace variable [namespace current]::options($k) w \
                [namespace current]::trace
    }

    lappend trace($k) $script
}

proc ssj::trace {name1 name2 op} {
    variable trace

    set new {}
    foreach script $trace($name2) {
        if {[catch { eval $script } result]} {
            debugmsg ssj "$result -- $script"
        } else {
            lappend new $script
        }
    }
    set trace($name2) $new
}
