/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 500 $
 * $Date: 2013-08-16 19:04:13 +0800 (Fri, 16 Aug 2013) $
 */
#ifndef INDII_TINT_MODEL_CLUSTERMODELOBSERVER_HPP
#define INDII_TINT_MODEL_CLUSTERMODELOBSERVER_HPP

#include "ClusterModelObservable.hpp"

namespace indii {
/**
 * Observer of ClusterModel.
 */
class ClusterModelObserver {
public:
  /**
   * Constructor.
   */
  ClusterModelObserver();

  /**
   * Destructor.
   */
  virtual ~ClusterModelObserver();

  /**
   * Observe an object.
   */
  void watch(ClusterModelObservable* o);

  /**
   * Stop observing the object.
   */
  void ignore();

  /**
   * Notify change specific to a single cluster.
   *
   * @param i The cluster.
   */
  virtual void notifyClusterChange(const unsigned i);

  /**
   * Notify change of hue rotation to a single cluster.
   *
   * @param i The cluster.
   */
  virtual void notifyHueChange(const unsigned i);

  /**
   * Notify change of saturation adjustent to a single cluster.
   *
   * @param i The cluster.
   */
  virtual void notifySatChange(const unsigned i);

  /**
   * Notify change of lightness adjustment to a single cluster.
   *
   * @param i The cluster.
   */
  virtual void notifyLightChange(const unsigned i);

  /**
   * Notify change of alpha adjustment to a single cluster.
   *
   * @param i The cluster.
   */
  virtual void notifyAlphaChange(const unsigned i);

  /**
   * Notify change in number of clusters.
   */
  virtual void notifyNumClustersChange();

  /**
   * Notify change in hard/soft cluster assignment.
   */
  virtual void notifyIsHardChange();

  /**
   * Notify change in number of repetitions.
   */
  virtual void notifyNumRepetitionsChange();

  /**
   * Notify change in saturation threshold.
   */
  virtual void notifySaturationThresholdChange();

  /**
   * Notify change in maximum pixels for clustering.
   */
  virtual void notifyMaxPixelsChange();

  /**
   * Notify change in saturation decay.
   */
  virtual void notifySaturationDecayChange();

  /**
   * Notify change in centroid decay.
   */
  virtual void notifyCentroidDecayChange();

  /**
   * Notify change in saturation softness.
   */
  virtual void notifySaturationSoftnessChange();

  /**
   * Notify change in centroid softness.
   */
  virtual void notifyCentroidSoftnessChange();

  /**
   * Notify of destruction.
   */
  void notifyDestroy();

private:
  /**
   * Observed object.
   */
  ClusterModelObservable* o;

  /**
   * Are we currently observing an object?
   */
  bool watching;
};
}

inline void indii::ClusterModelObserver::notifyClusterChange(const unsigned i) {
  //
}

inline void indii::ClusterModelObserver::notifyHueChange(const unsigned i) {
  notifyClusterChange(i);
}

inline void indii::ClusterModelObserver::notifySatChange(const unsigned i) {
  notifyClusterChange(i);
}

inline void indii::ClusterModelObserver::notifyLightChange(const unsigned i) {
  notifyClusterChange(i);
}

inline void indii::ClusterModelObserver::notifyAlphaChange(const unsigned i) {
  notifyClusterChange(i);
}

inline void indii::ClusterModelObserver::notifyNumClustersChange() {
  //
}

inline void indii::ClusterModelObserver::notifyIsHardChange() {
  //
}

inline void indii::ClusterModelObserver::notifyNumRepetitionsChange() {
  //
}

inline void indii::ClusterModelObserver::notifySaturationThresholdChange() {
  //
}

inline void indii::ClusterModelObserver::notifyMaxPixelsChange() {
  //
}

inline void indii::ClusterModelObserver::notifySaturationDecayChange() {
  //
}

inline void indii::ClusterModelObserver::notifyCentroidDecayChange() {
  //
}

inline void indii::ClusterModelObserver::notifySaturationSoftnessChange() {
  //
}

inline void indii::ClusterModelObserver::notifyCentroidSoftnessChange() {
  //
}

inline void indii::ClusterModelObserver::notifyDestroy() {
  this->watching = false;
}

#endif
