/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 512 $
 * $Date: 2013-09-29 15:03:45 +0800 (Sun, 29 Sep 2013) $
 */
#include "ThumbHSL.hpp"

#include "wx/wx.h"
#include "wx/gbsizer.h"

using namespace indii;

#define SPOW 1.0f
#define SLIDER_RES 4096.0f

ThumbHSL::ThumbHSL(wxWindow *parent, const unsigned int cluster) :
    wxPanel(parent), model(NULL), cluster(cluster) {
  static const int PADDING = 4;

  wxGridBagSizer* sizer = new wxGridBagSizer(0, 0);
  wxGridBagSizer* slideSizer = new wxGridBagSizer(0, 0);

  thumb = new ThumbImage(this, cluster);

  slideH = new wxSlider(this, ID_SLIDE_H, 0, -SLIDER_RES, SLIDER_RES,
      wxDefaultPosition, wxSize(256, -1), wxSL_HORIZONTAL);
  slideS = new wxSlider(this, ID_SLIDE_S, 0, -SLIDER_RES, SLIDER_RES,
      wxDefaultPosition, wxSize(256, -1), wxSL_HORIZONTAL);
  slideL = new wxSlider(this, ID_SLIDE_L, 0, -SLIDER_RES, SLIDER_RES,
      wxDefaultPosition, wxSize(256, -1), wxSL_HORIZONTAL);
  //slideA = new wxSlider(this, ID_SLIDE_A, 0, 0, 255, wxDefaultPosition,
  //    wxSize(256,-1), wxSL_HORIZONTAL);

  spectH = new SpectrumImage(this, cluster, SpectrumImage::H);
  spectS = new SpectrumImage(this, cluster, SpectrumImage::S);
  spectL = new SpectrumImage(this, cluster, SpectrumImage::L);
  //spectA = new SpectrumImage(this, cluster, SpectrumImage::A);

  sizer->Add(thumb, wxGBPosition(0, 0), wxDefaultSpan, wxALIGN_TOP | wxALL,
      PADDING);

  slideSizer->Add(spectH, wxGBPosition(1, 0), wxDefaultSpan, wxALIGN_TOP);
  slideSizer->Add(spectS, wxGBPosition(3, 0), wxDefaultSpan, wxALIGN_TOP);
  slideSizer->Add(spectL, wxGBPosition(5, 0), wxDefaultSpan, wxALIGN_TOP);
  //slideSizer->Add(spectA, wxGBPosition(7,0), wxDefaultSpan, wxALIGN_TOP);

  slideSizer->Add(slideH, wxGBPosition(0, 0), wxDefaultSpan,
      wxALIGN_TOP | wxEXPAND | wxTOP, PADDING);
  slideSizer->Add(slideS, wxGBPosition(2, 0), wxDefaultSpan,
      wxALIGN_TOP | wxEXPAND | wxTOP, PADDING);
  slideSizer->Add(slideL, wxGBPosition(4, 0), wxDefaultSpan,
      wxALIGN_TOP | wxEXPAND | wxTOP, PADDING);
  //slideSizer->Add(slideA, wxGBPosition(6,0), wxDefaultSpan,
  //    wxALIGN_TOP|wxEXPAND|wxTOP, PADDING);

  sizer->Add(slideSizer, wxGBPosition(0, 1), wxDefaultSpan,
      wxALIGN_CENTRE_VERTICAL);

  //sizer->SetFlexibleDirection(wxHORIZONTAL);
  //sizer->AddGrowableCol(2);
  SetSizerAndFit(sizer);
  //unsigned naturalWidth = this->GetSize().GetWidth();
  //unsigned naturalHeight = this->GetSize().GetHeight();
  //SetMinSize(wxSize(naturalWidth, naturalHeight));
  //SetBestSize(wxSize(naturalWidth, naturalHeight));
}

void ThumbHSL::setModel(ClusterModel* model) {
  this->model = model;
  ignore();
  thumb->setModel(model);
  spectH->setModel(model);
  spectS->setModel(model);
  spectL->setModel(model);
  //spectA->setModel(model);

  if (model != NULL) {
    if (cluster < model->getNumClusters()) {
      if (!model->hasLight(cluster)) {
        slideH->Disable();
        spectH->Disable();
        slideS->Disable();
        spectS->Disable();
      } else if (!model->hasSat(cluster)) {
        slideH->Disable();
        spectH->Disable();
      }
    }

    watch(model);
    notifyNumClustersChange();
    notifySaturationDecayChange();
    notifySaturationSoftnessChange();
  }

  Show(model != NULL && cluster < model->getNumClusters());
}

void ThumbHSL::notifyNumClustersChange() {
  if (cluster < model->getNumClusters()) {
    notifyHueChange(cluster);
    notifySatChange(cluster);
    notifyLightChange(cluster);
    notifyAlphaChange(cluster);
  }
}

void ThumbHSL::notifySaturationDecayChange() {
//  if (cluster < model->getNumClusters()) {
//    Refresh();
//  }
}

void ThumbHSL::notifySaturationSoftnessChange() {
//  if (cluster < model->getNumClusters()) {
//    Refresh();
//  }
}

void ThumbHSL::notifyClusterChange(const unsigned int i) {
  notifyHueChange(i);
  notifySatChange(i);
  notifyLightChange(i);
  notifyAlphaChange(i);
}

void ThumbHSL::notifyHueChange(const unsigned i) {
  if (i == cluster) {
    int val = intround(model->getHue(cluster) * SLIDER_RES * 2.0f);
    if (slideH->GetValue() != val) {
      slideH->SetValue(val);
    }
  }
}

void ThumbHSL::notifySatChange(const unsigned i) {
  if (i == cluster) {
    float s = model->getSat(cluster);
    float sign = (s < 0.0f) ? -1.0f : 1.0f;
    int val = intround(sign * SLIDER_RES * std::pow(std::abs(s), 1.0f / SPOW));
    if (slideS->GetValue() != val) {
      slideS->SetValue(val);
    }
    if (!model->hasSat(cluster)) {
      slideH->Disable();
      spectH->Disable();
    } else if (model->hasLight(cluster)) {
      slideH->Enable();
      spectH->Enable();
    }
    spectH->Refresh();
  }
}

void ThumbHSL::notifyLightChange(const unsigned i) {
  if (i == cluster) {
    float l = model->getLight(cluster);
    int val = intround(SLIDER_RES * l);
    if (slideL->GetValue() != val) {
      slideL->SetValue(val);
    }
    if (!model->hasLight(cluster)) {
      slideH->Disable();
      spectH->Disable();
      slideS->Disable();
      spectS->Disable();
    } else {
      if (model->hasSat(cluster)) {
        slideH->Enable();
        spectH->Enable();
      }
      slideS->Enable();
      spectS->Enable();
    }
    spectH->Refresh();
    spectS->Refresh();
  }
}

void ThumbHSL::notifyAlphaChange(const unsigned i) {
  //if (i == cluster) {
  //float a = model->getAlpha(cluster);
  //int val = static_cast<int>(255.0f*a);
  //if (slideA->GetValue() != val) {
  //  slideA->SetValue(val);
  //}
  //}
}

void ThumbHSL::OnSlideH(wxScrollEvent& evt) {
  float h = 0.5f * slideH->GetValue() / SLIDER_RES;
  h = bound(-0.5f, h, 0.5f);
  model->setHue(cluster, h);
}

void ThumbHSL::OnSlideS(wxScrollEvent& evt) {
  float s = slideS->GetValue() / SLIDER_RES;
  float sign = (s < 0.0f) ? -1.0f : 1.0f;
  s = sign * std::pow(std::abs(s), SPOW);
  s = bound(-1.0f, s, 1.0f);
  model->setSat(cluster, s);
}

void ThumbHSL::OnSlideL(wxScrollEvent& evt) {
  float l = slideL->GetValue() / SLIDER_RES;
  l = bound(-1.0f, l, 1.0f);
  model->setLight(cluster, l);
}

void ThumbHSL::OnSlideA(wxScrollEvent& evt) {
  //float a = slideA->GetValue()/SLIDER_RES;
  //a = bound(0.0f, a, 1.0f);
  //model->setAlpha(cluster, a);
}

BEGIN_EVENT_TABLE(ThumbHSL, wxPanel)
EVT_COMMAND_SCROLL(ID_SLIDE_H, ThumbHSL::OnSlideH)
EVT_COMMAND_SCROLL(ID_SLIDE_S, ThumbHSL::OnSlideS)
EVT_COMMAND_SCROLL(ID_SLIDE_L, ThumbHSL::OnSlideL)
EVT_COMMAND_SCROLL(ID_SLIDE_A, ThumbHSL::OnSlideA)
END_EVENT_TABLE()
