/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 511 $
 * $Date: 2013-09-28 15:50:18 +0800 (Sat, 28 Sep 2013) $
 */
#include "PreProcessPane.hpp"

#include "../../gui/Help.hpp"

#include "wx/wx.h"
#include "wx/gbsizer.h"
#include "wx/valtext.h"

using namespace indii;

PreProcessPane::PreProcessPane(wxWindow* parent) :
    Pane(parent), model(NULL) {
  static const int PADDING = 4;

  textMaxPixels = NULL;
  wxString maxPixels;
  maxPixels << 10000;

  /* options page */
  wxGridBagSizer* sizer = new wxGridBagSizer();

  wxStaticText *labelReps, *labelMaxPixels, *labelSat;
  labelReps = new wxStaticText(this, -1, _("Reps:"));
  labelMaxPixels = new wxStaticText(this, -1, _("Pixels:"));
  labelSat = new wxStaticText(this, -1, _("Threshold:"));

  spinReps = new wxSpinCtrl(this, ID_SPIN_REPS, wxString::Format(_T("%d"), 12),
      wxDefaultPosition, wxSize(80,-1), wxSP_ARROW_KEYS, 1, 20);
  textMaxPixels = new wxTextCtrl(this, ID_TEXT_MAX_PIXELS,
      maxPixels, wxDefaultPosition, wxSize(80,-1),
      wxTE_PROCESS_ENTER, wxTextValidator(wxFILTER_NUMERIC, &maxPixels));
  spinSat = new wxSpinCtrl(this, ID_SPIN_SAT,
      wxString::Format(_T("%d"), 32), wxDefaultPosition, wxSize(80,-1),
      wxSP_ARROW_KEYS, 0, 255, 32);

  sizer->Add(labelReps, wxGBPosition(0,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxLEFT|wxTOP|wxRIGHT, PADDING);
  sizer->Add(labelMaxPixels, wxGBPosition(1,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxLEFT|wxTOP|wxRIGHT, PADDING);
  sizer->Add(labelSat, wxGBPosition(2,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxALL, PADDING);
  
  sizer->Add(spinReps, wxGBPosition(0,1), wxDefaultSpan,
      wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxRIGHT, PADDING);
  sizer->Add(textMaxPixels, wxGBPosition(1,1), wxDefaultSpan,
      wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxRIGHT, PADDING);
  sizer->Add(spinSat, wxGBPosition(2,1), wxDefaultSpan,
      wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxRIGHT|wxBOTTOM, PADDING);

  sizer->SetFlexibleDirection(wxHORIZONTAL);
  sizer->AddGrowableCol(1);
  sizer->AddGrowableCol(0);
  SetSizerAndFit(sizer);
  setModel(NULL);
}

void PreProcessPane::setModel(ClusterModel* model) {
  this->model = model;
  ignore();

  Enable(model != NULL);
  spinReps->Enable(model != NULL);
  textMaxPixels->Enable(model != NULL);
  spinSat->Enable(model != NULL);
  
  if (model != NULL) {
    watch(model);
    notifyNumRepetitionsChange();
    notifyMaxPixelsChange();
    notifySaturationThresholdChange();
  }
}

void PreProcessPane::notifyNumRepetitionsChange() {
  /* pre-condition */
  assert (model != NULL);

  spinReps->SetValue(model->getNumRepetitions());
}

void PreProcessPane::notifyMaxPixelsChange() {
  /* pre-condition */
  assert (model != NULL);

  wxString maxPixels;
  maxPixels << model->getMaxPixels();

  textMaxPixels->SetValue(maxPixels);
}

void PreProcessPane::notifySaturationThresholdChange() {
  /* pre-condition */
  assert (model != NULL);

  spinSat->SetValue(intround(model->getSaturationThreshold()*255.0f));
}

void PreProcessPane::OnSpinReps(wxSpinEvent& evt) {
  if (model != NULL) {
    model->setNumRepetitions(evt.GetPosition());
  }
}

void PreProcessPane::OnSpinSat(wxSpinEvent& evt) {
	if (model != NULL) {
	  model->setSaturationThreshold(evt.GetPosition()/255.0f);
  }
}

void PreProcessPane::OnTextMaxPixels(wxCommandEvent& evt) {
  if (model != NULL && textMaxPixels != NULL) {  
    unsigned long n;
    if (textMaxPixels->GetValue().ToULong(&n)) {
      model->setMaxPixels(n);
    }
  }
}

void PreProcessPane::OnHelp(wxHelpEvent& evt) {
  Help::show(530);
}

BEGIN_EVENT_TABLE(PreProcessPane, Pane)
EVT_SPINCTRL(ID_SPIN_REPS, PreProcessPane::OnSpinReps)
EVT_SPINCTRL(ID_SPIN_SAT, PreProcessPane::OnSpinSat)
EVT_TEXT(ID_TEXT_MAX_PIXELS, PreProcessPane::OnTextMaxPixels)
EVT_HELP(wxID_ANY, PreProcessPane::OnHelp)
END_EVENT_TABLE()
