/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 512 $
 * $Date: 2013-09-29 15:03:45 +0800 (Sun, 29 Sep 2013) $
 */
#include "PostProcessPane.hpp"

#include "../../gui/Help.hpp"

#include "wx/wx.h"
#include "wx/gbsizer.h"

using namespace indii;

#define SLIDER_RES 4096.0f

PostProcessPane::PostProcessPane(wxWindow* parent) :
    Pane(parent), model(NULL) {
  static const int PADDING = 4;

  wxGridBagSizer* sizer = new wxGridBagSizer();

  slideSatDecay = new wxSlider(this, ID_SLIDE_SAT_DECAY, 0, 0, SLIDER_RES,
      wxDefaultPosition, wxSize(120,-1));
  slideSatSoft = new wxSlider(this, ID_SLIDE_SAT_SOFT, 0, 0, SLIDER_RES,
      wxDefaultPosition, wxSize(120,-1));
  slideHard = new wxSlider(this, ID_SLIDE_HARD, 0, 0, SLIDER_RES,
      wxDefaultPosition, wxSize(120,-1));

  wxStaticText *labelSatDecay, *labelSatSoft, *labelHard;
  labelSatDecay = new wxStaticText(this, -1, _("Decay"));
  labelSatSoft = new wxStaticText(this, -1, _("Edge"));
  labelHard = new wxStaticText(this, -1, _("Hardness"));

  sizer->Add(labelSatDecay, wxGBPosition(0,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelSatSoft, wxGBPosition(1,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelHard, wxGBPosition(2,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT|wxBOTTOM, PADDING);

  sizer->Add(slideSatDecay, wxGBPosition(0,1), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxTOP|wxRIGHT, PADDING);
  sizer->Add(slideSatSoft, wxGBPosition(1,1), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxTOP|wxRIGHT, PADDING);
  sizer->Add(slideHard, wxGBPosition(2,1), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxTOP|wxRIGHT|wxBOTTOM, PADDING);

  sizer->SetFlexibleDirection(wxHORIZONTAL);
  sizer->AddGrowableCol(1);
  SetSizerAndFit(sizer);
  setModel(NULL);
}

void PostProcessPane::setModel(ClusterModel* model) {
  this->model = model;
  ignore();

  Enable(model != NULL);
  
  /* must explicitly disable all controls, not just panel, on Mac */
  slideSatDecay->Enable(model != NULL);
  slideSatSoft->Enable(model != NULL);
  slideHard->Enable(model != NULL);
  
  if (model != NULL) {
    watch(model);
    notifySaturationDecayChange();
    notifySaturationSoftnessChange();
    notifyIsHardChange();
  }
}

void PostProcessPane::notifySaturationDecayChange() {
  /* pre-condition */
  assert (model != NULL);

  int val = intround(model->getSaturationDecay()*SLIDER_RES);
  if (slideSatDecay->GetValue() != val) {
    slideSatDecay->SetValue(val);
  }
}

void PostProcessPane::notifySaturationSoftnessChange() {
  /* pre-condition */
  assert (model != NULL);

  int val = intround(model->getSaturationSoftness()*SLIDER_RES);
  if (slideSatSoft->GetValue() != val) {
    slideSatSoft->SetValue(val);
  }
}

void PostProcessPane::notifyIsHardChange() {
  /* pre-condition */
  assert (model != NULL);

  int val = intround(std::pow(model->getHard(), 0.25f)*SLIDER_RES);
  if (slideHard->GetValue() != val) {
    slideHard->SetValue(val);
  }
}

void PostProcessPane::OnSlideSatDecay(wxScrollEvent& evt) {
	if (model != NULL) {
	  model->setSaturationDecay(slideSatDecay->GetValue()/SLIDER_RES);
  }
}

void PostProcessPane::OnSlideSatSoft(wxScrollEvent& evt) {
	if (model != NULL) {
	  model->setSaturationSoftness(slideSatSoft->GetValue()/SLIDER_RES);
  }
}

void PostProcessPane::OnSlideHard(wxScrollEvent& evt) {
	if (model != NULL) {
	  model->setHard(std::pow(slideHard->GetValue()/SLIDER_RES, 4.0f));
  }
}

void PostProcessPane::OnHelp(wxHelpEvent& evt) {
  Help::show(540);
}

BEGIN_EVENT_TABLE(PostProcessPane, Pane)

EVT_COMMAND_SCROLL(ID_SLIDE_SAT_DECAY, PostProcessPane::OnSlideSatDecay)
EVT_COMMAND_SCROLL(ID_SLIDE_SAT_SOFT, PostProcessPane::OnSlideSatSoft)
EVT_COMMAND_SCROLL(ID_SLIDE_HARD, PostProcessPane::OnSlideHard)

EVT_HELP(wxID_ANY, PostProcessPane::OnHelp)

END_EVENT_TABLE()
