/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 506 $
 * $Date: 2013-09-25 21:48:47 +0800 (Wed, 25 Sep 2013) $
 */
#ifndef INDII_MODEL_MODEL_HPP
#define INDII_MODEL_MODEL_HPP

#include "Observable.hpp"
#include "../image/ImageManipulation.hpp"
#include "../image/ImageResource.hpp"
#include "../image/ColourSpace.hpp"

#include "wx/image.h"
#include "wx/dcbuffer.h"

namespace indii {
/**
 * Abstract model.
 */
class Model: public Observable {
public:
  /**
   * Constructor.
   */
  Model(ImageResource* res);

  /**
   * Get underlying image resource.
   */
  ImageResource* getImageResource();

  /**
   * Get red channel proportion for conversion to greyscale.
   *
   * @return Red channel proportion.
   */
  float getRedMix() const;

  /**
   * Get green channel proportion for conversion to greyscale.
   *
   * @return Green channel proportion.
   */
  float getGreenMix() const;

  /**
   * Get blue channel proportion for conversion to greyscale.
   *
   * @return Blue channel proportion.
   */
  float getBlueMix() const;

  /**
   * Set red channel proportion for conversion to greyscale.
   *
   * @param r Red channel proportion.
   */
  void setRedMix(const float r);

  /**
   * Set red channel proportion for conversion to greyscale.
   *
   * @param g Green channel proportion.
   */
  void setGreenMix(const float g);

  /**
   * Set blue channel proportion for conversion to greyscale.
   *
   * @param b Blue channel proportion.
   */
  void setBlueMix(const float b);

  /**
   * Set greyscale mix.
   *
   * @param r Red channel proportion.
   * @param g Green channel proportion.
   * @param b Blue channel proportion.
   */
  void setChannelMix(const float r, const float g, const float b);

  /**
   * Set default parameters.
   */
  virtual void setDefaults() = 0;

  /**
   * Set parameters for dialog.
   */
  virtual void setForDialog() = 0;

  /**
   * Calculate foreground of subimage.
   *
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param scale Inverse scale of image.
   * @param[out] o Output image.
   */
  virtual void calcFg(const wxRect& rect, const int scale, Image& o);

  /**
   * Calculate foreground of whole image.
   *
   * @param scale Inverse scale of image.
   * @param[out] o Output image.
   */
  virtual void calcFg(const int scale, Image& o);

  /**
   * Calculate foreground of image.
   *
   * @param[out] o Output image.
   */
  virtual void calcFg(Image& o);

  /**
   * Calculate foreground of subimage.
   *
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param scale Inverse scale of image.
   * @param[out] o Output image.
   */
  virtual void calcFg(const wxRect& rect, const int scale, wxImage& o);

  /**
   * Calculate foreground of whole image.
   *
   * @param scale Inverse scale of image.
   * @param[out] o Output image.
   */
  virtual void calcFg(const int scale, wxImage& o);

  /**
   * Calculate foreground of image.
   *
   * @param[out] o Output image.
   */
  virtual void calcFg(wxImage& o);

  /**
   * Calculate annotation layer of subimage.
   *
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param scale Inverse scale of image.
   * @param[out] dc Device context to annotate.
   */
  virtual void calcAn(const wxRect& rect, const int scale, wxDC& dc);

  /**
   * Calculate annotation layer of whole image.
   *
   * @param scale Inverse scale of image.
   * @param[out] dc Device context to annotate.
   */
  virtual void calcAn(const int scale, wxDC& dc);

protected:
  /**
   * Image resource.
   */
  ImageResource* res;

  /**
   * Colour space model.
   */
  ColourSpace cs;

  /**
   * Greyscale red mixing.
   */
  float greyR;

  /**
   * Greyscale green mixing.
   */
  float greyG;

  /**
   * Greyscale green mixing.
   */
  float greyB;
};
}

inline indii::ImageResource* indii::Model::getImageResource() {
  return res;
}

inline float indii::Model::getRedMix() const {
  return greyR;
}

inline float indii::Model::getGreenMix() const {
  return greyG;
}

inline float indii::Model::getBlueMix() const {
  return greyB;
}

inline void indii::Model::calcFg(const wxRect& rect, const int scale,
    Image& o) {
  //
}

inline void indii::Model::calcFg(const int scale, Image& o) {
  /* pre-condition */
  assert(o.getWidth() == res->getWidth()/scale);
  assert(o.getHeight() == res->getHeight()/scale);

  wxRect rect(0, 0, o.getWidth(), o.getHeight());
  calcFg(rect, scale, o);
}

inline void indii::Model::calcFg(Image& o) {
  calcFg(1, o);
}

inline void indii::Model::calcFg(const wxRect& rect, const int scale,
    wxImage& o) {
  Image img(o.GetWidth(), o.GetHeight());
  calcFg(rect, scale, img);
  convert(img, o);
}

inline void indii::Model::calcFg(const int scale, wxImage& o) {
  Image img(o.GetWidth(), o.GetHeight());
  calcFg(scale, img);
  convert(img, o);
}

inline void indii::Model::calcFg(wxImage& o) {
  Image img(o.GetWidth(), o.GetHeight());
  calcFg(img);
  convert(img, o);
}

inline void indii::Model::calcAn(const wxRect& rect, const int scale,
    wxDC& dc) {
  //
}

inline void indii::Model::calcAn(const int scale, wxDC& dc) {
  wxRect rect(0, 0, res->getWidth()/scale, res->getHeight()/scale);
  calcAn(rect, scale, dc);
}

#endif
