/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 517 $
 * $Date: 2013-09-30 23:33:29 +0800 (Mon, 30 Sep 2013) $
 */
#include "ImageManipulation.hpp"

#include "wx/wx.h"

void indii::hide(wxImage& image) {
  int x, y;
  const int width = image.GetWidth();
  const int height = image.GetHeight();

  if (!image.HasAlpha()) {
    image.InitAlpha();
  }

  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++) {
      image.SetAlpha(x, y, (unsigned char)0);
    }
  }
}

void indii::show(wxImage& image) {
  int x, y;
  const int width = image.GetWidth();
  const int height = image.GetHeight();

  if (!image.HasAlpha()) {
    image.InitAlpha();
  }

  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++) {
      image.SetAlpha(x, y, (unsigned char)255);
    }
  }
}

void indii::show(wxImage& image, const wxRect& rect) {
  int x, y;

  if (!image.HasAlpha()) {
    image.InitAlpha();
  }

  for (y = rect.y; y < rect.y + rect.height; ++y) {
    for (x = rect.x; x < rect.x + rect.width; ++x) {
      image.SetAlpha(x, y, (unsigned char)255);
    }
  }
}

#ifndef __PLUGIN__
void indii::convert(const Image& in, wxImage& out) {
  /* pre-condition */
  assert(in.getWidth() == out.GetWidth() && in.getHeight() == out.GetHeight());

  int x, y;
  for (y = 0; y < in.getHeight(); ++y) {
    for (x = 0; x < in.getWidth(); ++x) {
      out.SetRGB(wxRect(x, y, 1, 1), uround(in.r(y, x)), uround(in.g(y, x)), uround(in.b(y, x)));
    }
  }
}
#endif

void indii::convert(const wxImage& in, Image& out) {
  /* pre-condition */
  assert(in.GetWidth() == out.getWidth() && in.GetHeight() == out.getHeight());

  int x, y;
  for (x = 0; x < in.GetWidth(); ++x) {
    for (y = 0; y < in.GetHeight(); ++y) {
      out.r(y, x) = in.GetRed(x, y)/255.0f;
      out.g(y, x) = in.GetGreen(x, y)/255.0f;
      out.b(y, x) = in.GetBlue(x, y)/255.0f;
    }
  }
}

wxString indii::wildcards(const bool save) {
  wxList handlers = wxImage::GetHandlers();

  wxImageHandler* handler;
  wxString res, name, ext, all;
  wxList::iterator iter;
  bool first = true;

  for (iter = handlers.begin(); iter != handlers.end(); ++iter) {
    handler = dynamic_cast<wxImageHandler*>(*iter);

    name = handler->GetName();
    ext = handler->GetExtension();

    /* only certain file types can be written... */
    if (!save || ext.IsSameAs(_("bmp")) || ext.IsSameAs(_("jpg")) ||
        ext.IsSameAs(_("pcx")) || ext.IsSameAs(_("png")) ||
        ext.IsSameAs(_("pnm")) || ext.IsSameAs(_("tif")) ||
        ext.IsSameAs(_("xpm"))) {
      ext = _("*.") + ext;
      #ifndef WX_28
      unsigned i;
      for (i = 0; i < handler->GetAltExtensions().GetCount(); ++i) {
        ext += _(";*.");
        ext += handler->GetAltExtensions()[i];
      }
      #endif
      name += _(" (");
      name += ext;
      name += _(")");

      #ifdef __WXGTK__
      /* do something (not perfect...) about case sensitivity */
      ext += _(";");
      ext += ext.Upper();
      #endif

      all += ext;
      all += _(";");

      if (!first) {
        res += _("|");
      }
      res += name + _("|") + ext;
      first = false;
    }
  }
  #ifndef __WXGTK__
  if (!save) {
    res += _("|All files|*.*");
    res = _("Image files|") + all + _("|") + res;
  }
  #else
  res += _("|All files|*.*");
  res = _("Image files|") + all + _("|") + res;
  #endif

  return res;
}
