/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 517 $
 * $Date: 2013-09-30 23:33:29 +0800 (Mon, 30 Sep 2013) $
 */
#include "ImageCache.hpp"

#include "ImageManipulation.hpp"

indii::ImageCache::ImageCache(Image* original) :
    original(original) {
  displayOriginal = new wxImage(original->getWidth(), original->getHeight(), false);
  convert(*original, *displayOriginal);
}

indii::ImageCache::~ImageCache() {
  cache_t::iterator iter1;
  for (iter1 = cache.begin(); iter1 != cache.end(); ++iter1) {
    delete iter1->second;
  }

  display_cache_t::iterator iter2;
  for (iter2 = displayCache.begin(); iter2 != displayCache.end(); ++iter2) {
    delete iter2->second;
  }

  delete displayOriginal;
}

indii::Image* indii::ImageCache::get(const int scale) {
  /* pre-condition */
  assert(scale >= 1);
  
  Image* img;
  if (scale <= 1) {
    img = original;
  } else {
    cache_t::const_iterator find = cache.find(scale);
    if (find != cache.end()) {
      img = find->second;
    } else {
      img = new Image(original->getWidth()/scale, original->getHeight()/scale);
      for (int y1 = 0; y1 < img->getHeight(); ++y1) {
        for (int x1 = 0; x1 < img->getWidth(); ++x1) {
          float r = 0.0f, g = 0.0f, b = 0.0f;
          for (int x2 = 0; x2 < scale; ++x2) {
            for (int y2 = 0; y2 < scale; ++y2) {
              r += original->r(scale*y1 + y2, scale*x1 + x2);
              g += original->g(scale*y1 + y2, scale*x1 + x2);
              b += original->b(scale*y1 + y2, scale*x1 + x2);
            }
          }
          img->r(y1, x1) = r/(scale*scale);
          img->g(y1, x1) = g/(scale*scale);
          img->b(y1, x1) = b/(scale*scale);
        }
      }
      cache.insert(std::make_pair(scale, img));
    }
  }
  return img;
}

wxImage* indii::ImageCache::getDisplay(const int width) {
  /* pre-condition */
  assert(width >= 1);

  double ratio = static_cast<double>(displayOriginal->GetWidth())/displayOriginal->GetHeight();
  wxImage* img;
  if (width == displayOriginal->GetWidth()) {
    img = displayOriginal;
  } else {
    display_cache_t::const_iterator find = displayCache.find(width);
    if (find != displayCache.end()) {
      img = find->second;
    } else {
      int height = intround(width/ratio);
      img = new wxImage(displayOriginal->Scale(width, height, wxIMAGE_QUALITY_HIGH));
      displayCache.insert(std::make_pair(width, img));
    }
  }
  return img;
}
