/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 506 $
 * $Date: 2013-09-25 21:48:47 +0800 (Wed, 25 Sep 2013) $
 */
#ifndef INDII_IMAGE_COLOURSPACE_HPP
#define INDII_IMAGE_COLOURSPACE_HPP

#include "boost/numeric/ublas/vector.hpp"

namespace indii {
/**
 * Colour space.
 */
class ColourSpace {
public:
  /*
   * Types of colour tuples.
   */
  typedef boost::numeric::ublas::vector<float,
      boost::numeric::ublas::bounded_array<float, 3> > rgb_t;
  typedef boost::numeric::ublas::vector<float,
      boost::numeric::ublas::bounded_array<float, 3> > hsl_t;

  /**
   * Constructor.
   */
  ColourSpace();

  /**
   * Get red channel lightness proportion.
   *
   * @return Red channel proportion.
   */
  float getRedMix();

  /**
   * Get green channel lightness proportion.
   *
   * @return Green channel proportion.
   */
  float getGreenMix();

  /**
   * Get blue channel lightness proportion.
   *
   * @return Blue channel proportion.
   */
  float getBlueMix();

  /**
   * Set channel proportions for lightness.
   * 
   * @param r Red channel proportion.
   * @param g Green channel proportion.
   * @param b Blue channel proportion.
   */
  void setLightness(const float r, const float g, const float b);

  /**
   * Convert RGB to HSL coordinates using greyscale parameters.
   *
   * @param in Input RGB values.
   * @param[out] out Output HSL values.
   */
  void rgb2hsl(const rgb_t& rgb, hsl_t& hsl);

  /**
   * Convert RGB to L using greyscale parameters.
   *
   * @param r Red channel value.
   * @param g Green channel value.
   * @param b Blue channel value.
   *
   * @return l Lightness.
   */
  float rgb2l(const float r, const float g, const float b);

  /**
   * Convert RGB to HSL coordinates using greyscale parameters.
   *
   * @param in Input HSL values.
   * @param[out] out Output RGB values.
   */
  void hsl2rgb(const hsl_t& hsl, rgb_t& rgb);

private:
  /**
   * Lightness proportions.
   */
  float greyR, greyG, greyB;

};
}

#include "ImageManipulation.hpp"

#define BASE_R 1.0f
#define BASE_G 1.0f/3.0f
#define BASE_B 2.0f/3.0f

inline float indii::ColourSpace::getRedMix() {
  return greyR;
}

inline float indii::ColourSpace::getGreenMix() {
  return greyG;
}

inline float indii::ColourSpace::getBlueMix() {
  return greyB;
}

inline float indii::ColourSpace::rgb2l(const float r, const float g,
    const float b) {
  return greyR * r + greyG * g + greyB * b;
}

inline void indii::ColourSpace::rgb2hsl(const rgb_t& rgb, hsl_t& hsl) {
  float& h = hsl(0);
  float& s = hsl(1);
  float& l = hsl(2);

  const float& r = rgb(0);
  const float& g = rgb(1);
  const float& b = rgb(2);

  float k1, k2, k3;

  /* post-condition */
  assert(r >= 0.0f && r <= 1.0f);
  assert(g >= 0.0f && g <= 1.0f);
  assert(b >= 0.0f && b <= 1.0f);

  /* lightness */
  l = greyR * r + greyG * g + greyB * b;

  /* hue & saturation */
  if (r == g && g == b) {
    // special case for greys
    h = 0.0f;
    s = 0.0f;
  } else if (r >= g) {
    if (g >= b) {
      // r > g > b
      k1 = g - b;
      k2 = r - b;
      k3 = r + b;
      h = (k1 / k2)/6.0f;
      s = k2 / k3;
    } else if (r >= b) {
      // r > b > g
      k1 = b - g;
      k2 = r - g;
      k3 = r + g;
      h = BASE_R - (k1 / k2)/6.0f;
      s = k2 / k3;
    } else {
      // b > r > g
      k1 = r - g;
      k2 = b - g;
      k3 = b + g;
      h = BASE_B + (k1 / k2)/6.0f;
      s = k2 / k3;
    }
  } else {
    if (r >= b) {
      // g > r > b
      k1 = r - b;
      k2 = g - b;
      k3 = g + b;
      h = BASE_G - (k1 / k2)/6.0f;
      s = k2 / k3;
    } else if (g >= b) {
      // g > b > r
      k1 = b - r;
      k2 = g - r;
      k3 = g + r;
      h = BASE_G + (k1 / k2)/6.0f;
      s = k2 / k3;
    } else {
      // b > g > r
      k1 = g - r;
      k2 = b - r;
      k3 = b + r;
      h = BASE_B - (k1 / k2)/6.0f;
      s = k2 / k3;
    }
  }

  /* relaxed IEEE compliance, so make sure result bounded as expected */
  h = bound(0.0f, h, 1.0f);
  s = bound(0.0f, s, 1.0f);
  l = bound(0.0f, l, 1.0f);
}

inline void indii::ColourSpace::hsl2rgb(const hsl_t& hsl, rgb_t& rgb) {
  float& r = rgb(0);
  float& g = rgb(1);
  float& b = rgb(2);

  const float& h = hsl(0);
  const float& s = hsl(1);
  const float& l = hsl(2);

  float k1, k2, k3;

  assert(h >= 0.0f && h <= 1.0f);
  assert(s >= 0.0f && s <= 1.0f);
  assert(l >= 0.0f && l <= 1.0f);

  if (s == 0.0f) {
    // special case for greys
    r = l;
    g = l;
    b = l;
  } else {
    k2 = (s - 1.0f) / (s + 1.0f);
    switch (static_cast<unsigned char>(6.0f*h)) {
    case 0:
      // r > g > b
      k1 = 6.0f*h;
      k3 = k1 + k2 * (k1 - 1.0f);
      r = l / (greyR + greyG * k3 - greyB * k2);
      g = k3 * r;
      b = -k2 * r;
      break;
    case 1:
      // g > r > b
      k1 = 6.0f*(h - BASE_G);
      k3 = -k1 - k2 * (k1 + 1.0f);
      g = l / (greyG + greyR * k3 - greyB * k2);
      r = k3 * g;
      b = -k2 * g;
      break;
    case 2:
      // g > b > r
      k1 = 6.0f*(h - BASE_G);
      k3 = k1 + k2 * (k1 - 1.0f);
      g = l / (greyG + greyB * k3 - greyR * k2);
      b = k3 * g;
      r = -k2 * g;
      break;
    case 3:
      // b > g > r
      k1 = 6.0f*(h - BASE_B);
      k3 = -k1 - k2 * (k1 + 1.0f);
      b = l / (greyB + greyG * k3 - greyR * k2);
      g = k3 * b;
      r = -k2 * b;
      break;
    case 4:
      // b > r > g
      k1 = 6.0f*(h - BASE_B);
      k3 = k1 + k2 * (k1 - 1.0f);
      b = l / (greyB + greyR * k3 - greyG * k2);
      r = k3 * b;
      g = -k2 * b;
      break;
    case 5:
      // r > b > g
      k1 = 6.0f*(h - BASE_R);
      k3 = -k1 - k2 * (k1 + 1.0f);
      r = l / (greyR + greyB * k3 - greyG * k2);
      b = k3 * r;
      g = -k2 * r;
      break;
    default:
      assert(false);
    }
  }

  /* relaxed IEEE compliance, so make sure result bounded as expected */
  r = bound(0.0f, r, 1.0f);
  g = bound(0.0f, g, 1.0f);
  b = bound(0.0f, b, 1.0f);
}

#endif

