/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 500 $
 * $Date: 2013-08-16 19:04:13 +0800 (Fri, 16 Aug 2013) $
 */
#ifndef INDII_GUI_WINDOW_HPP
#define INDII_GUI_WINDOW_HPP

#include "WindowObserver.hpp"
#include "Document.hpp"
#include "InfoBar.hpp"

#include "wx/frame.h"
#include "wx/aui/aui.h"

namespace indii {
  class OpenRecentMenu;
  class WindowMenu;

/**
 * Main window.
 */
class Window : public wxFrame {
public:
  /**
   * Modes.
   */
  enum Mode {
    /**
     * Standalone mode. Includes Save and Save As buttons.
     */
    STANDALONE,

    /**
     * Embedded mode. Includes OK button.
     */
    EMBED
  };

  /**
   * Constructor.
   *
   * @param parent Parent.
   * @param mode Mode of operation.
   * @param obs Enclosing application, if not embedded mode.
   *
   * Call init() once the first document is ready.
   */
  Window(wxWindow* parent, const Mode mode, WindowObserver* obs);

  /**
   * Destructor.
   */
  ~Window();

  /**
   * Open file.
   */
  virtual Document* open(const wxString& file) = 0;

  /**
   * Open file and add to window.
   */
  void openAndAdd(const wxString& file);
  
  /**
   * Restore default layout.
   */
  virtual void layoutDefault() = 0;

  /**
   * Set model.
   *
   * @param model Set model.
   */
  virtual void setModel(Model* model) = 0;

  /**
   * Is a pane shown?
   *
   * @param i Index of the pane.
   *
   * @return True if the pane is shown, false otherwise.
   */
  virtual bool isShown(const unsigned i) = 0;

  /**
   * Show a pane.
   *
   * @param i Index of the pane.
   * @param show True to show the pane, false to hide it.
   */
  virtual void show(const unsigned i, const bool show) = 0;

  /**
   * Set exiting status. Used predominantly under Mac to indicate whether the
   * closing of the window should also terminate the program.
   *
   * @param exiting Is the program exiting?
   */
  void setExiting(const bool exiting = true);
    
  /**
   * Get return code.
   */
  int GetReturnCode();

  /**
   * Set return code.
   */
  void SetReturnCode(const int returnCode);

  /**
   * Close event.
   */
  void OnSystemClose(wxCloseEvent& evt);

  /**
   * Drag and drop permission event.
   */
  void OnAllowNotebookDnD(wxAuiNotebookEvent& evt);

  /**
   * Menu opened event.
   */
  void OnMenuOpen(wxMenuEvent& evt);

  /**
   * Open button event.
   */
  void OnButtonOpen(wxCommandEvent& evt);

  /**
   * Open recent menu event.
   */
  void OnOpenRecent(wxCommandEvent& evt);

  /**
   * Clear recent menu event.
   */
  void OnClearRecent(wxCommandEvent& evt);

  /**
   * Save button event.
   */
  void OnButtonSave(wxCommandEvent& evt);

  /**
   * Save As button event.
   */
  void OnButtonSaveAs(wxCommandEvent& evt);

  /**
   * Close button event.
   */
  void OnButtonClose(wxCommandEvent& evt);

  /**
   * Exit button event.
   */
  void OnButtonExit(wxCommandEvent& evt);

  /**
   * OK button event.
   */
  void OnButtonOk(wxCommandEvent& evt);

  /**
   * Cancel button event.
   */
  void OnButtonCancel(wxCommandEvent& evt);

  /*
   * Toolbar button events.
   */
  void OnButtonAbout(wxCommandEvent& evt);
  void OnButtonRegister(wxCommandEvent& evt);
  void OnButtonWindow(wxCommandEvent& evt);
  void OnButtonOverlay(wxCommandEvent& evt);
  void OnButtonHelp(wxCommandEvent& evt);
  void OnButtonDefault(wxCommandEvent& evt);
  void OnButZoomOut(wxCommandEvent& evt);
  void OnButZoomIn(wxCommandEvent& evt);
  void OnButZoomNormal(wxCommandEvent& evt);
  void OnButZoomFit(wxCommandEvent& evt);
  void OnPageClose(wxAuiNotebookEvent& evt);
  void OnPageChanged(wxAuiNotebookEvent& evt);

  /*
   * Window menu events.
   */
  void OnSelect(wxCommandEvent& evt);
  void OnDefault(wxCommandEvent& evt);

protected:
  /**
   * Initialise.
   *
   * @param doc Document to have open initially.
   */
  void init(Document* doc = NULL);

  /**
   * Initialise panes.
   */
  virtual void initPanes();

  /**
   * Enable or disable menu and toolbar items as appropriate.
   */
  virtual void enable();

  /**
   * Enable or disable menu and toolbar.
   *
   * @param enable True to enable, false to disable.
   */
  virtual void enable(const bool enable);

  /**
   * Add pane to window menu.
   */
  void addPane(const wxString& name);

  /**
   * Enclosing application.
   */
  WindowObserver* obs;

  /**
   * Return code.
   */
  int returnCode;

  /**
   * Is program exiting?
   */
  bool exiting;
  
  /**
   * Mode of operation.
   */
  Mode mode;

  /**
   * AUI manager.
   */
  wxAuiManager* manager;

  /**
   * Main pane for AUI.
   */
  wxPanel* main;

  /**
   * Notebook, for standalone mode.
   */
  wxAuiNotebook* notebook;

  /**
   * Document, for embedded mode.
   */
  Document* doc;

  /**
   * Info bar.
   */
  InfoBar* infoBar;
  
  /*
   * Menus.
   */
  OpenRecentMenu* mnRecent;
  WindowMenu* mnWindow;
  WindowMenu* mnPopupWindow;
  wxMenu* mnFile;
  wxMenu* mnEdit;
  wxMenu* mnView;
  wxMenu* mnHelp;

  /*
   * Control ids.
   */
  enum ID {
    ID_BUT_DEFAULT = 1, // can't have menu item id of 0 on Mac
    ID_BUT_REGISTER,
    ID_BUT_WINDOW,
    ID_BUT_OVERLAY,

    /* these must match those given in OpenRecentMenu */
    ID_CLEAR_RECENT = 764,
    ID_OPEN_RECENT,

    /* these must match those given in WindowMenu */
    ID_SELECT = 1432,
    ID_DEFAULTS = 2758
  };

  /**
   * Extra height for floating panels.
   */
  #ifdef __WXOSX__
  static const int EXTRA_PANE_WIDTH = 0;
  static const int EXTRA_PANE_HEIGHT = 16;
  #else
  static const int EXTRA_PANE_WIDTH = 0;
  static const int EXTRA_PANE_HEIGHT = 0;
  #endif
  static const int MIN_THUMB_PANE_WIDTH = 387;

  DECLARE_EVENT_TABLE()
};
}

inline void indii::Window::setExiting(const bool exiting) {
  this->exiting = exiting;
}

#endif
