/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 500 $
 * $Date: 2013-08-16 19:04:13 +0800 (Fri, 16 Aug 2013) $
 */
#ifndef INDII_GUI_PREVIEWIMAGE_HPP
#define INDII_GUI_PREVIEWIMAGE_HPP

#include "../image/ImageManipulation.hpp"
#include "../image/ImageResource.hpp"
#include "../model/Model.hpp"
#include "../model/Controller.hpp"
#include "../model/Observer.hpp"

#include "wx/scrolwin.h"
#include "wx/bitmap.h"
#include "wx/dcclient.h"

namespace indii {
/**
 * Image preview.
 */
class PreviewImage : public wxScrolledWindow, public Observer {
public:
  /**
   * Constructor.
   *
   * @param parent Parent window.
   * @param res Image resource manager. Caller retains ownership.
   * @param model Model. Caller retains ownership.
   * @param control Controller. Caller retains ownership.
   */
  PreviewImage(wxWindow* parent, ImageResource* res, Model* model,
      Controller* control = NULL);

  /**
   * Zoom in.
   */
  void zoomIn();

  /**
   * Zoom out.
   */
  void zoomOut();

  /**
   * Zoom to 1:1
   */
  void zoomNormal();

  /**
   * Zoom to fit.
   */
  void zoomFit();

  /**
   * Is overlay on?
   */
  bool getOverlay();

  /**
   * Set overlay on or off.
   */
  void showOverlay(const bool on);

  /**
   * Refresh view.
   */
  void refresh();

  /**
   * Clear the preview on next refresh (used under Windows to minimise
   * flickering)?
   */
  void clear(const bool on = true);

  /*
   * Model events.
   */
  virtual void notify();
    
  /**
   * Handle paint event.
   */
  void OnPaint(wxPaintEvent& evt);

  /**
   * Handle resize event.
   */
  void OnSize(wxSizeEvent& evt);

  /*
   * Mouse events.
   */ 
  void OnMouseDown(wxMouseEvent& evt);
  void OnMouseUp(wxMouseEvent& evt);
  void OnMouseDoubleClick(wxMouseEvent& evt);
  void OnMouseWheel(wxMouseEvent& evt);
  void OnMouseMove(wxMouseEvent& evt);
  void OnEraseBackground(wxEraseEvent& evt);

  /**
   * Context sensitive help.
   */
  void OnHelp(wxHelpEvent& evt);

private:
  /**
   * Get location of mouse event.
   *
   * @param evt Mouse event.
   * @param x0 X-coordinate on original image.
   * @param y0 Y-coordinate on original image.
   * @param x X-coordinate on zoomed image.
   * @param y Y-coordinate on zoomed image.
   */
  void getLocation(wxMouseEvent& evt, int* x0, int* y0, int* x, int* y);

    /**
   * Image resource manager.
   */
  ImageResource* res;

  /**
   * Model.
   */
  Model* model;

  /**
   * Controller.
   */
  Controller* control;

  /**
   * Zoom numerator.
   */
  int zoomNumerator;

  /**
   * Zoom denominator.
   */
  int zoomDenominator;

  /**
   * Should overlay be displayed?
   */
  bool overlay;
  
  /**
   * Device context should be cleared before next paint (used to control
   * flicker, only clear device context when not doing so may be a problem).
   */
  bool doClear;

  /*
   * Dragging variables.
   */
  int mouseX, mouseY;

  DECLARE_EVENT_TABLE()
};
}

inline bool indii::PreviewImage::getOverlay() {
  return this->overlay;
}

inline void indii::PreviewImage::showOverlay(const bool on) {
  this->overlay = on;
}

#endif
