/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 517 $
 * $Date: 2013-09-30 23:33:29 +0800 (Mon, 30 Sep 2013) $
 */
#include "ChannelMixerPane.hpp"

#include "Help.hpp"

#include "wx/wx.h"
#include "wx/gbsizer.h"

using namespace indii;

ChannelMixerPane::ChannelMixerPane(wxWindow* parent) :
    Pane(parent), model(NULL) {
  static const int PADDING = 4;

  wxGridBagSizer* sizer = new wxGridBagSizer();

  slideRed = new wxSlider(this, ID_SLIDE_RED, 76, 0, 255,
      wxDefaultPosition, wxSize(100,-1));
  slideGreen = new wxSlider(this, ID_SLIDE_GREEN, 150, 0, 255,
      wxDefaultPosition, wxSize(100,-1));
  slideBlue = new wxSlider(this, ID_SLIDE_BLUE, 29, 0, 255,
      wxDefaultPosition, wxSize(100,-1));

  wxStaticText *labelRed, *labelGreen, *labelBlue;
  labelRed = new wxStaticText(this, -1, _("Red:"));
  labelGreen = new wxStaticText(this, -1, _("Green:"));
  labelBlue = new wxStaticText(this, -1, _("Blue:"));

  sizer->Add(labelRed, wxGBPosition(0,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelGreen, wxGBPosition(1,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelBlue, wxGBPosition(2,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT|wxBOTTOM, PADDING);

  sizer->Add(slideRed, wxGBPosition(0,1), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxTOP|wxRIGHT, PADDING);
  sizer->Add(slideGreen, wxGBPosition(1,1), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxTOP|wxRIGHT, PADDING);
  sizer->Add(slideBlue, wxGBPosition(2,1), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxTOP|wxRIGHT|wxBOTTOM, PADDING);

  sizer->SetFlexibleDirection(wxHORIZONTAL);
  sizer->AddGrowableCol(1);
  SetMinSize(wxSize(240,-1));
  SetSizerAndFit(sizer);
  setModel(NULL);
}

void ChannelMixerPane::setModel(Model* model) {
  this->model = model;
  ignore();

  Enable(model != NULL);
  slideRed->Enable(model != NULL);
  slideGreen->Enable(model != NULL);
  slideBlue->Enable(model != NULL);
  
  if (model != NULL) {
    watch(model);
    notifyGreyscaleChange();
  }
}

void ChannelMixerPane::notifyGreyscaleChange() {
  /* pre-condition */
  assert (model != NULL);

  int r = intround(model->getRedMix()*255.0f);
  int g = intround(model->getGreenMix()*255.0f);
  int b = intround(model->getBlueMix()*255.0f);

  if (slideRed->GetValue() != r) {
    slideRed->SetValue(r);
    slideRed->Enable(r != slideRed->GetMax());
  }
  if (slideGreen->GetValue() != g) {
    slideGreen->SetValue(g);
    slideGreen->Enable(g != slideGreen->GetMax());
  }
  if (slideBlue->GetValue() != b) {
    slideBlue->SetValue(b);
    slideBlue->Enable(b != slideBlue->GetMax());
  }
}

void ChannelMixerPane::OnSlideRed(wxScrollEvent& evt) {
  if (model != NULL) {
	  float r = slideRed->GetValue() / 255.0f;
 	  model->setRedMix(r);
  }
}

void ChannelMixerPane::OnSlideGreen(wxScrollEvent& evt) {
  if (model != NULL) {
    float g = slideGreen->GetValue() / 255.0f;
    model->setGreenMix(g);
  }
}

void ChannelMixerPane::OnSlideBlue(wxScrollEvent& evt) {
  if (model != NULL) {
    float b = slideBlue->GetValue() / 255.0f;
    model->setBlueMix(b);
  }
}

void ChannelMixerPane::OnHelp(wxHelpEvent& evt) {
  Help::show(550);
}

BEGIN_EVENT_TABLE(ChannelMixerPane, Pane)
EVT_COMMAND_SCROLL(ID_SLIDE_RED, ChannelMixerPane::OnSlideRed)
EVT_COMMAND_SCROLL(ID_SLIDE_GREEN, ChannelMixerPane::OnSlideGreen)
EVT_COMMAND_SCROLL(ID_SLIDE_BLUE, ChannelMixerPane::OnSlideBlue)
EVT_HELP(wxID_ANY, ChannelMixerPane::OnHelp)
END_EVENT_TABLE()
