/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 515 $
 * $Date: 2013-09-29 19:04:13 +0800 (Sun, 29 Sep 2013) $
 */
#include "App.hpp"

#include "Registration.hpp"
#include "About.hpp"
#include "Help.hpp"
#include "OpenRecentMenu.hpp"
#include "../reg/Components.hpp"
#include "../image/FileImageResource.hpp"
#include "../images/indii.hpp"
#include "../../gnome/gnome.hpp"
#include "../meta.hpp"
#include "../image/ImageManipulation.hpp"

#include "wx/filesys.h"
#include "wx/fs_arc.h"
#include "wx/stdpaths.h"

#ifdef __WXMSW__
#include "omp.h"
#endif

using namespace indii;

App::App() : window(NULL), haveFile(false) {
  //
}

void App::MacOpenFiles(const wxArrayString& fileNames) {
  wxLogNull logNo;
  if (window != NULL) {
    unsigned i;
    for (i = 0; i < fileNames.GetCount(); ++i) {
      if (wxImage::CanRead(fileNames[i])) {
        window->openAndAdd(fileNames[i]);
      } else {
        wxMessageBox(_("The given file is of an unsupported type."),
            _("Unsupported type"), wxOK|wxICON_ERROR, NULL);
      }
    }
  }
}

bool App::OnInit() {
  return wxApp::OnInit(); // for command line parsing
}

void App::OnInitCmdLine(wxCmdLineParser& parser) {
  wxApp::OnInitCmdLine(parser);
  parser.AddParam(_("file name"), wxCMD_LINE_VAL_STRING, wxCMD_LINE_PARAM_OPTIONAL);
}

bool App::OnCmdLineParsed(wxCmdLineParser& parser) {
  if (wxApp::OnCmdLineParsed(parser)) {
    #ifndef __WXOSX__
    /* handled by MacOpenFiles() on Mac */
    if (parser.GetParamCount() > 0) {
      file = parser.GetParam(0);
      haveFile = true;
    }
    #endif
    return true;
  } else {
    return false;
  }
}

int App::OnRun() {
  #if defined(__WXOSX__) && defined(MAC_APP_STORE)
  if (!Components::isActivated(Components::APP)) {
    exit(173);
    return false;
  }
  #endif

  splash();
  wxInitAllImageHandlers();
  indii_initialize_images();
  gnome_initialize_images();
  
  #ifdef __WXOSX__
  wxApp::SetExitOnFrameDelete(false);
  //wxApp::SetMacPreferencesMenuItemId(wxID_NONE);
  wxApp::s_macPreferencesMenuItemId = wxID_NONE;
  
  /* permanent menu for Mac */
  wxMenuBar* mbMenu = new wxMenuBar();
  mnRecent = new OpenRecentMenu();
  
  wxMenu* mnFile = new wxMenu();
  mnFile->Append(wxID_OPEN, _("&Open...\tCtrl-O"));
  #ifndef MAC_APP_STORE
  mnFile->AppendSubMenu(mnRecent, _("Open Recent"));
  #endif
  mnFile->Append(wxID_EXIT); // moved to program menu
  
  wxMenu* mnHelp = new wxMenu();
  mnHelp->Append(wxID_HELP, _("&Help\tCtrl-?"));
  mnHelp->Append(wxID_ABOUT);
  if (!Components::isActivated(Components::APP)) {
    mnHelp->Append(ID_BUT_REGISTER, _("Register"));
  }
  
  mbMenu->Append(mnFile, _("&File"));
  mbMenu->Append(mnHelp, _("&Help"));
  
  wxMenuBar::MacSetCommonMenuBar(mbMenu);
  #endif
  
  window = open();
  Help::init(window);
  SetTopWindow(window);
  window->Show();
  window->Raise();

  if (haveFile) {
    wxLogNull logNo;
    if (wxImage::CanRead(file)) {
      window->openAndAdd(file);
    } else {
      wxMessageBox(_("The given file is of an unsupported type."),
          _("Unsupported type"), wxOK|wxICON_ERROR, NULL);
      haveFile = false;
    }
  }

  if (!Components::isActivated(Components::APP)) {
    if (!Components::isTrial(Components::APP)) {
      Registration* dlgRegister = new Registration(window, Components::APP);
      dlgRegister->Centre();
      if (dlgRegister->ShowModal() != wxID_OK) {
        dlgRegister->Destroy();
        window->Close(true);
        window = NULL;
        return false;
      } else {
        dlgRegister->Destroy();
        return wxApp::OnRun();
      }
    } else if (Components::daysLeft() <= Components::maxDays()/3) {
      Registration* dlgRegister = new Registration(window, Components::APP);
      dlgRegister->Centre();
      dlgRegister->ShowModal();
      dlgRegister->Destroy();
      return wxApp::OnRun();
    } else {
      return wxApp::OnRun();
    }
  } else {
    return wxApp::OnRun();
  }
}

int App::OnExit() {
  return wxApp::OnExit();
}

void App::OnButtonOpen(wxCommandEvent& evt) {
  if (window != NULL) {
    window->OnButtonOpen(evt);
  } else {
    wxString openPath;
    wxConfig config(_(PACKAGE_TARNAME), _("indii.org"));
    if (!config.Read(_("OpenPath"), &openPath)) {
      openPath = wxStandardPaths::Get().GetDocumentsDir();
    }

    wxString path = wxFileSelector(wxFileSelectorPromptStr, openPath,
        wxEmptyString, wxEmptyString, wildcards(),
        wxFD_OPEN|wxFD_CHANGE_DIR|wxFD_FILE_MUST_EXIST|wxFD_PREVIEW, NULL);
    if (!path.empty()) {
      wxLogNull logNo;
      if (wxImage::CanRead(path)) {
        wxFileName filename(path);
        if (filename.IsOk() && filename.FileExists()) {
          window = open(path);
          SetTopWindow(window);
          window->CentreOnScreen();
          window->Show();
          window->Raise();

          openPath = filename.GetPath();
          config.Write(_("OpenPath"), openPath);
          config.Flush();

          OpenRecentMenu::AddFile(path);
        } else {
          wxMessageBox(_("The selected file does not exist. Please try another."),
                       _("File does not exist"), wxOK|wxICON_ERROR, NULL);
          OpenRecentMenu::RemoveFile(path);
        }
      } else {
        wxMessageBox(_("The selected file is of an unsupported type. Please try another."),
                    _("Unsupported type"), wxOK|wxICON_ERROR, NULL);
      }
    }
  }
}

void App::OnOpenRecent(wxCommandEvent& evt) {
  if (window != NULL) {
    window->OnOpenRecent(evt);
  } else {
    int id = evt.GetId();
    if (id >= 300 && id < 310) {
      wxString path(OpenRecentMenu::GetFile(id));
      if (wxImage::CanRead(path)) {
        wxFileName filename(path);
        if (!path.empty() && filename.IsOk() && filename.FileExists()) {
          window = open(path);
          SetTopWindow(window);
          window->CentreOnScreen();
          window->Show();
          window->Raise();

          OpenRecentMenu::AddFile(path);
        } else {
          wxMessageBox(_("The selected file does not exist. Please try another."),
                       _("File does not exist"), wxOK|wxICON_ERROR, NULL);
          OpenRecentMenu::RemoveFile(path);
        }
      } else {
        wxMessageBox(_("The selected file is of an unsupported type. Please try another."),
                     _("Unsupported type"), wxOK|wxICON_ERROR, NULL);
        OpenRecentMenu::RemoveFile(path);
      }
    }
  }
}

void App::OnButtonAbout(wxCommandEvent& evt) {
  showAboutDialog();
}

void App::OnButtonRegister(wxCommandEvent& evt) {
  if (window == NULL) {
    if (Components::isActivated(Components::APP)) {
      wxMessageBox(_("Your " PACKAGE_NAME " app is already registered."), _("Already registered"), wxICON_INFORMATION);
    } else {
      Registration* dlgRegister = new Registration(NULL, Components::APP);
      dlgRegister->Centre();
      dlgRegister->ShowModal();
      dlgRegister->Destroy();
    }
  } else {
    window->OnButtonRegister(evt);
  }
}

void App::OnButtonHelp(wxCommandEvent& evt) {
  Help::show();
}

void App::OnButtonExit(wxCommandEvent& evt) {
  if (window != NULL) {
    window->OnButtonExit(evt);
  } else {
    #ifdef __WXOSX__
    ExitMainLoop();
    #endif
  }
}

void App::notifyClose(Window* window, const bool exiting) {
  assert (this->window == window);
  
  this->window = NULL;
  #ifdef __WXOSX__
  if (exiting) {
    ExitMainLoop();
  }
  #endif
}

BEGIN_EVENT_TABLE(App, wxApp)
#ifdef __WXOSX__
EVT_MENU(wxID_OPEN, App::OnButtonOpen)
EVT_MENU(wxID_ABOUT, App::OnButtonAbout)
EVT_MENU(wxID_HELP, App::OnButtonHelp)
EVT_MENU(ID_BUT_REGISTER, App::OnButtonRegister)
EVT_MENU(wxID_EXIT, App::OnButtonExit)

EVT_MENU(300, App::OnOpenRecent)
EVT_MENU(301, App::OnOpenRecent)
EVT_MENU(302, App::OnOpenRecent)
EVT_MENU(303, App::OnOpenRecent)
EVT_MENU(304, App::OnOpenRecent)
EVT_MENU(305, App::OnOpenRecent)
EVT_MENU(306, App::OnOpenRecent)
EVT_MENU(307, App::OnOpenRecent)
#endif
END_EVENT_TABLE()
