/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 500 $
 * $Date: 2013-08-16 19:04:13 +0800 (Fri, 16 Aug 2013) $
 */
#ifndef INDII_MODEL_CONTROLLER_HPP
#define INDII_MODEL_CONTROLLER_HPP

#include "wx/event.h"

namespace indii {
/**
 * Controller for handling mouse events from PreviewImage.
 */
class Controller {
public:
  /**
   * Constructor.
   */
  Controller();
  
  /**
   * Set overlay on or off.
   */
  void showOverlay(const bool on);
  
  /**
   * Called when the left mouse button is released on the preview image. To
   * be handled as the model sees fit.
   *
   * @param x0 X-coordinate of the pixel clicked on original scale.
   * @param y0 Y-coordinate of the pixel clicked on original scale.
   * @param x X-coordinate of the pixel clicked.
   * @param y Y-coordinate of the pixel clicked.
   * @param evt Associated mouse event.
   */
  virtual void up(const int x0, const int y0, const int x, const int y,
      wxMouseEvent& evt);

  /**
   * Called when the left mouse button is clicked down on the preview image.
   * To be handled as the model sees fit.
   *
   * @param x0 X-coordinate of the pixel clicked on original scale.
   * @param y0 Y-coordinate of the pixel clicked on original scale.
   * @param x X-coordinate of the pixel clicked.
   * @param y Y-coordinate of the pixel clicked.
   * @param evt Associated mouse event.
   */
  virtual void down(const int x0, const int y0, const int x, const int y,
      wxMouseEvent& evt);

  /**
   * Called when the left mouse button is double-clicked down on the preview
   * image. To be handled as the model sees fit.
   *
   * @param x0 X-coordinate of the pixel clicked on original scale.
   * @param y0 Y-coordinate of the pixel clicked on original scale.
   * @param x X-coordinate of the pixel clicked.
   * @param y Y-coordinate of the pixel clicked.
   * @param evt Associated mouse event.
   */
  virtual void doubleDown(const int x0, const int y0, const int x,
      const int y, wxMouseEvent& evt);

  /**
   * Called when the left mouse button is down and the mouse cursor is dragged
   * across the preview image. To be handled as the model sees fit.
   *
   * @param x0 X-coordinate of the pixel clicked on original scale.
   * @param y0 Y-coordinate of the pixel clicked on original scale.
   * @param x X-coordinate of the pixel clicked.
   * @param y Y-coordinate of the pixel clicked.
   * @param dx0 Change in original x-coordinate since last drag event.
   * @param dy0 Change in original y-coordinate since last drag event.
   * @param dx Change in x-coordinate since last drag event.
   * @param dy Change in y-coordinate since last drag event.
   * @param evt Associated mouse event.
   */
  virtual void drag(const int x0, const int y0, const int x, const int y,
      const int dx0, const int dy0, const int dx, const int dy,
      wxMouseEvent& evt);
  
protected:
  /**
   * Show overlay?
   */
  bool overlay;
};
}

inline indii::Controller::Controller() : overlay(true) {
  //
}

inline void indii::Controller::showOverlay(const bool on) {
  this->overlay = on;
}

inline void indii::Controller::up(const int x0, const int y0, const int x,
    const int y, wxMouseEvent& evt) {
  evt.Skip();
}

inline void indii::Controller::down(const int x0, const int y0, const int x,
    const int y, wxMouseEvent& evt) {
  evt.Skip();
}

inline void indii::Controller::doubleDown(const int x0, const int y0, const int x,
    const int y, wxMouseEvent& evt) {
  evt.Skip();
}

inline void indii::Controller::drag(const int x0, const int y0, const int x,
    const int y, const int dx0, const int dy0, const int dx, const int dy,
    wxMouseEvent& evt) {
  evt.Skip();
}

#endif
